(* This file contains various utility functions which are used elsewhere *)

val status = ref OS.Process.failure
fun quit() = OS.Process.exit (!status)
(* Sometimes (eg, if we're using "make" for automated testing)
   we want to keep going after failure; use option -k to 
   set status to OS.Process.success. *)

local
    val baseName = ref ""
in
    fun setBaseName s = baseName := s
    fun getBaseName () = !baseName
    fun innerClassName n = (!baseName) ^ "$" ^ n
end

local
    val verbose = ref false
    val debug = ref false
in
    fun vprint x = if !verbose then TextIO.print x else ()
    fun setVerbose t = verbose := t
    fun fprint x = TextIO.print x before TextIO.flushOut TextIO.stdOut
    fun debugPrint x = if !debug then fprint x else ()
    fun debugPrintln x = if !debug then fprint (x^"\n") else ()
    fun setDebug t = debug := t
end

fun makeFullFilename dir base ext =
    Path.joinDirFile
       {dir=dir,
	file=Path.joinBaseExt {base=base, ext=SOME ext}}
(* eg "camelot/src" "Util" "sml" -> "camelot/src/Util.sml",  taking care of trailing slashes etc. *)

fun extend filename extn = (* Add extension if necessary *)
   let
       val e = Path.ext filename;
   in
       if e = SOME extn
       then
	   filename
       else
	   Path.joinBaseExt {base=filename, ext=SOME extn}
   end


fun id x = x

fun member _ [] = false
  | member x (h::t) = x=h orelse member x t;

fun memberL _ [] = false
  | memberL x ((a,_)::t) = x=a orelse memberL x t;

fun makeSet [] = [] (* Don't like this *)
  | makeSet (h::t) =
	if member h t then makeSet t else h :: (makeSet t)

infixr 5 ::?

fun NONE ::? l = l
  | (SOME x) ::? l = x::l

fun fst (x,_) = x;
fun snd (_,y) = y;

fun appsnd f (a,b) = (a,f b)
fun mapmap f = map (map f)

fun truncate s n = String.substring (s , 0, size s - n)
fun chop s c = hd (String.tokens (fn a => a=c) s)


fun println s = print (s^"\n")

fun listToString toString separator l =
case l of [] => ""
        | [h] => toString h
        | h::t => (toString h) ^ separator ^ (listToString toString separator t)

fun plural n s =
    let
	val n' = Int.toString n
    in
	if n=1 then n' ^ " " ^ s else n' ^ " " ^ s ^ "s"
    end

fun fillString s n = (* Pad string with spaces up to length n;  not very efficient *)
    if n <= String.size s then s
    else fillString (s^" ") n

exception zip3error

fun zip3 ([], [], []) = []
  | zip3 (h1::t1, h2::t2, h3::t3) = (h1, h2, h3)::zip3(t1,t2,t3)
  | zip3 _ = raise zip3error

fun unzip3 [] = ([],[],[])
  | unzip3 ((h1,h2,h3)::t) =
    let
	val (l1, l2, l3) = unzip3 t
    in
	(h1::l1, h2::l2, h3::l3)
    end

fun unzip4 [] = ([],[],[],[])
  | unzip4 ((h1,h2,h3,h4)::t) =
    let
	val (l1, l2, l3, l4) = unzip4 t
    in
	(h1::l1, h2::l2, h3::l3, h4::l4)
    end

fun max x y = if x < y then y else x;

fun qualified x =
    List.exists (fn a => a = #".") (String.explode x)

fun realToString r = String.translate (fn #"~" => "-" | c => Char.toString c) (Real.toString r)

(* Stuff moved here from Type.sml *)
local
    open Absyn
in

fun tyListToArrowTy [] = UNITty
  | tyListToArrowTy [ty] = ty
  | tyListToArrowTy (ty::tys) = ARROWty(ty, tyListToArrowTy tys)

fun arrowTyToTyList (ARROWty(ty, ty')) =
    ty :: (arrowTyToTyList ty')
  | arrowTyToTyList x = [x]


(* Extract names from argument list, discarding unit arguments *)
fun getArgNames [] = []
  | getArgNames (h::t) =
    case h of
	UNITvar => getArgNames t
      | VAR (name,_) => name::(getArgNames t)
end

fun nameOf (n,_) = n

local
   val errtup = ref NONE : (string * BasicIO.instream * Lexing.lexbuf) option ref
   open Absyn
   val () = Absyn.required
in
   fun errPrint s = TextIO.output(TextIO.stdErr, s)

   fun setErr (str, stream, buf) = errtup := SOME (str, stream, buf)
   fun closeInStream () =
       case (!errtup) of
	   NONE => errPrint ("Error: failed to close input stream")
	 | SOME (_,stream,_) => BasicIO.close_in stream;


fun locToString (Loc.Loc(a,b)) =
    "<" ^ Int.toString a ^ "," ^ Int.toString b ^ ">"

fun getLoc a = Normsyn.getLoc a
(*
(* Should be in NAsyntfn,  but we get circular dependencies *)
    case a of
	LOC l => l
      | MONO (_,a) => getLoc a
      | PHI (_,a) => getLoc a
*)

and errLocation x u =
    let val loc = getLoc u
    in
	if loc = Loc.nilLocation then errPrint "Undetermined location\n"
	else  Loc.errLocation x loc
    end

and error l s =
    let val () = case (!errtup) of
		     NONE => errPrint ("Error: " ^ s ^ "\n")
		   | SOME x => (errLocation x l;  Loc.errPrompt (s^"\n"));
    in
	quit()
    end


val nowarn = ref false
fun setWarn b = nowarn := not b

fun warn l s =
    if !nowarn then () else
    let val () = case (!errtup) of
		     NONE => errPrint ("Warning: " ^ s ^ "\n")
		   | SOME x => (errLocation x l;  Loc.errPrompt
						      ("Warning: "^s^"\n"));
    in
	()
    end

fun errors l = errors' l 0
and errors' ((l,s)::t) n =
    let val () = case (!errtup) of
		     NONE => errPrint ("Error: " ^ s ^ "\n")
		   | SOME x => (errLocation x l;  Loc.errPrompt (s^"\n\n"));
    in
        errors' t (n+1)
    end
  | errors' [] n = (errPrint ((Int.toString n) ^ " errors.\n"); quit())

fun exit s =
    let
	val () = errPrint ("Camelot error: " ^ s ^ "\n")
    in
	quit()
    end

fun ierror s =
    let
	val () = errPrint ("Internal error " ^ s ^ "\n")
    in
	quit()
    end

end

