(* Much of this file from Pierce's fullrecon example; see *)
(* http://www.cis.upenn.edu/~bcpierce/tapl/checkers/      *)

infixr  5 ::?

val L = Loc.Loc (1,5)

local
    open Normsyn
    open Util
    open Env
    val () = Env.required
    val typeToString = NAsyntfn.typeToString
    val getU =  NAsyntfn.getU
    val getvU = NAsyntfn.getvU
    val typeToString = NAsyntfn.typeToString
in

fun t_error s = Util.ierror ("[Type.sml]: " ^ s)

fun id x = x

fun locOf a =
    case a of
	LOC l => a
      | MONO (_,a) => locOf a
      | PHI (_,a) => locOf a

val dbgConstraints = ref false
fun setDebugConstraints b = dbgConstraints := b


(* ConBind - add Ty list for type, e.g. 'a tree of Empty | ... would be *)
(* 'a tree of Empty * | ...  :   ("Empty", ConBind("tree", ["'a"], [])) *)

datatype binding = VarBind of  Ty * Instance * Annotation
                 | ConBind of string * Ty list * Ty list
                 | ClassBind of Ty * context * Annotation

withtype context = (string * binding) list

fun constraintToString (x, y, l) =
    typeToString x ^ " = "
    ^ typeToString y ^ " "
(*    ^ Util.locToString (Util.getLoc l)*)

fun constraintListToString l = (listToString constraintToString "\n" l) ^ "\n"

local
    fun f pad l =
	case l of
	    [] => ""
	  | (s,b)::tl =>
	    let in
		case b of
		    VarBind (t,i,_) => pad ^ "{V} " ^ s ^ ":"
				       ^ (typeToString t) ^ "\n" ^ f pad tl
		  | ConBind (n,ttys,ctys) =>
		    pad ^ "{D} "
		    ^ let in case ttys of
				 [] => ""
			       | _ => (listToString typeToString " " ttys) ^ " "
		      end
		    ^  n ^ " = " ^ s
		    ^ let in case ctys of
				 [] => ""
			       | _ => " of " ^ (listToString typeToString " * " ctys)
		      end
		    ^ "\n" ^ f pad tl
		  | ClassBind (c,ctx,_) => pad ^ "{C} " ^ s ^ ": [\n"
					   ^ f (pad ^ " ") ctx ^ pad ^ "]\n"
					  ^ f pad tl
	    end
in
   fun ctxToString c = f "" c
end

fun constraintError loc t1 t2 s = (* s unused but informative *)
let val spec = 
	case t2 of 
	    TVARty X => if isOrd X then "ordered "
			else if isEq X then "equality "
			else ""
	  | _ => ""
in
    error loc
          ( "Type clash: an expression of type\n!     " ^
 	    typeToString t1 ^
            "\n! was found in a context requiring "
	    ^ spec  ^ "type\n!     " ^
	    typeToString t2  ^ "\n[" ^ s ^ "]")(* *)
end
fun subclassError loc t1 t2 =
    error loc
          ( "Type clash: an expression of type\n!     " ^
 	    typeToString t1 ^
            "\n! was found in a context requiring a subclass of type\n!     " ^
	    typeToString t2 )



(* Rather informal - but see Pierce p325 *)
(* Second, imperative, context for class types we read in *)
local
    val x = ref 0
in

fun newTvar () =
    let
	val () = x:= !x+1
	val v = "'" ^ Int.toString (!x)
    in
	TVARty {name=v, eq=ref false, ord=ref false} 
    end

(* If we just use newTvar later we can generate monomorphised versions
   of functions for illegal types (eg (f x y = x <y): bool -> bool -> bool).
   However, trying to copy the equality/order status doesn't seem to work
   either. *)

fun newTvar' (eq,ord) = 
    let
	val () = x:= !x+1
	val v = "'" ^ Int.toString (!x)
    in
	TVARty {name=v, eq=ref(!eq),  ord=ref(!ord)}  (* copy refs or values ?? *)
	(* Interpreter.cmlt doesn't work if you use eq=eq, ord=ord *)
    end
end

local
    val class_ctx = ref ([]:context)
in
fun addClassBindings ctx' = class_ctx := ctx'@(!class_ctx)
fun getClassCtx () = !class_ctx
fun ctxlength ctx = List.length ctx
fun addbinding ctx x bind = (x,bind)::ctx
fun getbinding ctx x = List.find (fn (y,_)=>x=y) (ctx@(!class_ctx))
fun getbindings ctx x = List.filter (fn (y,_)=>x=y) (ctx@(!class_ctx))

fun getfbind1 ctx f ty =
    case ctx of [] => (print "Can't find it\n"; NONE)
	      | (g, b)::ctx' =>
		if f=g then
		    let
			val () = debugPrint("Looking for " ^ f ^ " with type " ^ typeToString ty ^ "\n")
		    in case b of
			   VarBind (ty',_,_) =>
			   (print ("Found " ^ g ^ " with type " ^ typeToString ty' ^ "\n");
			    if ty' = ty then
				SOME ty'
			    else getfbind1 ctx' f ty
			   )
			 | _ => getfbind1 ctx' f ty
		end
		else getfbind1 ctx' f ty

fun getfbind ctx f ty = (print ("Looking for " ^ f ^ " with type " ^ typeToString ty ^ "\n");
getfbind1 ctx f ty)
end

(* Adds a 0-argument initialiser to any class which has no initialiser. *)
(* FIX: this is really fixing the bug that user-defined classes rely on the default
   initialiser without saying so. This should be fixed by allowing proper
   initialiser definition. *)
fun addInit methods =
    if List.exists (fn VALdec (("<init>",_), _, _) => true | _ => false) methods
    then methods
    else (VALdec(("<init>",nowhere), ARROWty(UNITty, UNITty), INSTANCE))::methods

fun ctxify (VALdec((v,u),t,i)) = (v,VarBind (t,i,u))
  | ctxify (CLASSdec((c,u),s,intfs,t)) =
    (c,ClassBind (OBJECTty c, map ctxify (addInit t),u))

fun resolveClass cname =
    let
        val classes = ClassPath.getClassInfo cname
        val ctx = map ctxify classes
    in addClassBindings ctx end

fun nwh x = (x,nowhere)

fun getSuperclass cname =
    let
	val () = resolveClass cname
    in
	Option.map nwh (ClassPath.getSuperclass cname)
    end


fun uctxify (x, VarBind (ty,i,u)) = SOME (VALdec((x,u), ty,i))
  | uctxify (c, ClassBind (ty, ctx,u)) = (* FIX: dropping info. right way anyway? *)
     SOME (CLASSdec((c,u),  getSuperclass c, [], List.mapPartial id (map uctxify ctx)))
  | uctxify _ = NONE


(* Make type variables in ty generic by substituting fresh ones. Since
   the current ones may be repeated, we keep a list ass of old->fresh
   assignments *)

fun generify ty =
    let
	val ass = ref []

        fun g ty =
            case ty of
                TVARty v =>
		   let val t = #name v in
		       case List.find (fn (ty1,_) => ty1=t) (!ass) of
			   SOME (_, ty2) => ty2
			 | NONE =>
			   let
			       val v = newTvar'(#eq v, #ord v)
			       val () = ass := (t,v)::(!ass)
			   in
			       v
			   end
		   end
              | ARROWty(t1,t2) => ARROWty(g t1, g t2)
              | ARRAYty t => ARRAYty (g t)
              | PRODUCTty tys => PRODUCTty (map g tys)
              | CONty(tys, v) => CONty (map g tys, v)
              | _ => ty

        val ty' = g ty
    in
        (ty', !ass)
    end

(* Sort of illogical for constructors - is type of that constructor once applied *)
fun getVarTy cname ctx loc var =
    case getbinding ctx var of
        SOME (v,VarBind (t,_,_)) => t
      | SOME (v,ConBind (t, tys, _)) => CONty(tys, t)
      | SOME (v,ClassBind(t,ctx,_)) => t
      | NONE => error loc ("Unknown identfier: "^ var ) (*^  "in\n" ^ (ctxToString ctx))*)

(**** Object stuff ****)
datatype mname = INIT | MNAME of string

(* Get field type from class type, method name, and context *)
(* Class type must be fully resolved. *)
fun getFieldTy inst (OBJECTty cname) fname ctx loc =
    (resolveClass cname;
     case getbinding ctx cname of
         SOME (cname, ClassBind (classTy, cctx,_)) =>
         (case getbinding cctx fname of
              SOME (var,VarBind (t,i,_)) =>
	      (if i = inst then t
	       else
		   case inst of INSTANCE => error loc (fname ^ " is not an instance field")
			      | STATIC => error loc (fname ^ " is not a static value")
	      )
            | NONE =>
              (case getSuperclass cname of
                   SOME (cl,_) => getFieldTy inst (OBJECTty cl) fname ctx loc
                 | NONE => error loc ("Unknown field "^fname^" in "^ cname))
            | _ => error loc (fname^" not bound to field in "^cname))
       | _ => error loc ("No class binding for "
			 ^ cname
			 ^"\n"
			 ^ (ctxToString (getClassCtx()))))

  | getFieldTy inst ty fname ctx loc = error loc ("Couldn't resolve reference to field "^fname
						  ^": type " ^(typeToString ty)
						  ^ " does not resolve to class type")

(* Like the above, but get all types for mname in class and superclasses.  *)
(* Must still be checked for applicability with actual arguments *)
fun getMethodTypes cname mname ctx loc =
    let
	val methodName = case mname of INIT => "<init>" | MNAME x => x
	val () = resolveClass cname
	val () = debugPrint ("resolved "^cname^"\n")
	val cctx  = case getbinding ctx cname of
			SOME (cname, ClassBind (classTy, cctx,_)) => cctx
		      | _ => error loc ("No class binding for "
					^ cname
					^ "\n"
					^ (ctxToString (getClassCtx())))
	val tys = map (fn (_,VarBind (v,t,_)) => (v,t) | _ => t_error "getMethodTypes")
		      (getbindings cctx methodName)
	val sup_tys =
	    case mname of INIT => []
			| _ =>
			  (case getSuperclass cname of
			       SOME (cl,_) => getMethodTypes cl mname ctx loc
			     | NONE => []
			  )
	val () = debugPrint (methodName ^ ": " ^ (plural (length (tys@sup_tys)) "method" ) ^ "\n")
	val () = debugPrint (String.concat (map (fn (x,i) => (typeToString x ^ "\n")) (tys@sup_tys)))
    in
	tys@sup_tys
    end

fun typeIsSubclass t t' =
    let
        val (classes, truth) = ClassPath.typeIsSubclass t t'
        val ctx = map ctxify classes
        val () = addClassBindings ctx
    in truth end

fun typeSubsume (ty, superTy) =
    if ClassPath.isObjectType ty then typeIsSubclass ty superTy
    else ty = superTy
fun methodSubtype  ty1 ty2 =
    let
	fun dropLast xs = List.take(xs, (List.length xs) - 1)
	val (t1,t2) = (dropLast (arrowTyToTyList ty1), dropLast (arrowTyToTyList ty2))
    in
	length t1 = length t2 andalso List.all (typeSubsume) (ListPair.zip (t1,t2))
    end
fun applicableMethods ty cname mname instance ctx loc =
    map #1 (List.filter (fn (ty', i) => methodSubtype (conv ty) ty'
					andalso i = instance)
			(getMethodTypes cname mname ctx loc))

(* conv - result type thrown away *)
and conv [] = Util.exit "conv[]\n"
  | conv xs = tyListToArrowTy (xs@[OBJECTty "test"])

fun lessSpecific loc m1 m2 = methodSubtype m2 m1 andalso not (methodSubtype m1 m2)
fun maxSpecific loc methods = maxSpecificR loc methods []
and maxSpecificR loc (m::methods) acc =
    maxSpecificR loc (List.filter (lessSpecific loc m) methods) (m::acc)
  | maxSpecificR loc [] acc = maxSpecificL loc acc []
and maxSpecificL loc (m::methods) acc =
    maxSpecificL loc (List.filter (lessSpecific loc m) methods) (m::acc)
  | maxSpecificL loc [] acc = acc

fun mostSpecific loc foo methods =
    case maxSpecific loc methods of
	[m] => m
      |	[] => error loc ("No applicable "^foo)
      | _ => error loc ("No "^foo^" is most specific")

fun getMostSpecific tys cname mname ctx loc instance =
    let val meths = (applicableMethods tys cname (MNAME mname) instance ctx loc)
	val () = debugPrint (mname ^ ": "
			     ^ (plural (length meths) "applicable method")
			     ^ " <" ^ cname ^ ">\n")
	val () = debugPrint (String.concat (map (fn x => (typeToString x ^ "\n")) (meths)))
    in
	mostSpecific loc "method" (meths)
    end

fun getMostSpecificInit tys cname ctx loc =
    let val meths = (applicableMethods tys cname INIT INSTANCE ctx loc)
	val () = debugPrint ("Initialiser: "
			     ^ (plural (length meths) "applicable method")
			     ^ " [" ^ cname ^ "]\n")
	val () = debugPrint (String.concat (map (fn x => (typeToString x ^ "\n")) (meths)))
    in
	mostSpecific loc "constructor" (meths)
    end

(**** object stuff finished ****)



(* Note substTy and substTys differ in terminology *)
fun substTy X T ty = (* replace type var X by T in ty *)
    let
	val w = X
	fun f ty = case ty of
		       TVARty v => if #name v = w then T else ty
		     | ARROWty(s, t) => ARROWty(f s, f t)
		     | ARRAYty t => ARRAYty (f t)
		     |	CONty (tys, v) => CONty (map f tys, v)
		     | PRODUCTty tys => PRODUCTty (map f tys)
		     | _ => ty
    in f ty end

fun substTys ctr tyT =
    foldr (fn ((X,T),ty) => substTy X T ty) tyT ctr

fun substConstr X T constr = (* called 55222 times in interpreter example *)
    List.map (fn (ty1, ty2, loc) => 
		 (substTy X T ty1, substTy X T ty2, loc)) constr

(* val n = ref 0
fun substConstr X T constr = (print ("susbst " ^ Int.toString (!n) ^"\n"); n := !n+1;(
    List.map (fn (ty1, ty2, loc) => (substTy X T ty1,
					    substTy X T ty2, loc)) constr) before print "done\n")
*)


fun occursIn X T =
    let 
	val w = #name X 
	fun oc ty = case ty of
			TVARty {name,...} => name = w
		      |	ARROWty (s, t) => oc s orelse oc t
		      |	ARRAYty ty => oc ty
		      | PRODUCTty tys => List.exists oc tys
		      |	CONty (tyLs,_) => List.exists oc tyLs
		      | _ => false
    in oc T end


fun resolveType x assign =
    let fun vbChase' x =
            case x of
                (TVARty tvar) =>
                (case List.find (fn (TVARty tv, _, loc) => tv=tvar | _ => false) assign
                  of
                     SOME (_, ty, _) =>
                     if ty = x then ty
                     else vbChase' ty
                   | NONE => TVARty tvar)
              | ARROWty (ty1, ty2) =>
                ARROWty (vbChase' ty1, vbChase' ty2)
              | ARRAYty ty => ARRAYty (vbChase' ty)
              | PRODUCTty tys => PRODUCTty (map vbChase' tys)
              | CONty (tys, con) =>
                CONty (map vbChase' tys, con)
              | _ => x
    in vbChase' x end

fun hasTVar t =
    case t of
	TVARty _ => true
      | ARRAYty t => hasTVar t
      | PRODUCTty ts => List.exists hasTVar ts
      | ARROWty (ty, ty') => hasTVar ty orelse hasTVar ty'
      | CONty (tys, _) => List.exists hasTVar tys
      | _ => false

(* recon - type reconstruction *)

fun recon classname ctx exp cc env0  = (* Note 1 *)
    (* cc is accumulator for constraints *)
    let
	infixr 5 ++
	fun  (c as (t, t', _)) ++ l = if t=t' then l else c::l
        (* If we just add the constraints without doing this we seem
           to get LOTS which are just equalities (in one example, 172 out of 300) *)
        (* We still get some repetitions, but this is probably unavoidable *)


	val env = ref env0
	fun addToEnv (v,ty) =
	    let
(*		val () = print ("Adding " ^ v ^ ": " ^ typeToString ty ^ "\n")*)
	    in
		env := Splaymap.insert (!env, v, ty)
	    end

	fun recon' ctx exp cc =
            let
		val (Exp, ty, cc')  = reconInner ctx exp cc
		val cc'' = unify cc'
		val principalTy = resolveType ty cc''
		val () = debugPrint ("In recon', got type " ^ typeToString ty )
		val () = debugPrintln ("; after resolveType, this became " ^ typeToString principalTy)
            in
		(Exp, principalTy (* ty*), cc'')  (* FIX *)

            end

	and reconVal ctx x cc =
            let
		val (t,x') =
		    let
			fun i t = (t,x) in
			case x of
			    VARval (var, ext, l)  =>
			    let
				val ty = getVarTy classname ctx l var
				val () = debugPrint ("reconVal " ^ var ^ ": " ^ typeToString ty ^ "\n")
			    in case ty of
				   ARROWty _ =>
				   let val (ty,ass) = generify ty
 				       val m = (var,ass)
				   in
				       (ty, VARval(var, ext, MONO(m,locOf l)))
				   end
				 | _ => (ty,VARval((var, ext, MONO(("FIX_THIS",[]),locOf l))))
					    (* see TODO *)
					    (* formerly just said (ty,x) *)
			    end

			  | CHARval _       => i CHARty
			  | INTval _        => i INTty
 			  | FLOATval _      => i FLOATty
			  | STRINGval _     => i STRINGty
			  | BOOLval   _     => i BOOLty
			  | UNITval   _     => i UNITty
			  | NULLval (c,_)   => i (OBJECTty c)
		    end

		val cc' = unify cc
		val principalTy = resolveType t cc'
            in
		(principalTy, x', cc')
            end

	and reconArgs ctx vs name cc =
	    let
		fun do_one_arg (v, (tys, Vs, c1)) =
		    let
			val (tyT2, V, c2) = reconVal ctx v c1
			val () = if hasTVar tyT2 then
				     error (getvU v)
					   ("Couldn't resolve reference to argument "
					    ^ NAsyntfn.valToString v
					    ^ " for "
					    ^ name ^ ": type "
					    ^ (typeToString tyT2)
					    ^ " contains type variables")
				 else ()
		    in
			(tyT2::tys, V::Vs, c2)
		    end
	    in
		foldr do_one_arg ([], [], cc) vs (* NOT foldl! *)
	    end

        and maprecon ctx [] tt cc = ([], tt, cc)
          | maprecon ctx (h::t) tt cc =
	    let
		val (T, t, c) = maprecon ctx t tt cc
		val (H, ty1, c1) = recon' ctx (VALexp (h, getvU h)) cc
            in
		(H::T, ty1::t, c1)
            end


	and reconOp ctx oper v1 v2 cc loc =
	    let
		val (tyT1, v1', cc1) = reconVal ctx v1 cc
		val (tyT2, v2', cc2) = reconVal ctx v2 cc1

		fun setEq t  = 
		    case t of TVARty v => (#eq v := true; t)
			    | _ => if isEqType t then t else
				   error loc ("Type " ^ typeToString t 
					      ^ " does not admit equality")

		fun setOrd t  = 
		    case t of TVARty v => (#eq v := true; #ord v := true; t)
			    | _ => if isOrdType t then t else
				   error loc ("Type " ^ typeToString t 
					      ^ " is unordered")

			     
		fun g t t' T = (T, (tyT1, t, [getvU v1])++(tyT2, t', [getvU v2])++cc2)
		fun g3 t = g t t t
		fun gb t = g t t BOOLty
		fun ge () = (BOOLty, (setEq tyT1, setEq tyT2, [getvU v1])++cc2)
		fun go () = (BOOLty, (setOrd tyT1, setOrd tyT2, [getvU v1])++cc2)

		val (ty, cc') =
		    case oper of
			PLUSop    => g3 INTty
		      | MINUSop   => g3 INTty
		      | TIMESop   => g3 INTty
		      | DIVop     => g3 INTty
		      | MODop     => g3 INTty
		      | LANDop    => g3 INTty
		      | LORop     => g3 INTty
		      | LXORop    => g3 INTty
		      | LSLop     => g3 INTty
		      | LSRop     => g3 INTty
		      | ASRop     => g3 INTty
		      | FPLUSop   => g3 FLOATty
		      | FMINUSop  => g3 FLOATty
		      | FTIMESop  => g3 FLOATty
		      | FDIVop    => g3 FLOATty
		      | CONCATop  => g3 STRINGty
		      | EQUALSop  => ge ()
		      | LESSop    => go ()
		      | LEQop     => go ()
	    in
		(ty, v1', v2', cc')
	    end



	and reconInner ctx exp cc =
	    case exp of
		VALexp (v, loc) =>
		let val (ty,v',cc') = reconVal ctx v cc
		in (VALexp (v', loc), ty, cc')
		end
	      | IFexp (test as TEST(oper, v1, v2, tloc), e1, e2, loc) =>
		let
		    val (E2, tyT3, c2) = recon' ctx e2 cc
		    val (E1, tyT2, c1) = recon' ctx e1 c2
		    val (tyT1, V1, V2, nc) = reconOp ctx oper v1 v2 c1 tloc
		    val newconstraints = (tyT1, BOOLty, [tloc])++(tyT3, tyT2, [getU e2])++nc
		in
		    (IFexp (TEST(oper, V1, V2, tloc),
			    E1, E2, loc), tyT2, newconstraints)
		end
	      |	UNARYexp(oper, v, loc) =>
		let
		    val (tyT, V, cc1) = reconVal ctx v cc
		    fun g t t' = ((tyT, t, [getvU v])++cc1, t')
		    val (newconstraints, ty) =
			case oper of
			    NOTop =>  g BOOLty BOOLty (* input type, output type *)
			  | ISNULLop =>  t_error "ISNULLop not implemented in Type.sml"
		in
		    (UNARYexp (oper, V, loc), ty, newconstraints)
		end
	      | BINexp(oper, v1, v2, loc) =>
		let
		    val (ty, V1, V2, newconstraints) = reconOp ctx oper v1 v2 cc loc
		in
		    (BINexp (oper, V1, V2, loc), ty, newconstraints)
		end

	      | APPexp((fname,floc), vs, ext, eloc) => (* Note 2 *)
		if (ext <> EXTERN) then
 		let

		    val () = debugPrint ("APPexp: " ^ fname ^ " (" ^ listToString valToString "," vs ^ ")\n")

		    val () = debugPrint (extToString ext ^ "\n")

(* What happens when we're applying a local?
   Where does it get the type from??? *)

		    val ((tyT, ass), fname') =
			let
			    val ty = getVarTy classname ctx eloc fname
			in
			case ty of ARROWty _ =>
				   let val _ = debugPrint ("arrowty: " ^ fname
							   ^ ": " ^ typeToString ty ^ "\n")
				   in
				       if ext = LOCAL then ((ty, []), fname) (* LOCAL includes funargs *)
				       else
					   (generify ty, fname)
				   end
			| _ => (* It's local or a funarg: careful it doesn't clash with a global  *)
				 let val () = debugPrint ("Didn't find " ^ fname ^"\n")
				 in
(*				     ((getVarTy classname ctx eloc fname, [fname]), fname)*)
				     ((ty, []), fname)
				 end
			end

		    val () = debugPrint ("... function " ^ fname
				    ^ "/" ^ fname' ^ " has type "^ typeToString tyT ^ "\n")

		    fun xxx v =
			case v of
			    CHARval (_,u)       => (CHARty, u)
			  | INTval (_,u)        => (INTty, u)
 			  | FLOATval (_,u)      => (FLOATty, u)
			  | STRINGval (_,u)     => (STRINGty, u)
			  | BOOLval   (_,u)     => (BOOLty, u)
			  | UNITval   u     => (UNITty, u)
			  | NULLval (c,u)   => (OBJECTty c, u)
			  | VARval _ => raise Fail "Oops"


		    fun ar [] _ _ _ = error eloc ("function requires arguments [" ^ fname ^ "]")
		      | ar [h] x a_arg a_c =
			let
			    val (ty, H, c1) = reconVal ctx h a_c
			    val c1' =
				case h of
				    VARval (p,ext,loc) => (
				    debugPrint ("Found arg " ^ p ^ ":" ^ typeToString ty ^ "\n");
				    (ty, getVarTy classname ctx loc p, [loc])++c1
				    )
				  | _ => c1
(*				    let
					val (t,l) = xxx h
				    in
					(ty,t,[l])++ c1
				    end*)
			in
			    ([getvU h,eloc], ARROWty(ty, x), [H], c1')
			end
		      | ar (h::t) x a_arg a_c =
			let
			    val (Tlocs, tys, T, c2) = ar t x a_arg a_c
 			    val (ty, H, c1) = reconVal ctx h c2
			    val c1' =
				case h of
				    VARval (p,ext,loc) => (
				    debugPrint ("Found arg " ^ p ^ ":" ^ typeToString ty ^ "\n");
				    (ty, getVarTy classname ctx loc p, [loc])++c1
				    )
				  | _ => c1 (*
				    let
					val (t,l) = xxx h
				    in
					(ty,t,[l])++ c1
				    end *)
			in
			    ((getvU h)::Tlocs, ARROWty(ty, tys), H::T, c1')
			end




		    val x = newTvar ()
		    val (vlocs, ty, Vs, c1) = ar vs x [] cc
		    val () = debugPrint ("vlocs  = " ^ stars vlocs ^ "\n")
		    val newconstraint = (ty, tyT, vlocs)
		    val () = debugPrint ("newconstraint: " ^ typeToString ty ^ " = " ^ typeToString tyT
										     ^ "\n")
		    val () = debugPrint ("Constraint has " ^ plural (length vlocs) "location" ^ "\n")

		    val m = (fname, ass)    (* info for Mono *)
fun bindingToStr (a,t) = a ^ " -> " ^ (typeToString t)
fun bindingsToStr l = (listToString bindingToStr ", " l)
		    val() = debugPrintln ("m/ass: {" ^ bindingsToStr ass ^ "}")
		    val p = (fname, UNITty) (* info for Phi *)
		    val E = APPexp((fname,floc), Vs, ext, MONO(m,PHI(p, locOf eloc)))
		in
		    (E, x, newconstraint++c1)
		end

		else if ext = EXTERN then
		let
		    val longname = fname
		    val fname = List.last (String.fields (fn c => c = #".") longname)
		    val class = truncate longname (1+ size fname)

		    val (argtys, Vs, cc') = reconArgs ctx vs longname cc

		    val methodTy = getMostSpecific argtys class fname ctx eloc STATIC
		    val retTy = List.last (arrowTyToTyList methodTy)
		    val m = (longname, [])
		    val p = (longname, methodTy)
		    val E = APPexp ((longname,floc), Vs, ext, MONO(m,PHI(p, locOf eloc)))
		in
		    (E, retTy, cc')
		end

		else error eloc "unexpected internality"

	      | CONexp (C as (v,vloc), args, addr, loc) =>
 		let
		    val (tyT, ass) = generify (getVarTy classname ctx loc v)
		in
		    case tyT of
			CONty (tyargs,tname) =>
			let
			    val ctys = case (getbinding ctx v) of
					   SOME(_,ConBind(_, _, ctys)) => ctys
					 | _ => error loc "Internal compiler error 733g"
			    val () = if length ctys <> length args
				     then error loc "Internal compiler error 733h: arity mismatch"
				     else ()

(*			    val (types, constraints) = reconArgs2 ctx args "OOPS"  cc*)

			    val (Args, types, constraints) =
				maprecon ctx  args [] cc

			    val newconstraints =
				let
				    fun f [] [] [] acc = acc
				      | f (ty::t1) (cty::t2) (v::t3) acc
					= (ty, (substTys ass) cty, [getvU v])++(f t1 t2 t3 acc)
				      | f _ _ _ _ = t_error "length mismatch at CONexp"
				in
				    f types ctys args constraints
				end
				    (* Check that the constructor args have appropriate types *)

			    val tyT' = substTys ass tyT
			    val f = (tname, ass) (* info for monomorphising *)
			    val (acstr, Addr) =
				case addr of NONE => (NONE, NONE)
					   | SOME (a,al)
					     => (SOME (getVarTy classname ctx loc a, DIAMONDty "", [loc]),
						 SOME (a,al))
						    (* Also checks that a is in scope *)
			    val E =  CONexp (C,
					     args,
					     Addr,
					     MONO(f, locOf loc))
			in
			    (E, tyT', acstr ::? newconstraints)
			end
		      | _ => error loc ("Unknown constructor " ^ v)
		end

	      | MATCHexp (X as (x,xl), rules, eloc) =>
		let
		    val (tyT, X', c1) = reconVal ctx (VARval (x,LOCAL,xl))  (* FIX FIX FIX *)
						 cc (* FIX THIS *) (* X' discarded *)

		    fun doRule (MATCHrule(C as (cname, cloc), args, addr, e', loc'))  =
			let
                       	    val ((ty, ass), tys', t) =
				case getbinding ctx cname of
				    SOME (var, ConBind (t, tys, ctys)) =>
				    (generify (getVarTy classname ctx loc' cname), ctys, t) (***)
				  | _ => error loc' (cname ^ " is not a known constructor name")
                            val tys = (map (substTys ass) tys')

			    val argnames = map #1 args
                            val () = if length args <> length tys
                                     then error loc' "Wrong number of arguments for constructor"
                                     else ()
			    val () = app (fn x => (debugPrint ("Matching " ^ x ^"\n" ))) argnames
                            val ctx' = (ListPair.zip (argnames,
						      ListPair.map (fn (x,l)=>
								       VarBind (x, STATIC, l)) (tys, map #2 args))
				       ) @ ctx
				(* Try to get the locations in the VarBind *)

				       (* FIX: TIDY THIS UP *)

			    val () = ListPair.app addToEnv (argnames, tys)

			    val ctx'' =
				case addr of
				    SOMEWHERE (d,dloc) => let
					val () = addToEnv (d, DIAMONDty "")  (* Note 3 *)
				    in
					(d, VarBind (DIAMONDty "", STATIC, dloc))::ctx'
				    end
				  | _ => ctx'

                            val (E', tyT1, c1) = recon' ctx'' e' []
			in
                            (MATCHrule (C, args, addr, E', loc'),
			     tyT1, (tyT, ty, [eloc])++c1, ass,
			     CONty(tys', t))
			end
			    (* FIX: not finished. class should exist and be subclass of e's class *)
		      | doRule (OOMATCHrule(pat, e', loc)) =
			let
			    val binding=
				case pat of
				    ANYCLASSpat => NONE
				  | CLASSpat ((var,_), (class,_)) =>
				    let
					val () = addToEnv (var, OBJECTty class)
				    in
					SOME (var, VarBind (OBJECTty class, STATIC, loc))
				    end
			    val ctx' = binding ::? ctx
			    val (E', tyT1, c1) = recon' ctx' e' []
			in
			    (OOMATCHrule (pat, E', loc),
			     tyT1, c1, [], OBJECTty "java.lang.Object" (* unused *))
			end

			(* Suppose we have an object x which we know to belong to some class,
			   say javax.microedition.lcdui.Item.  We can match against the
                           various subclasses of Item,  but the default case seems to add
                           an extra constraint that x has type Object,  when we're certain
                           that it has class Item.  This may cause problems in Grail when
                           type consolidation is turned off. *)

		    val foo = map doRule rules
		    val Rules = map #1 foo
		    val tylocs = map #2 foo
		    val constraints = List.concat (map #3 foo)

		    val constraints' =
			case tylocs of
			    h::t => foldr (fn (a,l) => (h,a, [eloc])++l) constraints t
			  | [] => error eloc "Match expression must have at least one clause"

		    val ty = (hd tylocs)  (* right hand type *)
		    val cty = hd (map #5 foo)  (* left  hand type *)
		    val tname = case cty of
				    CONty(types, tname) => tname
				  | OBJECTty x => x
				  | _ => error eloc "Match expression must have constructed type"

(* Substitutions are all the same if we typecheck and after
   the later substitution, so use #1 *)

		    val c0 = hd (map #4 foo)
		    val m = (tname, c0)(* Mono info *)

(* NOT NEEDED ??? Anyway,  we've thrown away the tyargs
		    val () = addToEnv (x, cty (* CONty ([], tname) *) )
				      (* for "match x with ...", get type of x *)
*)

		    val M = MATCHexp (X, Rules, MONO(m, locOf eloc))
		in
                    (M, ty, constraints'@cc)
		end

	      | LETexp (X as (x,xloc), e1, e2, loc) =>
		let
		    val (E1, tyT1, c1) = recon' ctx e1 cc
		    val ctx' = addbinding ctx x (VarBind (tyT1, STATIC, xloc))
		    val () = debugPrintln ("LETexp: " ^x^ " has type " ^ typeToString tyT1)
		    val () = addToEnv (x, tyT1)
		    val (E2, t, c) = recon' ctx' e2 c1
		in
		    (LETexp(X, E1, E2, loc), t, c)
		end
	      | TYPEDexp (e, t, eloc) => (* NOT CHECKED THIS YET *)
                let
		    val (E, tyT1, c1) = recon' ctx e cc
		in
		    (TYPEDexp (E, t, eloc),
		     tyT1, (tyT1,t,[getU e])++c1)
		end
	      | ASSERTexp (e, as1, as2, eloc) =>
		let
		    val () = warn eloc "Assertion not type-checked."
		in
		    recon' ctx e cc
		end
	      | COERCEexp (e,t,eloc) => (* FIX: Not right yet *)
                (* Should probably introduce subtype constraint. Needs thought. *)
		let
		    val (E, tyT1, c1) = recon' ctx e cc
		    val () = if typeIsSubclass tyT1 t then ()
			     else subclassError eloc tyT1 t
		in
		    (COERCEexp(E, t, eloc), t, c1)
		end
	      | SGETexp(V as (v,_),loc) =>
		let
		    (* FIX: nasty, duplicated *)
	            val x = List.last (String.fields (fn c => c = #".") v)
		    val class = truncate v (1+ size x)
		in
		    (SGETexp(V,loc), getFieldTy STATIC (OBJECTty class) x ctx loc, cc)
		end
	      | UPDATEexp(X as (x,_), v, loc) =>
		let
		    val (tyT1, V, c1) = reconVal ctx v cc
		    val class = case getbinding ctx "this" of
				    SOME (_, VarBind(ty,i,_)) => ty
				  | _ => error loc "Variable binding for 'this' not found"
		    val ty = getFieldTy INSTANCE class x ctx loc
		in
		    (UPDATEexp (X, V, loc),
		     UNITty, (tyT1, ty, [getvU v])::cc)
		end
	      | GETexp(Obj as (obj, _), Fname as (fname,_), loc) =>
		let
		    val tyT1 = getVarTy obj ctx loc obj  (* ER... ??? *)
		    val ty = getFieldTy INSTANCE tyT1 fname ctx loc
		in
		    (GETexp(Obj, Fname, loc), ty, cc)
		end

	      | NEWexp(C as (class,_), vs, loc) =>
		let
		    val (argtys, Vs, cc') = reconArgs ctx vs "new" cc

		    val initTy = getMostSpecificInit argtys class ctx loc

		    val () = debugPrint ("Initialiser: chose "^(typeToString initTy)^"\n")
		    val p = ("<init>", initTy)
		in
		    (NEWexp (C, Vs, PHI(p,locOf loc)), OBJECTty class, cc')
		end

	      | SUPERMAKERexp (vs, loc) =>
		let
		    val (argtys, Vs, cc') = reconArgs ctx vs "super" cc

		    val superclass =
			case getSuperclass (valOf classname) of
			    SOME (x,_) => x
			  | _ => "java.lang.Object"

		    val initTy = getMostSpecificInit argtys superclass ctx loc

		    val p = (superclass^".<init>", initTy)
		in
		    (SUPERMAKERexp (Vs, PHI(p,locOf loc)), UNITty, cc')
		end

	      | INVOKEexp(Obj as (obj,_), Mname as (mname, mu), vs, loc) =>
		let
		    (*
		     fun argConstr ((exp as (loc, e), ty), (acc_c, acc_m, acc_v))  =
			 let
			     val (tyT1, c1, acc_m', acc_v') = recon' ctx exp acc_c acc_m acc_v
			     val acc_c' = (tyT1, ty, loc)++c1
			 in
			     (acc_c', acc_m', acc_v')
			 end

		     fun argSubsume ((exp as (loc, e), ty), (acc_c, acc_m, acc_v))  =
			 let
			     val (tyT1, c1, acc_m', acc_v') = recon' ctx exp acc_c acc_m acc_v
			     val acc_c' = if isObjectType ty then
					      (if typeIsSubclass tyT1 ty then ()
					       else subclassError loc tyT1 ty;
					       c1)
					  else (tyT1, ty, loc)++c1
			 in
			     (acc_c', acc_m', acc_v')
			 end
		     *)

                 val retTy = newTvar ()
		 val tyT1 = getVarTy obj ctx loc obj  (* ER...??? *)
		 val class = case tyT1 of OBJECTty cl => cl
					| _ =>
					  error loc ("Couldn't resolve reference to method "^mname
						     ^": type "
						     ^(typeToString tyT1) ^ " does not resolve to class type")

		 val (argtys, Vs, cc') = reconArgs ctx vs mname cc

		 val methodTy = getMostSpecific argtys class mname ctx loc INSTANCE

		 val retTy = List.last (arrowTyToTyList methodTy)
		 val p = (mname, methodTy)
		in
		    (INVOKEexp(Obj, Mname, Vs, PHI(p,locOf loc)), retTy, cc')
		end

    in
        (recon' ctx exp cc, !env)
    end  (* recon *)

and stars l = "[" ^ listToString (fn x => "*") ", " l ^ "]"

and promote (X,Y) =
    let 
	val eq =  isEq X orelse isEq Y
	val ord = isOrd X orelse isOrd Y
    in 
	(#eq X := eq; #eq Y := eq; #ord X := ord; #ord Y := ord)
    end


and unify constraints =
    let
	val () = debugPrint ("Unifying (" 
			     ^ plural (length constraints) " constraint" 
			     ^ ")...\n")
	exception typeError of Ty * Ty * Normsyn.Annotation list
	fun zzzip (l1, l2, loc) = map (fn (a,b) => (a,b,loc)) (ListPair.zip (l1,l2))

	fun hd [] = (print "WARNING: missing location in constraint\n"; nowhere)
	  | hd (h::_) = h

	fun u l =
	    case l of
		[] => []
	      | h::rest =>
		let
		    val (t1, t2, loc) = h
		    val ll = stars loc
		    val () = if (!dbgConstraints)
			     then debugPrint ("U: " ^ typeToString t1 ^ " & "
					      ^ typeToString t2
					      ^ " " ^ ll
					      ^ "\n")
			     else ()
		in
		    case h of
			(INTty, INTty, _) => u rest
		      | (CHARty, CHARty, _) => u rest
		      | (BOOLty, BOOLty, _) => u rest
		      | (FLOATty, FLOATty,_) => u rest
		      | (STRINGty, STRINGty, _) => u rest
		      | (STRINGty, OBJECTty "java.lang.String", _) => u rest
		      | (OBJECTty "java.lang.String", STRINGty, _) => u rest
		      | (UNITty, UNITty, _) => u rest
		      | (tyS as TVARty X, tyT as TVARty Y, loc) =>
			let 
			    val () = promote (X,Y)
			in
			    if X = Y then u rest (* names ? *)
			else
			    (tyS, tyT, loc)::(u (substConstr (#name X) tyT rest)) 
			end

		      | (tyS as TVARty X, tyT, loc) =>
			let 
			    val () = if isOrd X andalso not (isOrdType tyT) then 
					 constraintError (hd loc) tyT tyS "Order violation" 
				     else ()
			    val () = if isEq X andalso not (isEqType tyT) then 
					 constraintError (hd loc) tyT tyS "Equality violation" 
				     else ()
			in
			    if occursIn X tyT then
				error (hd loc) ("circular constraints: " ^ (typeToString tyS)
						^ " and " ^ (typeToString tyT)
						^ ". Polymorphic recursion?")
			    else
				(tyS, tyT, loc)::(u (substConstr (#name X) tyT rest)) 
			end
		      | (tyS, tyT as TVARty Y, loc) =>
			let 
			    val () = if isOrd Y andalso not (isOrdType tyS) then 
					 constraintError (hd loc) tyS tyT "Order violation" 
				     else ()
			    val () = if isEq Y andalso not (isEqType tyS) then 
					 constraintError (hd loc) tyS tyT "Equality violation" 
				     else ()
			in
			    if occursIn Y tyS then
				error (hd loc) ("circular constraints: " ^ (typeToString tyS)
						^ " and " ^ (typeToString tyT)
						^ ". Polymorphic recursion?")
			    else
				(tyT,tyS, loc)::(u (substConstr (#name Y) tyS rest))
			end

		      | (ARROWty(tyS1,tyS2), ARROWty(tyT1,tyT2), loc::locs) =>
			u ((tyS1,tyT1,[loc]) :: (tyS2,tyT2,locs) :: rest)
		      | (ARROWty(tyS1,tyS2), ARROWty(tyT1,tyT2), []) =>
			(debugPrint "Warning: no locations for arrow types during unification\n";
			u ((tyS1,tyT1,[nowhere]) :: (tyS2,tyT2,[nowhere]) :: rest))
		      | (PRODUCTty l1, PRODUCTty l2, loc) => u ((zzzip (l1, l2, loc))@rest)
		      | (ARRAYty tyX, ARRAYty tyY, loc) => u ((tyX,tyY,loc)::rest)
		      | (ty1 as CONty(tyArgs, c), ty2 as CONty(tyArgs', c'), loc) =>
			 if c=c' then
			     let val r = u rest
				 val r1= u r
				 val r' = u ((zzzip (tyArgs, tyArgs', loc))@r1)
			     in u r' end
			else constraintError (hd loc) ty2 ty1 "Type mismatch"
	                (* kwxm: previously there was no check for c=c' here: this meant that
                           you could apply functions to members of the wrong datatype and
	                   get an incorrect executable program *)
		      | (DIAMONDty s, DIAMONDty s', loc) => u rest
                      (* For now, all diamond types match (could be problematic with old-style diamonds).
	                 For multiple diamond types,  we should carry around the actual type of the object
	                 with which the diamond is associated.
	                 This might be tricky since we won't have constructed the datatypes
                         the first time we get here *)

		      | (ty as (OBJECTty c), ty' as (OBJECTty c'), loc) =>
			if c = c' then u rest
			else constraintError (hd loc) ty ty' "Class mismatch"

		      | x  => raise typeError x
		end

	val r = u constraints
	    handle typeError (tyS, tyT, loc)
		   => constraintError (hd loc) tyS tyT "Unsolvable constraints"
	val () = debugPrint "... done\n"


    in
	r
    end


(* TODO: check funs have not had incompatible type declarations *)
(* OK, LET'S DO THIS REAL SOON *) (* unify function type with context *) (* It's done? *)
(* Check funs are not being redeclared? - This is now done in the parser *)
(* check for type variables getting loose *)
(* match expressions - covers exactly *) (* Depending on what this means,  it's probably OK now *)


fun typeFunDefList classname (functx: context) flist (env0: Env.funEnv) =
    let
        val aty = tyListToArrowTy

        fun b [] ctx = ctx
          | b ((FUNdef((v,vl),args,_,_,_))::t) ctx =
            let
		val () = debugPrint ("@ Typing " ^ v ^ "\n")
		val () = debugPrint ("Context is\n" ^ ctxToString ctx ^ "\n")
                val bind = (v, VarBind (newTvar(), STATIC, vl))
            in
                b t (bind::ctx)
            end

        val ctx = b flist functx

        (* The next function is kind of messy.  We want to retain the types of variables
           for later use,  so we use a list of (varname, type) pairs.  We have to collect
           this information for function arguments as well,  which is what happens below.
           The information which we collect is pretty much the same as the context information,
           so we're probably doing the same thing twice;  oh well. *)

        fun getConstraintsForFun fctx (FUNdef((fname,floc),args,i,e,loc)) =
        let
            fun getArgInfo l ctx argEnv =
		case l of
		    [] => (ctx, argEnv)
		  | UNITvar::vs =>  getArgInfo vs ctx argEnv
                  | (VAR(v,tyOpt))::vs =>
		    let
			val bindTy =
			    case tyOpt of
                                SOME ty => ty
			      | NONE => newTvar ()

			val ctx' = addbinding ctx v (VarBind(bindTy, STATIC,loc))
			val argEnv' = Splaymap.insert(argEnv,v,bindTy)
		    in
			getArgInfo vs ctx' argEnv'
		    end

            val (ctx, argEnv) = getArgInfo args fctx (Env.newVarEnv ())

            val ((E, ret, constr), varEnv) = recon classname ctx e [] argEnv

	    val newFunDef = FUNdef ((fname,floc), args, i, E, loc)

	    fun g UNITvar = UNITty
	      | g (VAR(v,_)) = getVarTy classname ctx loc v

	    val tys = map g args

            val funty' = aty (tys@[ret])
            val funty = case funty' of
			    ARROWty _ => funty'
			  | _ => (ARROWty(UNITty, funty'))

            val constr' =  (getVarTy classname ctx loc fname, funty, [loc]) :: constr
                (* The above line unifies the function type with the
                arrow type of type variables, so *should* constrain
                the number of arguments without further code if
                unification is correct. *)

	    val constr'' =  (* FIX: do we need this ??? It looks rather like the bit above *)
		case getbinding functx fname of
                              SOME (_,VarBind (t,_,_)) =>
			          (getVarTy classname ctx loc fname, t, [loc]) ::constr'
                            | _  => constr'
		(* Unify with previously declared/inferred types. *)
        in
            (newFunDef, constr'', (fname, funty, varEnv))
        end

        val (Flist, cc, funInfo) = unzip3 (map (getConstraintsForFun ctx) flist)
        val constr = List.concat cc
	val () = debugPrint ("Unifying (" ^ Int.toString (length constr) ^" constraints) ...\n")
        val assign = unify constr (* could be time-consuming *)
	val () = debugPrint "... done\n"

        val () = debugPrint ("Context:{\n"
                  ^ (ctxToString ctx)
                  ^ "Constraints:\n"
                  ^ (constraintListToString constr)
                  ^ "Assignments:\n"
                  ^ (constraintListToString assign) ^ "}\n")



        fun varChase v = case List.find (fn (x, VarBind _) => x=v | _ => false) ctx of
            SOME (_, VarBind(vb,_,_)) =>
                SOME (resolveType vb assign)
          | _ => NONE

        val tys = map (fn FUNdef((f,floc),_,inst,_,_)
			  => (f, VarBind (valOf (varChase f), inst, floc))) Flist


        fun getVar (TVARty t) = SOME t | getVar _ = NONE

        fun resolveCtr ctr = map (fn (a, b) => (a, resolveType b assign)) ctr

	val resolveMonoInfo = NAsyntfn.mapMono (fn (info, x) => (info, resolveCtr x))

	val Flist' = map (NAsyntfn.mapFunDef resolveMonoInfo) Flist

	fun resolveTypes (fname, funty, types) =
	    (fname,
	     resolveType funty assign,
	     Splaymap.map (fn (v,t) => resolveType t assign) types)


	val funInfo' = map resolveTypes funInfo

	fun addInfo ((fname, funty, vtypes), env) =
	    Splaymap.insert (env, fname, (resolveType funty assign, vtypes))

	val env' = List.foldl addInfo env0 funInfo'


(*
        (* Now we have to resolve invocations and new operations. Fresh
           type variables are generated during constraint generation
           for these expressions, so we now have to work out the proper
           types and substitute for the tyvars. *)

	val = new_substitution = ... (typeOO[f] vars assign ctx)
*)

    in
        (Flist', tys, env')
    end

fun typeFunBlock classname (functx: context) (FUNblock flist) env =
    let
	val (flist, tys, env') = typeFunDefList classname functx flist env
    in
	(FUNblock flist, tys, env')
    end


fun typeType names (TYPEdec(tvars, (name,_), cons, loc)) =
    let
	fun chkName loc x = if member x names then ()
				else error loc ("[Type]: Unknown type name " ^ x)

	fun checkTypeCon (TYPEcon(c, (tys,tloc), h, l)) =
	    let

		fun chk (ARROWty(a,b)) = (chk a; chk b)
		  | chk (ARRAYty a) = chk a
                  | chk (PRODUCTty tys) = app chk tys
		  | chk (CONty(tys, cc)) =  (app chk tys; chkName l cc)
		  | chk (TVARty v) =
		    if List.exists (fn (var,_) => var = #name v) tvars
		    then () else
		    error tloc
			  ("[Type]: Unknown type variable " ^ #name v ^ " in constructor")
		  | chk _ = ()

	    in
		app chk tys
	    end

	val () = app checkTypeCon cons
        (* Should this be here?  Is Syncheck too early? *)

	val () = if cons = []
		 then error loc "Type must have at least one constructor"
		 else ()

	val cnames = map (fn TYPEcon((c,_),(ts,_),_,_) => (c,ts)) cons

	val tys = map (fn (x,_) => mkTvar x) tvars
    in
	(name, tys, cnames)
    end



(* TODO: check we don't clobber a type constructor name *)
(* Done in the parser? *)
fun typeTypes types =
    let
	val names = map (fn TYPEdec(_,(n,_),_,_) => n) types
    in
        map (typeType names) types
    end

fun getClasstype (CLASSdef(classname, super, intfs, vals, meths)) =
    let
	fun getVarType _ UNITvar = UNITty
	  | getVarType _ (VAR(x, SOME ty)) = ty
	  | getVarType loc (VAR(x, NONE)) = error nowhere ("Parameter " ^ x ^ " must have type annotation")
	fun getExpType loc (TYPEDexp(e, ty,_)) = ty
	  | getExpType loc _  = error loc ("Missing method return type annotation")
	fun getMethodType (FUNdef(f, vars, inst, e, loc)) =
	    let
		val tys = map (getVarType loc) vars
		val retTy = getExpType loc e
		val ty = tyListToArrowTy (tys@[retTy])
	    in
		VALdec(f, ty, inst)
	    end
    in
	CLASSdec(classname, super, intfs, vals@(map getMethodType meths))
    end

fun typeClassDefs' ctx cdefs CC funEnvs =
    case cdefs of
	[] => (CC, funEnvs)
      | CLASSdef(C as (classname,cloc), super, intfs, vals, meths) :: cdefs' =>
	let
	    fun instance (FUNdef(_,_,INSTANCE,_,_)) = true
	      | instance (FUNdef(_,_,STATIC,_,_)) = false
	    fun cl x = (SOME classname, x)
	    val this_ctx = addbinding ctx "this" (VarBind(OBJECTty classname, STATIC, cloc))

            val (Meths, _, envI) =
		typeFunDefList (SOME classname) this_ctx (List.filter instance meths) (Env.newFunEnv())

            val (Funs, _, envS) =
		typeFunDefList (SOME classname) ctx (List.filter (not o instance) meths) envI
	    val thisEnv = {class = CLASSenv classname, funenv = envS}

	    val Class = CLASSdef(C, super, intfs, vals, Meths@Funs)
	in
            typeClassDefs' ctx cdefs' (Class::CC) (thisEnv::funEnvs)
	end

and typeClassDefs ctx cdefs =
    let
	(* Could do getClasstype on classdefs here, but for now we want classes in
	scope in the main program, and the main program in scope in all classes *)

        val (Classes, envs) = typeClassDefs' ctx cdefs [] []
    in
	(Classes, envs, ctx)
    end

fun typeProg (prog as (PROG(datatypes, valdecs, classdefs, fblocks))) =
    let
	type Loc = Loc.Location
        val typeinfo = typeTypes datatypes:
       (
         string           (* typename *)
         * Ty list        (* type variables *)
         * (
             string       (* constructor name *)
              * Ty list   (* argument types *)
           ) list
       ) list


       fun doCon (name: string) (tys: Ty list)
		 ((c,ts): string* (Ty list)) = (c, ConBind(name, tys, ts))
       fun doType (name, tys: Ty list, cons) = map (doCon name tys) cons

       val cctx = List.concat (map doType typeinfo)


       val builtins = (* PROBABLY DON'T NEED THIS ANY MORE *)
            [ VALdec(("main",nowhere), (ARROWty (ARRAYty STRINGty, UNITty)), STATIC) ]
       val functx = map ctxify (valdecs@builtins@Perv.builtinTypes())


       (* functx contains the types of the functions used in the program.
          The first time round, functx will contain entries only for library functions
          which are used by the program (set in Parser via Perv.markUsed),  and
          functions for which the programmer has supplied an explicit type in a
          valdec.
          The second time we get here, functx will have an entry for every
          function defined in the program.  This will lead to duplicate types for most
          functions,  but we have to put up with this since maybe the programmer will
	  have specified types which are stronger or weaker than the inferred ones. *)

       (* Get class context; classes in scope in all classes and main program, main program
          in scope in all classes *)

	val classtypes = map getClasstype classdefs (* CLASSdec list *)
	val () = ClassPath.updateInfo classtypes    (* Marks internal classes as known etc. *)
	val classctx = map ctxify classtypes

        val (Fblocks, ctx, mainEnv) = let
            fun tfuns [] NewFuns ctx env = (NewFuns, ctx, {class=MAINenv, funenv=env})
              | tfuns (h::t) NewFuns ctx env =
	        let
                    val (H, c, env') = typeFunBlock NONE ctx h env
	        in
                    tfuns t (H::NewFuns) (ctx@c) env'
	        end
        in
	    tfuns fblocks [] (cctx@functx@classctx) (Env.newFunEnv())
        end


	val pvars = app (fn (v,t) =>
			    (debugPrint ("  " ^ v ^ ": "
					 ^ (typeToString t) ^ "\n")))


	fun dropL l =
	    case l of [] => []
		    | #"("::t => t
		    | _ => l

	fun dropR l =
	    case l of [] => []
		    | #")"::t => t
		    | _ => l

	val chop = implode o rev o dropR o rev o dropL o explode

	fun getn f = case getbinding ctx f of
			SOME (v,VarBind (t,i,_)) => chop (typeToString t)
		      | _ => "(* Type not found *)"

        val (Classdefs, cenvs, ctx') = typeClassDefs ctx classdefs

        val progEnv = ProgEnv (mainEnv::cenvs)
(*val () = Env.prEnv progEnv*)
        val r = List.mapPartial uctxify (getClassCtx()@ctx')

        fun uniq [] = []  (* FIX:  this is messy and munges locs? *)
          | uniq [x] = [x]
          | uniq ((a as (VALdec((v,_),_,_)))::(b as (VALdec((v',_),_,_)))::t) =
                  if v = v' then uniq (b::t) else a::(uniq (b::t))
          | uniq ((a as (CLASSdec((v,_),_,_,_)))::(b as (CLASSdec((v',_),_,_,_)))::t) =
                  if v = v' then uniq (b::t) else a::(uniq (b::t))
          | uniq (a::b::t) = a::(uniq (b::t))

        val valdecs' = r @ (List.filter (fn CLASSdec _ => true | _ => false) valdecs)

        fun compare ((VALdec((v,_),_,_)), (VALdec((v',_),_,_))) = String.compare(v,v')
          | compare ((CLASSdec((v,_),_,_,_)), (CLASSdec((v',_),_,_,_))) = String.compare(v,v')
          | compare ((VALdec _), (CLASSdec _)) = LESS
          | compare _ = GREATER

        val Valdecs = uniq (Listsort.sort compare valdecs')

    in
        (PROG(datatypes, Valdecs, Classdefs, rev Fblocks), progEnv, ctx)
    end


end (* local open Absyn *)


(* ctx has type (string*binding) list *)
fun instToStr Absyn.STATIC = "STATIC" | instToStr Absyn.INSTANCE = "INSTANCE"

val typeToString = NAsyntfn.typeToString
fun bindToStr b =
    case b of
	VarBind (t,i,_) => "Var " ^ typeToString t ^ " [" ^ instToStr i ^ "]"
      | ConBind (s, l1, l2) => "Con " ^ s ^ " [" ^ Util.listToString typeToString "," l1
			       ^ "] <" ^ Util.listToString typeToString "," l2 ^ ">"
      | ClassBind (t,c,_) => "ClassBind " ^ typeToString t ^ ctxToString c

and ctx1 (s, b) = s ^ " ---> " ^ bindToStr b ^ "\n"
and ctxToStr l = "CONTEXT\n" ^ String.concat (map ctx1 l) ^ "\n\n"



(* ================ Note 1 ================ *)

(* recon returns a quintuple (exp', etye, constraints, vars)

   where exp' is the expresssion which is being typed,  possibly with
              some annotations for Phi and Mono,
	 etype is the type which has been inferred for the expression
         constraints is a list of constraints generated during type inference
         vars is a list of pairs consisting of variables declared in the expression,
              together with their types;  the types have to be resolved once
              reconstruction is complete.  This list is never read during recon;
              it's only written to and then subsequently returned as part of the
              result.  Thus we don't have to be too careful about

(* Now it's a splaymap *)


   The constraints are triples (t,t', loc) , where t is the type which
   the expression at location loc is expected to have, and t' is the
   type which has been inferred (OK, not really, but more or less)

   The function requires accumulators for the output data as input.  Also
   we require that no variable is ever redeclared:  this will be the case
   after normalisation.

   Monomorphisation information consists of (I think) pairs (s, l) where is is
   a name of a potentially polymorphic object,  and l is a list of type
   assignments for the free types in the object.  This information is now
   stored in the syntax tree.  The name s only seems to be needed for
   constructor applications and match expressions,  where it's the name
   of the associated type.  Annotations are also produced for function
   applications, where s is set to the name of the function;  however this
   can be recovered from the syntax tree anyway.
 *)


(* ================ Note 2 ================ *)

(* There was a longstanding problem with inexact locations for type errors in function
   applications.  Unification is done via arrow types,  so the only location available
   was that of the entire APPexp.  I've changed the constraints so that they now have
   the form (t1, t2, l),  where l is a list of locations.  For everything except APPexps
   this list is a singleton,  but for APPexps it's a list of locations,  one for each
   argument.  This solution could be improved,  but it seems to work for the time being. *)

(* ... but not quite.  In the unification function,  unification of arrow types with
   empty location lists seems to occur during function declaration.  Could this be
   caused by valdecs from the first typecheck???  Anyway,  I've put in something to
   handle this. *)

(* It's still a bit strange.  If you have "let f n = n+1 and g () = f true" then
   you get an error message for g with the types the wrong way round.  If you
   replace "and" with "let" then it's ok. ... Changing the types round in
   unification of arrow types makes the first example ok and the second one wrong. *)


(* ================ Note 3 ================ *)

(* Strictly,  we should check that d hasn't been bound in the
 pattern;  however we are OK because in that case d will
 have been renamed during normalisation and will be the
 only d in scope.  I also think that we don't need a
 constraint for d (maybe) *)

(************************* stuff from start of APPexp case
		    val ((tyT, ass), fname') =
			if member fname flist   (* FIX FIX FIX *)
				  (* flist is list of names of funs in current funblock *)

				  (* Check the use of flist: odd errmsg if you use undef fname *)
			then let
				val ty = getVarTy classname ctx loc fname
				val () = (
				    debugPrint ("Found " ^ fname);
				    debugPrint ("\nType is " ^ (typeToString ty) ^ "\n")
				)
			    in
				((ty, []), fname)
			    end
			else (* It's local or a funarg: careful it doesn't clash with a global  *)
				 let val () = debugPrint ("Didn't find " ^ fname ^"\n")
				 in
				     ((getVarTy classname ctx loc fname, []), fname)
				 end
*)
(*		    val tyT = if  (*ext = BUILTIN*) then
				  getVarTy classname ctx loc fname else tyT
(* With the above line uncommented we get correct types for hofs, but can't compile
List.cmlt &c.  With it commented out we can compile List but get bad funtys. *)

***************************)
