fun phiError s = Util.ierror ("[Phi.sml]: " ^ s)

val experimental = ref false (* if true,  compile using new Grail case statement *)

fun pr s = print s   (* km: debugging *)
fun prl s = print (s^"\n")


fun printVarTy (s,t) = (pr "    "; pr s; pr ": "; prl (Asyntfn.typeToString t))

fun printTyItem (s,a,l,b) =
    (print "  "; prl s;
     case l of [] => prl "   ---"
	     | _ =>
	       app printVarTy l
    )

fun printInfo (a,l) =
    let
	val () = prl "==="
	val () = case a of NONE => prl "NONE"
			 | SOME s => prl s
	val () = app printTyItem l
    in
	()
    end

val noarglists = ref false
(* omit parameter lists if true *)
(* But use -o for proper parameter lists *)

local
    val nowhere = Loc.nilLocation
    structure A = Normsyn
    val () = Absyn.required (* just in case *)
    structure G = GrailAbsyn
    val () = GrailAbsyn.required
    open Normsyn
    open Util
in

infixr  5 ::?
datatype var = VAR of string
	     | RESULT of G.RTy

fun getArgTypes l =
    case l of [] => []
	    | G.INTty::t => getArgTypes t
	    | h::t => h::(getArgTypes t)


(* Function names *)
local
    val counter = ref 0
in
fun resetfnames () = counter := 0
fun genfname() =
    let
	val n = !counter
	val () = counter := (!counter) + 1
    in
	"f_" ^ Int.toString n
   end
end

(* DataOpt - datatype layout optimisation *)
local
    val nullCons = ref (Binaryset.empty String.compare)
    val tysWithNull = ref (Binaryset.empty String.compare)
    val intTypes = ref (Binaryset.empty String.compare)
    val intCons = ref (Binaryset.empty String.compare)
in
    fun addRepresentations(cons, types, iTypes, iCons) =
        (nullCons := Binaryset.addList(!nullCons, cons);
         tysWithNull := Binaryset.addList(!tysWithNull, types);
         intTypes := Binaryset.addList(!intTypes, iTypes);
         intCons := Binaryset.addList(!intCons, iCons))
    fun nullRepr C = Binaryset.member(!nullCons, C)
    fun typeHasNull ty = Binaryset.member(!tysWithNull, ty)
    fun intType ty = Binaryset.member(!intTypes, ty)
    fun intRepr C = Binaryset.member(!intCons, C)
end


(* Horrible stuff *)
local
    val progEnv = ref (Env.ProgEnv [])
    val ctx = ref []: Type.context ref
    val funs = ref (Binarymap.mkDict(String.compare))
    val meths = ref (Binarymap.mkDict
                         (fn ((c,m),(c',m')) =>
                             let val comp = String.compare(c,c') in
                                 case comp of EQUAL => String.compare(m,m')
                                            | _ => comp
                             end))
in
fun getfun f = Binarymap.find(!funs, f)
fun getfunopt f = Binarymap.peek(!funs, f)
fun addfun f ty = funs := Binarymap.insert(!funs, f, ty)
fun getEnv class = Env.getClassEnv class (!progEnv)
fun getCtx () = !ctx

(* fun fullClassName class = if Util.qualified class then class else (Util.getBaseName ()) ^ "$" ^ class *)

(* Now misnomer - we only use this for fields! - Due to use of info lists *)
fun getmethod cm = Binarymap.find(!meths, cm)
fun getmethodopt cm = Binarymap.peek(!meths, cm)
fun addmethod c (m, ty) = meths := Binarymap.insert(!meths, (c,m), ty)

fun getmethod (c,m) =
    case Binarymap.peek(!meths, (c,m)) of
        SOME mth => mth
      | NONE =>
        (case ClassPath.getSuperclass c of
             SOME c' => getmethod (c',m)
           | NONE => phiError ("Method "^m^" not found\n"))


exception translateError of string
exception seekError of string
val stringArray = G.ARRAYty (G.REFty "java.lang.String")

local
    val thisMethod = ref ""
    val thisClass = ref ""
    fun methodVars () =
	let val thisEnv =
	    if !thisClass = ""
	    then Env.getMainEnv (!progEnv)
	    else Env.getClassEnv (!thisClass) (!progEnv)
	in
	    Splaymap.listItems (Env.getVarEnv (!thisMethod) thisEnv)
	end
in

fun setMethodName mname = thisMethod := mname
fun setClassName cname = thisClass := cname
fun getClassName () = !thisClass

fun getVarTypesAndNames () =
    if !noarglists
    then []
    else List.mapPartial (fn (v,t) =>
                             case Diamond.tyToGTyOpt t of
                                 SOME t' => SOME (t',v)
                               | NONE => NONE)
                         (methodVars())
fun getVarNames () =
    if !noarglists then [] else map snd (getVarTypesAndNames ())

fun varType varname =
    let
	fun findVar [] =
	    phiError ("Can't find types for variable " ^ varname
		    ^ " in function " ^ (!thisMethod))
	  | findVar ((name,ty)::t) =
	    if name = varname then ty else findVar t (* Ouch *)
    in
        if varname = "this" then
            OBJECTty (getClassName())
        else
            findVar (methodVars())
    end

end


fun setctx c = ctx := c
fun setenv e = progEnv := e

end (* local ... in ... *)

fun qualify x =
    if Util.qualified x then
        x
    else
        Util.getBaseName() ^  "." ^ x


fun unarrow (ARROWty(t1, t2)) =  t1 :: (unarrow t2)
  | unarrow t = [t]

fun transFunType ty =
    let
	val u = unarrow ty
    in
	(Diamond.tyToGTyOpt (List.last u),
	List.mapPartial Diamond.tyToGTyOpt (List.take (u, length u - 1)) )
    end

fun transValDec (VALdec((var,_),ty,_)) = (var, transFunType ty)
  | transValDec _ = phiError "Nested classes!"

fun addFunctionType (VALdec((var,_), ty,_)) = addfun var (transFunType ty)
    (* make sure you don't apply this to builtins,
       otherwise you get polymorphic stuff at Grail level *)
  | addFunctionType (CLASSdec((class,_), sup, intfs, mdecs)) = app (addmethod class) (map transValDec mdecs)

fun retTy fname =
    case getfunopt fname of
	SOME (rty, tys) => rty
      | NONE => phiError ("Function "^fname^" not found: did you mean this#"^fname^"?")


fun mTy fname =
    case getfunopt fname of
	SOME (rty, tys) =>  (*getArgTypes *)tys
      | NONE => phiError ("Function "^fname^" not found: did you mean this#"^fname^"?")

fun makeNull(t, c) = G.NULLval(t, SOME(Diamond.typeOfConstructor c))

fun isAtom (VALexp _) = true
  | isAtom _ = false

fun isPrim e =
    case e of
	VALexp _ => true
      | BINexp(oper,a,b,_) =>
	let in
	    case oper of
		EQUALSop => false
	      | LESSop => false
	      | LEQop => false
	      | _ => true
	end
  | UNARYexp _ => true
  | APPexp _ => true
  | CONexp (_,_,NONE,_) => true
  | CONexp (_,_,_,_) => true
  | NEWexp _ => true
  | SUPERMAKERexp _ => true
  | INVOKEexp _ => true
  | UPDATEexp _ => true
  | GETexp _ => true
  | SGETexp _ => true
  | COERCEexp _ => true
  | _  => false

fun transAtom a =
    case a of
	VARval (v,_,_)  => G.VARval v
      | CHARval (i,_)   => G.INTval i
      | INTval (i,_)    => G.INTval i
      | FLOATval (f,_)  => G.FLOATval f
      | STRINGval (s,_) => G.STRINGval s
      | BOOLval (b,_)   => if b then G.INTval 1 else G.INTval 0
      | NULLval (c,_)   => G.NULLval (c, NONE) (* Object *)
      | UNITval _       => phiError "Trying to translate unit as a value"

fun unitval (VARval (v,_,_)) = varType v = UNITty
  | unitval (UNITval _ ) = true
  | unitval _ = false
fun unitexp (VALexp (v,_)) = unitval v
  | unitexp _ = false

fun valGType w =
    case w of
	VARval (v,_,_) => Diamond.tyToGTyOpt (varType v)
      | CHARval _      => SOME (G.INTty)
      | INTval _       => SOME (G.INTty)
      | FLOATval _     => SOME (G.FLOATty)
      | STRINGval _    => SOME (G.REFty ("java.lang.String"))
      | BOOLval _      => SOME (G.BOOLEANty)
      | UNITval _      => NONE
      | NULLval (c,_)  => SOME (G.REFty c)

fun unWrapExp (TYPEDexp(e,t,_)) = e
  | unWrapExp (COERCEexp(e,t,_)) = e
  | unWrapExp e = e


local
   val GrailString = G.REFty "java.lang.String"
in
fun transPrim (VALexp (exp,_)) = G.VALop(transAtom exp)
  | transPrim (BINexp(oper, a, b, _)) =
    let in case oper of
	PLUSop    => G.BINop(G.ADDop, transAtom a, transAtom b)
      |	MINUSop   => G.BINop(G.SUBop, transAtom a, transAtom b)
      | TIMESop   => G.BINop(G.MULop, transAtom a, transAtom b)
      |	DIVop     => G.BINop(G.DIVop, transAtom a, transAtom b)
      |	MODop     => G.BINop(G.MODop, transAtom a, transAtom b)
      | LANDop    => G.BINop(G.ANDop, transAtom a, transAtom b)
      | LORop     => G.BINop(G.ORop, transAtom a, transAtom b)
      | LXORop    => G.BINop(G.XORop, transAtom a, transAtom b)
      | LSLop     => G.BINop(G.SHLop, transAtom a, transAtom b)
      | LSRop     => G.BINop(G.SHRop, transAtom a, transAtom b)
      | ASRop     => G.BINop(G.USHRop, transAtom a, transAtom b)
      | FPLUSop   => G.BINop(G.ADDop, transAtom a, transAtom b)
      |	FMINUSop  => G.BINop(G.SUBop, transAtom a, transAtom b)
      | FTIMESop  => G.BINop(G.MULop, transAtom a, transAtom b)
      |	FDIVop    => G.BINop(G.DIVop, transAtom a, transAtom b)
      |	EQUALSop  => phiError "= found in Phi.transPrim"
      |	LESSop    => phiError "< found in Phi.transPrim"
      |	LEQop     => phiError "<= found in Phi.transPrim"
      | CONCATop  => Perv.translate "append_string" [transAtom a, transAtom b]
    end

  | transPrim (UNARYexp(oper, v, _)) =
    let in case oper of
	NOTop =>  G.BINop(G.SUBop, G.INTval(1), transAtom v)
      | ISNULLop => phiError "ISNULLop not implemented in Phi.transPrim"
    end

  | transPrim(APPexp((fname,loc), args, ext, u)) =
    let
	val (funName, funType) = NAsyntfn.getPhi u
        (* FIX: But what if an arg is variable x of type unit? *)
	fun f v = if unitval v
                              then NONE
                              else SOME (transAtom v)

	val vals = List.mapPartial f args
    in
	case ext of
	    GLOBAL => G.INVOKESTATICop (G.MDESC(retTy fname, qualify fname, mTy fname), vals)
	  | LOCAL => Util.error loc "unexpected LOCAL fun in Phi"
	  | EXTERN =>
	    let
		val (retTy, tys) = transFunType funType
		val md = G.MDESC(retTy, qualify fname, tys)
	    in
		G.INVOKESTATICop(md, vals)
	    end
	  | BUILTIN =>
	    let in
		case
		    fname of
		    "empty" => (* SPECIAL CASE since it requires a type as an argument *)
			let in case args of
				   [n, v] => G.EMPTYop(valOf (f n), valOf (valGType v))
				 | _ => phiError "Bad arguments for make"
			end
		  | "free" =>
		    let in
			(* another SPECIAL CASE;  diamond names might not be available in Perv *)
			case args of
			    [VARval (v,_,_)] =>
			    let
				val diaType = varType v
				val c = case diaType of
					    DIAMONDty s => s
					  | _ => phiError ("Attempting to free non-diamond " ^ v)
				val diaName = Diamond.getDiamondName c
			    in
				G.INVOKESTATICop( G.MDESC(NONE,
							  diaName ^ ".free",
							  [G.REFty diaName]),
						  [G.VARval v])
			    end
			  | _ => phiError "Bad argument for free"
		    end
		  | "diamond_info" =>
		    let in
			case args of
			    [UNITval _] =>
			    G.INVOKESTATICop( G.MDESC(SOME (G.REFty "java.lang.String"),
						      (Util.innerClassName "dia_0")^ ".diamond_info", []), [])
					    (* Fix the above if we ever have multiple diamonds *)
			  | _ => phiError "Bad argument for report"
		    end
		  | "error" =>
		    let
			val () = phiError "'error' unimplemented"  (* WE'D PROBABLY NEED A GRAIL PRIMOP *)
		    in
			case args of
			    [STRINGval (s,_)] =>
			    G.INVOKESTATICop (G.MDESC( NONE,
						       "Camelotlib.exception",
						       [G.REFty "java.lang.String"]),
						       [G.STRINGval s])
			  | _ => phiError "Bad argument for 'error'"
		    end
		  | _ => Perv.translate fname vals
	    end
    end

  | transPrim (CONexp(C as (cname,_), args, NONE,_)) =
    let
	val fDescs = (Diamond.findFieldDescs C)
	val argtys = map (fn G.FDESC(ty,_) => ty) fDescs

	fun f l = (* throw away unit values *)
	    case l of [] => []
		    | h::t =>
		      case h of UNITval _ => f t
			      | _ => h::(f t)

	val args' = map transAtom (f args)
	val diaName = Diamond.getDiamondName cname
	val md = G.MDESC(SOME (G.REFty diaName), diaName ^ ".make", G.INTty :: argtys)
	val tag = G.INTval(Diamond.getTagInfo C)
    in
        if nullRepr cname then
            G.VALop(makeNull(diaName, C))
        else if intRepr cname then
            G.VALop(tag)
        else
	    G.INVOKESTATICop(md, tag::args')
    end

  | transPrim (CONexp(C as (cname,_), args, SOME (a,_),_)) =
    let
	val atype =
	    case varType a (* should make sure this is consistent with type for cname *)
	     of
                DIAMONDty s => Diamond.getDiamondName s
	      | _ => phiError "error 535b"

	val diaName = atype
	val fDescs = Diamond.findFieldDescs C

	val args' = map transAtom args
	val argtys = map (fn G.FDESC(ty,_) => ty) fDescs
	val md = G.MDESC(SOME (G.REFty diaName), diaName ^ ".fill", (G.REFty diaName)::G.INTty :: argtys)
	val tag = G.INTval(Diamond.getTagInfo C)
    in
	G.INVOKESTATICop (md, (G.VARval a)::tag::args')
    end

  | transPrim (INVOKEexp((obj,_), (mname,_), args,u)) =
    let
	val (invokeName, invokeType) = NAsyntfn.getPhi u
(*
	val () = if invokeName <> mname
		 then phiError ("Name mismatch: " ^ invokeName ^ " / " ^ mname)
		 else ()
*)
	fun f v = if unitval v
                  then NONE
                  else SOME (transAtom v)

	val vals = List.mapPartial f args

        val class = case varType obj of
                        OBJECTty c => c
                      | _ => phiError ("Invocation not on object type - "^ (Asyntfn.typeToString (varType obj)))
	val tys = arrowTyToTyList invokeType
	val retTy = Diamond.tyToGTyOpt (List.last tys)
	val argTys = List.mapPartial Diamond.tyToGTyOpt (List.take(tys, length tys - 1))
        val md = G.MDESC(retTy, class^"."^mname, argTys)

	val () = debugPrint ("Phi: generating method "^mname^" of type "^(Asyntfn.typeToString invokeType)^"\n")
    in
	if ClassPath.isInterface class then
            G.INVOKEINTERFACEop (obj, md, vals)
	else
            G.INVOKEVIRTUALop (obj, md, vals)
    end
  | transPrim (NEWexp((class,_), args,u)) =
    let
	val (initName, newType) = NAsyntfn.getPhi u

	fun f v = if unitval v
                  then NONE
                  else SOME (transAtom v)

        (* Used to just use exact type of arguments *)

        val tys = List.mapPartial valGType args

	val () = debugPrint ("Phi: generating initialiser of type "^(Asyntfn.typeToString newType)^"\n")
        (* Now use type inferred in Type.sml to handle multiple constructors *)
	val tys = arrowTyToTyList newType
	val argTys = List.mapPartial Diamond.tyToGTyOpt (List.take(tys, length tys - 1))
        val md = G.MDESC(NONE, class, argTys)
        val vals = List.mapPartial f args
    in
        G.NEWop(md, vals)
    end
  | transPrim (SUPERMAKERexp(args,u)) =
    let

	val (superInitMeth, newType) = NAsyntfn.getPhi u
	fun f v = if unitval v
                  then NONE
                  else SOME (transAtom v)

	val () = debugPrint ("Phi: generating supermaker of type "^(Asyntfn.typeToString newType)^"\n")
        (* Now use type inferred in Type.sml to handle multiple constructors *)
	val tys = arrowTyToTyList newType
	val argTys = List.mapPartial Diamond.tyToGTyOpt (List.take(tys, length tys - 1))
        val md = G.MDESC(NONE, superInitMeth, argTys)
        val vals = List.mapPartial f args
    in
        G.INVOKESPECIALop("this", md, vals)
    end
  | transPrim (UPDATEexp((x,_), v,_)) =
    let
        val ty = valGType v
        val cname = getClassName ()
        val fd = G.FDESC(valOf ty, cname^"."^x)
        val vv = transAtom v
    in
        G.PUTFIELDop("this", fd, vv)
    end
  | transPrim (GETexp((obj,_),(x,_),_)) =
    let
        val cname = getClassName ()
        val ty = Diamond.tyToGTy (Type.getFieldTy INSTANCE (varType obj) x (getCtx()) nowhere)
        val fd = G.FDESC(ty,cname^"."^x)
    in
        G.GETFIELDop(obj,fd)
    end
  | transPrim (SGETexp ((var,_),_)) =
    let
	val x = List.last (String.fields (fn c => c = #".") var)
	val class = truncate var (1+ size x)
        val ty = case getmethodopt (class, x) of
                     SOME (SOME ty, tys) => ty
                   | _ => phiError "Z1: no type for static variable"
    in
        G.GETSTATICop(G.FDESC(ty, var))
    end

  (* We probably have to do CHECKCASTop, but perhaps not in all cases.
     At least it doesn't seem to be required for method arguments, and
     somewhere it is written that checkcast is needed when assigning to a variable.
     It's probably best to do a checkcast whenever the type differs *)
  | transPrim (COERCEexp(e,ty,_)) =
    let in case ty of (OBJECTty class) =>
		(case unWrapExp e of
		     VALexp(VARval (x,_,_), _) => G.CHECKCASTop(class, x)
		   | _ => phiError "Coercion improperly normalised"
		)
	      | _ => phiError "Coercion to non-object type"
    end
  | transPrim _ = phiError "Error Q43: no action for primitive expression"

end (* local *)


(* Note that F and L are stored in reverse order and
   hence must be reversed before their final use *)
(* We also thread info through phi *)
fun phi (LETexp((x,_),e,e',_)) alpha result omega F L =
    let in case varType x of
	UNITty =>
	if isPrim e then
	    let
		fun voiddecs r L = (G.VOIDdec r)::L
		val L'= if unitexp e then L
			else voiddecs (transPrim e) L
	    in
		phi e' alpha result omega F L'
	    end
	else
	    let
		val nu = genfname()
		val (F', L') = phi e alpha (VAR x) nu F L
	    in
		phi e' nu result omega F' L'
	    end

      | _ =>
	if isPrim e then
	    let
		fun valdecs x r L = G.VALdec(x, r)::L
		val L' = valdecs x (transPrim e) L
	    in
		phi e' alpha result omega F L'
	    end
	else
	    let
		val nu = genfname()
		val (F', L') = phi e alpha (VAR x) nu F L
		val () = case e of (LETexp _) => print "WARNING: bad let-floating?\n"
				 | _ => ()
	    in
		phi e' nu result omega F' L'
	    end
    end

  | phi (IFexp (TEST(oper, v1, v2, _), e1, e2, _)) alpha result omega F L =
    let  (* scope for optimisation here *)
	     (* OK, I've done some optimisation,  but it still needs tidied up a bit *)
	fun transCmp LESSop = G.Ltest
	  | transCmp LEQop = G.LEtest
	  | transCmp EQUALSop = G.EQtest
	  | transCmp _ = phiError "Comparison expected in if test"

(* There was previously a special case omitting the check and branch
   in things like "if true then ...".  This has (perhaps temporarily)
   disappeared with the introduction of normalised syntax *)

	fun opres r = G.OPres r
	val isResult = case result of
			   RESULT _ => true
			 | _ => false
	val r1  =
	    if isResult andalso isPrim e1
	    then
		case e1 of
		    VALexp (UNITval _, _) => G.VOIDres
		  | _ =>  opres (transPrim e1)
	    else
		let val nu1 = genfname() in
		    G.FUNres (nu1, getVarNames())
		end
	val r2 =
	    if isResult andalso isPrim e2
	    then
		case e2 of
		    VALexp (UNITval _, _) => G.VOIDres
		  | _ =>  opres (transPrim e2)
	    else
		let val nu2 = genfname() in
		    G.FUNres (nu2, getVarNames())
		end


	val res = G.CHOICEres(transAtom v1, transCmp oper, transAtom v2, r1, r2)

	val newfun = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, res))
	val F' = newfun :: F

        (* Don't generate bodies which will not be referenced, due to compile-time
           test resolution or direct returning of primitive results. *)

	fun phi_r r e result omega F =
	    case r of G.FUNres (nu, _) =>
		      let
			  val (F', _) =  phi e nu result omega F []
		      in F' end
		    | _ =>  F

        val F2 = let
	    val F1 = phi_r r1 e1 result omega F'
	in
	    phi_r r2 e2 result omega F1
	end

    in
	(F2, [])
    end

  | phi (MATCHexp (v, rules, _)) alpha result omega F L =
       if !experimental then phiMatch1 v rules alpha result omega F L
       else phiMatch v rules alpha result omega F L
    (* Just to get the code for matches next to the auxilliary functions it uses *)

  | phi (TYPEDexp(e,ty,_)) alpha result omega F L =
    phi e alpha result omega F L

  | phi (ASSERTexp(e,pre,post,_)) alpha result omega F L =
    (* ignore assertions inside expressions for the time being *)
    phi e alpha result omega F L

  | phi e alpha result omega F L =
    if isPrim e then
	let
	    fun voiddec r = SOME(G.VOIDdec r)
	    and valdec x r = SOME(G.VALdec(x, r))
	    and opres r = G.OPres r
	    val fdec =
		case result of
		    VAR v =>
		    let
			val elet =
                            case varType v of
                                UNITty => if unitexp e then NONE
                                          else voiddec (transPrim e)
                              | _ => valdec v (transPrim e)
			val letdecs = elet::?L
			val res = G.PRIMres(G.FUNres(omega, getVarNames()))
			val f = G.FUNbody(rev letdecs, res)
		    in
			G.FDEC(alpha, getVarTypesAndNames(), f)
		    end
		  | RESULT _ =>
                    let val res = if unitexp e then G.VOIDres
				  else opres (transPrim e)
                    in
			G.FDEC(alpha, getVarTypesAndNames(),
				G.FUNbody(rev L, G.PRIMres(res)))
                    end
	in
	    (fdec::F, [])
	end
    else

(* Is this right? or should we not be continuing with omega? *)

	let in case e of
	     BINexp(oper,a,b, _) =>
	     let
		 val test = (case oper of
				 EQUALSop => G.EQtest
			       | LESSop => G.Ltest
			       | LEQop => G.LEtest
			       | _ => phiError "Primitive expression or </= expected")
		 val true_const = G.VALop(G.INTval(1))
		 val false_const = G.VALop(G.INTval(0))
	     in
 		 case result of
		     VAR v =>
		     let
			 val alpha_t = genfname()
			 val alpha_f = genfname()
			 val res = G.CHOICEres(transAtom a, test, transAtom b,
					       G.FUNres(alpha_t, getVarNames()),
					       G.FUNres(alpha_f, getVarNames()))
			 val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L,res))
			 val lt = G.VALdec(v, true_const)
			 val lf = G.VALdec(v, false_const)
			 val rres = G.PRIMres(G.FUNres(omega, getVarNames()))
			 val ft = G.FDEC(alpha_t, getVarTypesAndNames(), G.FUNbody([lt], rres))
			 val ff = G.FDEC(alpha_f, getVarTypesAndNames(), G.FUNbody([lf], rres))
		     in
			 (ff::ft::f::F, [])
		     end
		   | RESULT _ =>
		     let
			 val res = G.CHOICEres(transAtom a, test, transAtom b,
					       G.OPres(true_const), G.OPres(false_const))
			 val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L,res))
		     in
			 (f::F, [])
		     end
	     end
	   | _ => phiError "Error '44': Expression improperly normalised?"
	end


and makeFunRes f = G.FUNres(f, getVarNames())

and phiMatch (obj,_) rules alpha result omega F L =
    let
	val isObject =
	    case varType obj of
		OBJECTty _ => true
	      | _ => false

	val tname =
	    case varType obj
	     of
		CONty (_, v) => v
	      | OBJECTty x => x
	      | _ => phiError "error 535"

	val () = debugPrint ("varType obj: " ^ (Asyntfn.typeToString (varType obj)) ^ "\n")
        val objtype = Diamond.typeNameToDiamondName tname

	val tag = Normalise.tempName()
	val getTag = G.VALdec(tag, G.GETFIELDop (obj, G.FDESC (G.INTty, objtype^".$")))
	val () = if intType tname then println (tname ^ " is intType") else ()
        val tag = if intType tname then obj else tag
	val L' =  if (typeHasNull tname        (* Can't get tag until after null test *)
		      orelse isObject          (* No tag required *)
		      orelse intType tname     (* tag = obj *)
		      orelse length rules = 1) (* Only one choice:  needn't look at tag *)
		  then L
                  else getTag::L
    in
	if typeHasNull tname then
	    let
		val (nrule, rules') =
		    let
			fun f [] acc = phiError "absence of nullity"
			  | f (h::t) acc = (* isolate rule for null constructor *)
			    let in
				case h of
				    MATCHrule ((c,_), args, dia, e, _) =>
				    if nullRepr c then (h, List.revAppend (acc, t))
				    else f t (h::acc)
				  | OOMATCHrule _ => phiError "Found OOMATCHrule with null constructor"
			    end
		    in
			f rules []
		    end
	    in
		nullMatchTest nrule rules' getTag alpha result omega obj objtype tag F L'
                (* Have to pass getTag since we mustn't emit it until after the null test *)
	    end
	else
	    matchTest rules alpha result omega obj objtype tag F L' []
    end


and nullMatchTest nrule rules getTag alpha result omega obj objtype tag F L =
    case nrule of
	MATCHrule(con,args,addr,mexp,_) =>
	let in
	    case rules of
		[] => phi mexp alpha result omega F L (* Some fool has defined a type t = !X *)
	      | [MATCHrule(c',v',a',e',_)] => (* Only one other rule: don't need tag *)
		let
		    val (f1, rhos)  = matchResult obj con args mexp addr []
		    val (f2, rhos') = matchResult obj c' v' e' a' rhos
		    val nulltest =
			G.CHOICEres(G.VARval obj,
				    G.EQtest,
				    makeNull (objtype, con),
				    makeFunRes f1,
				    makeFunRes f2)

		    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, nulltest))
		in
		    makeRhos result omega rhos' (f::F) []
		end

	      | _ =>
		let
		    val (f1, rhos)  = matchResult obj con args mexp addr []
		    val fnext = genfname()
		    val nulltest =
			G.CHOICEres(G.VARval(obj),
				    G.EQtest,
				    makeNull(objtype, con),
				    makeFunRes f1,
				    makeFunRes fnext)

		    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, nulltest))
		    val F' = f::F
		in
		    matchTest rules fnext result omega obj objtype tag F' [getTag] rhos
		end
	end
      | OOMATCHrule _ => phiError "Found OOMATCHrule in nullMatchTest"

and matchTest rules alpha result omega obj objtype tag F L rhos =
    case rules of
	[] => phiError "Ruleless matchTest"

      | [MATCHrule(con,args,addr,e,_)] =>
        (* Special case - one rule only *)
        (* Omit the instruction to retrieve the tag (we don't need it) and continue
           immediately by extracting the fields and then jumping to the computation
           of the "continuation" e *)
	(* There is/was is problem here with the interaction of this with localvar
           consolidation *)
        let
	    val (cn,_) = con
	    val () = if null rhos then ()
		     else phiError "Something peculiar has happened"
	    val G = rhoHeader obj con args addr
	in
	    phi e alpha result omega F (List.revAppend(G, L))
	end

      | [MATCHrule(c,v,a,e,_), MATCHrule(c',v',a',e',_)] => (* We've got to the final two rules *)
        let
            val (f1, rhos') = matchResult obj c v e a rhos
            val (f2, rhos'') = matchResult obj c' v' e' a' rhos'

	    val ifex =
		G.CHOICEres(G.VARval tag,
			    G.EQtest,
			    G.INTval(Diamond.getTagInfo c),
			    makeFunRes f1,
			    makeFunRes f2
			   )

	    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, ifex))
	val F' = f::F
	in
            makeRhos result omega rhos'' F' []
	end

      | MATCHrule(con,args,addr,mexp,_)::nextrules =>
	let
            val (f1, rhos')  = matchResult obj con args mexp addr rhos

	    val fnext = genfname()
	    val ifex =
		G.CHOICEres(G.VARval tag,
			    G.EQtest,
			    G.INTval(Diamond.getTagInfo con),
			    makeFunRes f1,
			    makeFunRes fnext
			   )

	    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, ifex))
	    val F' = f::F
	in
	    matchTest nextrules fnext result omega obj objtype tag F' [] rhos'
	end

  (* ---------------- Object rules ---------------- *)

  (* object default case, any rules after are unreached *)
  | OOMATCHrule(ANYCLASSpat, e,_)::_ => phi e alpha result omega F L

  (* object interesting case *)
  | OOMATCHrule(CLASSpat((var,_), (class,_)), e,_)::nextrules =>
    let
        (* Should maybe check that rhos is null *)
	val () = debugPrint ("OOMR: "^var)
	val fnext = genfname()
	val fexp = genfname()
	val tmp = Normalise.tempName()
	val () = debugPrint (", phi: "^fexp)

	val Lexp = [G.VALdec(var, G.CHECKCASTop(class, obj))]
	val (F', _) = phi e fexp result omega [] Lexp

	val L' = (G.VALdec(tmp, G.INSTANCEop(class, obj)))::L
	val ifex = G.CHOICEres(G.VARval tmp,
			       G.EQtest,
			       G.INTval(1),
			       makeFunRes fexp,
			       makeFunRes fnext)
	val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L', ifex))
	val F'' = f::F'@F

	val () = debugPrint (", fun: "^alpha)
    in
	matchTest nextrules fnext result omega obj objtype tag F'' [] []
    end


  (* ================ Experimental: use new Grail case statement ================ *)

and phiMatch1 (obj,_) rules alpha result omega F L =
    let
	val isObject =
	    case varType obj of
		OBJECTty _ => true
	      | _ => false

	val tname =
	    case varType obj
	     of
		CONty (_, v) => v
	      | OBJECTty x => x
	      | _ => phiError ("error 535")

	val () = debugPrint ("varType obj: " ^ (Asyntfn.typeToString (varType obj)) ^ "\n")
        val objtype = Diamond.typeNameToDiamondName tname

	val tag = Normalise.tempName()
	val getTag = G.VALdec(tag, G.GETFIELDop (obj, G.FDESC (G.INTty, objtype^".$")))

        val tag = if intType tname then obj else tag
	val L' =  if (typeHasNull tname        (* Can't get tag until after null test *)
		      orelse isObject          (* No tag required *)
		      orelse intType tname     (* tag = obj *)
		      orelse length rules = 1) (* Only one choice:  needn't look at tag *)
		  then L
                  else getTag::L
    in
	if typeHasNull tname then
	    let
		val (nrule, rules') =
		    let
			fun f [] acc = phiError "absence of nullity"
			  | f (h::t) acc = (* isolate rule for null constructor *)
			    let in
				case h of
				    MATCHrule ((c,_), args, dia, e, _) =>
				    if nullRepr c then (h, List.revAppend (acc, t))
				    else f t (h::acc)
				  | OOMATCHrule _ => phiError "Found OOMATCHrule with null constructor"
			    end
		    in
			f rules []
		    end
	    in
		nullMatchTest1 nrule rules' getTag alpha result omega obj objtype tag F L'
                (* Have to pass getTag since we mustn't emit it until after the null test *)
	    end
	else
	    matchTest1 rules alpha result omega obj objtype tag F L' []
    end


and nullMatchTest1 nrule rules getTag alpha result omega obj objtype tag F L =
    case nrule of
	MATCHrule(con,args,addr,mexp,_) =>
	let in
	    case rules of
		[] => phi mexp alpha result omega F L (* Some fool has defined a type t = !X *)

	      | [MATCHrule(c',v',a',e',_)] => (* Only one other rule: don't need tag *)
		let
		    val (f1, rhos)  = matchResult obj con args mexp addr []
		    val (f2, rhos') = matchResult obj c' v' e' a' rhos
		    val nulltest =
			G.CHOICEres(G.VARval(obj),
				    G.EQtest,
				    makeNull(objtype, con),
				    makeFunRes f1,
				    makeFunRes f2)

		    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, nulltest))
		in
		    makeRhos result omega rhos' (f::F) []
		end

	      | _ =>
		let
		    val (f1, rhos)  = matchResult obj con args mexp addr []
		    val fnext = genfname()
		    val nulltest =
			G.CHOICEres(G.VARval(obj),
				    G.EQtest,
				    makeNull (objtype, con),
				    makeFunRes f1,
				    makeFunRes fnext)

		    val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L, nulltest))
		    val F' = f::F
		in
		    matchTest1 rules fnext result omega obj objtype tag F' [getTag] rhos
		end
	end
      | OOMATCHrule _ => phiError "Found OOMATCHrule in nullMatchTest"

and matchTest1 rules alpha result omega obj objtype tag F L rhos =
    case rules of
	[] => phiError "Ruleless matchTest"

  | [MATCHrule(con,args,addr,e,_)] =>
    (* Special case - one rule only *)
    (* Omit the instruction to retrieve the tag (we don't need it) and continue
       immediately by extracting the fields and then jumping to the computation
       of the "continuation" e *)
    let
	val () = if null rhos then ()
		 else phiError "Something peculiar has happened"
	val G = rhoHeader obj con args addr
    in
	phi e alpha result omega F (List.revAppend(G, L))
    end

  | rules as ((MATCHrule _) :: _) =>
    let
	fun getTag (MATCHrule (c,_,_,_,_)) = Diamond.getTagInfo c
	  | getTag _ = phiError "OOMATCHrule 7"
	val rules' = Listsort.sort (fn (x,y) => Int.compare(getTag x, getTag y)) rules
	val low = getTag (hd rules')
	val args = getVarNames ()

	fun g n l results rhos =
	    case l of
		[] => (rev results, rhos, n)
	      | (MATCHrule(c,v,a,e,_)::t) =>
		let
		    val (fname, rhos') = matchResult obj c v e a rhos
		in
		    g (n+1) t ((n,fname,args)::results) rhos'
		end
	      | _ => phiError "Error c: unexpected OOMATCHrule "

	val (results, rhos', h) = g (getTag (hd rules')) rules' [] rhos
	val caseExp = G.CASEres (tag, low, h-1, results)

	val f = G.FDEC (alpha, getVarTypesAndNames(), G.FUNbody (rev L, caseExp))

	val F' = f::F
    in
        makeRhos result omega rhos' F' []
    end

  (* ---------------- Object rules ---------------- *)

  (* object default case, any rules after are unreached *)
  | OOMATCHrule(ANYCLASSpat, e,_)::_ => phi e alpha result omega F L

  (* object interesting case *)
  | OOMATCHrule(CLASSpat((var,_), (class,_)), e,_)::nextrules =>
    let
        (* Should maybe check that rhos is null *)
	val () = debugPrint ("OOMR: "^var)
	val fnext = genfname()
	val fexp = genfname()
	val tmp = Normalise.tempName()
	val () = debugPrint (", phi: "^fexp)

	val Lexp = [G.VALdec(var, G.CHECKCASTop(class, obj))]
	val (F', _) = phi e fexp result omega [] Lexp

	val L' = (G.VALdec(tmp, G.INSTANCEop(class, obj)))::L
	val ifex = G.CHOICEres(G.VARval tmp,
			       G.EQtest,
			       G.INTval(1),
			       makeFunRes fexp,
			       makeFunRes fnext)
	val f = G.FDEC(alpha, getVarTypesAndNames(), G.FUNbody(rev L', ifex))
	val F'' = f::F'@F

	val () = debugPrint (", fun: "^alpha)
    in
	matchTest nextrules fnext result omega obj objtype tag F'' [] []
    end

  (* ================ End experimental stuff ================ *)



(* Generate code to carry out actual computations: corresponds to \rho functions in document *)
(* l is a bunch of valdecs which extract the fields for the constructor corresponding to
   the expression e (possibly followed by some code to free the diamond object or bind
   it to a diamond variable).   p is the label for this block *)

and makeRhos result omega rhos F L =
    let
	fun f rhos F L =
	    case rhos of
		[] => (F, L)
	      | (l,e,p)::rhos' =>
		let
		    val (F', L') = phi e p result omega F l
		in
		    f rhos' F' L'
		end
    in
	case L of
	    [] => f (rev rhos) F L
	  | _ => phiError "Nonempty letdecs in makeRhos"
    end

and rhoHeader obj (C as (con,_)) args addr = (* G-sequence from document *)
    let
	fun matchExtract obj con vars =
	    let fun ext i l acc =
		    case l of [] => acc
			    | v::vs =>
			      case varType v of
				  UNITty => ext i vs acc
				| _ =>  ext (i+1) vs
					    (G.VALdec (v, G.GETFIELDop (obj, Diamond.getFieldDesc (con, i))) :: acc)
	    in
		ext 0 vars []
	    end
		(* It would be good to change this so that it omits fields
		   which aren't actually used *)
	val fields = matchExtract obj C (map nameOf args)
    in
	case addr of NOWHERE => fields
		   | SOMEWHERE (a,_) => (G.VALdec(a,G.VALop(G.VARval obj)))::fields
		   | DISPOSE =>
		     (* We allow @_ annotations for nil and integer constructors,
			for Steffen's convenience.  We have to make sure we don't
			try to free the diamond for such values *)
		     if (intRepr con orelse nullRepr con) then fields
		     else
			 let
			     val diaName = Diamond.getDiamondName con
			     val diaType = G.REFty diaName
			 in (G.VOIDdec(G.INVOKESTATICop(
				       G.MDESC(NONE,
					       diaName ^ ".free",
					       [diaType]),
				       [G.VARval obj]))) :: fields
			 end
    end


and matchResult obj con args e addr rhos =
    let
	val nu = genfname()
	val G = rhoHeader obj con args addr
    in
	(nu, (G, e, nu)::rhos)
    end

and revp name l =
    let val l' = rev l
	val () = (print "====\n"; print (name ^"\n"))
	val () = app printVarTy l'
    in l' end


fun f_instanceFlag INSTANCE = NONE
  | f_instanceFlag STATIC = SOME Classdecl.F_ACCstatic

fun m_instanceFlag INSTANCE = NONE
  | m_instanceFlag STATIC = SOME Classdecl.M_ACCstatic

(*
fun fixInit (mdef as G.MDEF(flags, retty, fname, tys, body)) super =
     let
         fun fixInitBody (G.MBODY([], funs, res)) =
             let
                 val resfn = genfname()
                 val superInit = case super of
                                     SOME class => class^".<init>"
                                   | NONE       => "java.lang.Object.<init>"
                 val func = G.FDEC(resfn, getVarTypesAndNames(),
                                   G.FUNbody([G.VOIDdec(G.INVOKESPECIALop("this ",
                                                                          G.MDE SC(NONE, superInit, []),
                                                                          []))] ,
                                             res))
             in
                 G.MBODY([], func::funs, G.PRIMres(G.FUNres(resfn, getVarNames( ))))
             end
           | fixInitBody _ = phiError "Method has lets"
     in
         if fname = "<init>" then G.MDEF(flags, retty, fname, tys, fixInitBody body)
         else mdef
     end
*)

fun transArgs l =
    case l of
	[] => []
      | h::t =>
	case h of
	    UNITvar => transArgs t
	  | Normsyn.VAR(name,_)=>
	    case varType name of
		UNITty => transArgs t
	      | ty => (Diamond.tyToGTy ty,name)::(transArgs t)

fun method (FUNdef((fname,_), params, inst, e, loc)) =
    let
	val () = setMethodName fname
        val () = setClassName "" (* FIX: ugly *)
	val () = resetfnames()
	val alpha = "f_" ^ fname
	val omega = "Omega"

	val (F, L) = phi e alpha (RESULT (retTy fname)) omega [] []
	val () = case L of
		     [] => ()
		   | _ => phiError ("Unused letdecs after phi("^fname^")");


	val m = G.MDEF(rev (m_instanceFlag inst ::? [Classdecl.M_ACCpublic]), (* to get "public static" *)
		       retTy fname, fname, transArgs params,
		       G.MBODY([], rev F, G.PRIMres(G.FUNres(alpha, getVarNames()))))
    in
	m
    end

fun funblock (FUNblock b) = map method b

fun transVal (VALdec((var,_),ty,inst)) =
    let
        val gty = Diamond.tyToGTy ty
    in
        G.FDEF(f_instanceFlag inst ::?[], gty, var)
    end
  | transVal (CLASSdec _) = phiError "Nested classes"

fun transMeth cname super (FUNdef((fname,_), params, inst, e, _)) =
    let
        val (rTy, _) = getmethod(cname,fname)
	val () = setMethodName fname
        val () = setClassName cname
	val () = resetfnames()
	val alpha = "f_" ^ fname
	val omega = "Omega"

	val (F, L) = phi e alpha (RESULT rTy) omega [] []
	val () = case L of
		     [] => ()
		   | _ => phiError "Unused letdecs after phi"


	val m = G.MDEF(m_instanceFlag inst ::? [Classdecl.M_ACCpublic],
		       rTy, fname,
		       transArgs params,
		       G.MBODY([], rev F, G.PRIMres(G.FUNres(alpha, getVarNames()))))
    in
	m
    end


(*
  | transPrim (SUPERMAKERexp(args,(loc,_,p))) =
    let

	val (superInitMeth, newType) = case p of SOME q => q | NONE => error loc "doublet"
	fun f v = if unitval v
                  then NONE
                  else SOME (transAtom v)

	val () = debugPrint ("Phi: generating supermaker of type "^(Asyntfn.typeToString newType)^"\n")
        (* Now use type inferred in Type.sml to handle multiple constructors *)
	val tys = arrowTyToTyList newType
	val argTys = List.mapPartial Diamond.tyToGTyOpt (List.take(tys, length tys - 1))
        val md = G.MDESC(NONE, superInitMeth, argTys)
        val vals = List.mapPartial f args
    in
        G.INVOKESPECIALop("this", md, vals)
    end
*)


fun class (CLASSdef((cname,_), super, intfs, vals, meths)) =
    let
        val fields = map transVal vals
        val gmeths = map (transMeth cname super) meths
    in
        G.CDEF([Classdecl.C_ACCpublic, Classdecl.C_ACCsuper],
               cname, Option.map nameOf super, map nameOf intfs,
               fields, gmeths, NONE)
    end

fun phi' (PROG(_, vals,classes, fblocks)(* :Loc.Location Program*) ) env ctx dataInfo =
let
    val () = setenv env
    val () = setctx ctx
    val vals' = List.filter Mono.safeDec vals
    (* The second typecheck puts some polymorphic valdecs back into the program;
       we have to get rid of them again. *)
    val () = app addFunctionType vals'

    val () = addRepresentations dataInfo

    val meths = List.concat (map funblock fblocks)
    val gclasses = map class classes

    val flags = []
    val fields = []
in
    (G.CDEF(flags, Util.getBaseName(), NONE, [], fields, meths, NONE) :: gclasses)
end

end (* local structure ...  *)
