(* Built-in functions *)

(* Explanation:  built-in functions are represented by triples
   (name, type, tr).  Surprisingly,  "name" is the name of the function
   and "type" is its Camelot (arrow) type.  "tr" is a function which
   when applied to a list of Grail argument values returns a chunk of Grail
   which computes the result of the function call.  For builtins which
   correspond to Grail primops these are written down explicitly.  For calls
   to things in Camelotlib there's a function which produces an appropriate
   invokestatic operation.  (It would be good to generate these automatically
   from the Java source but there isn't enough information to generate
   the Camelot types.) *)


local
    open Util
    open Absyn
    structure A = Absyn
    val () = Absyn.required
    structure G = GrailAbsyn
    val () = GrailAbsyn.required
    val GrailString = G.REFty "java.lang.String"
    structure P = Polyhash
in

exception basisError of string

val builtInHash: (string, A.Ty * (G.Value list -> G.PrimOp) * bool) P.hash_table
    = P.mkTable(P.hash,op=)(20,basisError "Bad constructor index");
(* boolean value is used to mark functions which have actually been used in the
   current program *)


fun isBuiltIn s = case P.peek builtInHash s of NONE => false | _ => true
fun isBuiltInL (s,_) = isBuiltIn s

fun translate f  = let val (_,tr,_) = P.find builtInHash f in tr end
  (* return function which when applied to a list of (Grail)
     arguments returns Grail code for computing f applied
     to the arguments.  Called from Phi.sml.
   *)

fun err(name) = Util.exit
            ("Internal error: predefined function " ^ name
	      ^ "\ncalled with wrong number of arguments")

val alpha = TVARty {name="perv", eq=ref false, ord=ref false}
(* For polymorphic functions.  I wonder if there's any danger of 'a
   interfering with types in the user program? *)

fun makeArrowType [] r = ARROWty (UNITty, r)
  | makeArrowType [t] r = ARROWty (t, r)
  | makeArrowType (h::t) r = ARROWty(h, makeArrowType t r)

fun makeLibraryCall (cname, argtypes, rtype, jname) =
    (cname, (* Camelot name *)
     makeArrowType argtypes rtype,
     fn args =>
	G.INVOKESTATICop (
	  G.MDESC (Diamond.tyToGTyOpt rtype, jname, map Diamond.tyToGTy argtypes),
	  args)
    )
(* Usually jname = "Camelotlib." ^ cname,  but I haven't done this automatically
   in case we need a library function whose name isn't a valid Java name *)


val builtIns = [

(* Things which translate to grail primops *)

("empty", ARROWty (INTty, ARROWty(alpha, ARRAYty alpha)),
    fn [v1, v2] =>  G.EMPTYop(v1, G.INTty)
     | _ => err("empty")),

(* The entry for "empty" is fake;  "empty" is dealt with as a special case
   in Phi.sml since it requires a type as a parameter,  rather than a list
   of values like everything else. *)

    ("make",
     ARROWty (INTty, ARROWty(alpha, ARRAYty alpha)),
     fn [v1, v2] =>  G.MAKEop(v1, v2) | _ => err("make")),

    ("get",
     ARROWty (ARRAYty alpha, ARROWty(INTty, alpha)),
     fn [v1, v2] => G.GETop(v1, v2) | _ => err("get")),

    ("set",
     ARROWty (ARRAYty alpha, ARROWty(INTty, ARROWty (alpha, UNITty))),
     fn [v1, v2, v3] => G.SETop(v1, v2, v3) | _ => err("set")),

    ("arraylength",
     ARROWty (ARRAYty alpha, INTty),
     fn [v] => G.LENGTHop v | _ => err("Array.length")),

    ("int_of_float",
     ARROWty (FLOATty, INTty),
     fn [v] => G.FTOIop v | _ => err("int_of_float")),

    ("float_of_int",
     ARROWty (INTty, FLOATty),
     fn [v] => G.ITOFop v | _ => err("float_of_int")),

    ("code",                     (* same as int_of_char *)
     ARROWty (CHARty, INTty),
     fn [v] => G.VALop v | _ => err("code")),

    ("int_of_char",
     ARROWty (CHARty, INTty),
     fn [v] => G.VALop v | _ => err("int_of_char")),

    ("chr",                      (* same as char_of_int *)
     ARROWty (INTty, CHARty),
     fn [v] => G.VALop v | _ => err("chr")),

    ("char_of_int",
     ARROWty (INTty, CHARty),
     fn [v] => G.VALop v | _ => err("char_of_int"))

]

@ map makeLibraryCall [
(* Things which translate to library calls *)
(* Instead of having the library we could just inline the code which
   we would call in the library;  this might give big code. *)
(* Also,  this is easier for experimentation *)


       ("free",          (* "free" method from diamond class *)
       [DIAMONDty ""],  (* This won't work with multiple diamond types *)
       UNITty,
       (Util.innerClassName "dia_0") ^ ".free"),

      (* This entry is here so that the signature is available to the typechecker.
         Calls to "free" are intercepted in Phi.sml and the code is generated there;
	 we can't do it here because if we have multiple diamonds it won't be possible
         to determine the name of the appropriate class at this point in compilation. *)

       ("diamond_info",          (* from diamond class *)
       [],  (* This won't work with multiple diamond types *)
       STRINGty,
       (Util.innerClassName "Dia_0") ^ ".diamond_info"),

      ("nullString",
       [UNITty],
       STRINGty,
       "Camelotlib.nullString"),

      ("print_int",
       [INTty],
       UNITty,
       "Camelotlib.print_int"),

      ("print_float",
       [FLOATty],
       UNITty,
       "Camelotlib.print_float"),

      ("print_string",
       [STRINGty],
       UNITty,
       "Camelotlib.print_string"),

      ("print_char",
       [CHARty],
       UNITty,
       "Camelotlib.print_char"),

      ("print_newline",
       [],
       UNITty,
       "Camelotlib.print_newline"),

      ("print_int_newline",
       [INTty],
       UNITty,
       "Camelotlib.print_int_newline"),

      ("print_float_newline",
       [FLOATty],
       UNITty,
       "Camelotlib.print_float_newline"),

      ("print_string_newline",
       [STRINGty],
       UNITty,
       "Camelotlib.print_string_newline"),

      ("print_char_newline",  (* This'll get used a LOT *)
       [CHARty],
       UNITty,
       "Camelotlib.print_char_newline"),

      ("int_of_string",
       [STRINGty],
       INTty,
       "Camelotlib.int_of_string"),

      ("string_of_int",
       [INTty],
       STRINGty,
       "Camelotlib.string_of_int"),

      ("string_of_char",
       [CHARty],
       STRINGty,
       "Camelotlib.string_of_char"),

      ("float_of_string",
       [STRINGty],
       FLOATty,
       "Camelotlib.float_of_string"),

      ("string_of_float",
       [FLOATty],
       STRINGty,
       "Camelotlib.string_of_float"),

      (* strings *)

      ("string_length",
       [STRINGty],
       INTty,
       "Camelotlib.string_length"),

      ("append_string", (* The lexer translates ^ into this *)
       [STRINGty, STRINGty],
       STRINGty,
       "Camelotlib.append_string"),

      ("same_string",
       [STRINGty, STRINGty],
       BOOLty,
       "Camelotlib.same_string"),

      ("string_compare",
       [STRINGty, STRINGty],
       INTty,
       "Camelotlib.string_compare"),

      ("substring",
       [STRINGty, INTty, INTty],
       STRINGty,
       "Camelotlib.substring"),

      ("getc",
       [STRINGty, INTty],
       CHARty,
       "Camelotlib.getc"),

      (* file input *)

      ("open_input",
       [STRINGty],
       UNITty,
       "Camelotlib.open_input"),

      ("close_input",
       [],
       UNITty,
       "Camelotlib.close_input"),

      ("read_line",
       [],
       STRINGty,
       "Camelotlib.read_line"),

      ("read_char",
       [],
       CHARty,
       "Camelotlib.read_char"),

      ("eof",
       [],
       BOOLty,
       "Camelotlib.eof"),

      ("sleep",
       [INTty],
       UNITty,
       "Camelotlib.sleep"),

      ("isnullobj",
       [OBJECTty "java.lang.Object"],
       BOOLty,
       "Camelotlib.isnullobj"),

      ("date",
       [],
       STRINGty,
       "Camelotlib.date"),

      ("error",
       [STRINGty],
       alpha,  (* This doesn't work since we'd need polymorphic Grail *)
       "Camelotlib.exception"
      )

]



fun initBuiltIns() =
    app (fn (name, ty, tr)
	    => (P.insert builtInHash (name, (ty,tr,false)))) builtIns

val () = initBuiltIns()  (* Presumably this is called at an appropriate time *)

fun markUsed s =
    case P.peek builtInHash s of
	NONE => err ("Can't find built-in function " ^s)
      | SOME (ty,tr,b) => P.insert builtInHash (s, (ty,tr,true))

fun markUsedL (s,_) = markUsed s

fun builtinTypes () =
    let
	val l = P.listItems builtInHash
	fun f [] acc = acc
	  | f ((name,(ty,_,false))::t) acc = f t acc
	  | f ((name,(ty,_,true))::t) acc =
	    f t (A.VALdec((name,nowhere), ty, STATIC)::acc)
    in
	f l []
    end

fun builtinArgSizes () =
    let
	val l = P.listItems builtInHash
	fun nargs (ARROWty(a,b)) = 1 + nargs b
	  | nargs _ = 0
	fun f [] acc = acc
	  | f ((name,(ty,_,false))::t) acc = f t acc
	  | f ((name,(ty,_,true))::t) acc = f t ((name, nargs ty)::acc)
    in
	f l []
    end


(* These functions export type information to the typechecker
   and other parts of the compiler. *)

end (* local structure G = ... *)
