(* Miscellaneous functions on abstract syntax *)

open Normsyn Util

fun collapse fblocks = List.concat (map (fn FUNblock b => b) fblocks)

val showExtern = ref false

val mkTvar       = Asyntfn.mkTvar
val isEq         = Asyntfn.isEq
val isOrd        = Asyntfn.isOrd
val isEqType     = Asyntfn.isEqType
val isOrdType    = Asyntfn.isOrdType
val typeToString = Asyntfn.typeToString
val extToString  = Asyntfn.extToString


fun realToString r = String.translate (fn #"~" => "-" | c => Char.toString c) (Real.toString r)
fun intToString n  = String.translate (fn #"~" => "-" | c => Char.toString c) (Int.toString n)

val printArgTypes = ref false   (* Steffen's program doesn't like argtype annotations *)
fun setPrintArgTypes b = printArgTypes := b

val printSpace = ref true
fun setPrintSpace b = printSpace := b

fun valToString v =
    case v of
	VARval (w,_,_) => w
      | INTval (n, _) => intToString n
      | FLOATval (x, _) => realToString x
      | CHARval (n, _) => "'" ^ (Char.toString (chr n)) ^ "'"
      | STRINGval (s, _) => "\"" ^ s ^ "\""
      | BOOLval (true, _) => "true"
      | BOOLval (false, _) => "false"
      | UNITval _ => "()"
      | NULLval (s, _) => "(null :> " ^s ^ ")"

fun getvU v =
    case v of
	VARval    (_,_,u) => u
      | CHARval   (_,u) => u
      | INTval    (_,u) => u
      | FLOATval  (_,u) => u
      | STRINGval (_,u) => u
      | BOOLval   (_,u) => u
      | UNITval      u  => u
      | NULLval   (_,u) => u

fun getU e =
    case e of
	VALexp (_,u) => u
      | UNARYexp (_,_,u) => u
      | BINexp (_,_,_,u) => u
      | IFexp (_,_,_,u) => u
      | MATCHexp (_,_,u) => u
      | LETexp (_,_,_,u) => u
      | APPexp (_,_,_,u ) => u
      | CONexp (_,_,_,u) => u
      | TYPEDexp (_,_,u) => u
      | COERCEexp (_,_,u) => u
      | NEWexp (_,_,u) => u
      | SUPERMAKERexp (_,u) => u
      | INVOKEexp (_,_,_,u) => u
      | UPDATEexp (_,_,u) => u
      | GETexp (_,_,u) => u
      | SGETexp (_,u) => u
      | ASSERTexp (_,_,_,u) => u


(* Operations on annotations *)

fun getMono a =
    case a of
	LOC l => error a "Missing monomorphisation annotation"
      | MONO (i,a) => i
      | PHI (_,a) => getMono a

fun mapMono f a =
    case a of
	LOC _ => a
      | MONO (m,a) => MONO (f m, a)  (* mapMono f a ? *)
      | PHI (p,a) => PHI (p, mapMono f a)

fun getPhi a =
    case a of
	LOC l => error a "Missing phi annotation"
      | MONO (_,a) => getPhi a
      | PHI (i,a) => i

fun mapPhi f a =
    case a of
	LOC _ => a
      | MONO (m,a) => MONO(m, mapPhi f a)
      | PHI (i,a) => PHI(f i, a)  (* mapPhi f a ? *)


(* Mapping over 'a Absyn *)

val mapName       = Asyntfn.mapName
and mapTy         = Asyntfn.mapTy
and mapHeapUsage  = Asyntfn.mapHeapUsage
and mapTypeCon    = Asyntfn.mapTypeCon
and mapTypeDec    = Asyntfn.mapTypeDec
and mapFunAssns   = Asyntfn.mapFunAssns
and mapHints      = Asyntfn.mapHints
and mapMatchDiam  = Asyntfn.mapMatchDiam
and mapOOMatchPat = Asyntfn.mapOOMatchPat

and mapValDec     = Asyntfn.mapValDec
and mapVarTy      = Asyntfn.mapVarTy

fun mapValue f v =
    case v of
	VARval (v, ext, l) => VARval (v,ext, f l)
      | INTval (n, l)      => INTval (n, f l)
      | FLOATval (x, l)    => FLOATval (x, f l)
      | CHARval (n, l)     => CHARval (n, f l)
      | STRINGval (s, l)   => STRINGval (s, f l)
      | BOOLval (true, l)  => BOOLval (true, f l)
      | BOOLval (false, l) => BOOLval (false, f l)
      | UNITval l          => UNITval (f l)
      | NULLval (s, l)     => NULLval (s, f l)

fun mapTest f (TEST (oper, v, w, u)) = TEST(oper, mapValue f v, mapValue f w, f u)

fun mapExp f e =
    case e of
	VALexp (v,u) => VALexp(mapValue f v, f u)
      | UNARYexp (oper, v, u) => UNARYexp (oper, mapValue f v, f u)
      | BINexp (oper, v, w, u) => BINexp (oper, mapValue f v, mapValue f w, f u)
      | IFexp (test, e1, e2, u) =>
	  IFexp (mapTest f test, mapExp f e1, mapExp f e2, f u)
      | MATCHexp (n,l,u) => MATCHexp (mapName f n, map (mapMatchRule f) l, f u)
      | LETexp (v, e, e',u) => LETexp (mapName f v, mapExp f e, mapExp f e', f u)
      | APPexp (n, l, ext, u) => APPexp (mapName f n, map (mapValue f) l, ext, f u)
      | CONexp (v, vs, addr, u) =>
	  CONexp (mapName f v, map (mapValue f) vs, Option.map (mapName f) addr, f u)
      | TYPEDexp (e,t,u) => TYPEDexp (mapExp f e, mapTy f t, f u)
      | COERCEexp (e,t,u) => COERCEexp (mapExp f e, mapTy f t, f u)
      | NEWexp (class, vs,u) => NEWexp (class, map (mapValue f) vs, f u)
      | SUPERMAKERexp (vs,x) => SUPERMAKERexp(map (mapValue f) vs, f x)
      | INVOKEexp (obj,mname,vs,u) => INVOKEexp (mapName f obj, mname, map (mapValue f) vs, f u)
      | UPDATEexp (a, v, u) => UPDATEexp (a, mapValue f v, f u)
      | GETexp (obj, var, u) => GETexp (mapName f obj, var, f u)
      | SGETexp (var, u) => SGETexp (var, f u)
      | ASSERTexp (e, as1, as2, u) => ASSERTexp (mapExp f e, as1, as2, f u)

and mapMatchRule f r =
    case r of
        MATCHrule (con, args, d, e, u)
	=> MATCHrule (mapName f con, map (mapName f) args, mapMatchDiam f d,
		      mapExp f e, f u)
      | OOMATCHrule (p, e, u) => OOMATCHrule (mapOOMatchPat f p, mapExp f e, f u)

fun mapFunDef f (FUNdef (fname, args, inst, e, u)) =
    FUNdef (mapName f fname, map (mapVarTy f) args,
	    inst, mapExp f e, f u)

fun mapFunBlock f (FUNblock l) =
    FUNblock (map (mapFunDef f) l)

fun mapClassDef f (CLASSdef (class, super, imps, vals, methods)) =
    CLASSdef (mapName f class,
	      Option.map (mapName f) super,
	      map (mapName f) imps,
	      map (mapValDec f) vals,
	      map (mapFunDef f) methods)

fun mapProgram f (PROG(typedecs,valdecs,classdefs,funblocks)) =
    PROG (map (mapTypeDec f) typedecs,
	  map (mapValDec f) valdecs,
	  map (mapClassDef f) classdefs,
	  map (mapFunBlock f) funblocks)


(* ---------------- Fix externals  ---------------- *)
(* Uses of local variables/function arguments as
   function names have to be marked with LOCAL.  This
   traverses the syntax and fixes things. I've just done
   this by quickly modifying Asyntfn.resolveClassNames,
   so there's lots of unnecessary stuff in here.
   FIX: tidy this up. *)

(* Now, variables are also marked with GLOBAL/LOCAL etc.
   This saves quite a bit of trouble later.  *)


val funs = ref []: string list ref

fun addFun (FUNdef ((fname,_),_,_,_,_)) = funs := fname::(!funs)
fun addFuns l = app (fn FUNblock b => app addFun b) l

val empty = Binaryset.empty String.compare
fun add v s = Binaryset.add(s,v)
fun addargs l s = Binaryset.addList (s,l)
fun used v s = Binaryset.member(s,v)
fun mkLocals args =
    let fun f (arg, s) =
	    case arg of
		UNITvar => s
	      | VAR(n,_) => add n s
    in
	List.foldl f empty args
    end


fun fixExt locals v ext =
    case ext of
	LOCAL => (* Default for stuff which can't be determined during parsing *)
	if used v locals then LOCAL else GLOBAL
      | _ => ext

fun fixLocalFuns p =
let
    fun fixName x = x
    fun fixNameL x = x

    fun fixTypeDec d = d
    fun fixTy t = t

    fun fixValue locals v =
	case v of VARval (w,ext,l) => VARval (w,fixExt locals w ext,l)
	  | _ => v

    fun fixFunAssns (FUNASSNS (name, ass1, ass2))
      = FUNASSNS (fixName name, ass1, ass2)

    fun fixHints (HINTS (decs, funassns))
      = HINTS (map fixTypeDec decs, map fixFunAssns funassns)

    fun fixTest locals (TEST (oper, v, w, u)) = TEST (oper, fixValue locals v, fixValue locals w, u)

    fun fixMatchDiam m =
	case m of
            NOWHERE => NOWHERE
	  | SOMEWHERE n => SOMEWHERE (fixName n)
	  | DISPOSE => DISPOSE

    and fixOOMatchPat p =
	case p of
            CLASSpat (n, n') => CLASSpat (fixName n, fixName n')
	  | ANYCLASSpat => ANYCLASSpat

    fun fixMatchRule locals r =
	case r of
            MATCHrule (con, args, d as SOMEWHERE (n,_), e, u)
	    => MATCHrule (fixName con, map fixName args, fixMatchDiam d,
			  fixExp (add n (addargs (map #1 args) locals)) e, u)
          | MATCHrule (con, args, d, e, u)
	    => MATCHrule (fixName con, map fixName args, fixMatchDiam d,
			  fixExp (addargs (map #1 args) locals) e, u)
	  | OOMATCHrule (p, e, u) => OOMATCHrule (fixOOMatchPat p, fixExp locals e, u)


    and fixExp locals e =
	case e of
	    VALexp (v,u) => VALexp(fixValue locals v, u)
	  | UNARYexp (oper, v, u) => UNARYexp (oper, fixValue locals v, u)
	  | BINexp (oper, v, w, u) => BINexp (oper, fixValue locals v, fixValue locals w, u)
	  | IFexp (test, e1, e2, u) =>
	       IFexp (fixTest locals test, fixExp locals e1, fixExp locals e2, u)
	  | MATCHexp (v,l,u) => MATCHexp (fixName v, map (fixMatchRule locals) l, u)
	  | LETexp (v, e, e',u) =>
	    let
		val locals' = add (#1 v) locals
	    in
		LETexp (fixName v, fixExp locals' e, fixExp locals' e', u)
	    end
	  | APPexp (n as (fname,_), l, ext, u) =>
	       APPexp (fixName n, map (fixValue locals) l, fixExt locals fname ext, u) (* <----- *)
	  | CONexp (v, vs, addr, u) =>
	    CONexp (fixName v, map (fixValue locals) vs, Option.map fixName addr, u)
	  | TYPEDexp (e,t,u) => TYPEDexp (fixExp locals e, fixTy t, u)
	  | COERCEexp (e,t,u) => COERCEexp (fixExp locals e, fixTy t, u)
	  | NEWexp (class, vs,u) => NEWexp (class, map (fixValue locals) vs, u)
	  | SUPERMAKERexp (vs,x) => SUPERMAKERexp(map (fixValue locals) vs, x)
	  | INVOKEexp (obj,mname,vs,u)
	    => INVOKEexp (fixName obj, mname, map (fixValue locals) vs, u)
	  | UPDATEexp (a, v, u) => UPDATEexp (a, fixValue locals v, u)
	  | GETexp (obj, var, u) => GETexp (fixName obj, var, u)
	  | SGETexp (var, u) => SGETexp (var, u)
	  | ASSERTexp (e, as1, as2, u) => ASSERTexp (fixExp locals e, as1, as2, u)


    fun fixValDec vd =
	case vd of
            Asyntfn.VALdec (name, ty, inst) =>
	    Asyntfn.VALdec (fixName name, fixTy ty, inst)
	  | Asyntfn.CLASSdec (cname, super, ns, vds) =>
	    Asyntfn.CLASSdec (fixName cname, Option.map fixName super,
		      map fixName ns, map fixValDec vds)

    fun fixVarTy Asyntfn.UNITvar = Asyntfn.UNITvar
      | fixVarTy (Asyntfn.VAR(s,t)) = Asyntfn.VAR(s, Option.map fixTy t)

    fun fixFunDef (FUNdef (fname, args, inst, e, u)) =
	FUNdef (fixName fname, map fixVarTy args, inst, fixExp (mkLocals args) e, u)

    fun fixFunBlock (FUNblock l) = FUNblock (map fixFunDef l)

    fun fixClassDef (CLASSdef (class, super, imps, vals, methods)) =
	CLASSdef (fixName class,
		  Option.map fixName super,
		  map fixName imps,
		  map fixValDec vals,
		  map fixFunDef methods)

    fun fixProgram (PROG(typedecs,valdecs,classdefs,funblocks)) =
	let
	    val () = addFuns funblocks
	in
	PROG (typedecs,
	      valdecs,
	      map fixClassDef classdefs,
	      map fixFunBlock funblocks)
	end

in
    fixProgram p
end



(* ---------------- Printing abstract syntax -------------------*)

(* Temporary stuff to dispose of "secret" functions for Steffen *)

local
    val hideArrayFuns = ref true
in
(* Set this to true if you don't want to see the main fn etc *)
(* Use the -show-all option to change this *)

fun setHide b = hideArrayFuns := b

fun arrayFree t =
    case t of
        ARROWty (t1, t2) => arrayFree t1 andalso arrayFree t2
      | ARRAYty t => false
      | PRODUCTty l => arrayFreeList l
      | CONty (l, v) => arrayFreeList l
      | _ => true

and arrayFreeList l =
    case l of [] => true
	    | h::t => if arrayFree h then arrayFreeList t else false

fun discardBadValdecs l =
    if !hideArrayFuns then
	List.filter (fn VALdec(v,t,_) => arrayFree t
		      | _ => true) l
    else l

fun secretFun (FUNdef ((name,_), _, _, _, _)) =
    name = "main"
    orelse String.isPrefix "atol" name
   (* Probably not precise enough *)

fun discardArrayFns (FUNblock l) =
    if !hideArrayFuns
    then FUNblock (List.filter (not o secretFun) l)
    else FUNblock l (* Caution:  funblock might be empty *)

end
(* end of temporary stuff *)


(* ================================ Printing syntax tree ================================ *)
(* Much of this is unchanged from Asyntfn *)


fun printProgAndInfo os prInfo prog =
let
fun print s = TextIO.output(os,s)

fun printList printItem separator l =
case l of [] => ()
        | [h] => printItem h
        | h::t => (printItem h; print separator; printList printItem separator t)

fun printExt ext =
    if !showExtern then
	print (" <"^ extToString ext ^ ">")
	else ()

fun printVec printItem l =
case l of [] => ()
	| _ => (print "("; printList printItem "," l; print ")")

fun printType t = print ((typeToString t) ^ " ")

fun printName (n,_) = print n

fun newline() = print "\n"

fun printValue v =
    case v of
	VARval (w,ext,u)      => (print w; printExt ext; prInfo u)
      | INTval (n,_)      => print (intToString n)
      | FLOATval (x,_)    => print (realToString x)
      | CHARval (n,_)     => print ("'" ^ (Char.toString (chr n)) ^ "'")
      | STRINGval (s,_)   => print ("\"" ^ s ^ "\"")
      | BOOLval (true,_)  => print "true"
      | BOOLval (false,_) => print "false"
      | UNITval _         => print "()"
      | NULLval (s,_)     => print ("(null :> " ^s ^ ")")

fun printUnaryOperator oper =
    let
	val s =
	    case oper of
		NOTop => "not"
	      | ISNULLop => "isnull"
    in
	print s
    end


fun printBinaryOperator oper =
    let
	val s =
	    case oper of
		PLUSop => " + "
	      | MINUSop => " - "
	      | TIMESop => " * "
	      | DIVop => " / "
	      | MODop => " mod "
	      | LANDop => " land "
	      | LORop => " lor "
	      | LXORop => " lxor "
	      | LSLop => " lsl "
	      | LSRop => " lsr "
	      | ASRop => " asr "
	      | FPLUSop => " +. "
	      | FMINUSop => " -. "
	      | FTIMESop => " *. "
	      | FDIVop => " /. "
	      | LESSop => " < "
	      | LEQop  => " <= "
	      | EQUALSop => " = "
	      | CONCATop => " ^ "
    in
	print s
    end

fun printTest (TEST(oper, v, w, _)) = (printValue v;
				       printBinaryOperator oper;
				       printValue w)

fun printExp e =
    let in
    case e of
	    VALexp (v,u) => (printValue v; prInfo u)
	  | UNARYexp (oper, v, u) =>
	    (printUnaryOperator oper; print "("; printValue v; print ")"; prInfo u)

	  | BINexp (oper, v, w, u) =>
	    (print "("; printValue v; printBinaryOperator oper; printValue w; print ")"; prInfo u)

	  | IFexp (test, e1, e2, u) => (print "if ";
					printTest test;
					print " then\n   "; printExp e1;
					print"\nelse (\n   "; printExp e2;
					print "\n)"; prInfo u)
	  | MATCHexp (n,l,u) => (print "( match ";
				 printName n;
				 print " with"; prInfo u; print "\n    ";
				 printList printMatchRule "  | " l;
				 print ")")
	  | LETexp (v, e, e',u) => (print "let "; printName v; print " = ";
				    printExp e; print "\nin "; printExp e'; prInfo u)
	  | APPexp (v, [], ext,u) => (printName v; printExt ext; print "() "; prInfo u)
	  | APPexp (v, l, ext,u ) => (printName v; printExt ext; print " ";
				    printList (fn v => (print "(";printValue v; print ")")) " " l;
				    print " "; prInfo u)
	  | CONexp (v, vs, addr,u) =>
	    let val () = (printName v; printVec printValue vs)
		val () = case addr of
		    NONE => print ""
		  | SOME w => (print "@"; printName w)
	    in
		 prInfo u
	    end
	  | TYPEDexp (e,t,u) => (printExp e; print ": "; printType t; prInfo u)
	  | COERCEexp (e,t,u) => (printExp e; print ":> "; printType t; prInfo u)
          | SUPERMAKERexp (vs,u) => (print "super "; printList printValue " " vs; prInfo u)
          | NEWexp (class, vs,u) => (print "new "; printName class; print " ";
				     printList printValue " " vs; prInfo u)
          | INVOKEexp (obj,mname,vs,u)
	    => (printName obj; print ("#" ^ nameOf mname ^ " "); printList printValue " " vs; prInfo u)
          | UPDATEexp (x, v, u) => (printName x; print " <- "; printValue v; prInfo u)
          | GETexp (obj, var, u) => (printName obj; print "#"; printName var; prInfo u)
          | SGETexp (var, u) => (printName var; prInfo u)
	  | ASSERTexp (e, as1, as2, u) =>
	    let
		val _ = warn nowhere "Assertion not printed"  (* No assertions in user syntax yet *)
	    in
		(printExp e; prInfo u)
	    end
    end


and printMatchRule (MATCHrule (c, args, addr, e, _)) =
    let
	val a = case addr of NOWHERE => ""
			   | DISPOSE => "@_"
			   | SOMEWHERE (v,_) => "@"^v
    in
	case args of
	    [] => (printName c; print (a ^ " -> "); printExp e; print "\n")
	  | _ => (printName c;
		  printVec printName args;
		  print a;
		  print (" -> ");
		  printExp e;
		  print "\n")
    end
| printMatchRule (OOMATCHrule (ANYCLASSpat, e, _)) =
  (print "_ -> "; printExp e; print "\n")
| printMatchRule (OOMATCHrule (CLASSpat(ob, cl), e, _)) =
  (printName ob; print " :> "; printName cl; print " -> "; printExp e; print "\n")

(*
fun printArg a =
    case a of
	UNITvar => print "()"
      | VAR(id, NONE) => print id
      | VAR(id, SOME ty) => (print "("; print id; print ": "; printType ty; print ")")
*)


(* Steffen doesn't like type annotations *)
fun printArg a =
    case a of
	UNITvar => print "u"
      | VAR(id, NONE) => print id
      | VAR(id, SOME ty) => print id

fun printFunDef d =
    case d of
	FUNdef (name, [], inst, body, _) =>
	(printName name;  print "() = \n"; printExp body; print "\n")
      | FUNdef (name, args, inst, body, _) =>
	(printName name;  print " "; printList printArg " " args;
	 print " = \n"; printExp body; print "\n")

fun printFunBlock (FUNblock l) =
    let
	fun p flg [] = ()
	  | p flg (h::t) = (print (if flg then "\nlet " else "\nand "); printFunDef h; print ""; p false t)
    in
	p true l
    end

fun printValDec v =
    case v of
	VALdec (v,l, STATIC) =>
	(print "val "; printName v; print ": "; printType l; newline())
      | VALdec (v,l, INSTANCE) =>
	(print "val "; printName v; print ": "; printType l; newline())
      | CLASSdec (c, sup, intfs, decs) =>
	(print "(*** Classtypes are currently unreliable";
	 print "\nclasstype "; printName c;
	 case sup of SOME s => (print " is "; printName s)
		   | NONE => ();
	 print ": "; newline();
	 app printValDec decs; print "end\n***)\n\n")

fun printField f =
    case f of
	VALdec (v,l, STATIC) =>
	(print "field static "; printName v; print ": "; printType l; newline())
  | VALdec (v,l, INSTANCE) =>
    (print "field "; printName v; print ": "; printType l; newline())
  | _ => Util.exit "printField:  field is CLASSdec"

fun printConstructor (TYPEcon (v, (types,_), heap,_)) =
    let val (nullrep, diamonds0) =
            case heap of HEAP => ("","(*1*)")
                       | NOHEAP => ("!", "")
	val diamonds = if !printSpace then diamonds0 else ""
    in
        case types of
	    [] => (print nullrep;printName v; print diamonds)
          | _ =>
	    (
	     print nullrep; printName v; print diamonds; print " of ";
	     printList printType "* " types
	    )
    end

fun printTypeDec (TYPEdec (tvars, tname, constrs,_)) =
    (print (case tvars of [] => "type" | _ => "type ");
     printVec printName tvars;
     print " "; printName tname; print " = \n    ";
     printList printConstructor "\n  | " constrs;
     newline ();
     newline ()
    )


fun printSuper e =
    case e of
	TYPEDexp (LETexp (("_",_), SUPERMAKERexp ([UNITval _],_), body',_),UNITty, _) => body'
      |	TYPEDexp (LETexp (("_",_), SUPERMAKERexp (args,_), body',_),ty, loc) =>
	(print ": super ";
	 printList printValue " " args;
	 body'
	)
      | _ => Util.error (getU e) "malformed supermaker"

(* The parser requires a return type annotation but then sticks it
   at the end of the body. We have to undo this. *)
fun printBody e =
    case e of
	TYPEDexp (e', ty, loc) =>
	(print ": ";
	 printType ty;
	 print " =\n";
	 printExp e'
	)
      | _ =>
	( print " =\n";
	  printExp e)

fun printMethod m =
    case m of
	FUNdef (("<init>",_), [], inst, body, _) =>
	(print "maker ()";
	 let val body' = printSuper body in
	     print " =\n"; printExp body'; print "\n\n"
	 end)
      | FUNdef (("<init>",_), args, inst, body, _) =>
	(print "maker ";
	 printList printArg " " args;
	 let val body' = printSuper body in
	     print " =\n"; printExp body'; print "\n\n"
	 end)
      | FUNdef (name, [], inst, body, _) =>
	(print "method "; printName name;
	 print " ()"; printBody body; print "\n\n")
      | FUNdef (name, args, inst, body, _) =>
	(print "method "; printName name; print " ";
	 printList printArg " " args;
	 printBody body; print "\n\n")

fun printClassDef (CLASSdef(class, super, imps, fields, methods)) =
    (print "class "; printName class;
     print " = ";
     case super of SOME cl =>
		   ( printName cl; print " with")
		 | NONE =>();
     newline ();

     printList (fn x => (print "implement "; printName x)) "\n" imps;
     newline ();
     app printField (Util.makeSet fields);
     (* get rid of duplicate valdecs:  these annoy lfd_infer *)
     app printMethod methods;
     print "end\n"
     )


fun printProg (PROG(typedecs,valdecs,classdefs,funblocks)) =
    let
	val valdecs' = Util.makeSet (discardBadValdecs valdecs)
    in
    ( print "(* ---------------- Start of program ---------------- *)\n";
      app printTypeDec typedecs;
      app printValDec valdecs';
      app printClassDef classdefs;
      app printFunBlock (map discardArrayFns funblocks);
      print "(* ---------------- End of program ---------------- *)\n"
    )
    end

in
    printProg prog
end

fun printProgram pri prg = (printProgAndInfo TextIO.stdOut pri prg; TextIO.flushOut TextIO.stdOut)
fun outputProgram os = printProgAndInfo os

