(* Monomorphisation *)

(* what about unit lists??? *)

local open Util Type Normsyn NAsyntfn in

fun constrToString (TYPEcon((c,_), ([],_),h, _)) = c
  | constrToString (TYPEcon((c,_), (tys,_),h,_))
    = c ^ " of " ^ (listToString typeToString " * ") tys

fun typedecToString (TYPEdec(vars, (tname,_), cons,_)) =
    (case vars of
	 [] => ""
       | _ => "(" ^ (listToString #1 " " vars) ^ ") "
    )
    ^ tname
    ^ " = "
    ^ listToString constrToString " | " cons

fun m_error s = Util.ierror ("[Mono.sml]: " ^ s)
fun poly_error s = Util.exit
		       (s ^
			"\nThe program may contain a polymorphic value such as an empty list:\n"
			^ "try adding type annotations."
		       )

type Loc = Loc.Location

infixr 5 ::?

local
    val counter = ref 0
    val fcounter = ref 0
    val vcounter = ref 0
in
    fun mkSuffix x = (counter := !counter + 1;
                         "_" ^ Int.toString (!counter))
    fun mkTypeName x = (counter := !counter + 1;
                        x ^ "_" ^ (Int.toString (!counter)))
    fun mkFunName x = (fcounter := !fcounter + 1;
                       x ^ "_" ^ (Int.toString (!fcounter)))
    fun mkVarName () = (vcounter := !vcounter + 1;
			"?b" ^ (Int.toString (!vcounter)))

end

fun substTy subst tyT =
    foldr (fn ((x,y),ty) => Type.substTy x y ty) tyT subst

fun substTys subst tys =
    map (substTy subst) tys

fun sameList p l l' = List.length l = List.length l'
		      andalso ListPair.all (fn (x,x') => p x x') (l,l')


fun nameOf (TYPEdec(_,(n,_),_,_)) = n

local
    val monoDecs = ref []
    val done = ref []
in
    fun saveTdec (tyArgs, tyName, tyDec) = (
	debugPrint ("Saving typeDec for " ^ typedecToString tyDec ^ "\n");
	done := (tyArgs, tyName) :: (!done);
	monoDecs := tyDec :: (!monoDecs)
    )
    fun alreadyDone (tyArgs, tyName) = member (tyArgs, tyName) (!done)
    fun getMonoDecs () = !monoDecs
end

local
val TYPEDECS = ref []: Absyn.Annotation TypeDec list ref


in

(*
fun makeBindings (tyVars, tyArgs) =
    ListPair.foldl (fn ((t,_),ty,m) => Map.insert(m,t,ty))
		   (Map.empty ()) (tyVars, tyArgs)

fun addBindings (tyVars, tyArgs) m =
    ListPair.foldl (fn ((t,_),ty,m) => Map.insert(m,t,ty))
		   m (tyVars, tyArgs)
*)

fun makeBindings (tyVars, tyArgs) =
    if length tyVars <> length tyArgs then m_error "Arity mismatch 2"
    else ListPair.zip (map #1 tyVars, tyArgs)


fun setTypeDecs l = TYPEDECS := l

fun getTypeDec tname =
    case List.find (fn d => nameOf d = tname) (!TYPEDECS)
     of SOME d => d
      | NONE => m_error ("Missing TYPEdec for " ^ tname)

fun getDecAndBindings tyArgs tyName =
    case List.find (fn d => nameOf d = tyName) (!TYPEDECS)
     of SOME (tdec as TYPEdec(tvars, _,_,_)) =>
	(tdec, makeBindings (tvars, tyArgs))
      | NONE => m_error ("Missing TYPEdec for " ^ tyName)

fun getTypeVars tname =
    case List.find (fn d => nameOf d = tname) (!TYPEDECS)
     of SOME (TYPEdec(args,_,_,_)) => args
      | NONE => m_error ("Missing TYPEdec for " ^ tname)

end (* local *)

fun sameType INTty INTty = true  (* equality for types,  regarding free types as equal *)
  | sameType CHARty CHARty = true
  | sameType BOOLty BOOLty = true
  | sameType FLOATty FLOATty = true
  | sameType STRINGty STRINGty = true
  | sameType UNITty  UNITty  = true
  | sameType (TVARty _) (TVARty _) = true
  | sameType (ARRAYty t) (ARRAYty t') = sameType t t'
  | sameType (PRODUCTty l) (PRODUCTty l') = sameList sameType l l'
  | sameType (ARROWty(t1,t2)) (ARROWty(t1',t2')) =
              sameType t1 t1' andalso sameType t2 t2'
  | sameType (CONty (l,v)) (CONty (l',v')) =
              v=v' andalso sameType (PRODUCTty l) (PRODUCTty l')
  | sameType (OBJECTty ty) (OBJECTty ty') = ty=ty'
  | sameType (DIAMONDty x) (DIAMONDty y) = x=y
  | sameType _ _ = false


(* We want this because sometimes we have instances of polymorphic functions
   with different type variables and we want them to be the same.  For example
   when monomorphising the polymorphic list length function there'll be
   an instance of this with type scheme (say) 'a list -> int in the pending
   list,  but the recursive call may have the scheme 'b list -> int,  where
   both 'a and 'b are being instantiated to string; we want to be able to
   realise that the copy of "length" which we generated for 'a will also
   suffice for 'b.  Does this make sense?  I thought not. *)


fun tysEqual (tys: Ty list) (tys': Ty list) = sameList sameType tys tys'

fun sameSpec ((tys, name), (tys', name')) =
    name = name' andalso tysEqual tys tys'


local
structure P = Polyhash
exception notFound
in

val suffixMap: (Ty list * string, string) P.hash_table  (* eg: [int] list -> "_5" *)
  = P.mkTable(P.hash, sameSpec)(20, notFound)

val inverseMap: (string, Ty list * string) P.hash_table (* inverts suffixMap *)
  = P.mkTable(P.hash, op=)(20, notFound)

fun dbg () =
(
	debugPrint "suffix map is [\n";
	P.apply (fn ((l,t),s) =>
		(debugPrint (" [" ^ (listToString typeToString ", " l) ^ "] "
			     ^ t ^ ": " ^ s ^"\n"))
	    ) suffixMap;
	debugPrint "]\n\n"
)


(* The following code "fixes" a problem with type instantiation.  The program
   "type 'a t = T of 'a; let f (x: int t t) = 0" failed to compile because
    int t t was instantiated to t_1, int t was instantiated to t_2, and then
    later x was given the type t_2 t, clashing with its other type of int t t.
    Now we "de-instantiate" t_2 t to (int t) t before proceeding. *)

fun unfoldBinding tname (binding as (tvar,btype)) =
    case btype of
	CONty (tys0, uname) => (* is tys0 ever nonempty? (partially instantiated) *)
	let in
	    case P.peek inverseMap uname of
		SOME (tys, uname') => (tvar, CONty (tys, uname'))
	      | NONE => binding
	end
      | _ => binding

fun bindingToStr (a,t) = a ^ " -> " ^ (typeToString t)
fun bindingsToStr l = (listToString bindingToStr ", " l)
fun tyconToStr tys tname = "["^ (listToString typeToString ", " tys) ^ "] " ^ tname

(* Get the suffix for a type when instantiated with variable->type mapping ctr.
   If this instantiation has not been seen before, generate a new name and add
   an entry to the suffixMap. *)

fun getSuffix_aux tname bindings tyVars =
    case tyVars of
	[] => ""
      | _ =>
	let
	    val bindings' = map (unfoldBinding tname) bindings
            val tys = map ((substTy bindings') o (fn (x,_) => mkTvar x)) tyVars
	in
	    case P.peek suffixMap (tys, tname) of
		SOME suffix => suffix
	      | NONE =>
		let
		    val suffix = mkSuffix tname
		    val () = P.insert suffixMap ((tys, tname), suffix)
		    val () = P.insert inverseMap (tname^suffix, (tys, tname))
		in
		    suffix
		end
	end

(* eg, looking for (int, string) pair *)
fun getTypeSuffix tys tname =
    let
        val tyVars = getTypeVars tname
	val bindings = makeBindings (tyVars, tys)
    in
	getSuffix_aux tname bindings tyVars
    end

(* eg, looking for suffix for ('a,'b) pair in presence of bindings
  'a -> int, 'b -> string, 'c -> bool, ... got from constructor annotation.
   Perhaps we could modify the annotation to contain the substituted types. *)

fun getConSuffix bindings tname =
    getSuffix_aux tname bindings (getTypeVars tname)



fun getSuffix tys tname =
    case P.peek suffixMap (tys, tname) of
	SOME suffix => suffix
      | NONE => m_error ("getSuffix: didn't find " ^ tyconToStr tys tname ^ " in suffixMap\n")


(* This is very unpleasant.  We've got a list of type instantiations which
   we're required to create.  In producing these,  there may be "hidden"
   polymorphic types embedded in the constructors, and we then have to
   generate monomorphic versions of these as well.  This is what getInnerTypes
   is about below,  but I'm not sure if it does everything that's required.
   The original problematic program was

     type 'a foo = F of 'a
     type expr = E of int foo foo
     let reduce (e: expr) = 0

   which caused trouble because of the embedded version of int foo.  This
   example now works,  but I'm not convinced that the code below will deal
   with all programs correctly;  sufficiently complex nesting of polymorphic
   types may show up more bugs. *)

local
fun getInnerTypes bindings ty =
    let fun f t =
	    case t of
		CONty (tys0,c) =>
		let
		    val tys = substTys bindings tys0
		    val () = app f tys
		    val _  = getTypeSuffix tys c (* Only called for side-effect of making new entry in map *)
		in
		    ()
		end
	      | ARRAYty t1 => f t1
	      | ARROWty (t1, t2) => (f t1; f t2)
	      | PRODUCTty tys => app f tys
	      | _ => ()
    in
	f ty
    end
in

fun findInnerTypes ((tyArgs,tyName),_) =
    let
	val (TYPEdec (tvars, t_name, constrs, _), bindings) = getDecAndBindings tyArgs tyName
	val () = app (getInnerTypes bindings) tyArgs
	val () = app (fn TYPEcon(_,(argtys, _),_,_)
			 => app (getInnerTypes bindings) argtys) constrs
    in
	()
    end
end


(* The next function takes an entry from the suffixMap and produces an
   appropriate monomorphic type declaration. *)

fun makeMonoDec ((tyArgs, tyName), suffix) =
    let
        val () = debugPrint ("\n% makeMonoDec ("
			     ^ "[" ^ (listToString typeToString ", " tyArgs) ^ "], "
			     ^ tyName ^ ", \"" ^ suffix ^ "\")\n")

	val (TYPEdec(tvars, (t,_), cons, u), bindings) = getDecAndBindings tyArgs tyName

	val () = debugPrint ("got typedec for ["
			     ^ listToString (fn (x,_)=>x) ", " tvars ^ "] " ^ t ^ "\n")

	fun monoType ty =
	    (debugPrint ("monoType " ^ typeToString ty ^ "\n");
	     case ty of
		 CONty(tys0, c) =>
		 let
		     val tys = substTys bindings tys0
		     val () = debugPrint ("CONty(" ^ tyconToStr tys c ^ ")\n")
		 in
		     CONty([], c ^ getSuffix tys c)
		 end

	       | TVARty {name=t,...} =>
		 (case List.find (fn (x,_)=>x=t) bindings of
		      SOME (_,ty) =>
		      (debugPrint ("Replacing "^t^" with " ^(typeToString ty) ^"\n");
		       monoType ty)
		    | NONE => poly_error ("Free type variable "
			      ^ t
			      ^ " found during monomorphisation.")
				 )
	       | ARRAYty t => ARRAYty (monoType t)
	       | ARROWty(t1, t2) => ARROWty(monoType t1, monoType t2)
	       | PRODUCTty tys => PRODUCTty (map monoType tys)

	       | _ => ty
	    )


        fun monoCon (TYPEcon((c,loc),(argtys,v),h,u)) =
	    TYPEcon((c ^ suffix,loc), (map monoType argtys, v), h, u)

	val newDec = TYPEdec([],(tyName^suffix,nowhere), map monoCon cons, u)
    in
	saveTdec (tyArgs, tyName, newDec)
    end



fun getFunTy f [] = NONE
  | getFunTy f (VALdec((v,_), tys,_)::vals) =
    if f = v
        then SOME tys
    else getFunTy f vals
  | getFunTy f ((CLASSdec _)::vals) = getFunTy f vals


fun getFunDef [] f = m_error ("Fundef '" ^f^ "' not found")
  | getFunDef ((fd as FUNdef((v,_),_,_,_,_))::funs) f =
    if f = v
        then fd
    else getFunDef funs f

fun getInfo f [] =
    m_error ("No info for " ^ f)
  | getInfo f ((f',ty,_,i)::is) =
    if f = f' then i else getInfo f is

(* We're looking for the name of a function instantiation. *)
(* First we look in the monomorphised functions which we've
   already generated, then in a list of pending functions.
   If we don't find the instantiation we're interested in then we
   generate a new one and add an entry for it to the pending
   list. The pending list now has a front part and a back part. *)

fun getFunName fname spec bindings done pending pending2 =
let
    fun beenDone (f,tys,_) = f = fname andalso sameType tys spec
    fun inPending (f,tys,_,_) = f = fname andalso sameType tys spec
in
    case List.find beenDone done of
	SOME (_,_,FUNdef((fname',_), _, _, _, _)) => (fname', NONE)
      | NONE =>
	case List.find inPending pending of
	    SOME (_,_,_,fname') => (fname', NONE)
	  | NONE =>
	    case List.find inPending pending2 of
		SOME (_,_,_,fname') => (fname', NONE)
	      | NONE =>
		let
		    val fname' = mkFunName fname
		in
		    (fname', SOME (fname, spec, bindings, fname'))
		end
end


fun instantiateTy ty =
    case ty of
	CONty(tys, tname) =>
	      CONty([], tname ^ getTypeSuffix tys tname )
      | ARROWty (t1, t2) => ARROWty (instantiateTy t1, instantiateTy t2)
      | ARRAYty t => ARRAYty (instantiateTy t)
      | PRODUCTty l => PRODUCTty (map instantiateTy l)
      | _ => ty


fun isString v env =
    case v of 
	STRINGval _ => true
      | VARval (v,_,u) =>
	let 
	    val ty = Env.getVarTy v env 
	    val bindings = #2 (getMono u)
	    val ty' = substTy bindings ty
	in 
	    ty' = STRINGty
	end
      | _ => false

fun isUnit v env =
    case v of 
	UNITval _ => true
      | VARval (v,_,u) =>
	let 
	    val ty = Env.getVarTy v env 
	    val bindings = #2 (getMono u)
	    val ty' = substTy bindings ty
	in 
	    ty' = UNITty
	end
      | _ => false

fun instantiate e pending pending2 env (vals, done)  =
let
    val pending2' = ref pending2

    fun monoVar (x,ext,l) =
	if Perv.isBuiltIn x then (x,ext,l) (* FIX: polymorphic builtins??? *)
	else
	    if Env.isLocal x env then (x,ext,l)
	    else
		case getFunTy x vals of
		    NONE => Util.error l ("Can't find type for variable " ^ x)
		  | SOME ty =>
		    let
			val () = debugPrintln ("\nmonoVar " ^ x)
			val (_,bindings) = getMono l
			val () = debugPrint ("bindings = " ^ bindingsToStr bindings ^ "\n")
			val ty' = substTy bindings ty
				    val () = debugPrint ( "ty  = " ^ typeToString ty ^ "\n")
				    val () = debugPrint ( "ty' = " ^ typeToString ty' ^ "\n")
			val (x',new) = getFunName x ty' bindings done pending (!pending2')
			val () = case new of NONE => ()
					   | SOME p => pending2' := p::(!pending2')
		    in
			(x',ext,l)
		    end

    fun monoVal v =
	case v of
	    VARval w => VARval (monoVar w)
	  | _ => v

    fun inst expr =
	case expr of
	    VALexp (v,l)    => VALexp (monoVal v, l)
	  | UNARYexp _      => expr
	  | BINexp (oper, v, w, u) => 
	    if isString v env (* resolve string comparisons *)
	    then let 
		    fun mkMono s  = MONO ((s,[]), u)
		    val X = mkVarName ()
		 in case oper of
			EQUALSop => (Perv.markUsed "same_string";
					     APPexp (("same_string", u), 
						     [v,w], BUILTIN, mkMono "same_string"))
		      | LESSop => (Perv.markUsed "string_compare";
				   LETexp ((X,mkMono ""),
					   APPexp (("string_compare", u), 
						   [v,w], BUILTIN, mkMono "string_compare"),
					   BINexp (LESSop, VARval(X, LOCAL, u),
						   INTval(0,u), u), u))

		      | LEQop => (Perv.markUsed "string_compare";
				  LETexp ((X,mkMono ""),
					  APPexp (("string_compare", u), 
						  [v,w], BUILTIN, mkMono "string_compare"),
					  BINexp (LEQop, VARval(X, LOCAL, u),
						  INTval(0,u), u), u))
		      | _ => expr
		 end
	    else if isUnit v env 
	    then 
		case oper of
		    EQUALSop => VALexp (BOOLval (true, u), u)
		  | _ => m_error "Unexpected operator while resolving unit comparison"
	    else 
		expr

	  | INVOKEexp _     => expr
	  | NEWexp _        => expr
	  | GETexp _        => expr
	  | SGETexp _       => expr
	  | UPDATEexp _     => expr
	  | SUPERMAKERexp _ => expr

	  | APPexp((fname,floc), vs, ext, u) =>
	    if Env.isLocal fname env then expr
	    else
	    let
		val () = debugPrint ("\nmono APP: " ^ fname ^ "\n")
		val (n, bindings) = getMono u

		val () = debugPrint ("n = " ^ n ^ "\nbindings = " ^ bindingsToStr bindings ^ "\n")

		val () = if n=fname then () else
			 m_error ("name mismatch in annotation ["
				  ^ fname ^ "]")
				 (* val eu = getU e *)


		fun gft f [] = NONE
		  | gft f (VALdec((v,_), tys,_)::r) =
		    if f = v
		    then SOME (f,tys)
		    else gft f r
		  | gft f (_::r) = gft f r

		fun ft v =
		    case v of VARval (f,_,_) => gft f vals
			    | _ => NONE


		val fname'= (* only place pending is modified *)
                    case ext
		     of
                        GLOBAL =>
			let in
			    case getFunTy fname vals (* only place vals is used *)
			     of NONE => error floc "[Mono]: function type not found"
			      | SOME ty =>
				let
				    val ty' = substTy bindings ty
				    val () = debugPrint ( "ty  = " ^ typeToString ty ^ "\n")
				    val () = debugPrint ( "ty' = " ^ typeToString ty' ^ "\n")
				    val (fname', new) =
					getFunName fname ty' bindings done pending (!pending2')
				    val () = case new of NONE => ()
						       | SOME p => pending2' := p::(!pending2')
				in
				    fname'
				end
			end
		      | LOCAL => fname
		      | BUILTIN => fname
		      | EXTERN =>  fname

		val () = debugPrint ("after MA: " ^ fname' ^ "\n")

		val vs' = map monoVal vs

	    in
		APPexp((fname',floc), vs', ext, u)
	    end

	  | CONexp((cname,cloc),vs,a, u) =>
	    let  (* DO SOMETHING ABOUT @ a *)
		val (tname, bindings) = getMono u
		val cname' =
		    let
			val sfx = getConSuffix bindings tname
			val () = debugPrint ("Monoing con app " ^ tname ^ "/" ^ cname ^ ": " ^ sfx ^ "\n")
		    in
			cname^sfx
		    end
	    in
		CONexp((cname',cloc), vs, a, u)
	    end


	  | LETexp(x,e1,e2,u) => LETexp (x, inst e1, inst e2, u)

	  | IFexp(tst as TEST(oper, v, w, tu) ,e1,e2,u) => 
            (* Let's resolve string comparisons as well *)
	    if isString v env 
	    then let 
		    fun mkMono s = MONO ((s,[]), tu)
		    val X = mkVarName ()
		 in case oper of
			EQUALSop => (Perv.markUsed "same_string";
				     LETexp ((X,mkMono ""),
					     APPexp (("same_string", tu), 
						     [v,w], BUILTIN, mkMono "same_string"),
					     IFexp (
					     TEST (EQUALSop, VARval(X, LOCAL, tu),
						   BOOLval(true,tu), tu),
					     inst e1, inst e2, u),u))
		      | LESSop => (Perv.markUsed "string_compare";
				   LETexp ((X,mkMono ""),
					   APPexp (("string_compare", tu), 
						   [v,w], BUILTIN, mkMono "string_compare"),
					   IFexp (
					   TEST (LESSop, VARval(X, LOCAL, tu),
						 INTval(0,tu), tu),
					   inst e1, inst e2, u),u))
		      | LEQop => (Perv.markUsed "string_compare";
				  LETexp ((X,mkMono ""),
					  APPexp (("string_compare", tu), 
						  [v,w], BUILTIN, mkMono "string_compare"),
					  IFexp (
					  TEST (LEQop, VARval(X, LOCAL, tu),
						INTval(0,tu), tu),
					     inst e1, inst e2, u),u))
		      | _ => m_error "Unexpected operator while resolving string comparison"
		 end
	    else if isUnit v env 
	    then 
		case oper of
		    EQUALSop => inst e1
		  | _ => m_error "Unexpected operator while resolving unit comparison"
	    else 
		IFexp(tst, inst e1, inst e2, u)

	  | MATCHexp(v,rules,u) =>
	    let
		val (tname, bindings) = getMono u

		fun fRule (MATCHrule((cname,cloc),vs,a,e,u)::rules) acc =
		    let
			val () = debugPrint (bindingsToStr bindings ^ " ##\n")
			val cname' = cname ^ (getConSuffix bindings tname)
			val e' = inst e
			val rule = MATCHrule((cname',cloc), vs, a, e', u)
			val () = debugPrint ("Monoing match " ^ tname ^ "/" ^ cname ^ ": " ^ cname' ^ "\n")
		    in
			fRule rules (rule::acc)
		    end
		  | fRule (OOMATCHrule(pat, e, u)::rules) acc =
		    let
			val e' = inst e
			val rule = OOMATCHrule(pat, e', u)
		    in
			fRule rules (rule::acc)
		    end
		  | fRule [] acc = rev acc

		val rules' = fRule rules []
	    in
		MATCHexp(v, rules', u)
	    end

	  | TYPEDexp (e,ty,u) => TYPEDexp (inst e, instantiateTy ty, u)

	  | COERCEexp (e,ty,u) => COERCEexp (inst e, instantiateTy ty, u)

	  | ASSERTexp (e, as1, as2, u) =>
	    m_error "Assertions not supported in monomorphisation yet"

in
    (inst e, !pending2')
end


fun fixtype bindings ty =
let
    val () = println ("Calling fixtype on " ^ typeToString ty)
    val () = println ("Bindings: " ^ bindingsToStr bindings)
    val fix = substTy bindings
    fun fixty ty =
	case ty of
	    CONty(tys0, c) => CONty (map fixty tys0, c)
	  | TVARty {name=t,...} =>
	    (case List.find (fn (x,_)=>x=t) bindings of
		 SOME (_,ty) =>
		 (debugPrint ("fixtype: Replacing "^t^" with " ^(typeToString ty) ^"\n");
		  ty)
	       | NONE => poly_error ("Free type variable "
				     ^ t
				     ^ " found in fixtype.")
	    )
	  | ARRAYty t => ARRAYty (fixty t)
	  | ARROWty(t1, t2) => ARROWty(fixty t1, fixty t2)
	  | PRODUCTty tys => PRODUCTty (map fixty tys)
	  | _ => ty
in
    fixty ty
end



local

    fun instVar v =
	case v of
	    UNITvar => UNITvar (* HOPE THIS IS OK *)
	  | VAR(x, NONE) => v
	  | VAR(x, SOME ty) =>
	    let val ty' = instantiateTy ty
		val v' = VAR(x, SOME ty')
	    in
		v'
	    end

    (* IS THERE ANY DANGER of finding unseen inner types in monoFuntype ?? *)
    fun monoFuntype ty0 fname =
	let
	    val () = debugPrintln ("monoFuntype at " ^ typeToString ty0 )
	    fun aux ty =
		case ty of
		    CONty (tys, tyName) =>
		    let
			val tvars = getTypeVars tyName
			val tys' = map aux tys
			val suffix = getTypeSuffix tys' tyName
			val () = debugPrint ("Got type name "
					    ^ tyName ^ suffix ^ " for " ^ typeToString ty ^ "\n")
			val () = dbg ()
		    in
			CONty ([], tyName^suffix)
		    end
		  | TVARty x => (* ty *)
		    m_error ("\nTrying to monomorphise function '" ^ MonoUtil.trimSuffix fname
			     ^ "' at polymorphic type\n "
			     ^ typeToString ty0)
		  | ARRAYty ty => ARRAYty (aux ty)
		  | ARROWty (ty, ty') => ARROWty (aux ty, aux ty')
		  | PRODUCTty tys => PRODUCTty (map aux tys)
		  | _ => ty
	in
	    aux ty0
	end
in

fun monoFuns pending pending2 done fundefs valDecs newValDecs env =
    case pending of
	[] =>
	let in
	    case pending2 of
		[] => (map #3 done,  newValDecs)
	      | _ => monoFuns (rev pending2) [] done fundefs valDecs newValDecs env
	end

      | (ofname,ftype,fbindings,fname)::rest =>
	let
        (* Substitute correct types for this inst. of the function's tyvars *)
        fun fixBindings (n,bindings) =
            (n, (map (fn (x,y) => (x, substTy fbindings y)) bindings) @ fbindings)

	val fixInfo = mapMono fixBindings

        val () = debugPrint ("Doing " ^ ofname ^ " as " ^ fname ^ " with type " ^ typeToString ftype ^ ".\n"
                             ^ "Pending:(\n"
                             ^ (String.concat
                                    (map (fn (f,ty,_,f') =>
					     " > " ^ f
					     ^ ":"^ (typeToString ty)^ " as "^ f'^ "\n")
                                       (pending @ pending2)))
                             ^ ")\n")
	val FUNdef(_, vs, inst, e, u) =
	    NAsyntfn.mapFunDef fixInfo (getFunDef fundefs ofname)

	val fenv = Env.getVarEnv ofname env
(*	val argnames = List.mapPartial (fn UNITvar => NONE | VAR(s,_) => SOME s) vs *)
        val (e', pending2') = instantiate e pending pending2 fenv (valDecs, done)

        (* It's important to use the list of pending functions here
           (not rest), so that we can see the current function.
           However, we're going to throw away the head of the pending
           list when we recursively call monoFuns, so new entries can't
           be added to the head of the pending list.  We deal with
           this by having a list pending2 which is the reverse of the
           second "half" of the pending list (and new entries are
           added to pending2). When pending is empty we replace it
           with rev pending2. *)

	val vs = map instVar vs
        val ty = monoFuntype ftype fname
	val val' = VALdec((fname,nowhere), ty, STATIC)
    in
        monoFuns rest pending2'
		 ((ofname, ftype, FUNdef((fname,nowhere), vs,inst,e',u))::done)
		 fundefs valDecs (val'::newValDecs) env
    end
end (* local *)


(* Was monomorphic(Ty|Dec), but that became a misnomer *)
(* Consider both "(int) lst" and "'a lst" to be bad, want just "lst_1" &c. *)
fun safeTy (TVARty _)   = false
  | safeTy (ARRAYty ty) = safeTy ty
  | safeTy (PRODUCTty tys) = List.all safeTy tys
  | safeTy (CONty(tys,_))  = tys = [] (* List.all safeTy tys *)
  | safeTy (ARROWty(ty,ty')) = safeTy ty andalso safeTy ty'
  | safeTy _ = true

fun safeDec (VALdec(_,ty,_)) = safeTy ty
  | safeDec (CLASSdec _) = true

local

    (* ---------------- Debugging ---------------- *)

    fun constrToString (TYPEcon((c,_), ([],_),h, _)) = c
      | constrToString (TYPEcon((c,_), (tys,_),h,_))
        = c ^ " of " ^ (listToString typeToString " * ") tys

    fun typedecToString (TYPEdec(vars, (tname,_), cons,_)) =
	(case vars of
	     [] => ""
	   | _ => "(" ^ (listToString #1 " " vars) ^ ") "
	)
	^ tname
	^ " = "
	^ listToString constrToString " | " cons

    fun dbg1 r =
	 debugPrint ("About to call monoFuns: rootFuns is\n"
		     ^ String.concat (
		     map
			 (fn (f,t,_,f') => (f ^ ":" ^ (typeToString t)^"\n"))
                         r)
		     ^ "\n")

    fun dbg2 fns types = (
	debugPrint ("\nFinished monomorphising functions: "
                    ^ (Int.toString (length fns))
                    ^ " functions generated\n"
		    ^ String.concat (map (fn FUNdef((f,_),_,_,_,_) => " " ^ f ^ "\n")
					 fns));

	debugPrint ("\nMonomorphising types ("
		    ^ (Int.toString (P.numItems suffixMap)) ^ " in suffixMap, "
		    ^ (Int.toString (length types)) ^ " in types)\n");

	dbg()
    )

    fun dbg3 Types =
	 debugPrint ("Finished monomorphising types: "
                     ^ plural (length Types) "type"
                     ^ " generated\n "
		     ^ (listToString typedecToString "\n " Types)
		     ^"\n")
    (* ---- Check that we do in fact have a monomorphic program ---- *)

    fun htv (TVARty _) = true
      | htv (ARROWty (t1, t2)) = htv t1 orelse htv t2
      | htv (ARRAYty ty) = htv ty
      | htv (CONty (tys, _)) = List.exists htv tys
      | htv (PRODUCTty tys) = List.exists htv tys
      | htv _ = false

    fun hcon (TYPEcon (l,(types,_),h,_)) = List.exists htv types

    fun checkMono l =
	case List.find (fn TYPEdec(_,_,constrs,_) => (List.exists hcon constrs)) l
	 of NONE => ()
	  | SOME d => poly_error ("Polymorphic type\n\n "
				  ^ typedecToString d
				  ^ "\n\nfound after monomorphisation.")

    fun checkValDec (VALdec ((name,_), ty, _)) =
	if htv ty
	then print ("WARNING: " ^ name ^ " has type " ^ typeToString ty ^ "\n")
	else ()
      | checkValDec _ = ()
in

fun monomorphise (PROG(typeDecs, valDecs, classes, fblocks)) progenv =
    let
	val () = setTypeDecs typeDecs

	val l = map (fn TYPEdec(v,(t,_),_,_) => (v,t)) typeDecs

	val () = app (fn ([],d) => P.insert suffixMap (([], d),"") | _ => ()) l
		 (* WHAT ABOUT INVERSEMAP ?? *)

        val allFunDefs = NAsyntfn.collapse fblocks

	fun getFunTy' loc f =
	    case getFunTy f valDecs of
		NONE => error loc "[Mono]: missing valdec"
	      | SOME ty => ty

        val rootFuns = List.filter (fn (_,ty,_,_) => not (htv ty))
            (map (fn FUNdef((f,floc), _,_,_,_) => (f, getFunTy' floc f , [], f))
		 allFunDefs)

	val () = dbg1 rootFuns

        val (monoFns, monoValDecs) =
	    monoFuns rootFuns [] [] allFunDefs valDecs [] (Env.getMainEnv progenv)

	val () = app checkValDec monoValDecs
val dbg = ref false

	val monoBlocks =
	    if !dbg then
		map (fn x => FUNblock [x]) monoFns
	    else
		MonoUtil.makeFunBlocks monoFns allFunDefs []

	val () = dbg2 monoFns typeDecs

        val ValDecs = MonoUtil.reorderValdecs ((List.filter safeDec valDecs) @ monoValDecs)

	val () = P.apply findInnerTypes suffixMap

	val () = P.apply makeMonoDec suffixMap

	val monodecs = getMonoDecs ()
        val TypeDecs = MonoUtil.reorderDecs monodecs typeDecs

(*	val () = checkMono TypeDecs*)
        (* Better:  make sure that polymorphic datatypes are eliminated by the time we get here *)

	val () = dbg3 TypeDecs
    in
        PROG(TypeDecs, ValDecs, classes, monoBlocks)
    end
end (* local fun ... *)
end (* local structure P = Polyhash ... *)
end (* local open *)


(*
        and instantiateList [] todo0 todo ftodo0 ftodo es' = (List.rev es', todo, ftodo)
          | instantiateList (e::es) todo0 todo ftodo0 ftodo es' =
            let
                val (e', todo', ftodo') = instantiate (todo0@todo) (ftodo0@ftodo) e
            in
                instantiateList es todo0 (todo'@todo) ftodo0 (ftodo'@ftodo) (e'::es')
            end

*)

(*    and instantiateList [] toood oodot ootod ooodt todoo = (List.rev todoo, oodot, ooodt)
      | instantiateList (toood::dooot) odoot dotoo ootod dtooo otdoo =
        let
            val (toodo, dooto, oodto) = inst (odoot@dotoo) (ootod@dtooo) toood
        in
            instantiateList dooot odoot (dooto@dotoo) ootod (oodto@dtooo) (toodo::otdoo)
        end
*)
