open Classdecl

exception Metadata of string

(* Attribute names needed by gf and gdf *)

val funInfoAttrName  = "uk.ac.ed.lfcs.mrg.GrailFunctionInfo"
val nullInfoAttrName = "uk.ac.ed.lfcs.mrg.NullInfo"
val typeInfoAttrName = "uk.ac.ed.lfcs.mrg.TypeInfo"
val layoutInfoName   = "uk.ac.ed.lfcs.mrg.LayoutInfo"


	  (* ---------------- Metadata output ---------------- *)



val attrName = ref ""
exception noMarkers
exception Metadata of string

local
fun err s = raise Fail ("Metadata output error: " ^ s ^ " [" ^ !attrName ^ "]")
val out = ref []
in

fun w8fromInt n = if n<0 orelse n>255 
		  then raise Fail ("integer too big for Word8 [" ^ !attrName ^ "]")
		  else Word8.fromInt n
						 

fun initOutput s = 
    (
     attrName := s;
     out := []
    )

fun putInt n = out := (w8fromInt n) :: (!out)

fun putString s = 
    let
	fun putl l acc = 
	    case l of 
		[] => acc
	      | h::t => putl t ((w8fromInt (Char.ord h))::acc)

	val n = String.size s
    in 
	if n > 255 then err "string too long"
	else (
	    putInt n;
	    out := putl (String.explode s) (!out)
	    )
    end

fun putList put_item l = 
    let 
	fun put l = 
	    case l of 
		[] => ()
	      | h::t => (put_item h; put t)
    in
	(putInt (length l); put l)
    end

fun getOutput () = ATTR {attr = !attrName, info = Word8Vector.fromList (rev (!out))}

end (* local *)




                 (* ---------------- Metadata input ---------------- *)

fun findAttr name l =
    case l of 
	[] => NONE
      | a::t => 
	let in 
	    case a of 
		ATTR {attr, info} => 
		if attr = name
		then SOME info
		else findAttr name t
	      | _ => findAttr name t
	end
	
local 
    val v = ref (Word8Vector.fromList [])
    val p = ref 0
    fun err () = raise Fail ("Corrupt metadata? [" ^ !attrName ^ "]")
in

fun initInput name attrs = 
    case findAttr name attrs of
	NONE => false
      | SOME w => 
	(
	 v := w;
	 p := 0;
	 attrName := name;
	 true
	)

fun getInt () =
    let 
	val n = Word8Vector.sub(!v,!p) handle Subscript => err ()
	val () = p := !p+1
    in
	Word8.toInt n
    end
    
fun getString n =
    let 
	val len = getInt n
	val s = String.implode (
		Word8Vector.foldri (fn (_, c, l) => (chr (Word8.toInt c)::l)) [] (!v, !p, SOME len))
	    handle Subscript => err ()
	val () = p := !p + len
    in
	s
    end
    
fun getList get_item =
    let
	fun getn n acc = 
	    if n=0 then rev acc
	    else
		getn (n-1) (get_item()::acc)
		
	val n = getInt()
    in
	getn n []
    end

end (* local *)



                (* ---------------- Individual metadata types ---------------- *)


             (* ---------------- Diamond class layout information ---------------- *)

fun makeLayoutInfo w = (* layout = string * (string * int * (string * string) list) list *)
    case w of 
	NONE => ((*TextIO.print "WARNING: no layout info produced\n" ;*) NONE)
      | SOME l =>
	let
	    fun doArg (s1,s2) = (putString s1; putString s2)
	    fun doCon (cname, tag, args) = 
		(putString cname; putInt tag; putList doArg args)
					   
	    fun doItem (tname, constructors) = 
		(putString tname; putList doCon constructors)

	    val () = initOutput layoutInfoName
	    val () = putList doItem l
	in
	    SOME (getOutput ())
	end
    
fun getLayoutInfo l =
    if 
	initInput layoutInfoName l 
    then
	let 
	    fun getConArg () = (getString(), getString())
			       
	    fun getCon () =
		let 
		    val cname = getString ()
		    val tag = getInt ()
		    val args = getList getConArg
		in
		    (cname, tag, args)
		end
		
		
	    fun getType () =
		let 
		    val tname = getString ()
		    val constructors = getList getCon
		in
		    (tname, constructors)
		end
	in
	    SOME (getList getType)
	end
    else 
	NONE
	 


            (* ---------------- Function-argument metadata ---------------- *)

fun makeFunInfo findVar funLabels =
    (* Function-argument metadata:  serialised version of (fun_name * (type*param_name)list)list *)
let
    val () = initOutput funInfoAttrName

    fun locOfArg name =
	let val (_, register) = findVar name
	in
	    Localvar.toInt register
	end
	
    fun putArg (_,name) = putInt (locOfArg name)
			    
    fun putItem (fname, (_,args,_)) = (putString fname; putList putArg args)

    val () = putList putItem funLabels
in	
    getOutput ()
end


fun getFunInfo l typeAndNameOfVar =
    if 
	initInput funInfoAttrName l
    then
	let
	    fun getArg () = typeAndNameOfVar (getInt())
	    fun getItem () = (getString(), getList getArg)
	in
	    getList getItem
	end
    else
	[]
    (* <init> doesn't have this attribute;  should it? *)
	


       (* ---------------- Metadata for types of null values ---------------- *)

(* We create a map from names of null (Camelot) constructors
   to integers (called markers);  each class has its own map.
   For each method we also save a list of markers,
   one for every null value appearing in the method.  Non-annotated
   null values (arising from Camelot objects, not datatypes) have marker 0.
   All of this information is subsequently serialised in metadata.  We 
   assume that the program has at most 255 distinct null constructors. *)

local 
    structure P = Polyhash
    val markers = ref []
    fun pushMarker n = markers := n::(!markers)
    fun getMarkers () = !markers

    val mhash: (string, int) P.hash_table ref
      = ref (P.mkTable(P.hash,op=)(20, Metadata "unknown marker"))

    val currentMarker = ref 0
			
    fun newMarker c =
	let 
	    val () = currentMarker := !currentMarker + 1;
	    val m = !currentMarker
(*	    val () = print (c ^ " -> " ^ Int.toString m ^ "\n")*)
	    val () = if m > 255 then raise Metadata "Too many markers"
		     else P.insert (!mhash) (c, m)
	in
	    pushMarker m
	end
in

fun resetMarkers () = markers := []

fun resetTypes () = (
    currentMarker := 0;
    mhash := P.mkTable(P.hash,op=)(20, Metadata "unknown marker")
)

fun saveMarker c = 
    case c of 
	NONE => pushMarker 0
      | SOME d => 
	let in
	    case P.peek (!mhash) d of 
		NONE => newMarker d (* haven't seen this one before *)
	      | SOME n => pushMarker n
	end

fun makeNullInfo () =
    let 
	val info = Word8Vector.fromList (map w8fromInt (!markers))
    in
	ATTR {attr=nullInfoAttrName, info=info}
    end
    (* info looks like:  marker_1, ..., marker_n  *)
    (* Note that the order isn't reversed,  because of 
       the way we use an accumulator to produce bytecode. *)

fun makeTypeInfo () =
    let
	val () = initOutput typeInfoAttrName

	val l = P.listItems (!mhash)

	fun putItem (tname, marker) = (putString tname; putInt marker)

	val () = putList putItem l
    in
	getOutput ()
    end
    (* info looks like:
         num_types, (* k, say *)
         length of name_1, bytes of name_1, marker_for_name_1,
         ...,
         length of name_k, bytes of name_k, marker_for_name_k,
     *)

end

local
    val mtab = ref (Intmap.empty(): string Intmap.intmap)
    val markersFound = ref false
    val markers = ref (Word8Vector.fromList [])
    val pos = ref 0
in

fun readTypeInfo l =
    if initInput typeInfoAttrName l 
    then
	let 
	    val numNulls = getInt ()

	    fun makeMap j hash = 
		if j = 0 then hash
		else
		let 
		    val name = getString ()
		    val marker = getInt ()
		    val hash' = Intmap.insert (hash, marker, name)
(*                    val () = print (name ^ " -> " ^ Int.toString marker ^ "\n")*)
		in
		    makeMap (j-1) hash'
		end
		
	    val hash = makeMap numNulls (Intmap.empty())
			    
	    val () = mtab := hash
	in
	    ()
	end
	else
	    ()

fun readNullInfo l =
    case findAttr nullInfoAttrName l of
	NONE => markersFound := false
      | SOME v => (markersFound := true; pos := 0; markers := v)

fun nextMarker () =
    if !markersFound then 
	Intmap.peek (!mtab, Word8.toInt (Word8Vector.sub(!markers, !pos)))
	before pos := !pos + 1
	handle Subscript => raise Metadata "ran out of type markers: corrupt metadata?"
    else
	raise noMarkers
end