(* Perform linearity checking *)

open Normsyn Util

fun linearityError s = Util.ierror ("[Linearity.sml]: " ^ s)

fun heapfree t =
    case t of
	INTty => true
      | CHARty => true
      | BOOLty => true
      | FLOATty => true
      | STRINGty => true (* Not really,  but let's pretend *)
      | OBJECTty _ => true (* Boggle. But diamond-free. *)
      | UNITty => true
      | TVARty _ => false  (* Shouldn't happen *)
      | ARRAYty t => true  (* OK, it isn't.  Let's pretend that it is *)
      | ARROWty (t,t') => heapfree t andalso heapfree t'
	(* Probably irrelevant without h.o. fns *)
      | PRODUCTty  _ => false
      | CONty _ => false
      | DIAMONDty _ => false

fun mergeCase ((l,e), (l',e')) = (l@l',e@e')
fun mergeCaseL ls = foldr mergeCase ([],[]) ls

fun bind x l = List.filter (fn y=>x=y) l
(* bind not needed because of normalisation? *)
(* Problem. Error messages will be confusing when they talk of x#3
when the variable was x, due to normalisation. Locations should be
fine, and so the printed context will be okay. Perhaps associate x#3
with x or chop off #... in error messages. *)
(* OK,  I did this.  Can there be a problem with names like ?t4 ? *)


fun chkLinearExp exp env =
    let
        fun typeOf x = Env.getVarTy x env

        fun usesHeap x = (not o heapfree o typeOf) x
        fun merge u (([],err), (l,err')) = (l,err@err')
          | merge u ((h::t,err), (l,err')) =
            if usesHeap h andalso List.exists (fn x=>x=h) l  then
                let val error = (u, ("Variable " ^ (Util.chop h #"#")^ " of type " ^
				       Asyntfn.typeToString (typeOf h) ^ " used non-linearly")) in
                    merge u ((t,err), (l, error::err'))
                end
            else merge u ((t,err), (h::l,err'))
        and mergeL u ls = foldr (merge u) ([],[]) ls

	fun chk e =
	    let fun g v =
		    case v of
			VARval (x,_,_) => ([x], [])
		      | _ => ([],[])
	    in
		case e of
		    VALexp (v,_) => g v
		  | UNARYexp (_, v,_) => g v
		  | BINexp (_, v1, v2, l) => merge l (g v1, g v2)
		  | IFexp (TEST(_, v1, v2, _), e1, e2, l) =>
		    let val m = merge l (g v1, g v2)
		    in
			merge l (m, mergeCase (chk e1, chk e2)) (* used l twice ?? *)
		    end
		  | MATCHexp ((x,_), rules,l) => merge l (([x],[]), mergeCaseL (map chkMRule rules))
		  | LETexp (v, e, e',l) => merge l (chk e, chk e')
		  | APPexp (v, vs, ext,l) => mergeL l (map g vs)
		  | CONexp (c, args, dia, l) =>
		    let val args' =
			    case dia of
				NONE => args
			      | SOME (d,dl) => (VARval (d,LOCAL,dl))::args  (* EEEK *)
		    in
			mergeL l (map g args')
		    end
		  (* You're not allowed to use the diamond again.  Maybe we
                     should enforce this even if linearity isn't on?  Maybe not.
		     ### Bad location ###. *)
                  | TYPEDexp (e,_,_) => chk e
		  | COERCEexp (e,_,_) => chk e
		  | NEWexp(c,vs,l) => mergeL l (map g vs)
		  | SUPERMAKERexp(vs,l) => mergeL l (map g vs)
		  | INVOKEexp(f,mname,vs,l) => mergeL l (map g vs)
		  | UPDATEexp(x,v,l) => g v
		  | GETexp(obj,x,l) => ([], []) (* DONT KNOW IF THIS IS RIGHT *)
		  | SGETexp x => ([],[])
		  | ASSERTexp (e,_,_,_) => chk e
	    end

        and chkMRule (MATCHrule(_, _, _, e,_)) = chk e
          | chkMRule (OOMATCHrule(_, e, _)) = chk e

         (* At this point we could check that if there's a diamond then it's
            actually used in (all branches of) e,  thus spotting potential
            memory leaks.  [They're not real memory leaks since the JVM garbage
            collector will eventually reclaim them (maybe),  but we could give
            a warning].  This might be tricky if the diamond is passed as a function
            argument which never gets used:  how can you tell if the diamond
            is really assigned to in all possible situations? Strict linearity?
            (But there's a problem with unused arguments anyway).  *)
	 (* If linearity is turned on then nondestructive matches can only
            lead to memory leaks since if you have "match l with h::t => ..."
	    you can't refer to the name l any more,  so you've lost the only
            handle you have on the heap space which l occupies.  Maybe we should
            demand that all match rules involve @,  or at least give a warning
            if they don't *)
	 (* Also,  for Steffen's analysis you should never allocate any memory:
	    in theory the main function should take a list of diamonds which
            is used for all required allocation.  For truly linear memory usage
            we'd have to prohibit undecorated constructors.  Perhaps we could have
            a function to initially allocate a given amount of memory on the freelist
            and then work only with that; this would require a proof that the freelist
            was never exhausted (Steffen's stuff could help with this),  or we
            could raise an exception if we ran out of freelist. *)
         (* Also we could check that in an expression such as "let _ = free d in e",
            d isn't used in e (since d will have been returned to the freelist).  This
            can't happen if linearity is enforced,  but when it isn't we should probably
            give a warning (in Asyntfn?). *)
         (* Do we need to do anything special for null constructors like !Nil ? *)
         (* Oops, didn't get around to that. *)
    in
        chk exp
    end


fun chkLinearFunDef env (FUNdef((f,_),vs,inst,exp,l)) =
    let
        val f_env = Env.getVarEnv f env
    in
        chkLinearExp exp f_env
    end


fun checkProgram (p as PROG(types, valdecs, classdefs, funblocks)) env =
    let
        val mainenv = Env.getMainEnv env
        fun concat2 l = foldr (fn ((a,b),(a',b')) => (a@a',b@b')) ([],[]) l
        val (_, errs) = concat2 (map (chkLinearFunDef mainenv) (NAsyntfn.collapse funblocks))
        val () = if errs <> [] then errors errs else ()
    in () end
