(* Lambda-lifting: replace expressions (fun x -> ...) with global functions *)

(* eg:

   let f l =
   let y = 7 in
   map (fun x -> x+y) l

  --->

   let lam$0 y x = x+y

   let f l =
   let y = 7 in
   map (lam$0 y) l

*)


local

open Absyn
exception lamError of string
val counter = ref 0
fun newLamName () = ("lam$" ^ Int.toString (!counter)) before counter := !counter + 1

val lams = ref []

val empty = Splayset.empty (String.compare)
fun add s v = Splayset.add (s,v)
fun addlist s l = Splayset.addList (s,l)
fun addArgs s l =
    case l of [] => s
	    | h::t =>
	      case
		  h of VAR(x,_) => addArgs (add s x) t
		     | _ => addArgs s t

(* Get the free variables within the body of a lambda-expression:  these will
   be required as extra arguments for the lifted version *)

(* This may not be entirely OK *)

fun fv expr (vars as (bound, free)) =
    case expr of
    VALexp (v,_) =>
    let in
	case v of
	    VARval (var,_,_) => if Splayset.member (bound,var)
			      then vars else (bound, add free var)
	  | _ => vars
    end
  | UNARYexp(_,e,_) => fv e vars
  | BINexp(_,e,e',_) =>fv e' (fv e vars)
  | IFexp (e,e1,e2,_) => fv_list [e,e1,e2] vars
  | MATCHexp(e,l,_) => 	fv_mrules l (fv e vars)
  | CONexp(c,l,a,_) => fv_list l vars
  | APPexp(e,es,_,_) => fv_list es (fv e vars)
  | TYPEDexp (e,t, _) => fv e vars
  | COERCEexp (e,t, _) => fv e vars
  | NEWexp (class, es, _) => fv_list es vars
  | SUPERMAKERexp (es, _) => fv_list es vars
  | INVOKEexp (e, _, es, _) => fv_list es (fv e vars)
  | UPDATEexp (_,e,_) => fv e vars
  | GETexp (e,_, _) => fv e vars
  | SGETexp _ => vars
  | ASSERTexp (e,as1,as2, _) => fv e vars
  | LETexp((x,_),e1,e2,_) => fv e2 (fv e1 (add bound x, free))
  | LAMexp (args, e, _) => fv e (addArgs bound args, free)

and fv_list es acc = foldl (fn (vars,e) => fv vars e) acc es

and fv_mrules l vars =
    let fun doRule (r, (bound,free)) =
	    case r of
		MATCHrule (c,args,_,e,_) => fv e (addlist bound (map #1 args), free)
	      | OOMATCHrule (_,e,_) => fv e (bound,free)
    in
	foldl doRule vars l
    end


fun mkArg v = VAR (v, NONE)
fun mkValExp l v = VALexp (VARval(v,LOCAL,l),l)


in

fun getLams () = !lams before lams := []

fun addlam expr =
    case expr of
	LAMexp (largs,e,loc) =>
	let
	    val (b,f) = fv expr (empty, empty)
	    val free = Splayset.listItems f

	    val name = newLamName ()
	    val vargs = (map mkArg free)@largs
	    val lifted = FUNdef ((name,loc), vargs, STATIC, e, loc)
	    val () = lams := lifted :: (!lams)  (* save it *)

	    val args = map (fn x => VALexp(VARval (x,LOCAL,loc), loc)) free
	    val app = (* Replace the lamexp with the name of the lifted version,
                         possibly supplying some free variables as args *)
		case args of
		    [] => VALexp (VARval(name,LOCAL,loc),loc)
		  | _ => APPexp (VALexp(VARval(name,LOCAL,loc),loc), args, GLOBAL, loc)
	in
	    app  (* should we return the Normsyn version? No: expr may contain embedded lambdas *)
	end

      | _ => Util.error (Asyntfn.getU expr) "Trying to lamdba-lift non-function"

end
