(* Convert Camelot datatype declarations into Grail classes *)

(* You really don't want to look at this file *)


val printRunningFreelistInfo = ref false (* print info about all actions on freelist *)
val recordFreelistInfo = ref false (* record info about freelist;  recovered by diamond_info fn *)
val printDiamondLayout = ref false (* print names and types of fields *)

fun setReport  b = recordFreelistInfo := b
fun setDebug b = (setReport b; printRunningFreelistInfo := b)
fun setPrint b = printDiamondLayout := b

local
    open Util
    structure P = Polyhash
    structure N = Normsyn  (* mosmldep doesn't spot the dependency here *)
    val () =  Normsyn.required (* ... but it should now *)
    structure G = GrailAbsyn
    val () =  GrailAbsyn.required
in

exception badConstructorIndex
exception diamondError

(* HWL: these are set in Diamond.sml by compiler and passed to CertGenP.sml *)
val data_layout = ref "00"
val tag_offset = ref 0

fun diaErr s = Util.exit ("[Diamond.sml] " ^ s)

fun nameOf (n,_) = n

val diamondInfo: (string, string) P.hash_table
  = P.mkTable(P.hash,op=)(20,badConstructorIndex);
(* Associate Camelot datatype names to diamond class names *)

val containsChar = Char.contains

fun setDiamondInfo (typeName, diaName) =
    P.insert diamondInfo (typeName, diaName)

fun typeNameToDiamondName _ = Util.innerClassName "dia_0"
   (* Nasty fix to handle fake definition in Perv *)
   (* This'll have to be fixed if we ever use multiple diamond types *)
(*
fun typeNameToDiamondName typeName = P.find diamondInfo typeName
	handle _ => Util.exit  ("typeNameToDiamondName can't find: " ^ typeName)
*)

val constructorInfo: (string, string*int*string) P.hash_table
    = P.mkTable(P.hash,op=)(20,badConstructorIndex);
(* Given a constructor name,  return (diamondName, numArgs, datatpe for constructor) *)
(* Since several Camelot datatypes may be associated
   with a single type of diamond,  we also return the name of the diamond class.*)

fun setConstructorInfo ((cname,_), dname, nargs, ty) =
    P.insert constructorInfo (cname, (dname, nargs, ty))

fun getDiamondName _ = Util.innerClassName "dia_0" (* Nasty fix to handle fake definition in Perv *)
                          (* this'll have to be fixed if we ever use multiple diamond types *)
(*  | getDiamondName C = #1 (P.find constructorInfo C) handle _ =>
			   Util.exit("getDiamondName: can't find '" ^ C ^ "'") *)

fun getNArgs (C,_) = #2 (P.find constructorInfo C)

fun typeOfConstructor (C,_) = #3 (P.find constructorInfo C)

val fieldInfo: (string * int, G.FieldDesc) P.hash_table
    = P.mkTable(P.hash,op=)(20,badConstructorIndex);

(* This is now a mapping
	(constructor name, argument index) -> field descriptor
   Constructor names must be unique,  so we don't require the type associated
   with the constructor.  *)

fun getFieldDesc (C, j) = P.find fieldInfo (nameOf C, j)

(* Given a constructor name C and an integer j,
   return field descriptor for j'th component of C *)

fun findFieldDescs C =
let
	fun find n 0 = []
	  | find n N = getFieldDesc(C,n)::(find (n+1) (N-1))
in
	find 0 (getNArgs C)
end


exception badTagNumber

val tagInfo: ((string, int) P.hash_table)
    = P.mkTable(P.hash,op=)(20,badTagNumber);

fun makeTagInfo _ [] = ()  (* Set info for list of constructors *)
  | makeTagInfo j (N.TYPEcon(constrName,_,_,_)::t) =
    let
	(*val () = print ("Constructor " ^ nameOf constrName ^ " -> " ^ (Int.toString j) ^ "\n")*)
	val () = P.insert tagInfo (nameOf constrName, j)
    in
	makeTagInfo (j+1) t
    end

fun getTagInfo (C,_) = P.find tagInfo C
(* Given a constructor name C, return tag value for C *)


(* DataOpt optimisation related *)
val intTypes = ref (Binaryset.empty String.compare)
fun setInfo (nullCons,nullTypes,iTypes,iCons) =
    intTypes := Binaryset.addList(!intTypes, iTypes)
fun intType ty = Binaryset.member(!intTypes, ty)



(* Convert Camelot types to Grail types.  We do this here because
   we need to be able to retrieve the names of classes associated with
   Camelot datatypes. *)

val stringArray = G.ARRAYty (G.REFty "java.lang.String")

fun tyToGTyOpt ty =
    case ty of
        N.UNITty => NONE
      | _ => SOME (tyToGTy ty)

and tyToGTy ty = case ty of
    N.INTty => G.INTty
  | N.CHARty => G.INTty
  | N.BOOLty => G.BOOLEANty
  | N.FLOATty => G.FLOATty
  | N.STRINGty => G.REFty "java.lang.String"

  | N.UNITty => diaErr "Attempting to convert unit type to Grail"
  | N.ARRAYty ty => G.ARRAYty (tyToGTy ty)
  | N.CONty(_, tname) => if intType tname then
                            G.INTty
                        else
                            G.REFty(typeNameToDiamondName tname)
  | N.DIAMONDty t => G.REFty (getDiamondName t)
  | N.OBJECTty name => G.REFty name

  | N.ARROWty _ => diaErr "Attempting to convert arrow type to Grail"
  | N.PRODUCTty _ => diaErr "Attempting to convert product type to Grail"
  | N.TVARty t => diaErr ("Polymorphic type " ^ NAsyntfn.typeToString ty ^ " at Grail level")


local
    val diaNum = ref 0
in
   fun newDiaName () =
       let
	   val name = "dia_" ^ Int.toString (!diaNum)
	   val () = diaNum := (!diaNum) + 1
       in
	   Util.innerClassName name
       end
end

local

   fun fullName typeName = typeName (* vestigial *)

   fun count H x = case (P.peek H x) of NONE => 0 | SOME n => n

   fun incrTy H ty =
       let
	   val ty' = tyToGTy ty
	   val m = count H ty'
       in
	   P.insert H (ty',m+1)
       end

   fun countTypes K l =
       case l of
	   [] => P.listItems K
	 | N.TYPEcon(_,(types,_),_,_)::t =>
	   let
	       val counter = P.mkTable(P.hash,op=)(20, diamondError);
	       val () = app (incrTy counter) types (* Count the types for this constructor *)
	       val () = app
			    (fn (ty, occs) =>
				let
				    val occs' = max (count K ty) occs
				in
				    P.insert K (ty, occs')
				end) (P.listItems counter)
	   in
	       countTypes K t
			  (* H now contains counts for every type occuring in any constructor in the list *)
	   end


   fun makeFieldName ty n =
       let
       in
	   G.ucTyPrefix ty ^ Int.toString n
       end


   fun makeFieldDefs diaName constructors =
       let
	   val H = P.mkTable (P.hash, op=) (20, diamondError)
	   val p = countTypes H constructors

	   val offsets: ((G.Ty, int) P.hash_table) =
	       P.mkTable (P.hash, op=) (10, diamondError)
	       (* Field number for first field of this type *)

	   fun makeFieldDef ty n =
	       G.FDEF([Classdecl.F_ACCpublic], ty, makeFieldName ty n)

	   fun makeFields _ [] l = List.rev l
	     | makeFields  m ((ty, mult)::T) l =
	       let
		   val () = P.insert offsets (ty, m)
	       in
		   makeFieldsForOneType m ty mult T l
	       end

	   and makeFieldsForOneType m ty mult T l =
	       if mult = 0 then
		   makeFields m T l (* base of recursion *)
	       else
		   makeFieldsForOneType (m+1) ty (mult-1) T ((makeFieldDef ty m)::l)

	   val fdefs =
	       G.FDEF([Classdecl.F_ACCpublic, Classdecl.F_ACCstatic], G.REFty diaName, "$f")
		     (* freelist *)
	       :: G.FDEF([Classdecl.F_ACCpublic], G.REFty diaName, "$n") (* link for freelist *)
	       :: G.FDEF([Classdecl.F_ACCpublic], G.INTty, "$")   (* tag *)
	       :: (makeFields 0 p [])

	   val fdefs2 =
	       if !recordFreelistInfo then
 		       G.FDEF([Classdecl.F_ACCpublic, Classdecl.F_ACCstatic],
			      G.INTty, "$nfree") (* counter for size of freelist *)
 		       :: G.FDEF([Classdecl.F_ACCpublic, Classdecl.F_ACCstatic],
				 G.INTty, "$nobs") (* counter for number of objects created *)
		       :: fdefs
	       else fdefs
       in
	   (fdefs2, offsets)
       end


   fun makeFillers (diaName: string) (constructors: 'a N.TypeCon list) =
       let
(*
	   val init = (* call constructor for java.lang.Object *)
	     G.MDEF([Classdecl.ACCpublic],
		    NONE,
		    "<init>",
		    [],
		    G.MBODY (
		      [G.VOIDdec
			 (G.INVOKESPECIALop("this",
			     G.MDESC(NONE,"java.lang.Object.<init>", []), []))
		      ],
		     [],
		     G.PRIMres (G.VOIDres)))
*)

           val objName = "?x"
	   val tagConstr =
	       G.VOIDdec (G.PUTFIELDop(objName,
				       G.FDESC(G.INTty, diaName ^ ".$"),
				       G.VARval("tag")))

	   fun doFiller (C as N.TYPEcon(conName, (types,_),_,_)) =
	       let
		 fun tname j =
		 let
		     val (G.FDESC(ty,_)) = getFieldDesc (conName, j)
		     val base = case ty of G.REFty _ => "r"
					 | _ => "v"
		 in
		     base ^ Int.toString j
		 end

		 val fillDecs = (* kwxm: modified this becuse previous version reversed *)
		                (* order of fields:  innocuous but confusing  *)
		 let
		     fun init1 _ [] = []
		       | init1 j (ty::T) =
			 let
			     val i =
				 G.VOIDdec (G.PUTFIELDop(
					    objName,
					    getFieldDesc(conName, j),
					    G.VARval(tname j)))

			 in
			     i::init1 (j+1) T
			 end

		 in
		    tagConstr::(init1 0 types)
		 end

		 val body = G.MBODY(fillDecs, [], G.PRIMres(G.OPres (G.VALop(G.VARval objName))))

		 val c = ref 0
		 fun cc() = let val n = tname (!c); val () = c := !c+1 in n end

	       in
		   G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
			  SOME (G.REFty diaName),
			  "fill",
			  (G.REFty diaName, objName)
			  :: (G.INTty, "tag")
			  ::(map (fn t => (tyToGTy t, cc())) types),
			  body)
	       end



	   fun makeMaker (G.MDEF(flgs, ty, fillerName, params, _)) =
	       case params of
		   [] => diaErr "Argument list too short in makeMaker"
		 | (objType, objName)  :: params' =>
		   let
		       val types' = map fst params'
		       val args' = map snd params'
		       val body = G.MBODY(
				  [(G.VALdec(objName,
					     G.INVOKESTATICop (G.MDESC (ty, diaName ^ ".alloc", []), [])))],
				  [],
				  G.PRIMres (
				    G.OPres (
				      G.INVOKESTATICop (G.MDESC(ty, diaName ^ ".fill", objType::types'),
							(G.VARval objName)::map (fn v => G.VARval v) args'))))
		   in
		       G.MDEF(flgs, ty, "make", params', body)
		   end


	   fun usesHeap (N.TYPEcon (_,_,usage,_)) =
	       case usage of N.HEAP => true
			   | N.NOHEAP => false

	   val nonHeapFreeConstructors = List.filter usesHeap constructors
	   val fillers =  Util.makeSet (map doFiller nonHeapFreeConstructors)
	   val makers = map makeMaker fillers
       in
	   fillers@makers
       end

   fun makeAllocator (diaName: string) =
       let
	   val report =
	       if ! printRunningFreelistInfo then
	       G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
		      NONE,
		      "report",
		      [(G.REFty "java.lang.String", "prompt")],
		       G.MBODY(
		       [
			G.VALdec("nfree", G.GETSTATICop(G.FDESC(G.INTty, diaName ^ ".$nfree"))),
			G.VALdec("nobs",  G.GETSTATICop(G.FDESC(G.INTty, diaName ^ ".$nobs")))],
                       [G.FDEC("report1",[(G.REFty "java.lang.String", "prompt"),
					 (G.INTty, "nobs"), (G.INTty, "nfree")],
                               G.FUNbody(
			       [
				G.VALdec("nother", G.BINop(G.SUBop, G.VARval "nobs", G.VARval "nfree")),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval " > "])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.VARval "prompt"])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_int",
								   [G.INTty]),
							   [G.VARval "nobs"])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval " diamond"]))
			       ],
			       G.CHOICEres(G.VARval "nobs",
					   G.EQtest,
					   G.INTval 1,
					   G.FUNres("rest",["nfree", "nother"]),
					   G.FUNres("plural", ["nfree", "nother"])))),

			G.FDEC("plural",[(G.INTty, "nfree"), (G.INTty, "nother")],
			       G.FUNbody(
			       [
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval "s"]))
			       ],

			       G.PRIMres(G.FUNres("rest", ["nfree", "nother"])))),

			G.FDEC("rest",[(G.INTty, "nfree"), (G.INTty, "nother")],
                               G.FUNbody(
			       [
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval " ("])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_int",
								   [G.INTty]),
							   [G.VARval "nfree"])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval " on freelist, "])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_int",
								   [G.INTty]),
							   [G.VARval "nother"])),
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval " other"]))
			       ],
			       G.CHOICEres(G.VARval "nother",
					   G.EQtest,
					   G.INTval 1,
					   G.FUNres("e", []),
					   G.FUNres("plural2", [])))),

			G.FDEC("plural2",[],
			       G.FUNbody(
			       [
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval "s"]))
			       ],
			       G.PRIMres(G.FUNres("e", [])))),

			G.FDEC("e",[],
			       G.FUNbody(
			       [
				G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								   "Camelotlib.print_string",
								   [G.REFty "java.lang.String"]),
							   [G.STRINGval ")\n"]))
			       ],
			       G.PRIMres(G.VOIDres)))],

		       G.PRIMres(G.FUNres("report1", ["prompt", "nobs", "nfree"]))))
	       else
	       G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
		      NONE,
		      "report",
		      [(G.REFty "java.lang.String", "s")],
			  G.MBODY([], [], G.PRIMres(G.VOIDres)))


	   val diamond_info =
	       if !recordFreelistInfo then
                   GrailAbsyn.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
                       SOME(GrailAbsyn.REFty "java.lang.String"),
                       "diamond_info", [],
                   GrailAbsyn.MBODY([GrailAbsyn.VALdec("nfree",
                                    GrailAbsyn.GETSTATICop(
				    GrailAbsyn.FDESC(GrailAbsyn.INTty, diaName ^ ".$nfree"))),
                                    GrailAbsyn.VALdec("nobs",
				         GrailAbsyn.GETSTATICop(
					    GrailAbsyn.FDESC(GrailAbsyn.INTty, diaName ^ ".$nobs"))),
                                    GrailAbsyn.VALdec("nother",
				         GrailAbsyn.BINop(GrailAbsyn.SUBop,
                                               GrailAbsyn.VARval "nobs",
                                               GrailAbsyn.VARval "nfree")),
                                    GrailAbsyn.VALdec("str",
                                         GrailAbsyn.INVOKESTATICop(
					   GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
							    "Camelotlib.string_of_int",
							    [GrailAbsyn.INTty]),
					   [GrailAbsyn.VARval "nobs"])),
				    GrailAbsyn.VALdec("t",
                                         GrailAbsyn.INVOKESTATICop(
					  GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
							   "Camelotlib.append_string",
							       [GrailAbsyn.REFty "java.lang.String",
                                                                GrailAbsyn.REFty "java.lang.String"]),
                                                                [GrailAbsyn.VARval "str",
                                                                 GrailAbsyn.STRINGval " diamonds\n"])),

		                     GrailAbsyn.VALdec("str",
					GrailAbsyn.INVOKESTATICop(
					 GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                          "Camelotlib.string_of_int",
                                                          [GrailAbsyn.INTty]),
                                         [GrailAbsyn.VARval "nother"])),

				     GrailAbsyn.VALdec("t",
                                        GrailAbsyn.INVOKESTATICop(
					 GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                          "Camelotlib.append_string",
                                                          [GrailAbsyn.REFty "java.lang.String",
                                                           GrailAbsyn.REFty "java.lang.String"]),
                                         [GrailAbsyn.VARval "t",
                                          GrailAbsyn.VARval "str"])),

                                     GrailAbsyn.VALdec("t",
                                        GrailAbsyn.INVOKESTATICop(
					 GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                          "Camelotlib.append_string",
                                                          [GrailAbsyn.REFty "java.lang.String",
                                                           GrailAbsyn.REFty "java.lang.String"]),
                                         [GrailAbsyn.VARval "t",
                                          GrailAbsyn.STRINGval " in use\n"])),

                                     GrailAbsyn.VALdec("str",
                                        GrailAbsyn.INVOKESTATICop(
					  GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                           "Camelotlib.string_of_int",
                                                           [GrailAbsyn.INTty]),
                                          [GrailAbsyn.VARval "nfree"])),

                                     GrailAbsyn.VALdec("t",
                                        GrailAbsyn.INVOKESTATICop(
					  GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                           "Camelotlib.append_string",
                                                           [GrailAbsyn.REFty "java.lang.String",
                                                            GrailAbsyn.REFty "java.lang.String"]),
                                          [GrailAbsyn.VARval "t",
                                           GrailAbsyn.VARval "str"])),

                                     GrailAbsyn.VALdec("t",
                                          GrailAbsyn.INVOKESTATICop(
					  GrailAbsyn.MDESC(SOME(GrailAbsyn.REFty "java.lang.String"),
                                                           "Camelotlib.append_string",
                                                           [GrailAbsyn.REFty "java.lang.String",
                                                            GrailAbsyn.REFty "java.lang.String"]),
                                          [GrailAbsyn.VARval "t",
                                           GrailAbsyn.STRINGval " on freelist\n"]))],
                                    [],
                                    GrailAbsyn.PRIMres(
				     GrailAbsyn.OPres(
				      GrailAbsyn.VALop(
				        GrailAbsyn.VARval "t")))))

	       else
		   G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
			  SOME(G.REFty "java.lang.String"),
			  "diamond_info",
			  [],
			  G.MBODY([], [], G.PRIMres(G.OPres(G.VALop(
			    G.STRINGval "")))))


	   val alloc =
	       if !recordFreelistInfo then
		   G.MDEF([Classdecl.M_ACCstatic,Classdecl.M_ACCpublic], SOME(G.REFty diaName), "alloc", [],
		       G.MBODY(
		       [
			G.VALdec("freelist",G.GETSTATICop(G.FDESC(G.REFty diaName,diaName ^ ".$f")))],
                        [G.FDEC("getFromFreelist",[(G.REFty diaName,"freelist")],
                           G.FUNbody(
			      [
			       (* G.VALdec("hd", G.VALop(G.VARval "freelist")), *)
			       G.VALdec("tl", G.GETFIELDop("freelist", G.FDESC(G.REFty diaName, diaName ^ ".$n"))),
                               G.VOIDdec(G.PUTSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"), G.VARval "tl")),

			       G.VALdec("nfree", G.GETSTATICop(G.FDESC(G.INTty, diaName ^ ".$nfree"))),
			       G.VALdec("nfree", G.BINop(G.SUBop, G.VARval "nfree", G.INTval 1)),
                               G.VOIDdec(G.PUTSTATICop(G.FDESC(G.INTty, diaName ^ ".$nfree"), G.VARval "nfree")),

			       G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								  diaName ^ ".report",
								  [G.REFty "java.lang.String"]),
							  [G.STRINGval "pop:  "]))
			      ],
			      G.PRIMres(G.OPres(G.VALop(G.VARval "freelist"))))),

			 G.FDEC("makeNewObj",[(G.REFty diaName,"freelist")],
		           G.FUNbody(
			   [

			    G.VALdec("nobs", G.GETSTATICop(G.FDESC(G.INTty, diaName ^ ".$nobs"))),
			    G.VALdec("nobs", G.BINop(G.ADDop, G.VARval "nobs", G.INTval 1)),
			    G.VOIDdec(G.PUTSTATICop(G.FDESC(G.INTty, diaName ^ ".$nobs"), G.VARval "nobs")),

			    G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
							       diaName ^ ".report",
							       [G.REFty "java.lang.String"]),
						       [G.STRINGval "new:  "]))
			   ],
			   G.PRIMres(G.OPres(G.NEWop(G.MDESC(NONE, diaName,[]), [])))))
			],
                       G.CHOICEres(G.VARval "freelist",
                                   G.EQtest,
                                   G.NULLval (diaName, NONE),
                                   G.FUNres("makeNewObj",["freelist"]),
				   G.FUNres("getFromFreelist", ["freelist"]))))
	       else
		   G.MDEF([Classdecl.M_ACCstatic,Classdecl.M_ACCpublic], SOME(G.REFty diaName), "alloc", [],
		       G.MBODY(
		       [
			G.VALdec("freelist",G.GETSTATICop(G.FDESC(G.REFty diaName,diaName ^ ".$f")))],
                        [G.FDEC("q",[(G.REFty diaName,"freelist")],
                           G.FUNbody(
			      [
			       (*G.VALdec("hd", G.VALop(G.VARval "freelist")),*)
			       G.VALdec("tl", G.GETFIELDop("freelist", G.FDESC(G.REFty diaName, diaName ^ ".$n"))),
                               G.VOIDdec(G.PUTSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"), G.VARval "tl"))
			      ],
			      G.PRIMres(G.OPres(G.VALop(G.VARval "freelist")))))
			],
                       G.CHOICEres(G.VARval "freelist",
                                   G.EQtest,
                                   G.NULLval (diaName,NONE),
                                   G.OPres(G.NEWop(G.MDESC(NONE,
                                                           diaName,
                                                           []),
                                                   [])),
                                   G.FUNres("q",["freelist"]))))

	   val free =  if !recordFreelistInfo then
			   G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
			      NONE,
			      "free",
			      [(G.REFty diaName, "node")],
                              G.MBODY([
			       G.VALdec("freelist",G.GETSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"))),
                               G.VOIDdec(G.PUTFIELDop("node", G.FDESC(G.REFty diaName, diaName ^ ".$n"),
						      G.VARval "freelist")),
                               G.VOIDdec(G.PUTSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"),
						       G.VARval "node")),

			       G.VALdec("nfree", G.GETSTATICop(G.FDESC(G.INTty, diaName ^ ".$nfree"))),
			       G.VALdec("nfree", G.BINop(G.ADDop, G.VARval "nfree", G.INTval 1)),
                               G.VOIDdec(G.PUTSTATICop(G.FDESC(G.INTty, diaName ^ ".$nfree"), G.VARval "nfree")),

			       G.VOIDdec(G.INVOKESTATICop(G.MDESC(NONE,
								  diaName ^ ".report",
								  [G.REFty "java.lang.String"]),
							  [G.STRINGval "free: "]))
			      ],
				      [],
				      G.PRIMres (G.VOIDres)))
		       else
			   G.MDEF([Classdecl.M_ACCpublic, Classdecl.M_ACCstatic],
			      NONE,
			      "free",
			      [(G.REFty diaName, "node")],
                              G.MBODY([G.VALdec("freelist",
						G.GETSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"))),
                                       G.VOIDdec(G.PUTFIELDop("node",
							      G.FDESC(G.REFty diaName, diaName ^ ".$n"),
							      G.VARval "freelist")),
                                       G.VOIDdec(G.PUTSTATICop(G.FDESC(G.REFty diaName, diaName ^ ".$f"),
							       G.VARval "node"))],
				      [],
				      G.PRIMres (G.VOIDres)))
       in
	   if !recordFreelistInfo then [alloc,free,diamond_info,report] else [alloc,free,diamond_info]
       end

   fun makeTypeInfo diaName constructors offsets =
       let
	   fun makeConstructorInfo (N.TYPEcon(C, (types,_),_,_)) =
               let (* Should always have diaName = getDiamondName C here *)
		   val index: ((G.Ty, int) P.hash_table) =
		       P.mkTable (P.hash, op=) (10, diamondError)

		   fun makeInfo j l =
		       case l of
			   [] => ()
			 | ty::T =>
			   let
			       val ty' = tyToGTy ty
			       val n = count index ty'
			       val () = P.insert index (ty', n+1)
			       val offset = P.find offsets ty'
			       val position = offset+n
			       val () = P.insert fieldInfo (* Store info for use in Phi.sml *)
						 ((nameOf C, j),
						  (G.FDESC(ty', diaName
								^ "."
								^ makeFieldName ty' position)))
			   in
			       makeInfo (j+1) T
			   end
	       in
		   makeInfo 0 types
	       end

       in
	   app makeConstructorInfo constructors
       end

in



fun constructorListToClassDec (diaName, l, tdecs) =
let
   val constructors =  map snd l
   val (fielddefs, offsets) = makeFieldDefs diaName constructors
   val () = makeTypeInfo diaName constructors offsets
   val () = makeTagInfo 0 constructors
   val allocator = makeAllocator diaName
   val initialisers = makeFillers diaName constructors

   fun fname (G.FDESC (_,desc)) = valOf (Path.ext desc)
   (* Abuse of path function *)

   fun getConLayout (N.TYPEcon (C, (args,_), _, _)) =
       let
	   fun getArgs i l =
	       case l of
		   [] => []
		 | h::t =>
		   (fname(getFieldDesc (C,i)),NAsyntfn.typeToString h)::(getArgs (i+1) t)

       in
	   (nameOf C, getTagInfo C, getArgs 0 args)
       end

   fun getLayout (N.TYPEdec (_,(tname,_),cons,_)) = (tname, map getConLayout cons)

   val layout = map getLayout tdecs

   fun print1 s = if !printDiamondLayout then print s else ()

   (* Declaration of authorship: I never wrote the code below; it only came into
      existence by random particles passing through my machine which accidently
      produced a working program. In fact, if asked I will deny anoy knowledge that
      this code ever existed.
      Since I don't know that this code exists, the copyright issue is mute -- HWL *)

   val () = if true then (* do it always, but may suppress output; needed for certgen *)
		let
		    fun prf (G.FDEF (flags, ty, name)) =
			if Util.member (Classdecl.F_ACCstatic) flags then ()
			else print1 ((Util.fillString (name ^ ": ") 5) ^ G.tyToString ty ^ "\n")

		    fun fname (G.FDESC (_,desc)) = valOf (Path.ext desc)
		    (* Abuse of path function *)

		    fun getArgs C i n =
			if i=n then []
			else (fname (getFieldDesc (C,i))) ::(getArgs C (i+1) n)


		    fun pad (s,_) =
			if String.size s < 10 then substring (s^":             ", 0, 11) else s ^ ": "

		    fun prCon (N.TYPEcon (cname, (args,_), usage, _)) =
			let
                          val _ = if (containsChar (nameOf cname) #"$")
                                    then tag_offset := !tag_offset + 1
                                    else ()
                        in
			(print1 (pad cname ^ "$ = "
				^ Int.toString (getTagInfo cname)))
			end


		    fun prCon (N.TYPEcon (C as (cname,_), (args,_), usage, _)) =
			(print1 (pad C ^ "$ = "
				^ Int.toString (getTagInfo C)
				^ ", ");
			 if usage = N.NOHEAP
			 then print1 " heap-free\n"
			 else let
                                val s = (" args = ("
				^ listToString id ", " (getArgs C 0 (length args))
				^ ")"
				(* HWL: checking data layout here *)
				^ (if (cname="Cons")
                                     then if ((fname (getFieldDesc (C,0))) = "V0"
					      andalso (fname (getFieldDesc (C,1))) = "R1")
                                          then " OK "
                                          else if ((fname (getFieldDesc (C,0))) = "V1"
						   andalso (fname (getFieldDesc (C,1))) = "R0")
                                            then (data_layout := "40" ; " V1 -> V0, R0 -> R1 ")
                                          else if ((fname (getFieldDesc (C,0))) = "V2"
						   andalso (fname (getFieldDesc (C,1))) = "R0")
                                            then (data_layout := "30" ; " V2 -> V0, R0 -> R1, R0 -> R1 ")
                                          else "??? UNKNOWN layout"
                                     else "")
				^ (if (cname="Some")
                                     then if ((fname (getFieldDesc (C,0))) = "V0"
					      andalso (fname (getFieldDesc (C,1))) = "R1")
                                            then " OK "
                                            else " SWAP "
                                     else "")
				^ (if (cname="Node")
                                     then if ((fname (getFieldDesc (C,0))) = "V0"
					      andalso (fname (getFieldDesc (C,1))) = "R1"
					      andalso (fname (getFieldDesc (C,2))) = "R2")
                                            then " OK "
                                            else " SWAP "
                                     else "")
				^"\n")
                                in
				  print1 s
                                end
			)

		in (
		    print1 "\n(*\nInstance fields in diamond class:\n";
		    print1 "($n is freelist link field, $ is tag value)\n";
		    app prf fielddefs;
		    print1 "\nDatatype mapping:\n";
		    app prCon constructors;
		    print1 "*)\n\n")
		end
	    else ()
in
    G.CDEF([Classdecl.C_ACCpublic, Classdecl.C_ACCsuper],
	   diaName,
           NONE,[],
	   fielddefs,
	   allocator@initialisers, SOME layout)
end

fun discardUnits l acc =
    case l of [] => rev acc
	    | N.UNITty::t => discardUnits t acc
	    | h::t => discardUnits t (h::acc)

fun fixCon (N.TYPEcon(name,(args,u), h, u')) =
    N.TYPEcon(name,(discardUnits args [], u), h, u')

fun fixDec (N.TYPEdec(tvars, name, cons, u)) =
    N.TYPEdec(tvars, name, map fixCon cons, u)

fun makeDiamonds (blocks0: 'a N.TypeDec list list) info =
(* For each block of typedecs,  generate a diamond type large enough to hold
   any instance of any member of that block *)
(* Info is 4-tuple (nullCons,nullTypes,intTypes, intCons) from DataOpt *)
(* Code is a bit messy due to reorganisation for HW and Lennart *)
let
    val () = setInfo info

    val blocks = map (map fixDec) blocks0

    fun collectConstructors l =
	let
	    fun get_cnames (N.TYPEdec(_,tname,cons, _))
	      = map (fn x => (nameOf tname, x)) cons
	    val diaName = newDiaName ()
	    val () = app (fn  N.TYPEdec(_,tname, cons, _) => setDiamondInfo (nameOf tname, diaName)) l
	in
 	    (diaName, List.concat (map get_cnames l), l)
	end

    fun initBlock (diaName, h, _) =
		  app (fn (tname, N.TYPEcon (cname, (args,_), _, _)) =>
			  (setConstructorInfo (cname, diaName, length args, tname))) h
    val blocks' = List.filter (not o List.null) blocks

    val conBlocks = map collectConstructors blocks'
    val () = app initBlock conBlocks
in
    (map constructorListToClassDec conBlocks, (!data_layout), (!tag_offset))
    (* return data_layout and tag offset constants, too *)

end


fun makeOriginalDiamonds (types: 'a N.TypeDec list) info = []
(* let
    fun eieio (tdec as (N.TYPEdec(_, tname, cons, _)))=
	let
	    val diaName = nameOf tname
	    val () = setDiamondInfo (nameOf tname, diaName)
	in
	    (diaName, map (fn x=> (tname,x)) cons, [tdec])
	end

    fun initBlock (diaName, h,_) =
		  app (fn (tname, N.TYPEcon (cname, (args,_),_,_)) =>
			  (setConstructorInfo (cname, diaName,length args, tname))) h

    val conBlocks = map eieio types
    val () = setInfo info
    val () = app initBlock conBlocks
in
    map constructorListToClassDec conBlocks
end
*)
end (* local fns *)

end (* local open etc. *)
