local
    structure P = Polyhash
    structure S = Splaymap
    open Normsyn Util NAsyntfn
in

val testing = ref false


exception DefuncError of string
fun d_error s = Util.ierror ("[Defunc.sml]: " ^ s)

fun conName n x = "T" ^ Int.toString n ^ "_" ^ x
fun appName n = "apply$" ^ Int.toString n
fun typeName n = "t$" ^ Int.toString n


(* ---------------- Mapping arrow types to datatypes ----------------*)

local
    val tIndex = ref 0
in
    fun newIndex () = !tIndex before tIndex := !tIndex + 1
end

val typeSuffix: (Ty, int) P.hash_table =
    P.mkTable (P.hash, op=) (20, DefuncError "typeSuffix")

val invSuffix: (int, Ty) P.hash_table =
    P.mkTable (P.hash, op=) (20, DefuncError "invSuffix ($ in ident?)")


fun saveSuffix ty j =
    (debugPrintln ( "saveSuffix: " ^ typeToString ty 
		    ^ " => t$" ^ Int.toString j );
     P.insert typeSuffix (ty, j);
     P.insert invSuffix  (j, ty)
    )

fun canonical ty =
    case ty of
	ARROWty (s,t) =>
	let val s' = canonical s
	    val t' = canonical t
	    val ty' = ARROWty (s', t')

	    val n = case P.peek typeSuffix ty' of
		   SOME n => (debugPrint ("cfound " ^ Int.toString n ^ "\n"); n)
		 | NONE =>
		   let
		       val i = newIndex ()
		       val () = saveSuffix ty' i
		       val () = debugPrint ("cmade " ^ Int.toString i ^ "\n")
		   in
		       i
		   end
	in
	    CONty ([], typeName n)
	end
      | _ => ty


fun getTySuffix ty =
    case ty of
	ARROWty (s,t) =>
	let 
	    val s' = canonical s
	    val ty' = ARROWty (s', t)
	in 
	    case P.peek typeSuffix ty' of
	       SOME n => (debugPrint ("gfound " ^ Int.toString n ^ "\n"); n)
	     | NONE =>
	       let
		   val i = newIndex ()
		   val () = saveSuffix ty' i
		   val () = debugPrint ("gmade " ^ Int.toString i ^ "\n")
	       in
		   i
	       end
	end
      | CONty (_,tname) => (* DO WE REALLY NEED THIS ??? *)
	if String.isPrefix "t$" tname
	then
	    let
		val n = valOf (Int.fromString (String.extract (tname, 2, NONE)))
	    in
		n
	    end
	else d_error ("[getTySuffix]: found " ^ typeToString ty)
      | _ => d_error ("[getTySuffix]: found " ^ typeToString ty)

fun makeCanon ty =
    case ty of
	ARROWty (s,t) =>
	let
	    val s' = canonical s
	in case t of
	       ARROWty (u,v) => ARROWty (s', ARROWty (makeCanon u, v))
	     | _ => ARROWty (s', t)
	end
      | _ => ty



fun getType ty =
    case ty of
	ARROWty (s,t) =>
	let (*
	    val s1 = getType s  (* just to get the types in the suffix map *)
	    val t1 = getType t
	     *)
	in
	    CONty ([], typeName (getTySuffix ty))
	end
      | _ => ty


fun getDomain n =
    let val ty = P.find invSuffix n
    in
	 ty
    end


fun prTypeSuffixes () = (
    vprint "Type suffixes\n";
    vprint "-------------\n";
    Polyhash.apply (fn (t,n) => vprint (typeToString t ^ ": " 
					^ Int.toString n ^ "\n")) typeSuffix;
    vprint "-------------\n"
)



(* ---------------- Fiddling with arrows ---------------- *)

fun nArrows ty =
    case ty of
	ARROWty (s,t) => 1 + nArrows t
      | _ => 0

fun arrowToList ty =
    case ty of
	ARROWty (s,t) => s::(arrowToList t)
      | _ => [ty]

fun arrowInput ty =
    case ty of
	ARROWty (s,t) => s::(arrowInput t)
      | _ => []

local fun split ty n acc =
    if n=0 then (rev acc, ty)
    else
	case ty of
	    ARROWty (s,t) => split t (n-1) (s::acc)
	  | _ => d_error "Arrow too short to split"
in
fun splitArrow ty n =
    split ty n []
end

fun tyTos t = 
    case t of ARROWty (t1, t2) => "Ar(" ^ tyTos t1 ^ ", " ^ tyTos t2 ^ ")"
	    | _ => typeToString t

fun makeArrow l r =
    case l of [] => raise DefuncError "empty domain in makeArrow"
	    | _ => foldr ARROWty r l

fun rTy ty =
    case ty of
	ARROWty (s,t) => rTy t
      | _ => ty


fun isArrow t = case t of ARROWty _ => true | _ => false
fun nonArrow t = case t of ARROWty _ => false | _ => true

(* ---------------- Fiddling with types ---------------- *)

fun getFunTy f [] = NONE
  | getFunTy f (VALdec((v,_), tys,_)::vals) =
    if f = v
        then SOME tys
    else getFunTy f vals
  | getFunTy f ((CLASSdec _)::vals) = getFunTy f vals


fun fixTy1 ty =
    case P.peek typeSuffix ty of
	NONE => (case ty of ARROWty (t,u) => ARROWty(fixTy1 t, fixTy1 u)
			 | _ => ty)
      | SOME n => CONty ([], typeName n)

fun fixTy ty =
    case ty of ARROWty (t,u) => ARROWty (fixTy1 t, fixTy1 u)
	     | _ => fixTy1 ty

fun fixTy2 ty =
    CONty ([], typeName (getTySuffix ty))


fun forceFixTy ty =
    case ty of ARROWty (t,u) => ARROWty (fixTy2 t, fixTy2 u)
	     | _ => ty


fun fixFunt2 name funt nargs =
    let
	val (bound, rest) = splitArrow funt nargs
	val () = debugPrintln ("#2 IN " ^ name ^ ", bound is ["
			       ^ listToString typeToString ", " bound ^ "]")
	val () = debugPrintln ("#2 rest is " ^ typeToString rest)
(*
	val funt' = (if isArrow rest 
                     (* doing this in all cases seems to cause problems *)
		     then makeArrow (map canonical bound) (canonical rest)
		     else funt)
*)
	val funt' = makeArrow (map canonical bound) (canonical rest)
	val () = if funt = funt' then ()
		 else debugPrintln ("Fixfunt2: type of " ^ name 
				    ^ " altered from " ^ tyTos funt
				    ^ " to " ^ tyTos funt')
    in
	funt'
    end



(* ---------------- Creating "closure" datatypes ---------------- *)


val tCons: (int, (string * Ty list * int option) list) P.hash_table =
    P.mkTable (P.hash, op=) (20, DefuncError "tCons")

fun addCon n s ev args ty =
    let
	val args' = map getType args
	val () = 
	    (debugPrintln ("> con " ^ s );
	     debugPrintln ("> type:  " ^ typeToString ty);
	     debugPrintln ("> args:  " ^ listToString typeToString "," args);
	     debugPrintln ("> args': " ^ listToString typeToString "," args')
	    )
    in
	case P.peek tCons n of
	    NONE => P.insert tCons (n, [(s,args',ev)])
	  | SOME l => if member (s,args',ev) l 
		      then () 
		      else P.insert tCons (n, (s,args',ev)::l)
    end

fun makeCon n (f,tys,_) = TYPEcon ((conName n f, nowhere), 
				   (map getType tys, nowhere), HEAP, nowhere)

fun makeTy (n, l) =
    TYPEdec ([], (typeName n, nowhere), map (makeCon n) l, nowhere)


fun makeTypes () = map makeTy (P.listItems tCons)


fun prCon (s,tys,ext) =
    case tys of [] => vprint (" " ^ s ^ "\n")
	      | _  => vprint (" " ^ s ^ " of " 
			      ^ listToString typeToString " * " tys ^ "\n")

fun prTypes () = (
    vprint "Typedefs\n";
    vprint "-------------\n";
    Polyhash.apply (fn (n,cons) => 
		       (vprint (Int.toString n ^ ":\n"); app prCon cons)) 
		   tCons;
    vprint "-------------\n"
)


(* ---------------- Application functions ---------------- *)

local
    val c = ref 0
    fun nxtc() = Int.toString (!c) before c:= !c+1
in

fun mklist prefix n =
    List.tabulate (n, fn j => prefix ^ nxtc())
end

val mkvals = map (fn v => VARval (v, LOCAL, nowhere))  (* FIX: LOCAL ? *)
val mkmargs = map (fn v => (v, nowhere))

fun mkRule n args (f,boundtys,ev) =
    let
	val names = mklist "x$" (length boundtys)
	val boundargnames = mkmargs names
	val boundargs =  mkvals names

	val allargs = boundargs @ args

	val action =
	    case ev of
		NONE => 
		let 
		    val ext = if Perv.isBuiltIn f then BUILTIN else GLOBAL
		in
		    APPexp ((f,nowhere), allargs, ext, nowhere)
		end
	      | SOME m => 
		let val aname = appName m
		in
		    APPexp ((aname,nowhere), boundargs@args, GLOBAL, nowhere)
		end
    in
	MATCHrule ((conName n f, nowhere), boundargnames, 
		   NOWHERE, action, nowhere)
    end


fun badArrow ty =
    case ty of ARROWty (s,t) => (
	       case s of ARROWty _ => true
		       | _ => badArrow t
	       )
	     | _ => false


fun mkfargs args types = ListPair.map (fn (v,ty) => 
					  VAR(v, NONE (*SOME (getType ty)*))) 
				      (args, types)

fun makeApp (n,l) =
    let
	val dom = getDomain n
	val nargs = nArrows dom
	val argnames = mklist "y$" nargs
	val appargs = mkfargs argnames (arrowToList dom)
	val appvars = mkvals argnames  (* UNIT problems ?? *)
    in
	FUNblock [
	FUNdef ((appName n, nowhere),
		VAR ("f_$",NONE)::appargs,
		STATIC,
		MATCHexp (("f_$",nowhere), map (mkRule n appvars) l, nowhere),
		nowhere)
	]
    end

fun makeApps () = map makeApp (P.listItems tCons)



(* ---------------- Funarg counts ---------------- *)

local
    val argCounts: (string, int) P.hash_table =
	P.mkTable (P.hash, op=) (20, DefuncError "argCount")

    fun setfargcount (FUNdef ((fname,l), args,_,_,_)) = 
	P.insert argCounts (fname, length args)
in

fun setArgCounts b = app (fn FUNblock l => app setfargcount l) b

fun getArgCount f =
    case P.peek argCounts f of
	NONE => raise Fail ("Couldn't get arg count for " ^ f)
      | SOME n => n
end


(* ---------------- Variable type tables ---------------- *)

(* Do this once at the start.  Maybe the types of global functions
   can change.  What about mutually recursive hofs?  Can the type of the
   second function (which we use to defunc the first) change when we
   get round to doing it? *)

fun getGlobalTypes mainenv =
let
    val empty = Splaymap.mkDict String.compare

    val builtins = Perv.builtinTypes ()
    fun addBty (d,m) =
	case d of VALdec((name,_), ty, _) => Splaymap.insert(m, name, ty)
		| _ => m

    val builtinInf = List.foldl addBty empty builtins

        (* Modify return type for functions returning functions  *)
	(* Also modifying the argtypes causes problems sometimes *)   
    fun fixFunt fname funt =  
	let                   
	    val nargs = getArgCount fname
	    val (bound, rest) = splitArrow funt nargs
	    val () = debugPrintln ("## IN " ^ fname ^ ", bound is ["
				   ^ listToString typeToString ", " bound ^ "]")
	    val () = debugPrintln ("## rest is " ^ typeToString rest)

	    val funt' = makeArrow  ( (*map canonical*)  bound ) (canonical rest)  

	    val () = if funt = funt' then ()
		     else debugPrintln ("type of "^ fname ^ " changed from " 
					^ typeToString funt
					^ " to " ^ typeToString funt')
	in
	    funt'
	end

    fun addGty (fname, (fty,_), m) = 
	Splaymap.insert (m, fname, fixFunt fname fty)

in
    Splaymap.foldl addGty builtinInf mainenv
end  

fun getGlobalTypes2 mainenv vdecs =  
(* Previous version didn't work for externals sometimes: try this instead *)
let
    val empty = Splaymap.mkDict String.compare

    val builtins = Perv.builtinTypes ()
    fun addBty (d,m) =
	case d of VALdec((name,_), ty, _) => Splaymap.insert(m, name, ty)
		| _ => m

    val builtinInf = List.foldl addBty empty builtins
    val extInf = List.foldl addBty builtinInf vdecs  (* Duplication: FIX *)

        (* Modify return type for functions returning functions  *)
	(* Also modifying the argtypes causes problems sometimes *)   
    fun fixFunt fname funt =  
	let                   
	    val nargs = getArgCount fname
	    val (bound, rest) = splitArrow funt nargs
	    val () = debugPrintln ("## IN " ^ fname ^ ", bound is ["
				   ^ listToString typeToString ", " bound ^ "]")
	    val () = debugPrintln ("## rest is " ^ typeToString rest)

	    val funt' = makeArrow  ( (*map canonical*)  bound ) (canonical rest)  

	    val () = if funt = funt' then ()
		     else debugPrintln ("type of "^ fname ^ " changed from " 
					^ typeToString funt
					^ " to " ^ typeToString funt')
	in
	    funt'
	end

    fun addGty (fname, (fty,_), m) = 
	Splaymap.insert (m, fname, fixFunt fname fty)
in
    Splaymap.foldl addGty extInf mainenv
end  


fun getVarInf env globalInf (FUNdef((fname,_), args, _,_,_)) =
let
    fun getArgNames l acc =
	case l of [] => acc
		| UNITvar::t => getArgNames t acc
		| VAR(v,_)::t => getArgNames t (v::acc)

    val argnames = getArgNames args []

    val thisEnv = Env.getVarEnv fname env 

    fun addAty (arg,m) = (* Do we really have to do args separately? *)
	Splaymap.insert (m,arg, Env.getVarTy arg thisEnv)
    val argInf = List.foldl addAty globalInf argnames

    fun addVty (v,t,m) =
	if member v argnames then 
	    (debugPrintln ("Found arg " ^ v ^ " with type " ^ tyTos t);
	     m)
	else Splaymap.insert (m,v,t)

    val varInf = Splaymap.foldl addVty argInf thisEnv
in
    varInf
end



(* ---------------- Fresh names ---------------- *)

local
    val counter = ref 0
in
    fun resetTempNames () = counter := 0
    fun tempName s =
    let
	val n = !counter
    in
	"?" ^ s ^ Int.toString n before counter := n + 1
    end
end


(* ---------------- Defunctionalistion ---------------- *)

local
in

fun defunExp varInf e args =
let
    val varInfR = ref varInf

    fun getVtype v =
	let
	    val ty =
		Splaymap.find (!varInfR, v) 
		handle _ => raise Fail ("[Defunc.getVtype]: couldn't find " ^ v)
	in
	    case ty of CONty (_,tname) => (* DO WE REALLY NEED THIS ??? *)
(* We've already converted the arrow type to a datatype: let's invert it *)
	       if String.isPrefix "t$" tname
	       then
		   let
		       val n = valOf (Int.fromString (String.extract (tname, 2, NONE)))
		       val () = debugPrint ("looking for " ^ tname ^ "\n")
		       val ty' = getDomain n
		       val () = debugPrint ("found " ^ typeToString ty' ^ "\n")
		   in
		       ty'
		   end
	       else ty
		     | _ => ty
	end


    fun update v ty = varInfR := Splaymap.insert (!varInfR, v, ty)

    fun makeLets e l u =
	case l of [] => (debugPrint "makeLets done\n"; e)
		| (x,c)::t => makeLets (LETexp(x,c,e,u)) t u

    fun doArgs args T mkExp u =
    let
	fun doArgs' args acc_v acc_norm =
	    case args of
		[] =>
		let
		in
		    makeLets (mkExp (rev acc_v)) acc_norm u
		end
	      | (h as VARval(x,xext,l))::t =>
		let 
		in
		case xext of
		    LOCAL =>
		    let
		    in
			doArgs' t (h::acc_v) acc_norm
		    end
		  | _ =>
		    let
			val ty = getVtype x
			val n = getTySuffix ty
			val X = conName n x
			val () = addCon n x NONE [] ty
			val x' = tempName "u"
			val H = VARval (x',xext,l)
		    in
			doArgs' t (H::acc_v)
				(((x',l), CONexp((X,l),[],NONE,l))::acc_norm) 
		    end
		end
	      | h::t => doArgs' t (h::acc_v) acc_norm
			
	val e' = doArgs' args [] []
    in
	(e', T) before debugPrintln "doArgs done"
    end
    
	
    fun defun exp =
    case exp of
	VALexp (v,_)    =>
	let in
	    case v of
		VARval(x,ext,l) =>
		let
		    val ty = getVtype x
		in
		    case ext of
			LOCAL => exp
				 before debugPrintln "defun VALexp 1 done\n"
		      | _ =>
			if nonArrow ty then exp 
                             (* FIX: (maybe) what if we've got eg an array of fns? *)
			else
			    let (* similar to later code *)
(*			         val ty = canonical ty*)
                                 val n = getTySuffix ty
				 val X = conName n x
				 val () = addCon n x NONE [] ty
			    in
				CONexp ((X,l), [], NONE, l)
				before debugPrintln "defun VALexp 0 done\n"
			    end
		end
	      | _ => exp
		     before debugPrintln "defun VALexp 2 done\n"
	end

      | APPexp ((fname,fl), args,ext,u) =>
	let
	    val (exp',_) = doApp exp
	in
	    exp' before debugPrintln "defun APPexp done\n"
	end

      | LETexp(x as (xx,_), e1, e2, u) =>
	let
	    val xty = getVtype xx
	    val () = debugPrintln ("Type of " ^ xx ^ " is " ^ typeToString xty)

	in
	    case e1 of
		APPexp _ => let
		    val (e1', ty) = doApp e1
		    val () = debugPrintln 
				 ("LETexp [" ^ xx ^"]: after doApp, ty is " 
				  ^ typeToString ty )
		    val () = update (#1 x) ty
		in
		    LETexp (x, e1', defun e2, u) 
		       before debugPrintln "defun LETexp 0 done\n"
		end

	      | VALexp (VARval (y,yext,yloc), loc) =>
		(* This code is very much like that for VALexp above. *)
		let 
		    val tyY = getVtype y
		    val e1' =
			case yext of
			    LOCAL => e1
			  | _ =>
			    if nonArrow tyY then e1
			    else
				let
				    val () = 
					debugPrint ("*** " ^ y ^ ": "
						    ^ typeToString tyY ^ "\n")
(*val _ = canonical tyY*)
				    val n = getTySuffix tyY
				    val Y = conName n y
				    val () = addCon n y NONE [] tyY
				    val () = update xx (CONty ([], typeName n))
				in
				    CONexp ((Y,yloc), [], NONE, loc)
				end
		in
		    LETexp (x, e1', defun e2, u)
		    before debugPrintln "defun LETexp 1 done\n"
		end

	      | _ => LETexp (x, defun e1,  defun e2, u)
		     before debugPrintln "defun LETexp 2 done\n"
	end

      | IFexp(tst,e1,e2,u) => IFexp (tst, defun e1, defun e2, u)

      | MATCHexp(v,rules,u) =>
	let
	    fun getR (MATCHrule((cname,cloc),vs,a,e,u)) =
		MATCHrule ((cname,cloc),vs,a,defun e,u)
	      | getR r = r
	in
	    MATCHexp (v, map getR rules, u)
	end

      | TYPEDexp (e,ty,u) => TYPEDexp (defun e, ty, u)  (* Type may change? *)
      | COERCEexp (e,ty,u) => COERCEexp (defun e, ty, u)

      | UNARYexp _      => exp
      | BINexp _        => exp
      | INVOKEexp _     => exp
      | NEWexp _        => exp
      | GETexp _        => exp
      | SGETexp _       => exp
      | UPDATEexp _     => exp
      | SUPERMAKERexp _ => exp
      | CONexp _        => exp
      | ASSERTexp (e, as1, as2, u) => raise Fail "assertion in defunc"


    and doApp exp =
    case exp of
	APPexp((fname,floc), args, ext, u) =>
	let
	    val () = debugPrint ("\n[appexp] "
				 ^ fname ^ " "
				 ^ listToString valToString " " args
				 ^ " <" ^ extToString ext
				 ^ ">\n")
		     
	    val ty = getVtype fname
	    val nargs = length args
	    val nArr = nArrows ty
			       
	    val ty = 
		if true 
		then ty 
		else case ext of 
			 LOCAL => 
			 if nargs <= nArr 
			 then
			     (debugPrintln ("doApp: " ^ fname
					    ^ " underapplied: calling fixFunt2");
			      fixFunt2 fname ty nargs )
			 else ty
		       | _ => ty

           (* SOMETIMES we want to do fixFunt2, sometimes we don't.
              How do we tell which is which????
              We also have to be careful about what we do in fixArgs.
              The two things interact messily.
*)

		       
	    val () = (
		debugPrintln ("type = " ^ tyTos ty);
		debugPrintln ("nargs = " ^ Int.toString nargs);
		debugPrintln ("nArr = " ^ Int.toString nArr);
                debugPrintln ""
	    )

	in
	    case ext of
	    LOCAL =>
	    if nargs = nArr then
		doArgs args (rTy ty)
		(fn l => APPexp ((appName (getTySuffix ty),floc),
				 VARval (fname, LOCAL, floc)::l, GLOBAL, u)) u
			                                      (* LOCAL ?? *)
			before debugPrintln "doApp done <L0>"
	    else if nargs < nArr
	    then
		let

		    val () = debugPrint ("-> " ^ fname ^ ": " 
					 ^ typeToString ty ^ "\n")
		    val (bound, rest) = splitArrow ty nargs
		    val () = debugPrint ("bound: ["
					 ^ listToString tyTos ", " bound
					 ^ "]\n")
		    val () = debugPrint ("rest: " ^ tyTos rest ^ "\n")
		    val n = getTySuffix rest
		    val () = debugPrint ("bound: ["
					 ^ listToString typeToString ", " bound
					 ^ "]\n")
		    val () = debugPrint ("rest: " ^ tyTos rest ^ "\n")
		    val m = getTySuffix ty
		    val () = debugPrintln ("Looking for Fsuffix: " ^ tyTos ty )
		    val ty' = CONty ([], typeName m)
		    val () = addCon n (typeName m) (SOME m) (ty'::bound) ty
		    val T' = fixTy rest (* CONty ([], typeName n)*)
		in
		    doArgs args T' 
			   (fn l => 
			       CONexp ((conName n (typeName m), floc),
				       VARval (fname, LOCAL, floc)::l, NONE, u)) u
			   (* LOCAL ?? *)
		    before debugPrintln ("doApp done <L1>")
		end
		
	    else
		let
		    val () = debugPrint ("[Defunc.sml]: " ^ fname 
					 ^ " over-applied")
		    val args1 = List.take (args, nArr)
		    val args2 = List.drop (args, nArr)
		    val nvar = tempName "f"
		    val () = update nvar ty
		    val E = defun (LETexp (
				   (nvar,floc), 
				   APPexp ((fname, floc), args1, ext, u),
				   APPexp ((nvar, floc), args2, LOCAL, u),u))
			    
		in
		    (E, ty) before debugPrintln ("doApp done <L2>")
		end
		
	  | _ =>
	    if nArr = nargs then
		doArgs args (rTy ty) (fn l => APPexp ((fname,floc), l, ext, u)) u
		before debugPrintln "doApp done <G0>"
	    else if nargs < nArr
	    then
		let
		    val () = debugPrint ("-> " ^ fname ^ ": " 
					 ^ typeToString ty ^ "\n")
		    val (bound, rest) = splitArrow ty nargs
		    val () = debugPrint ("bound: ["
					 ^ listToString typeToString ", " bound
					 ^ "]\n")
		    val () = debugPrint ("rest: " ^ typeToString rest ^ "\n")
		    val n = getTySuffix rest
			    
		    val () = addCon n fname NONE bound ty
		    val T' = fixTy rest (* CONty ([], typeName n)*)
			     
		in
		    doArgs args T' 
			   (fn l => 
			       CONexp ((conName n fname, floc), l, NONE, u)) u
		    before debugPrintln ("doApp done <G1>")
		end
		
	    else
		let
		    val () = debugPrint ("[Defunc.sml]: " ^ fname 
					 ^ " over-applied")
		    val args1 = List.take (args, nArr)
		    val args2 = List.drop (args, nArr)
		    val nvar = tempName "f"
		    val () = update nvar ty
		    val E = defun 
				(LETexp ((nvar,floc), 
				    APPexp ((fname, floc), args1, ext, u),
				    APPexp ((nvar, floc), args2, LOCAL, u),u))
			    
		in
		    (E, ty) before debugPrintln ("doApp done <G2>")
		end
		
	end
	
      | _ => Util.error (getU exp) "bad argument for doApp"
	     
	     
    val e' = Normalise.float_lets (defun e)

    fun fixArg a =
	case a of
	    UNITvar => UNITvar
	  | VAR (x,_) =>
	    let fun makeCanon t = t
		val ty = getVtype x
		val () = debugPrintln "fixArg: calling makeCanon"
		val ty' = getType (makeCanon ty)  (* was just ty  *)
	    in
		VAR (x, SOME ty')
	    end
in
    (e', map fixArg args) before debugPrintln "defunExp done\n"
end

end (* local *)

fun doFun env globalInf (fdef as FUNdef ((fname,l), args, i, e, u)) =
    let
	val () = debugPrintln ("\n\nDefunctionalising " ^ fname ^ "\n")
	val varInf = getVarInf env globalInf fdef
	val (e', args') = defunExp varInf e args
    in
	FUNdef ((fname,l), args', i, e', u)
    end

fun doBlock env globalInf (FUNblock l) = FUNblock (map (doFun env globalInf) l)

fun funName (FUNdef ((fname,_),_,_,_,_)) = fname


fun fixVdec v =
    case v of
	VALdec (v,ty,inst) => VALdec (v, fixTy ty, inst)
      | _ => v


fun defunctionalise (PROG (tdecs, vdecs, cdefs, fblocks)) progenv =
    let
	fun prvd (VALdec ((n,_), ty,_ )) = vprint (n ^ ": " ^ tyTos ty ^ "\n")
	  | prvd (CLASSdec((n,_),_,_,_)) = vprint ("[CLASS " ^ n ^ "]\n")
	val () = app prvd vdecs


	val () = app (fn VALdec ((n,_), ty,_)
			 => debugPrint (n ^": "^ typeToString ty ^ "\n")
		       | _ => ()) vdecs

	val () = setArgCounts fblocks

	val mainenv = Env.getMainEnv progenv

	val globalInf = getGlobalTypes2 mainenv vdecs

	val fblocks'  = map (doBlock mainenv globalInf) fblocks

	val () = prTypeSuffixes ()
	val () = prTypes()
	val apps = makeApps ()
	val appnames = map (fn (FUNblock (FUNdef ((fname,_),_,_,_,_)::_)) => fname 
			     | _ => "") 
			   apps
	val funs' = NAsyntfn.collapse (apps @ fblocks') 
				      (* don't do makeApps until after doBlock &c *)

	val () = debugPrint "---------------- funs' ----------------\n"
	val () = app (fn x => debugPrint (funName x ^"\n")) funs'
	val () = debugPrint "---------------------------------------\n"

	fun make1fb f = FUNblock [f]

	val p = PROG (tdecs@(makeTypes()), [], cdefs, map make1fb funs')

	val () = if !testing
		 then  NAsyntfn.printProgram (fn _ => ()) p
		 else ()

	val Fblocks = MonoUtil.makeFunBlocks 
			  funs' 
			  (NAsyntfn.collapse fblocks) 
			  appnames

        (* We should keep the ones whose type hasn't changed *)

	(* The old valdecs are no use since the types may have changed.  
	   We can't rely on typechecking to get the correct types since 
	   it might give you something too general (eg,  id: int -> int 
           may become polymorphic).  However,  the fundefs now have typed
	   arguments which will hopefully allow the typechecker to infer
	   correct types for everything. *)
    in
 	PROG (tdecs@(makeTypes()), [], cdefs, Fblocks)
    end

end

