(* Typecheck programs for ! annotations denoting heap-free constructors *)

(*

Safety conditions for type t = A of a1 * ... * aN | ... :

* At most one constructor is heap-free
  (HF constructor represented by null)
 OR
* All constructors heap-free          *** To be implemented ***
  (datatype represented by int)

* Any heap-free constructor is nullary
* No heap-free constructor C is constructed as C@d or matched with C@d

There is really no reason one couldn't supply an unnecessary diamond,
with that diamond placed on the free list instead, but we don't do
that yet. Similarly diamonds marked for disposal rather than
explicitly used.

Datatypes decorated with unsafe ! annotations produce warnings,
and those annotations are removed.

Undecorated datatypes are arbitrarily decorated in a safe
manner. These decorations may be sub-optimal, and in any case it may
be more clear if the programmer supplies an explicit decoration.

*)


local
    structure S=Binaryset
    open Normsyn
    open Util
in

fun mywarn loc s =
    if getLoc loc = Loc.nilLocation then () (* Just silently change builtins *)
    else warn loc s

(* LOOK AT THIS *)

fun inspectExp (e, (matches,apps: string S.set) ) =
    let
        fun inspectRule (MATCHrule((c,_),_,diamond,e,_), (m,a)) =
            let
                val (m',a') = inspectExp(e,(m,a))
            in
                case diamond of NOWHERE => (m',a')
                              | _ => (S.add(m',c),a')
            end
	  | inspectRule (OOMATCHrule(pat,e,_), (m,a)) = inspectExp(e,(m,a))
    in
        case e of
            IFexp(_,eT,eF,_) => inspectExps [eT,eF] matches apps
          | LETexp(x,e,e',_) => inspectExps [e,e']    matches apps
          | TYPEDexp(e,_,_) => inspectExp(e, (matches, apps))
          | COERCEexp(e,_,_) => inspectExp(e, (matches, apps))
          | ASSERTexp(e,_,_,_) => inspectExp(e, (matches, apps))
          | MATCHexp(_,rules,_) =>
                foldl inspectRule (matches,apps) rules
          | _ => (matches, apps)
    end
and inspectExps exps matches apps =
    List.foldl inspectExp (matches,apps) exps
fun inspectFundef( (FUNdef(_,_,_,e,_)), (m,a) ) = inspectExp(e, (m,a))
fun inspectFunBlock (FUNblock b, (m,a)) =
    List.foldl inspectFundef (m,a) b
fun inspectFunBlocks matches apps fblocks =
    List.foldl inspectFunBlock (matches, apps) fblocks

fun heapFree (TYPEcon(c,tys,NOHEAP,_)) = true
  | heapFree _ = false

fun makeDatatype matches apps do_opt (TYPEdec(tvars, tname, clist, u)) =
(*     TYPEcon of Var * (Ty list) * HeapUsage *)
    let
        fun revert clist =
            map (fn TYPEcon(c,tys,_,l) => TYPEcon(c,tys,HEAP,l)) clist
        fun nullary (TYPEcon(c,([],_),_,_)) = true
          | nullary _ = false
        fun usedHF (TYPEcon((c,_),_,_,_)) =
            not (S.member(matches,c) orelse S.member(apps,c))

        val hf = (List.filter heapFree clist)
        val err = if length hf > 1 andalso length hf < length clist then
                      SOME "more than 1 constructor (but not all) heap-free"
                  (* next actually prohibited syntactically *)
                  else if not (List.all nullary hf) then
                      SOME "non-nullary constructers may not be heap free"
		  (*
                  else if not (List.all usedHF hf) then
                      SOME "heap-free constructor used as diamond" (* May repeat warning from Syncheck *)
		  *)
                  else NONE

        val clist' = if do_opt then
                         case err of NONE => clist
                                   | SOME err =>
                                     (mywarn u ("invalid '!' annotation: " ^ err);
                                      revert clist)
                     else (revert clist)
    in
        TYPEdec(tvars,tname,clist',u)
    end

and makeDatatypes datatypes matches apps do_opt =
    map (makeDatatype matches apps do_opt) datatypes

fun optimise (PROG(datatypes, valdecs,classdefs, fblocks)) do_opt =
    let
	val matches = S.empty String.compare
	val apps = S.empty String.compare
	val (matches', apps') = if do_opt then inspectFunBlocks matches apps fblocks
				else (matches,apps)
	val datatypes' = makeDatatypes datatypes matches' apps' do_opt
    in
        PROG(datatypes',valdecs,classdefs,fblocks)
    end

fun getInfoDT (TYPEdec(tvars, (tname,_), clist, loc)) =
    let
        val hf = (List.filter heapFree clist)
        val hfNames = map (fn TYPEcon((c,_),_,_,_) => c) hf
    in
	(* Do we want 'type t = C' to give null or int?
	if length hf = 1 andalso length clist = 1
	then ([], [], [tname], hfNames)
	else  *)
        if length hf = 1 then
            (hfNames, [tname], [], [])
        else if length hf > 1 then
            ([], [], [tname], hfNames)
        else
            ([], [], [], [])
    end

and getInfo (PROG(datatypes, valdecs, classdefs, funs)) =
    let
        fun unzip4 [] = ([],[],[],[])
          | unzip4 ((h1,h2,h3,h4)::t) =
            let
	        val (l1, l2, l3, l4) = unzip4 t
            in
	        (h1::l1, h2::l2, h3::l3, h4::l4)
            end

        val infos = map getInfoDT datatypes
        val (nullConsL, nullTypesL, intTypesL, intConsL) = unzip4 infos
        val nullCons = List.concat nullConsL
        val nullTypes = List.concat nullTypesL
        val intTypes = List.concat intTypesL
        val intCons = List.concat intConsL
    in
        (nullCons,nullTypes,intTypes, intCons)
    end
end
