local
    open Normsyn Util
    structure A = Absyn
    structure G = GrailAbsyn
    val () = Absyn.required
    val () = GrailAbsyn.required

    (* Associate name of class with names of superclass/ interfaces *)
    val superTable = ref (Binarymap.mkDict String.compare)
    val interfaceTable = ref (Binarymap.mkDict String.compare)
    val ttable = ref (Splayset.empty String.compare)
    (* Table of names of "classes" which are really interfaces *)
in

fun cpError s = Util.ierror ("[Classpath.sml]: " ^ s)

datatype classholder = DIR of string | ZIP of ZipReader.zipfile
local
    val cp: string list option ref  = ref NONE

    val classHolders = ref []
    val loaded = ref false

    fun setClassPath' s = (* s is a colon-separated list of directory names *)
    let
	val files = String.tokens (fn c => c = #":") s
        (* Note that we haven't trimmed trailing slashes; we always
           manipulate paths and filenames using the functions in Path,
           and these make sure that slashes (or other separators)
           are in the correct places *)
    in
	cp := SOME files
    end

    fun initClasspath () =
	case (!cp) of
	    NONE => (* not set from command-line, so look elsewhere *)
	    let
		val pwd = getOpt (Process.getEnv "PWD", ".")
		val default =
		    case Process.getEnv "CAMELOT_CLASSPATH"
		     of NONE => (* try ${HOME}/classes *)
			let in
			    case Process.getEnv "HOME"
			     of NONE => pwd
			      | SOME home =>
				let val f = Path.joinDirFile
						{dir=home, file="classes"}
				in if Nonstdio.file_exists f then f ^ ":" ^ pwd else pwd end
			end
		      | SOME s => s
	    in
		setClassPath' default
	    end
	  | SOME p => ()
in

fun classpath () = valOf (!cp)
fun setClassPath s = setClassPath' s


(* The next function can be quite time consuming,  so we don't want to call
   it unless we're definitely using OO stuff;  this is why the stuff with
   'loaded' is there. *)

fun getClassHolders () =
    if (!loaded) then !classHolders
    else
    let
	val () = loaded := true
	val () = initClasspath ()
	fun doFile f =
	    case (#ext (Path.splitBaseExt f)) of
		SOME "jar" => ZIP (ZipReader.open_in f)
	      |	SOME "zip" => ZIP (ZipReader.open_in f)
	      | _ => DIR f
    in
	classHolders := map doFile (valOf (!cp));
	!classHolders
    end

fun closeJars () = app (fn ZIP f => ZipReader.close_in f | _ => ()) (!classHolders)
end  (* local *)


local
    val knownClasses = ref (Binaryset.empty String.compare)
in
fun isKnownClass c = Binaryset.member(!knownClasses,c)
and addKnownClass c = knownClasses := Binaryset.add(!knownClasses,c)
end


(* FIX: add checking. cycles not permitted. *)
(* USE IMPERATIVE HASHTABLES?? *)
fun getInfo (VALdec _, (superTable,interfaceTable)) = (superTable,interfaceTable)
  | getInfo (CLASSdec((cname, _), NONE, intfs, meths), (superTable,interfaceTable)) =
    (addKnownClass cname;
     (superTable, Binarymap.insert(interfaceTable, cname, map nameOf intfs)))
  | getInfo (CLASSdec((cname,_), SOME (super,_), intfs, meths), (superTable,interfaceTable)) =
    (addKnownClass cname;
     (Binarymap.insert(superTable, cname, super),
      Binarymap.insert(interfaceTable,cname, map nameOf intfs)))

fun updateInfo vals =
    let
        val (t,it) = foldl getInfo (!superTable,!interfaceTable) vals
    in (superTable := t; interfaceTable := it) end

fun constructInfo (PROG(types, vals, classdefs, fundefs)) =
    updateInfo vals

fun recordInterface c = (ttable := Splayset.add(!ttable,c))
fun isInterface c = Splayset.member(!ttable,c)

exception badGrailType

fun gtyToTy ty = case ty of
    G.INTty => A.INTty
  | G.BOOLEANty => A.BOOLty
  | G.FLOATty => A.FLOATty
  | G.REFty "java.lang.String" => STRINGty
  | G.REFty name => OBJECTty name
  | G.ARRAYty ty => A.ARRAYty (gtyToTy ty)
  | _ => raise badGrailType

fun cnameToFilename cname = String.translate (fn #"." => "/" | x => str x) cname
fun filenameToClass fname = String.translate (fn #"/" => "." | x => str x) fname

fun hasFlag flag flags = List.exists (fn x => x = flag) flags
fun f_public flags = hasFlag Classdecl.F_ACCpublic flags
fun f_static flags = if hasFlag Classdecl.F_ACCstatic flags then STATIC else INSTANCE
fun m_public flags = hasFlag Classdecl.M_ACCpublic flags
fun m_static flags = if hasFlag Classdecl.M_ACCstatic flags then STATIC else INSTANCE

open Jvmtype

fun qualName class =
    let
        fun f (a::b::t) = f ((a ^"."^b)::t)
          | f [x] = x
          | f [] = ""
        val toks = (Jvmtype.packages class)@[Jvmtype.className class]
    in f toks end

fun jtyToTy Tint = G.INTty  (* Used to be in Compile; maybe somewhere better ? *)
  | jtyToTy Tboolean = G.BOOLEANty
  | jtyToTy Tfloat = G.FLOATty
  | jtyToTy (Tclass c) = G.REFty (qualName c)
  | jtyToTy (Tarray ty) = G.ARRAYty (jtyToTy ty)
  | jtyToTy Tchar = G.CHARty
  | jtyToTy Tbyte = G.BYTEty
  | jtyToTy Tshort = G.SHORTty
  | jtyToTy Tlong = G.LONGty
  | jtyToTy Tdouble = G.DOUBLEty

fun jtyToRTy (SOME t) = SOME (jtyToTy t)
  | jtyToRTy NONE = NONE


fun mdeclToValdec {flags, name, msig = (tys, rty), attrs} =
    (let
         val (tysG, rtyG) = (map jtyToTy tys, jtyToRTy rty)
         val tysC = map gtyToTy tysG
	 val rtyC = case rtyG of SOME tyG => gtyToTy tyG
			       | NONE => UNITty
	 val tysC' = case tysC of [] => [UNITty,rtyC]
				| xs => xs@[rtyC]
     in
	 if m_public flags then
             SOME (VALdec((name,nowhere), tyListToArrowTy tysC', m_static flags))
	 else NONE
     end) handle badGrailType => NONE  (* What's going on here? *)

fun fdeclToValdec {flags, name, ty, attrs} =
    (let
         val tyC = (gtyToTy o jtyToTy) ty
     in
	 if f_public flags then
             SOME (VALdec((name,nowhere), tyC, f_static flags))
	 else NONE
     end) handle badGrailType => NONE


fun getTypesFile classfile =
    (let
         val {flags, this, super, ifcs,
	      fdecls, mdecls, attrs, ...} = Decompile.toClassDecl classfile

         (* Could maybe modify toClassDecl so that it (optionally) doesn't
            decompile the bytecode:  this might speed things up. *)

	 fun qualNameL c = (qualName c, nowhere)

         val this = qualNameL this
         val super = Option.map qualNameL super
         val ifcs = List.map qualNameL ifcs
         val valdecs = (List.mapPartial fdeclToValdec fdecls) @
		      (List.mapPartial mdeclToValdec mdecls)
         val () = if hasFlag Classdecl.C_ACCinterface flags
		  then recordInterface (nameOf this)
		  else ()
    in
        CLASSdec(this, super, ifcs, valdecs)
    end)
    handle (Io _) =>
	   error nowhere ("Io exception in ClassPath.getTypesFile")

fun findClassFile cname [] =
    error nowhere ("Can't find class " ^ cname ^
	       " in [" ^ (listToString (fn x=>x) "," (classpath())) ^ "]")
  | findClassFile cname (h::t) =
    case h of
	DIR d =>
	let
	    val fullName = Util.makeFullFilename d cname "class"
	in
	    if Nonstdio.file_exists (fullName)
	    then
		getTypesFile (Classfile.inputClassFile fullName)
	    else findClassFile cname t
	end
      | ZIP z =>
	let
	    val fullName = Util.makeFullFilename "" cname "class"
	in
	    case ZipReader.inputMember (z, fullName) of
		NONE => findClassFile cname t
	      | SOME v => getTypesFile (Classfile.vectorToClass v)
	end

fun getClassInfo cname =
    if isKnownClass cname then [] else
    let
        val () = vprint ("Loading " ^ cname ^ "\n")
        val class = findClassFile (cnameToFilename cname) (getClassHolders())
        val (this, super) = case class of CLASSdec(t,s,_,_) => (t,s) | _ => cpError "Not classdec"
        val () = debugPrint (nameOf this
			     ^ " <: "
			     ^ (case super of SOME (x,_) => x | NONE => "java.lang.Object")
			     ^ "\n")
        val () = updateInfo [class]
    in
        (class :: (case super of SOME (c,_) => getClassInfo c | NONE => []))
    end

fun isSubclass classname supername =
    (* WATCH OUT: conceivably class and superclass could be in different directories *)
    let
	val info = getClassInfo classname
        (* Could avoid reading info in first two cases, but no great saving *)
    in
	if supername = "java.lang.Object" then (info, true)
	else if classname = supername then (info, true)
	else
       	    case Binarymap.peek(!superTable,classname) of
                SOME cl =>
                if cl = supername then (info, true)
                else
		    let
			val (i,t) = isSubclass cl supername
		    in
			if t then (info@i, true) else
			let val (i', t) = isSubclassI info classname supername in (info@i@i', t) end
		    end
	      | NONE => isSubclassI info classname supername
    end

and isSubclassI info classname supername =
    let
	fun isSC (intf::intfs) info =
	    let
		val (info',t) = isSubclass intf supername
		val info'' = info'@info
	    in
		if t then (info'', true) else isSC intfs info''
	    end
	  | isSC [] info = (info, false)
    in
	case Binarymap.peek(!interfaceTable,classname) of
	    SOME (intfs) => isSC intfs info
	  | NONE => (info, false)
    end

fun getSuperclass classname
  = if classname = "java.lang.Object" then NONE
    else case Binarymap.peek(!superTable, classname) of
             SOME cl => SOME cl
           | NONE => SOME "java.lang.Object"

fun isObjectType (OBJECTty _) = true
  | isObjectType (STRINGty)   = true
  | isObjectType _ = false
fun uniformObjectType (OBJECTty c) = (OBJECTty c)
  | uniformObjectType (STRINGty) = (OBJECTty "java.lang.String")
  | uniformObjectType x = x
and typeIsSubclass t1 t2 =
    let
        val t1' = uniformObjectType t1
        val t2' = uniformObjectType t2
    in
        case (t1', t2') of (OBJECTty c1, OBJECTty c2) => isSubclass c1 c2
                         | _ => ([], false)
    end


(*
fun findClasses classdefs fundefs =
    let
        fun cmap = List.concat o map
        fun classDef (CLASSdef(c,_,_,_,vds,funs)) = cmap funDef funs
        fun funDef (FUNdef(f,vx,e,_)) = exp e
        fun exp (INVOKEexp())
    in
        (cmap classDef classdefs)@(cmap funDef fundefs)
    end
*)

end (* open *)
