(* Camelot.sml: main compilation thing *)

val version = "Camelot compiler version 4.2 [temporary] (27th April 2005)"

val deel_uapl_prg = "check_sharing"

val usage = [version,
"",
"Usage:",
" camelot [options] file",
"",
" OPTIONS",
"   All options are accepted with either one or two dashes in the long form.",
"",
" -h, -help",
"   Output this help message.",
" -a1, -absyn1",
"   Print normalised abstract syntax.",
" -a2, -absyn2",
"   Print monomorphised normalised abstract syntax.",
" -a2, -absyn2",
"   Print monomorphised normalised abstract syntax after DEEL sharing inference.",
" -ab, -asdl-bin",
"   Like -a2 but write the syntax to an ASDL binary pickle file (.asdl.bin).",
" -as, -asdl-sexp",
"   Like -ab but write an ASDL S-expression pickle file (.asdl.sexp).",
" -b, -debug",
"   Debug mode. Gives some probably useless information.  Pages of output",
"   per function, mainly type inference and monomorphisation related.",
" -c, -classnames",
"   Rather than saying \"compiled ...\",  just print out the names of",
"   the classes which have been generated (this was for use in the online",
"   demo)",
" -cp <dirlist>, -classpath <dirlist>",
"   This option is for use with object-oriented Camelot extensions.",
"   <dirlist> consists of a colon-separated list of directory names",
"   in which the compiler will look for JVM classes (including the",
"   standard Java libraries).   If the compiler needs to look for classfiles",
"   and no -cp option is given then directories listed in the environment",
"   variable CAMELOT_CLASSPATH will be used;  if this variable is not set",
"   then the compiler will use ${HOME}/classes:. .  If you're using the shell",
"   variable,  don't put a ~ in any of the paths:  use $HOME instead.",
"   See also the -j and -m options.",
" -C, --certgen",
"   Enable certificate generation",
" -d <dir>",
"   Write the output classes into directory <dir>.",
" -D, -diamond-layout",
"   Print information about structure of diamond class",
" -f, -freelist",
"   Print debugging information about behaviour of freelist allocation",
"   during execution.",
" -g, -grail",
"   Output Grail to .gr file (no class file)",
" -grail-cases",
"   Compile Camelot matches using the experimental Grail case construct",
" -hints, -proof-hints",
"   Output a file ...-HINTS.asdl.bin containing proof hints",
" -j, -java-rt",
"   Same as -cp /group/project/mrg/java_rt_sun-1.4.1:.",
"   (For compiling with the standard Java class hierarchy on DICE machines in Edinburgh)",
" -J, -jar, --jar",
"   Create a jar file out of .class file and produced certificate; needs -C to make sense",
" -k",
"   Return with exit status 0 even if compilation fails",
" -l, -linear",
"   Turn on linearity checking",
" -lfd",
"   Create .lfd file and call lfd_infer",
" -lfd-opts",
"   Pass following string as lfd_infer option (arguments may need to be quoted)",
" -ll, -layered",
"   Turn on layered sharing checking and inference (needs " ^ deel_uapl_prg ^ ")",
" -lu, -usage-aspects, -readonly",
"   Turn on usage aspects checking and inference (needs " ^ deel_uapl_prg ^ ") (NOT YET)",
" -lb, -debug-usage",
"   Debug the layered sharing and usage aspect inferences",
" -m, -midp",
"  Same as -cp /group/project/mrg/midp-1.0.3_classes:.",
"   (Useful for compiling with the MIDP 1.0 API on DICE machines in Edinburgh)",
" -N, -nodiamonds",
"   Use the original datatype scheme:  one Java class for each datatype",
"   (may not work any more).",
" -n, -no-tail-call-elimination",
"   Turn off tail-call elimination.",
" -nc, -no-consolidation",
"   Turn off Grail local variable consolidation.",
" -nw, -no-warnings",
"   Suppress warning messages",
" -od, -optimise-destruction, -auto-destruction",
"   Turn on destruction inference (works only with -ld or -lu) (NOT YET).",
" -r, -record-freelist, --record-freelist",
"   Record information about the freelist.  Programs can recover this",
"   information by calling diamond_info: unit -> string",
" -readable",
"   Omit argument lists from Grail output:  the program probably",
"   won't execute,  but it's easier to read.",
" -s, -silent",
"   Silent mode.  Suppress all textual output when compiling correct programs.",
" -show-all",
"   Show adpator functions when printing abstract syntax.",
" -t, -time",
"   Print the current date and time",
" -thy,",
"   Emit Isabelle theory files",
" -v, -verbose",
"   Set verbose mode (useful with -lfd)",
" -V, -version",
"   Print current compiler version and exit",
" -x, -exit",
"   Exit from compiler immediately"]

datatype flag =
  ASDL_BIN
| ASDL_SEXP
| AUTODESTR (* TODO *)
| A_NORMALISE
| CERTGEN
| CLASSNAMES_ONLY
| DEBUG0
| DEBUG
| DEBUGFREELIST
| DEBUGUSAGE
| DEEL
| GRAILOUT
| JAR
| LFD
| LINEAR
| NODIAMONDS
| NOCONSOLIDATION
| NOTAILCALLOPT
| PRINTABSYN0
| PRINTABSYN1
| PRINTABSYN2
| PRINTABSYN3
| PRINTABSYN4
| PRINTENV
| PROOFHINTS
| RECORDFREELIST
| SHOW_ALL
| SILENT
| THEORY
| USAGEASPECTS (* TODO *)
| VERBOSE

local
    val flags = ref []: flag list ref
in
    fun flagSet f = List.exists (fn x=>x=f) (!flags)
    fun setFlag f = flags := (f::(!flags))
end

val outputDir = ref ""
(* Nasty,  but I'm in a hurry *)

val lfd_opts = ref ""
fun set_lfd_opts args =
    case args of
	[] => Util.exit "Missing argument for -lfd-opts"
      | h::t => (lfd_opts := h; t)

val lfd_dir = ref ""
val lfd_prg = ref "lfd_infer"

val dPrint0 = ref false
val dPrint = ref false

val printInfo = ref false

fun prMono (f,ass) = print (" <mono {" ^ f ^ "}: [" ^ Mono.bindingsToStr ass ^ "]>")
fun prPhi p = print ""
fun prInfo i =
    if !printInfo then
	case i of
	    Absyn.LOC (Loc.Loc (a,b)) => print ("<<" ^ Int.toString a ^ ", " ^ Int.toString b ^ ">>")
	  | Absyn.MONO (m,i') => (prMono m; prInfo i')
	  | Absyn.PHI (p,i') => (prPhi p; prInfo i')
    else ()

val noEnv = Env.ProgEnv []

fun printProg prg env = (
    NAsyntfn.printProgram prInfo prg;
    if flagSet PRINTENV then Env.prEnv env else ()
)

fun noInfo _ = ()
fun outputProgram os = NAsyntfn.outputProgram os noInfo

fun debugPrint x = if (!dPrint) then TextIO.print x else ()
fun debugPrint0 x = if (!dPrint0) then Util.fprint x
		    else debugPrint ("\n" ^ x)
fun debugPrintDone () = debugPrint0 "* OK\n"

fun debugPrintAbsyn x = if (!dPrint) then printProg x NONE else ()

fun println s = TextIO.print (s^"\n")

fun addExt fname ext = Path.joinBaseExt {base=fname, ext=SOME ext}


fun compile fname =
    let
        (* Don't want messages mixed with syntax going to stdout *)
	fun printToStdErr s =
	    if not (flagSet SILENT) then
		TextIO.output(TextIO.stdErr, s)
	else ()

	(* HWL hacked here; for picking pragmas out of Camelot file *)
        fun preprocess fname =
             let
               val wrapper_file = fname ^ ".Wrap"
               val status = Process.system ("cat " ^ fname ^ " | sed '/^(\\*#/,/^#\\*)/!d' | sed 's/^M$//'  > " ^ wrapper_file);
               val _ = if (FileSys.fileSize wrapper_file = 0)
                         then FileSys.remove wrapper_file
                         else ()
             in
               () (* YUCK *)
             end

 	fun print s =
	    if not (flagSet SILENT) then
		TextIO.print s
	else ()

        (* HWL: bundle up all our stuff into a jar file *)
        fun postprocess cname =
             let
               val jar_file = cname ^ ".jar"
               val cert_file = cname ^ "Certificate.thy"
               val manifest_file = "MANIFEST"

               val s = "# create MANIFEST\n" ^
                       "#echo Main-Class: MRGOnly\n" ^
                       "MRG-ClassName: " ^cname^"\n" ^
                       "MRG-CertificateName: "^cname^"Certificate\n"

	       val os = TextIO.openOut (Util.makeFullFilename (!outputDir) manifest_file "")
               val _  = TextIO.output(os,s)
	       val os = TextIO.closeOut os
	       val os = TextIO.openOut (Util.makeFullFilename (!outputDir) manifest_file "")
               (* --- create a jar file *)
	       val _ = debugPrint ("* Building jar file: |jar -cmf "^manifest_file^" "^jar_file^" "^cert_file^" *.class|")
	       val status = Process.system("jar -cmf "^manifest_file^" "^jar_file^" "^cert_file^" *.class")
               val _ = printToStdErr ("Wrote " ^ jar_file ^ "\n")
	       val _ = FileSys.remove manifest_file
	    in
             ()  (* YUCK *)
            end

	fun makeDiamonds  (Normsyn.PROG(types, vals, classes, funs)) info =
	    if flagSet NODIAMONDS then
		(Diamond.makeOriginalDiamonds types info, "00", 0)
	    else
		Diamond.makeDiamonds [types] info

	fun writeJVML (richTypes,data_layout,tag_offset) (cdef as GrailAbsyn.CDEF(_,cname,_,_,_,_,_)) =
	let
            val lyt = case (Int.fromString (data_layout)) of
                          NONE => 0
                        | SOME x => x
            (* this is now a tuple! *)
            (*
	    val thySyntax = if (flagSet THEORY)
                              then ((flagSet CERTGEN), lyt, tag_offset, 5 (*default logic*))
                              else ((flagSet CERTGEN), lyt, tag_offset, 0 (*NO thy gen*))
            val _ = TextIO.print ("Logic version is "^(Int.toString (!Predicate.thy_syntax))^"\n")
            val _ = TextIO.print ("Tactic flavour is "^(Int.toString (!Predicate.tactic_flavour)))
            *)
            val thySyntax = ((flagSet CERTGEN), lyt, tag_offset, (!Predicate.thy_syntax), (!Predicate.tactic_flavour))
	    val classname = Compile.compile fname cdef (!outputDir) thySyntax richTypes
	in

	    if flagSet CLASSNAMES_ONLY then
		printToStdErr (cname^"\n")
	    else
		(printToStdErr ("Wrote " ^ addExt cname "class" ^ "\n");
		if (not ((!Predicate.thy_syntax)=0)) then printToStdErr ("Wrote " ^ addExt cname "thy" ^"\n") else () )
	end

	fun writeGrail (cdef as GrailAbsyn.CDEF(_,cname,_,_,_,_,_)) =
	    let
		val os = TextIO.openOut (Util.makeFullFilename (!outputDir) cname "gr")
	    in
	 	(GrailAbsyn.setOut os;
		 if flagSet CLASSNAMES_ONLY then
		     printToStdErr (addExt cname "gr" ^ "\n")
		 else
		     printToStdErr ("Wrote " ^ addExt cname "gr" ^ "\n");
		 GrailAbsyn.prClassDef cdef;
		 TextIO.closeOut os)
	    end

	fun getLayout classes =
	    case classes of
		[] => NONE
	      | (GrailAbsyn.CDEF (_,_,_,_,_,_,layout))::_ => layout

	fun setLayout classes layout =
	    case classes of
		[] => Util.exit "Can't find any classes"
	      | h::t =>
		let
		    val GrailAbsyn.CDEF(flags, name, super, intfs, fields, methods, _) = h
		    val h' = GrailAbsyn.CDEF(flags, name, super, intfs, fields, methods, layout)
		in
		    h'::t
		end

        (* The diamond layout is properly attached to the corresponding diamond class
           (in case we ever have multiple diamonds).  For certificate generation it's
           convenient to have the info in the main class, so this code does this. We
           depend on the fact that Phi puts the main class at the front of the list. *)



	(* ----------------  Let's go ---------------- *)

	(* ---------------- Deal with flags ---------------- *)

	val () = if flagSet DEBUG0         then dPrint0 := true          else ()
	val () = if flagSet DEBUG          then dPrint := true           else ()
	val () = if flagSet DEBUG          then Util.setDebug true       else ()
	val () = if flagSet DEBUGFREELIST  then Diamond.setDebug true    else ()
	val () = if flagSet RECORDFREELIST then Diamond.setReport true   else ()
        val () = if flagSet VERBOSE        then Util.setVerbose true     else ()


        (* ---------------- Set up various filename components ---------------- *)

	val extn = "cmlt"
	val filename = Util.extend fname extn
        val basename = Path.base (Path.file filename)

	val () = Util.setBaseName basename

	val () = if Nonstdio.file_exists filename
		 then ()
		 else Util.exit ("can't open " ^ filename)

	(* ---------------- Preprocess the input file  ---------------- *)

        val _ = preprocess filename (* HWL *)

	(* ---------------- Parse the input file  ---------------- *)

	val ast = MainParser.parse filename
	    (* Side-effect: sets stuff in util for error reporting *)

	val ast1 = Asyntfn.resolveClassNames ast
	    (* Resolve class names - vital *)

	val () = NAsyntfn.setHide (not (flagSet SHOW_ALL))  (* FIX THIS *)
	(* Don't print out the automatically-generated functions (for Steffen's benefit) *)


        (* ---- Add list and option types, and "main" function if necessary ---- *)

        val ast2 = Lib.addStandard ast1
(*	val () = debugPrintAbsyn ast2*)


        (* ---------------- Check that match rules are well-formed etc. ---------------- *)

	val () = Syncheck.syncheck ast2  (* We have to do this before normalisation *)


        (* ---------------- Normalisation ---------------- *)

	val () = debugPrint0 "* Normalising...\n"
	val normedAst = Normalise.normProg ast2
	val () = debugPrintDone ()
	val () = debugPrintAbsyn normedAst

	val () = if flagSet PRINTABSYN0
		 then printProg normedAst NONE else ()


        (* ---------------- Resolve external OO stuff ---------------- *)

	val () = debugPrint0 "* Class resolution...\n"
        val () = ClassPath.constructInfo normedAst
	val () = debugPrintDone ()

        (* ---------------- First typecheck ---------------- *)

	val () = debugPrint0 "* First typecheck...\n"
	val (typedAst1, env1, ctx1) = Type.typeProg normedAst
	val () = debugPrintDone ()

	val () = if flagSet PRINTABSYN1
		 then printProg typedAst1 (SOME env1) else ()


	val vprint = Util.vprint
	fun newline() = vprint "\n"
	fun printValDec (Normsyn.VALdec ((v,_),l, Normsyn.STATIC)) =
	    (vprint "val "; vprint v; vprint ": "; vprint (NAsyntfn.typeToString l); newline())
	  | printValDec (Normsyn.VALdec ((v,_),l, Normsyn.INSTANCE)) =
	    (vprint "val "; vprint v; vprint ": "; vprint (NAsyntfn.typeToString l); newline())
	  | printValDec (Normsyn.CLASSdec (c, sup, intfs, decs)) = vprint "** CLASSdec\n"

	fun vd (Normsyn.PROG(_,vds,_,_)) = vds

	val () = app printValDec (vd typedAst1)
(*
	val () = print (Type.ctxToStr ctx1)
	val () = Type.prEnv env1
*)
        (* ---------------- Datatype optimisation ---------------- *)
	(* Does this actually do anything?  Maybe we want the programmer to do it explicitly *)

        val optAst = typedAst1 (*DataOpt.optimise typedAst1 false*)
	val () = debugPrintAbsyn (optAst)

	(* ---------------- Monomorphisation ---------------- *)

	val () = debugPrint0 "* Monomorphising...\n"
	val () = debugPrint "\nMonomorphising...\n\n\n"
	val monAst = Mono.monomorphise optAst env1
        (* Previously -a2 printed monAst here; now we do another typecheck to get the env *)

(*	val () = if flagSet PRINTABSYN2 then printProg monAst NONE else ()*)
	val () = debugPrintDone ()


	(* ---------------- Typecheck again ---------------- *)

	val () = debugPrint0 "* Second typecheck...\n"
	val (typedAst2, env2, ctx2) = Type.typeProg monAst
	val () = if flagSet PRINTABSYN2 orelse flagSet DEBUG
		 then printProg typedAst2 (SOME env2) else ()
	val () = debugPrintDone ()

        (* ---------------- Defunctionalisation ---------------- *)

	val () = debugPrint0 "* Defunctionalising...\n"
	val firstOrderAst = Defunc.defunctionalise typedAst2 env2
(*	val () = if flagSet PRINTABSYN3 then printProg firstOrderAst NONE else ()*)

	val () = debugPrintDone ()


	val () = debugPrint0 "* Third typecheck...\n"
	val (typedAst3, env3, ctx3) = Type.typeProg (DataOpt.optimise firstOrderAst true)
	val () = debugPrintDone ()

(* We really want this one,  but sometimes we don't see errors *)
	val () = if flagSet PRINTABSYN3
		 then printProg (typedAst3) (SOME env3) else ()

	(* ---------------- Linearity check ---------------- *)

        val () = if (flagSet LINEAR) then
		     (debugPrint0 "* Linearity check...\n";
                      Linearity.checkProgram typedAst3 env3;
		      debugPrintDone ()
		     )
                else ()

(* IS THIS USED ??? *)
(*
        (* -------- ASDL absyn output for DEEL inference -------- *)

	(* Do we really want to do this if we're not calling any external programs? *)

	val () = if (flagSet ASDL_BIN) then
		     let
			 val pkl_out = BinIO.openOut (basename^".asdl.bin")
			 val _ = Camelot_absyn_ASDLUtil.write_program asdlAst pkl_out
			 val _ = BinIO.closeOut pkl_out
		     in () end
		 else ()

	val () = if (flagSet ASDL_SEXP) then
		     let
			 val pkl_out = TextIO.openOut (basename^".asdl.sexp")
			 val _ = Camelot_absyn_ASDLUtil.sexp_wr_program (AsdlUtil.ASDL_Program monAst) pkl_out
			 val _ = TextIO.closeOut pkl_out
		     in () end
		 else ()
*)
        (* ----------------  DEEL inference and checking  ---------------- *)

        (* This is a pain in the arse.  Michal's separation analysis throws away many of
           the locations,  so we can't report errors accurately after doing the
           analysis.  To fix this we would have to alter asdl/src/Absyn.asdl (to include
           more locations in the asdl pickle),  AsdlUtil.sml (which converts Camelot absyn
	   to the asdl equivalent), and most of the files in ../../SepDestr/*.hs,  which
           use yet another version of the absyn and require stuff to convert it to and from
           the asdl form.  All of these would have to be modified to include the locations
           which they ignore.  For the present,  I'm just filling in the missing locations
           with 'nowhere'. *)


	val (typedAst4, env4, ctx4) =
	    if flagSet DEEL orelse flagSet USAGEASPECTS
	    then
	    let
		val () = debugPrint0 "* DEEL sharing analysis...\n\n"
		val tmp_asdl_1 = FileSys.tmpName ()
		val tmp_asdl_2 = FileSys.tmpName ()
		val pkl_out = BinIO.openOut tmp_asdl_1
		val asdlAst = AsdlUtil.ASDL_Program monAst
		val _ = Camelot_absyn_ASDLUtil.write_program asdlAst pkl_out
		val _ = BinIO.closeOut pkl_out
		val command =
		    deel_uapl_prg
		    ^ (if (flagSet DEEL) then " --layered" else "")
		    ^ (if (flagSet USAGEASPECTS) then " --usage-aspects" else "")
		    ^ (if (flagSet AUTODESTR) then " --optimize" else "")
		    ^ (if flagSet DEBUG orelse flagSet DEBUGUSAGE
		       then " --html=" ^ basename ^ ".html"
		       else "")
		    ^ " " ^ tmp_asdl_1
		    ^ " " ^ tmp_asdl_2
				(* val _ = print ("Command: " ^ command ^ "\n")*)
		val status = Process.system command
		val _ = FileSys.remove tmp_asdl_1
	    in
		if status <> Process.success then Util.quit()
		else
		    let
			val pkl_in = BinIO.openIn tmp_asdl_2
			val asdl_deelAst = Camelot_absyn_ASDLUtil.read_program pkl_in
			val () = if flagSet PROOFHINTS then
				     ProofHints.output_hints asdl_deelAst (basename ^ "-HINTS.asdl.bin")
				 else ()
			val () = BinIO.closeIn pkl_in
			val () = FileSys.remove tmp_asdl_2
			val () = debugPrintDone ()
			val deelAst = AsdlUtil.Camelot_Program asdl_deelAst
			val () = debugPrint0 "* Third typecheck...\n"
			val (typedDeelAst,denv,dctx) = Type.typeProg deelAst (* reannotate *)
			val () = debugPrintDone ()
		    in
			(typedDeelAst, denv, dctx)
		    end
	    end

	    else
		(typedAst3, env3, ctx3)

	val () = if flagSet PRINTABSYN4
		 then printProg typedAst4 (SOME env4) else ()


        (* ---------------- Construct the diamond class ---------------- *)

        val dataInfo = DataOpt.getInfo typedAst4
	val () = debugPrint0 "* Constructing diamond class...\n"
	val (diamonds,diamond_layout,tg) = makeDiamonds typedAst4 dataInfo 
							(* Lots of side-effects in Diamond.sml *)
	val () = debugPrintDone ()


	(* ---------------- Resource inference ---------------- *)

	val Normsyn.PROG(_,_,_,funblocks) = typedAst4
	val funs = NAsyntfn.collapse funblocks

	(* Perform the Hofmann-Jost analysis and import the rich types *)

	val richTypes =
	    if (flagSet LFD)
	    then
		let
		    val () = debugPrint0 ("* Resource inference...\n")

		    fun read_rvs is acc =
			if BinIO.endOfStream is then acc
			else read_rvs is ((Camelot_absyn_ASDLUtil.read_rich_valdec is)::acc)

		    val fname = Util.makeFullFilename (!outputDir) basename "lfd"
		    val pickle_file = Util.makeFullFilename (!outputDir) "valdecs" "pkl"
		    val os = TextIO.openOut fname
		    val () = outputProgram os typedAst4  (* monAst? typedAst3? *)
		    val () = TextIO.closeOut os

                    val maybe_lfd_dir = if ((!lfd_dir)="") then "" else (!lfd_dir) ^ "/"
                    (* val maybe_tmp_dir = if ((!outputDir)="") then "" else " -d " ^ (!outputDir) ^ "/"			      *)

                    (* not needed any more! -- HWL
                    val tmpname = Util.makeFullFilename (!outputDir) "qqq" "lfd"
                    val _ = Process.system ("cat " ^ fname ^" | sed 's/==/=/' > " ^ tmpname)
                    val _ = Process.system ("mv " ^ tmpname ^ " " ^ fname)
		    *)

                    val tmpname = Util.makeFullFilename (!outputDir) "qqq" "txt"
                    val tmp2name = Util.makeFullFilename (!outputDir) "qqw" "txt"
                    val exec_str = maybe_lfd_dir ^ (!lfd_prg) ^ " " ^ (!lfd_opts) ^
				   " -pkl " ^ fname ^
                                   "> "^tmpname

                    val _ = debugPrint ("Trying to call lfd as |" ^ exec_str ^"|\n")
		    val _ = Process.system exec_str

                    val _ = Process.system ("cat "^tmpname^" | sed '/Memory leak detected/!d' > "^tmp2name)
                    val leak_detected = FileSys.fileSize tmp2name > 0
                    val _ = FileSys.remove tmpname
                    val _ = FileSys.remove tmp2name
                    val _ = if flagSet CERTGEN andalso leak_detected
                              then (TextIO.print "Memory leak detected: validation can't handle such certificates!\n";
				    CertGenP.leak_detected := true)
                              else ()
			    
		    val is = BinIO.openIn pickle_file
		    val rvaldecs = read_rvs is []
		    val () = BinIO.closeIn is
		    val () = FileSys.remove pickle_file

		    val _ = debugPrintDone ()

		    val _ = debugPrint0 ("* Predicate generation...\n")
		    val pred = Predicate.makePredicates basename funs rvaldecs
                    (* fill in types to be used in CertGenP *)
		    val _ = CertGenP.reportedTypes := Predicate.reportTypes basename funs rvaldecs
		    val _ = debugPrintDone ()

		    val () = if flagSet VERBOSE
			     then
			     (
			      print "\n#################### Rich types from lfd_infer #################\n";
			      app AsdlUtil.printRichValDec rvaldecs;
			      print "################################################################\n"
			     )
			     else ()
		    (* val () = print pred *)

		in
                  pred (* results from LFD *)
		end
	    else (* !LFD *)
             Binarymap.mkDict (String.compare)


	(* ---------------- Translate to Grail---------------- *)

	val () = debugPrint0 "* Generating Grail...\n"
	val grailClasses0 = Phi.phi' typedAst4 env4 ctx4 dataInfo
	val grailClasses = setLayout grailClasses0 (getLayout diamonds)

	val () = debugPrintDone ()

        (* ---------------- Close files ---------------- *)

	(* The source file is still open because error-reporting
           functions need to be able to print bits of source code *)

	val () = Util.closeInStream ()
	val () = ClassPath.closeJars ()


        (* ----------------- Grail optimisations ----------------- *)

	(* ---------------- Tail-call elimination ---------------- *)


	val optGrailClasses = if flagSet NOTAILCALLOPT then
			       grailClasses
			   else
			       (debugPrint0 "* Tail-call elimination...\n";
				map Optimise.optimise grailClasses 
				before debugPrintDone ())

	(* ---------------- Local variable consolidation ---------------- *)

	val finalClasses =  if flagSet NOCONSOLIDATION then optGrailClasses else
			    (debugPrint0 "* Local variable consolidation...\n";
			     map Dataflow.consolidate optGrailClasses
			     before debugPrintDone ())


	(* ---------------- Create the output file ---------------- *)


        val richTypes' = (richTypes, diamond_layout, tg) (* pass in data layout info from makeDiamonds *)
        val writeClasses = if flagSet GRAILOUT then writeGrail else writeJVML richTypes'

	val _ = app writeClasses (finalClasses@diamonds)

	(* ---------------- Postprocess the input file  ---------------- *)

        val GrailAbsyn.CDEF(_,cname,_,_,_,_,_) = List.hd finalClasses
        val _ = if (flagSet JAR)
                  then  postprocess cname (* HWL *)
                  else  ()
    in
     () (* great, let's do function-less programming !*)
    end



(* -------------------- Parse arguments then call compiler ------------------- *)

val warn = ref true

fun parseArgs [] files =
    let in case files of [] => if (!warn) then Util.exit "No input files" else ()
		| [file] => compile file
		| _ => Util.exit "Too many input files"
    end
  | parseArgs (arg::args) files =
    if (String.substring (arg,0,1)) = "-" then
	if arg = "-h" orelse arg = "-help" orelse arg = "--help"
	   orelse arg = "-usage" orelse arg = "--usage" then
	    app println usage
	else if arg = "-argtypes" orelse arg = "--argtypes" then
	    (NAsyntfn.setPrintArgTypes true; parseArgs args files)  
                 (* Args have type annotations after defunctionalisation *)
	else if arg = "-a0" orelse arg = "-absyn0" orelse arg = "--absyn0" then
	    (setFlag PRINTABSYN0; setFlag SILENT; parseArgs args files)
	else if arg = "-a1" orelse arg = "-absyn1" orelse arg = "--absyn1" then
	    (setFlag PRINTABSYN1; setFlag SILENT; parseArgs args files)
	else if arg = "-a2" orelse arg = "-absyn2" orelse arg = "--absyn2" then
	    (setFlag PRINTABSYN2; setFlag SILENT; parseArgs args files)
	else if arg = "-a3" orelse arg = "-absyn3" orelse arg = "--absyn3" then
	    (setFlag PRINTABSYN3; setFlag SILENT; parseArgs args files)
	else if arg = "-a4" orelse arg = "-absyn4" orelse arg = "--absyn4" then
	    (setFlag PRINTABSYN4; setFlag SILENT; parseArgs args files)
	else if arg = "-ab" orelse arg = "-asdl-bin" orelse arg = "--asdl-bin" then
	    (setFlag ASDL_BIN; setFlag SILENT; parseArgs args files)
	else if arg = "-as" orelse arg = "-asdl-sexp" orelse arg = "--asdl-sexp" then
	    (setFlag ASDL_SEXP; setFlag SILENT; parseArgs args files)
	else if arg = "-A" then
	    (setFlag A_NORMALISE; parseArgs args files)
	else if arg = "-b0" orelse arg = "-debug0" orelse arg = "--debug0" then
	    (setFlag DEBUG0; parseArgs args files)
	else if arg = "-b" orelse arg = "-debug" orelse arg = "--debug" then
	    (setFlag DEBUG; parseArgs args files)
	else if arg = "-bc" then
	    (setFlag DEBUG; Type.setDebugConstraints true; parseArgs args files)
	else if arg = "-c" orelse arg = "-classnames" orelse arg = "--classnames" then
	    (setFlag CLASSNAMES_ONLY; parseArgs args files)
	else if arg = "-cp" orelse arg = "-classpath" orelse arg = "--classpath" then
	    case args of
		[] => Util.exit "No classpath specified"
	      | h::t =>
		let
		    val () = ClassPath.setClassPath h
		in
		    parseArgs t files
		end
	else if arg = "-C" orelse arg = "--certgen" then
	    (setFlag CERTGEN; (* setFlag LFD; *) parseArgs args files)
	else if arg = "-d" then
	    case args of
		[] => Util.exit "No output directory specified for -d"
	      | h::t =>
		let
		    val () = outputDir := h
		    val () = lfd_opts := !lfd_opts ^ " -d " ^ h
		in
		    parseArgs t files
		end
	else if arg = "-D" orelse arg = "-diamond-layout" orelse arg = "--diamond-layout" then
	    (Diamond.setPrint true; parseArgs args files)
	else if arg = "-e" orelse arg = "-env" orelse arg = "--env" then
	    (setFlag PRINTENV; parseArgs args files)
	else if arg = "-ext" then
	    (NAsyntfn.showExtern:= true; parseArgs args files)
	else if arg = "-f" orelse arg = "-freelist" orelse arg = "--freelist" then
	    (setFlag DEBUGFREELIST; parseArgs args files)
	else if arg = "-g" orelse arg = "-grail" orelse arg = "--grail" then
	    (setFlag GRAILOUT; parseArgs args files)
	else if arg = "-grail-cases" orelse arg = "--grail-cases" then
	    (Phi.experimental := true; parseArgs args files)
	else if arg = "-i" orelse arg = "-info" orelse arg = "--info" then
	    (printInfo := true; parseArgs args files)  (* So we can see monoinfo for debugging *)
	else if arg = "-j" orelse arg = "-java-rt" orelse arg = "--java-rt" then
	    (ClassPath.setClassPath "/group/project/mrg/java_rt_sun-1.4.1:.";
	     parseArgs args files)
	else if arg = "-J" orelse arg = "-jar" orelse arg = "--jar" then
	    (setFlag JAR; parseArgs args files) (* HWL: create a jar file *)
	else if arg = "-k" then 
	    (Util.status := OS.Process.success; parseArgs args files)
	else if arg = "-l" orelse arg = "-linear" orelse arg = "--linear" then
	    (setFlag LINEAR; parseArgs args files)
	else if arg = "-ll" orelse arg = "-layered" orelse arg = "--layered" then
	    (setFlag DEEL; parseArgs args files)
	else if arg = "-lu"
		orelse arg = "-usage-aspects"
		orelse arg = "--usage-aspects"
		orelse arg = "-readonly"
		orelse arg = "--readonly" then
	    (setFlag USAGEASPECTS; parseArgs args files)
	else if arg = "-lb" orelse arg = "-debug-usage" orelse arg = "--debug-usage" then
	    (setFlag DEBUGUSAGE; parseArgs args files)
	else if arg = "-hints"
		orelse arg = "--hints"
		orelse arg = "-proof-hints"
		orelse arg = "--proof-hints" then
	    (setFlag PROOFHINTS; parseArgs args files)
	else if arg = "-lfd" then
	    (setFlag LFD; (* setFlag SILENT; *) parseArgs args files)
	else if arg = "-lfd-opts" orelse arg = "-lfdopts" orelse arg = "--lfd-opts" then
	    (let val args' = set_lfd_opts args in parseArgs args' files end)
	else if arg = "-lfd-dir" orelse arg = "-lfddir"  orelse arg = "--lfd-dir" then
	    case args of
		[] => Util.exit "No working directory specified for -lfddir"
	      | h::t =>
		let
		    val () = lfd_dir := h
		in
		    parseArgs t files
		end
	else if arg = "-lfd-prg" orelse arg = "-lfdprg" orelse arg = "--lfd-prg" then
	    case args of
		[] => Util.exit "No program name specified for doing space inference (-lfdprg <prgname>)"
	      | h::t =>
		let
		    val () = lfd_prg := h
		in
		    parseArgs t files
		end
	else if arg = "-m" orelse arg = "-midp" orelse arg = "--midp" then
	    (ClassPath.setClassPath "/group/project/mrg/midp-1.0.3_classes:.";
	     parseArgs args files)
	else if arg = "-n" orelse arg = "-no-tail-call-elimination"
		orelse arg = "--no-tail-call-elimination" then
	    (setFlag NOTAILCALLOPT; parseArgs args files)
	else if arg = "-nc" orelse arg = "-no-consolidation" orelse arg = "--no-consolidation" then
	    (setFlag NOCONSOLIDATION; Compile.lax := true; parseArgs args files)
	else if arg = "-nod" orelse arg = "--nod" then
	    (NAsyntfn.setPrintSpace false; parseArgs args files)
	else if arg = "-nw" orelse arg = "-no-warnings" orelse arg = "--no-warnings" then
	    (Util.setWarn false; parseArgs args files)
	else if arg = "-N" orelse arg = "-nodiamonds" orelse arg = "--nodiamonds" then
	    (setFlag NODIAMONDS; parseArgs args files)
	else if arg = "-o" orelse arg = "-optimise" orelse arg = "--optimise" then
	    (TextIO.print "*** The -o option is now deprecated ***\n";
	     TextIO.print "Tail-call optimisation is on by default;  use -n to turn it off\n";
	     parseArgs args files)
	else if arg = "-od"
		orelse arg = "-optimise-destruction"
		orelse arg = "--optimise-destruction"
		orelse arg = "-auto-destruction"
		orelse arg = "--auto-destruction" then
	    (setFlag AUTODESTR; parseArgs args files)
        else if arg = "-q" orelse arg = "-quack" orelse arg = "--quack" then
	    (TextIO.print ("quack\n"); warn := false; parseArgs args files)
        else if arg = "-Q" orelse arg = "-Quack" orelse arg = "--Quack" then
	    (TextIO.print ("Quack\n"); warn := false; parseArgs args files)
	else if arg = "-r" orelse arg = "-record-freelist" orelse arg = "--record-freelist" then
            (setFlag RECORDFREELIST; parseArgs args files)
	else if arg = "-readable" orelse arg = "--readable" then
	    (Phi.noarglists := true; parseArgs args files)
	else if arg = "-s" orelse arg = "-silent" orelse arg = "--silent" then
	    (setFlag SILENT; ToyGrailAbsyn.shut_up := true ; parseArgs args files)
        else if arg = "-show-all" orelse arg = "--show-all" then
	    (setFlag SHOW_ALL; parseArgs args files)
	else if arg = "-t" orelse arg = "-time" orelse arg = "--time" then
	    (TextIO.print (Date.toString(Date.fromTimeLocal(Time.now()))^"\n");
			  if null args then () else parseArgs args files)
	else if arg = "-test" then
	    (Defunc.testing := true; parseArgs args files)
	else if arg = "-test2" then
	    (Dataflow.setDebug true; parseArgs args files)
	else if arg = "-thy" orelse arg = "--thy" then
          if null args
            then (Predicate.thy_syntax := Predicate.DEFAULT_THY_SYNTAX ; 
                  Predicate.tactic_flavour := Predicate.DEFAULT_TACTIC_FLAVOUR ; 
                  parseArgs args files)
            else (* set the style for the theory file; must match settings in gf.sml *)
              let 
                val x = Int.fromString (hd args)
              in 
                case x of 
                   NONE => (Predicate.thy_syntax := Predicate.DEFAULT_THY_SYNTAX ; 
                            Predicate.tactic_flavour := Predicate.DEFAULT_TACTIC_FLAVOUR ; 
                            parseArgs args files)
                 | (SOME x') => (Predicate.thy_syntax := x'; 
                                 parseArgs (tl args) files)
              end
	else if arg = "-v" orelse arg = "-verbose" orelse arg = "--verbose" then
	    (setFlag VERBOSE; parseArgs args files)
	else if arg = "-V" orelse arg = "-version" orelse arg = "--version" then
	    (TextIO.print (version ^ "\n"))
        else if arg = "-x" orelse arg = "-exit" orelse arg = "--exit" then ()
	else
	    Util.exit ("Unknown option " ^ arg)

    else
	parseArgs args (arg::files)

val _ = parseArgs (CommandLine.arguments()) []

