(* Miscellaneous functions on abstract syntax *)

open Absyn Util

fun collapse fblocks = List.concat (map (fn FUNblock b => b) fblocks)
(* Collapse funblocks: temporary until we deal with them properly *)

fun realToString r = String.translate (fn #"~" => "-" | c => Char.toString c) (Real.toString r)
fun intToString n  = String.translate (fn #"~" => "-" | c => Char.toString c) (Int.toString n)

fun extToString ext =
    case ext of GLOBAL => "GLOBAL"
	      | LOCAL => "LOCAL"
	      | BUILTIN => "BUILTIN"
	      | EXTERN => "EXTERN"

fun valToString v =
    case v of
	VARval (w,_,_) => w
      | INTval (n, _) => intToString n
      | FLOATval (x, _) => realToString x
      | CHARval (n, _) => "'" ^ (Char.toString (chr n)) ^ "'"
      | STRINGval (s, _) => "\"" ^ s ^ "\""
      | BOOLval (true, _) => "true"
      | BOOLval (false, _) => "false"
      | UNITval _ => "()"
      | NULLval (s, _) => "(null :> " ^s ^ ")"

fun isEq  (v: tyVar) = !(#eq v)
fun isOrd (v: tyVar) = !(#ord v)
fun tVar  (v: tyVar) = #name v
fun mkTvar x = TVARty {name = x, eq = ref false, ord = ref false}

fun isEqType ty = 
    case ty of
	INTty       => true
      | CHARty      => true
      | BOOLty      => true
      | FLOATty     => true
      | STRINGty    => true
      | UNITty      => true
      | TVARty v    => isEq v
      | ARRAYty _   => false
      | PRODUCTty _ => false
      | ARROWty _   => false
      | CONty _     => false
      | DIAMONDty _ => false
      | OBJECTty _  => false

fun isOrdType ty = 
    case ty of
	INTty       => true
      | CHARty      => true
      | BOOLty      => false
      | FLOATty     => true
      | STRINGty    => true
      | UNITty      => false
      | TVARty v    => isOrd v
      | ARRAYty _   => false
      | PRODUCTty _ => false
      | ARROWty _   => false
      | CONty _     => false
      | DIAMONDty _ => false
      | OBJECTty _  => false


fun typeToString ty =
    case ty of
	INTty => "int"
      | CHARty => "char"
      | BOOLty => "bool"
      | FLOATty => "float"
      | STRINGty => "string"
      | UNITty => "unit"
      | TVARty v =>
	if isOrd v then "''" ^ #name v
	else if isEq v then "'" ^ #name v 
	else #name v
      | ARRAYty t => typeToString t ^ " array"
      | ARROWty (t1, t2) =>
	let in case t2 of
		   ARROWty(s1,s2) => "(" ^ typeToString t1
				     ^ " -> " ^ typeToString s1
				     ^ " -> " ^ typeToString s2 ^ ")"
		 | _ => "(" ^ (typeToString t1) ^ " -> " ^ typeToString t2 ^ ")"
	end
      | PRODUCTty l => listToString typeToString " * " l
      | CONty (tys, c) =>
	let in case tys of
		   [] => c
		 | [t] => typeToString t ^ " " ^ c
		 | _  =>  "(" ^ (listToString typeToString ", " tys) ^ ") " ^ c
	end
      | DIAMONDty t => "<>" (*"<"^t^">"*)
      | OBJECTty t => t

fun getU e =
    case e of
	VALexp (_,u) => u
      | UNARYexp (_,_,u) => u
      | BINexp (_,_,_,u) => u
      | IFexp (_, _, _, u) => u
      | MATCHexp (_,_,u) => u
      | LETexp (_,_,_,u) => u
      | APPexp (_,_,_,u ) => u
      | CONexp (_,_,_,u) => u
      | TYPEDexp (_,_,u) => u
      | COERCEexp (_,_,u) => u
      | NEWexp (_,_,u) => u
      | SUPERMAKERexp (_,u) => u
      | INVOKEexp (_,_,_,u) => u
      | UPDATEexp (_,_,u) => u
      | GETexp (_,_,u) => u
      | SGETexp (_,u) => u
      | ASSERTexp (_,_,_,u) => u
      | LAMexp (_, _, u) => u


(*
val _ = getU (VALexp(INTval (5,nowhere),nowhere))
fun getvU v =
    case v of
	VARval    ((_,u),_) => u
      | CHARval   (_,u) => u
      | INTval    (_,u) => u
      | FLOATval  (_,u) => u
      | STRINGval (_,u) => u
      | BOOLval   (_,u) => u
      | UNITval      u  => u
      | NULLval   (_,u) => u
      | LAMval  (_,_,u) => u
(* Mapping over 'a Absyn *)
*)

fun mapName f x = x
fun mapNameL f (x,u) = (x, f u)

and mapTy f t =
    case t of
         INTty => INTty
       | CHARty => CHARty
       | BOOLty => BOOLty
       | FLOATty => FLOATty
       | STRINGty => STRINGty
       | UNITty => UNITty
       | TVARty s => TVARty s
       | ARRAYty t => ARRAYty (mapTy f t)
       | PRODUCTty ts => PRODUCTty (map (mapTy f) ts)
       | ARROWty (t, t') => ARROWty (mapTy f t, mapTy f t')
       | CONty (ts,s) => CONty (map (mapTy f) ts, s)
       | DIAMONDty s => DIAMONDty s
       | OBJECTty s => OBJECTty s

and mapHeapUsage f h =
    case h of
        HEAP => HEAP
      | NOHEAP => NOHEAP

and mapTypeCon f (TYPEcon (cname, (tys,y), h, x)) =
    TYPEcon (mapName f cname, (map (mapTy f) tys, f y), mapHeapUsage f h, f x)

and mapTypeDec f (TYPEdec (tvars, name, cons, x))
  = TYPEdec (map (mapNameL f) tvars, mapName f name, map (mapTypeCon f) cons, f x)

and mapFunAssns f (FUNASSNS (name, ass1, ass2))
  = FUNASSNS (mapName f name, ass1, ass2)

and mapHints f (HINTS (decs, funassns))
  = HINTS (map (mapTypeDec f) decs, map (mapFunAssns f) funassns)

and mapValue f v =
    case v of
	VARval (w, ext, l) => VARval (w, ext, f l)
      | INTval (n, l)      => INTval (n, f l)
      | FLOATval (x, l)    => FLOATval (x, f l)
      | CHARval (n, l)     => CHARval (n, f l)
      | STRINGval (s, l)   => STRINGval (s, f l)
      | BOOLval (true, l)  => BOOLval (true, f l)
      | BOOLval (false, l) => BOOLval (false, f l)
      | UNITval l          => UNITval (f l)
      | NULLval (s, l)     => NULLval (s, f l)

and mapMatchDiam f m =
    case m of
        NOWHERE => NOWHERE
      | SOMEWHERE n => SOMEWHERE (mapName f n)
      | DISPOSE => DISPOSE

and mapOOMatchPat f p =
    case p of
        CLASSpat (n, n') => CLASSpat (mapName f n, mapName f n')
      | ANYCLASSpat => ANYCLASSpat

and mapMatchRule f r =
    case r of
        MATCHrule (con, args, d, e, x)
	=> MATCHrule (mapName f con, map (mapName f) args, mapMatchDiam f d,
		      mapExp f e, f x)
      | OOMATCHrule (p, e, x) => OOMATCHrule (mapOOMatchPat f p, mapExp f e, f x)

and mapExp f e =
    case e of
	VALexp (v,x) => VALexp(mapValue f v, f x)
      | UNARYexp (oper, e, x) => UNARYexp (oper, mapExp f e, f x)
      | BINexp (oper, e, e',x) => BINexp (oper, mapExp f e, mapExp f e', f x)
      | IFexp (e, e1, e2, x) => IFexp (mapExp f e, mapExp f e1, mapExp f e2, f x)
      | MATCHexp (e,l,x) => MATCHexp (mapExp f e, map (mapMatchRule f) l, f x)
      | LETexp (v, e, e',x) => LETexp (mapName f v, mapExp f e, mapExp f e', f x)
      | APPexp (e, l, ext, x) => APPexp (mapExp f e, map (mapExp f) l, ext, f x)
      | CONexp (v, es, addr, x) =>
	  CONexp (mapName f v, map (mapExp f) es, Option.map (mapName f) addr, f x)
      | TYPEDexp (e,t,x) => TYPEDexp (mapExp f e, mapTy f t, f x)
      | COERCEexp (e,t,x) => COERCEexp (mapExp f e, mapTy f t, f x)
      | NEWexp (class, es,x) => NEWexp (class, map (mapExp f) es, f x)
      | SUPERMAKERexp (es,x) => SUPERMAKERexp(map (mapExp f) es, f x)
      | INVOKEexp (obj,mname,es,x) => INVOKEexp (mapExp f obj, mname, map (mapExp f) es, f x)
      | UPDATEexp (a, e, x) => UPDATEexp (a, mapExp f e, f x)
      | GETexp (obj, var, x) => GETexp (mapExp f obj, var, f x)
      | SGETexp (var, x) => SGETexp (var, f x)
      | ASSERTexp (e, as1, as2, x) => ASSERTexp (mapExp f e, as1, as2, f x)
      | LAMexp (w, e, l)   => LAMexp (mapName f w, mapExp f e, f l)

and mapValDec f vd =
    case vd of
        VALdec (name, ty, inst) => VALdec (mapName f name, mapTy f ty, inst)
      | CLASSdec (cname, super, ns, vds) =>
	 CLASSdec (mapName f cname, Option.map (mapName f) super,
		  map (mapName f) ns, map (mapValDec f) vds)

and mapVarTy f v =
    case v of
	UNITvar => UNITvar
      | VAR(s,t) => VAR(s, Option.map (mapTy f) t)

and mapFunDef f (FUNdef (fname, args, inst, e, x)) =
    FUNdef (mapName f fname, map (mapVarTy f) args, inst, mapExp f e, f x)

and mapFunBlock f (FUNblock l) =
    FUNblock (map (mapFunDef f) l)

and mapClassDef f (CLASSdef (class, super, imps, vals, methods)) =
    CLASSdef (mapName f class,
	      Option.map (mapName f) super,
	      map (mapName f) imps,
	      map (mapValDec f) vals,
	      map (mapFunDef f) methods)

and mapProgram f (PROG(typedecs,valdecs,classdefs,funblocks)) =
    PROG (map (mapTypeDec f) typedecs,
	  map (mapValDec f) valdecs,
	  map (mapClassDef f) classdefs,
	  map (mapFunBlock f) funblocks)


(* ---------------- Class name resolution ---------------- *)
(* Internally-defined Class names are initially treated as
   datatype names: this stuff traverses the absyn and replaces
   appropriate CONty values with OBJECTty values once we know
   the names of all the classes. *)
(* For defunctionalisation, we also have to have APPexp's marked
   with LOCAL if the thing being applied is not a global function
   or a method.  I've stuck some stuff in here to do this,  but
   so far only for the main program. *)

val classnames = ref []: string list ref
fun addClass c = classnames := c::(!classnames)
fun isClass c = member c (!classnames)


fun resolveClassNames p =
let
    fun resolveName x = x
    and resolveNameL (x,u) = (x,u)

    and resolveTy t =
	case t of
            INTty => INTty
	  | CHARty => CHARty
	  | BOOLty => BOOLty
	  | FLOATty => FLOATty
	  | STRINGty => STRINGty
	  | UNITty => UNITty
	  | TVARty s => TVARty s
	  | ARRAYty t => ARRAYty (resolveTy t)
	  | PRODUCTty ts => PRODUCTty (map resolveTy ts)
	  | ARROWty (t, t') => ARROWty (resolveTy t, resolveTy t')
	  | DIAMONDty s => DIAMONDty s
	  | OBJECTty s => OBJECTty s
	  | CONty (ts,s) => (* This is the one that matters *)
	    if isClass s
	    then
		case ts of [] => OBJECTty s
			 | l => Util.error nowhere ("Arity error for class {("
						    ^ (listToString typeToString ", " l)
						    ^ ") " ^ s ^ "}")
	    else
		CONty (map resolveTy ts, s)

    and resolveHeapUsage h = h

    and resolveTypeCon (TYPEcon (cname, (tys, y), h, x)) =
	TYPEcon (resolveName cname, (map resolveTy tys, y), resolveHeapUsage h, x)

    and resolveTypeDec (TYPEdec (tvars, name, cons, x))
      = TYPEdec (map resolveNameL tvars, resolveName name, map resolveTypeCon cons, x)

    and resolveFunAssns (FUNASSNS (name, ass1, ass2))
      = FUNASSNS (resolveName name, ass1, ass2)

    and resolveHints (HINTS (decs, funassns))
      = HINTS (map resolveTypeDec decs, map resolveFunAssns funassns)

    and resolveMatchDiam m =
	case m of
            NOWHERE => NOWHERE
	  | SOMEWHERE n => SOMEWHERE (resolveName n)
	  | DISPOSE => DISPOSE

    and resolveOOMatchPat p =
	case p of
            CLASSpat (n, n') => CLASSpat (resolveName n, resolveName n')
	  | ANYCLASSpat => ANYCLASSpat

    and resolveMatchRule r =
	case r of
            MATCHrule (con, args, d, e, x)
	    => MATCHrule (resolveName con, map resolveName args, resolveMatchDiam d,
			  resolveExp e, x)
	  | OOMATCHrule (p, e, x) => OOMATCHrule (resolveOOMatchPat p, resolveExp e, x)

    and resolveValue v = v

    and resolveExp e =
	case e of
	    VALexp (v,x) => VALexp(resolveValue v, x)
	  | UNARYexp (oper, e, x) => UNARYexp (oper, resolveExp e, x)
	  | BINexp (oper, e, e',x) => BINexp (oper, resolveExp e, resolveExp e', x)
	  | IFexp (e, e1, e2, x) => IFexp (resolveExp e, resolveExp e1, resolveExp e2, x)
	  | MATCHexp (e,l,x) => MATCHexp (resolveExp e, map resolveMatchRule l, x)
	  | LETexp (v, e, e',x) => LETexp (resolveName v, resolveExp e, resolveExp e', x)
	  | APPexp (e, l, ext, x) => APPexp (resolveExp e, map resolveExp l, ext, x)
	  | CONexp (v, es, addr, x) =>
	    CONexp (resolveName v, map resolveExp es, Option.map resolveName addr, x)
	  | TYPEDexp (e,t,x) => TYPEDexp (resolveExp e, resolveTy t, x)
	  | COERCEexp (e,t,x) => COERCEexp (resolveExp e, resolveTy t, x)
	  | NEWexp (class, es,x) => NEWexp (class, map resolveExp es, x)
	  | SUPERMAKERexp (es,x) => SUPERMAKERexp(map resolveExp es, x)
	  | INVOKEexp (obj,mname,es,x) => INVOKEexp (resolveExp obj, mname, map resolveExp es, x)
	  | UPDATEexp (a, e, x) => UPDATEexp (a, resolveExp e, x)
	  | GETexp (obj, var, x) => GETexp (resolveExp obj, var, x)
	  | SGETexp (var, x) => SGETexp (var, x)
	  | ASSERTexp (e, as1, as2, x) => ASSERTexp (resolveExp e, as1, as2, x)
	  | LAMexp (args, e, x)   => LAMexp (args, resolveExp e, x)

    and resolveValDec vd =
	case vd of
            VALdec (name, ty, inst) => VALdec (resolveName name, resolveTy ty, inst)
	  | CLASSdec (cname, super, ns, vds) =>
	    CLASSdec (resolveName cname, Option.map resolveName super,
		      map resolveName ns, map resolveValDec vds)
    and resolveVarTy UNITvar = UNITvar
      | resolveVarTy (VAR(s,t)) = VAR(s, Option.map resolveTy t)

    and resolveFunDef (FUNdef (fname, args, inst, e, x)) =
	FUNdef (resolveName fname, map resolveVarTy args, inst, resolveExp e, x)

    and resolveFunBlock (FUNblock l) = FUNblock (map resolveFunDef l)

    and resolveClassDef (CLASSdef (class, super, imps, vals, methods)) =
	CLASSdef (resolveName class,
		  Option.map resolveName super,
		  map resolveName imps,
		  map resolveValDec vals,
		  map resolveFunDef methods)

    and resolveProgram (PROG(typedecs,valdecs,classdefs,funblocks)) =
	PROG (map resolveTypeDec typedecs,
	      map resolveValDec valdecs,
	      map resolveClassDef classdefs,
	      map resolveFunBlock funblocks)
in
    resolveProgram p
end



(* ---------------- Printing abstract syntax -------------------*)

(* Temporary stuff to dispose of "secret" functions for Steffen *)

local
    val hideArrayFuns = ref false
in
(* Set this to true if you don't want to see the main fn etc *)

fun setHide b = hideArrayFuns := b

fun arrayFree t =
    case t of
        ARROWty (t1, t2) => arrayFree t1 andalso arrayFree t2
      | ARRAYty t => false
      | PRODUCTty l => arrayFreeList l
      | CONty (l, v) => arrayFreeList l
      | _ => true

and arrayFreeList l =
    case l of [] => true
	    | h::t => if arrayFree h then arrayFreeList t else false

fun discardBadValdecs l =
    if !hideArrayFuns then
	List.filter (fn VALdec(v,t,_) => arrayFree t
		      | _ => true) l
    else l

fun secretFun (FUNdef ((name,_), _, _, _, _)) = String.isPrefix "main" name
				    orelse String.isPrefix "atol" name

fun discardArrayFns (FUNblock l) =
    if !hideArrayFuns
    then FUNblock (List.filter (not o secretFun) l)
    else FUNblock l (* Caution:  funblock might be empty *)

end
(* end of temporary stuff *)


(* ================================ Printing syntax tree ================================ *)

fun printProgAndInfo os prInfo prog =
let

fun print s = TextIO.output(os,s)

fun printList printItem separator l =
case l of [] => ()
        | [h] => printItem h
        | h::t => (printItem h; print separator; printList printItem separator t)

fun printVec printItem l =
case l of [] => ()
	| _ => (print "("; printList printItem "," l; print ")")

fun printType t = print ((typeToString t) ^ " ")

fun printName (n,_) = print n
fun printNameL (n,_) = print n

fun newline() = print "\n"

fun printValue v =
    print (valToString v)  (* fix *)

fun printArg a =
    case a of
	UNITvar => print "()"
      | VAR(id, NONE) => print id
      | VAR(id, SOME ty) => (print "("; print id; print ": "; printType ty; print ")")

fun printUnaryOperator oper =
    let
	val s =
	    case oper of
		NOTop => "not"
	      | ISNULLop => "isnull"
    in
	print s
    end


fun printBinaryOperator oper =
    let
	val s =
	    case oper of
		PLUSop    => " + "
	      | MINUSop   => " - "
	      | TIMESop   => " * "
	      | DIVop     => " / "
	      | MODop     => " mod "
	      | LANDop    => " land "
	      | LORop     => " lor "
	      | LXORop    => " lxor "
	      | LSLop     => " lsl "
	      | LSRop     => " lsr "
	      | ASRop     => " asr "
	      | FPLUSop   => " +. "
	      | FMINUSop  => " -. "
	      | FTIMESop  => " *. "
	      | FDIVop    => " /. "
	      | LESSop    => " < "
	      | LEQop     => " <= "
	      | EQUALSop  => " = "
	      | CONCATop  => " ^ "
    in
	print s
    end

fun printExp e =
    case e of
	    VALexp (v,u) => printValue v
	  | UNARYexp (oper, e,u) => (printUnaryOperator oper; print "("; printExp e; print ")"; prInfo u)
	  | BINexp (oper, e, e',u) =>
	    (print "("; printExp e; printBinaryOperator oper; printExp e';print ")"; prInfo u)
(*	  | IFexp (e, e1, e2,u) => (print "if "; printExp e;
				  print " then\n   "; printExp e1;
				  print"\nelse begin\n   "; printExp e2; print "\nend"; prInfo u)
	  | MATCHexp (e,l,u) => (print "begin match ";
				 printExp e;
				 print " with"; prInfo u; print "\n";
				 printList printMatchRule "  | " l;
				 print "end")
*)
	  | IFexp (e, e1, e2,u) => (print "if "; printExp e;
				  print " then\n   "; printExp e1;
				  print"\nelse (\n   "; printExp e2; print "\n)"; prInfo u)
	  | MATCHexp (e,l,u) => (print "( match ";
				 printExp e;
				 print " with"; prInfo u; print "\n";
				 printList printMatchRule "  | " l;
				 print ")")
	  | LETexp (v, e, e',u) => (print "let "; printName v; print " = ";
				    printExp e; print "\nin "; printExp e'; prInfo u)
	  | APPexp (e, [], _,u) => (printExp e; print "() "; prInfo u)
	  | APPexp (e, l, _,u ) => (printExp e; print " ";
				    printList (fn e => (print "(";printExp e; print ")")) " " l;
				    print " "; prInfo u)
	  | CONexp (v, es, addr,u) =>
	    let val () = (printName v; printVec printExp es)
		val () = case addr of
		    NONE => print ""
		  | SOME w => (print "@"; printName w)
	    in
		 prInfo u
	    end
	  | TYPEDexp (e,t,u) => (printExp e; print ": "; printType t; prInfo u)
	  | COERCEexp (e,t,u) => (printExp e; print ":> "; printType t; prInfo u)
          | NEWexp ((class,_), es,u) => (print "new "; print class; print " "; printList printExp " " es; prInfo u)
          | SUPERMAKERexp (es,u) => (print "super "; printList printExp " " es; prInfo u)
          | INVOKEexp (obj,(mname,_),es,u)
	    => (printExp obj; print ("#" ^ mname ^ " "); printList printExp " " es; prInfo u)
          | UPDATEexp (x, e, u) => (printName x; print " <- "; printExp e; prInfo u)
          | GETexp (obj, var, u) => (printExp obj; print "#"; printName var; prInfo u)
          | SGETexp (var, u) => (printName var; prInfo u)
	  | ASSERTexp (e, as1, as2, u) =>
	    let
		val _ = warn nowhere "Assertion not printed"  (* No assertions in user syntax yet *)
	    in
		(printExp e; prInfo u)
	    end
	  | LAMexp (args,e,u) => (print "fun "; app printArg args; print " -> "; printExp e)


and printMatchRule r =
    case r of
	MATCHrule (c, args, addr, e, _) =>
	let
	    val a = case addr of NOWHERE => ""
			       | DISPOSE => "@_"
			       | SOMEWHERE (v,_) => "@"^v
	in
	    case args of
		[] => (printName c; print (a ^ " -> "); printExp e; print "\n")
	      | _ => (printName c;
		      printVec printName args;
		      print a;
		      print (" -> ");
		      printExp e;
		      print "\n")
	end
      | OOMATCHrule (ANYCLASSpat, e, _) =>
	(print "_ -> "; printExp e; print "\n")
      | OOMATCHrule (CLASSpat(ob, cl), e, _) =>
	(printName ob; print " :> "; printName cl; print " -> "; printExp e; print "\n")

fun printFunDef d =
    case d of
	FUNdef (name, [], inst, body, _) =>
	(printName name;  print "() = \n"; printExp body; print "\n")
      | FUNdef (name, args, inst, body, _) =>
	(printName name;  print " "; printList printArg " " args;
	 print " = \n"; printExp body; print "\n")

fun printFunBlock (FUNblock l) =
    let
	fun p flg [] = ()
	  | p flg (h::t) = (print (if flg then "\nlet " else "\nand "); printFunDef h; print ""; p false t)
    in
	p true l
    end


fun printValDec v =
    case v of
	VALdec (v,l, STATIC) =>
	(print "val "; printName v; print ": "; printType l; newline())
      | VALdec (v,l, INSTANCE) =>
	(print "val "; printName v; print ": "; printType l; newline())
      | CLASSdec (c, sup, intfs, decs) =>
	(print "(*** Classtypes are currently unreliable";
	 print "\nclasstype "; printName c;
	 case sup of SOME s => (print " is "; printName s)
		   | NONE => ();
	 print ": "; newline();
	 app printValDec decs; print "end\n***)\n\n")

fun printField (VALdec (v,l, STATIC)) =
    (print "field static "; printName v; print ": "; printType l; newline())
  | printField (VALdec (v,l, INSTANCE)) =
    (print "field "; printName v; print ": "; printType l; newline())
  | printField _ = Util.exit "printField:  field is CLASSdec"

fun printConstructor (TYPEcon (v, (types,_), heap,_)) =
    let val (whee, diamonds) =
            case heap of HEAP => ("","(*1*)")
                       | NOHEAP => ("!", "")
    in
        case types of
	    [] => (print whee;printName v; print diamonds)
          | _ =>
	    (
	     print whee; printName v; print diamonds; print " of ";
	     printList printType "* " types
	    )
    end

fun printTypeDec (TYPEdec (tvars, tname, constrs,_)) =
    (print (case tvars of [] => "type" | _ => "type ");
     printVec printNameL tvars;
     print " "; printName tname; print " = \n  ";
     printList printConstructor "\n  | " constrs;
     newline ();
     newline ()
    )


fun printSuper e =
    case e of
	TYPEDexp (LETexp (("_",_), SUPERMAKERexp ([VALexp(UNITval _, _)],_), body',_),UNITty, _) => body'
      |	TYPEDexp (LETexp (("_",_), SUPERMAKERexp (args,_), body',_),ty, loc) =>
	(print ": super ";
	 printList printExp " " args;
	 body'
	)
      | _ => Util.error (getU e) "malformed supermaker"

(* The parser requires a return type annotation but then sticks it
   at the end of the body. We have to undo this. *)
fun printBody e =
    case e of
	TYPEDexp (e', ty, loc) =>
	(print ": ";
	 printType ty;
	 print " =\n";
	 printExp e'
	)
      | _ =>
	( print " =\n";
	  printExp e)

fun printMethod m =
    case m of
	FUNdef (("<init>",_), [], inst, body, _) =>
	(print "maker ()";
	 let val body' = printSuper body in
	     print " =\n"; printExp body'; print "\n\n"
	 end)
      | FUNdef (("<init>",_), args, inst, body, _) =>
	(print "maker ";
	 printList printArg " " args;
	 let val body' = printSuper body in
	     print " =\n"; printExp body'; print "\n\n"
	 end)
      | FUNdef (name, [], inst, body, _) =>
	(print "method "; printName name;
	 print " ()"; printBody body; print "\n\n")
      | FUNdef (name, args, inst, body, _) =>
	(print "method "; printName name; print " ";
	 printList printArg " " args;
	 printBody body; print "\n\n")

fun printClassDef (CLASSdef(class, super, imps, fields, methods)) =
    (print "class "; printName class;
     print " = ";
     case super of SOME cl =>
		   ( printName cl; print " with")
		 | NONE =>();
     newline ();

     printList (fn x => (print "implement "; printName x)) "\n" imps;
     newline ();
     app printField (Util.makeSet fields);
     (* get rid of duplicate valdecs:  these annoy lfd_infer *)
     app printMethod methods;
     print "end\n"
     )


fun printProg (PROG(typedecs,valdecs,classdefs,funblocks)) =
    let
	val valdecs' = Util.makeSet (discardBadValdecs valdecs)
    in
    ( print "(* ---------------- Start of program ----------------*)\n";
      app printTypeDec typedecs;
      app printValDec valdecs';
      app printClassDef classdefs;
      app printFunBlock (map discardArrayFns funblocks);
      print "(* ---------------- End of program ----------------*)\n"
    )
    end



fun fullTy ty = (* Maybe we don't need this with the new diamond names *)
    case ty of
	INTty => ty
      | CHARty => ty
      | BOOLty => ty
      | FLOATty => ty
      | STRINGty => ty
      | UNITty => ty
      | TVARty v => ty
      | ARRAYty t => ARRAYty (fullTy t)
      | ARROWty (t, t') => ARROWty (fullTy t, fullTy t')
      | PRODUCTty l => PRODUCTty (map fullTy l)
      | CONty (ts, v) => CONty (map fullTy ts, innerClassName v)
      | DIAMONDty t => ty
      | OBJECTty t => ty


in
    printProg prog
end

fun printProgram pri prg = printProgAndInfo TextIO.stdOut pri prg
fun outputProgram os = printProgAndInfo os

