(* Various utility functions for ASDL abstract syntax (conversions etc) *)

exception oops of string

(* Stuff for manipulating ASDL representations of Steffen's rich types *)
local
    structure A = Absyn_ASDL
in

fun richTypeToString t = case t of
	 A.RICH_INTty =>  "int"
       | A.RICH_CHARty =>  "char"
       | A.RICH_BOOLty =>  "bool"
       | A.RICH_FLOATty =>  "float"
       | A.RICH_STRINGty =>  "string"
       | A.RICH_UNITty =>  "unit"
       | A.RICH_TVARty s =>  "'" ^ s
 (*    | A.RICH_LISTty t => (richTypeToString t) ^ " list" *)
       | A.RICH_ARRAYty t => (richTypeToString t) ^ " array"
       | A.RICH_PRODUCTty l => Util.listToString richTypeToString " * " l
       | A.RICH_ARROWty (t1,t2) => richTypeToString t1 ^ "\n -> " ^ richTypeToString t2
       | A.RICH_CONty (targs, tname, constrs) => (* targs omitted *)
	     tname ^ "[" ^ (Util.listToString richConstructorToString "|" constrs) ^ "]"
       | A.RICH_DIAMONDty s => "<>"
       | A.RICH_OBJECTty s => "(** OBJECT " ^ s ^ " **)"
       | A.RICH_SELFty => "#"

and richConstructorToString (A.RICH_CON (s, args, space)) =
    case args of
	[] => s ^ "{" ^ Int.toString space ^ "}"
      | _ => s ^ "{" ^ (Util.listToString richTypeToString "," args) ^ "," ^ Int.toString space ^ "}"

(* Constructor name, arguments, space annotation *)

and richValDecToString (A.RICH_VALDEC (fname, isp, rt, osp)) =
    fname ^ ":\n " ^ Int.toString isp ^ ", " ^ richTypeToString rt ^ ", " ^ Int.toString osp

fun printRichValDec v = print ((richValDecToString v) ^ "\n\n")

end (* local *)

(* ----------------------------- Conversions ----------------------------------- *)

(* Convert polymorphic Camelot syntax to monomorphic ASDL syntax and back *)

fun toStdLoc u =
    case Util.getLoc u of Loc.Loc(m,n) => Location.Loc(m,n)

fun fromStdLoc (Location.Loc(m,n)) = Absyn.LOC (Loc.Loc(m,n))
(* We lose any other annotations,  so have to typecheck again *)

local
open Normsyn Util
structure AA = Absyn_ASDL
val () = Absyn_ASDL.required
val nowhere = Location.nilLocation
in

fun ASDL_Name (x,u) = x
fun ASDL_NameL (x,a) = (toStdLoc a, x)

fun ASDL_Ty t =
    case t of
         INTty => AA.INTty
       | CHARty => AA.CHARty
       | BOOLty => AA.BOOLty
       | FLOATty => AA.FLOATty
       | STRINGty => AA.STRINGty
       | UNITty => AA.UNITty
       | TVARty v => AA.TVARty (#name v)  (* FIX: discarding eqtype info *)
       | ARRAYty t => AA.ARRAYty (ASDL_Ty t)
       | PRODUCTty ts => AA.PRODUCTty (map ASDL_Ty ts)
       | ARROWty (t, t') => AA.ARROWty (ASDL_Ty t, ASDL_Ty t')
       | CONty (ts,s) => AA.CONty (map ASDL_Ty ts, s)
       | DIAMONDty s => AA.DIAMONDty s
       | OBJECTty s => AA.OBJECTty s

fun ASDL_HeapUsage h =
    case h of
        HEAP => AA.HEAP
      | NOHEAP => AA.NOHEAP

fun ASDL_TypeCon (TYPEcon (cname, (tys, tyloc), h, loc)) =
    (toStdLoc loc, AA.TYPEcon (ASDL_Name cname, map ASDL_Ty tys, ASDL_HeapUsage h))

fun ASDL_TypeDec (TYPEdec (tvars, name, cons, loc))
  = (toStdLoc loc, AA.TYPEdec (map ASDL_NameL tvars, ASDL_Name name, map ASDL_TypeCon cons))

fun ASDL_FunAssns (FUNASSNS (name, ass1, ass2))
  = AA.FUNASSNS (ASDL_Name name, ass1, ass2)

fun ASDL_Hints (HINTS (decs, funassns))
  = AA.HINTS (map ASDL_TypeDec decs, map ASDL_FunAssns funassns)

fun ASDL_External e =
    case e of
	GLOBAL  => AA.GLOBAL
      | LOCAL   => AA.LOCAL
      | BUILTIN => AA.BUILTIN
      | EXTERN  => AA.EXTERN

fun ASDL_Value w =
    case w of
        VARval (v,ext,_)=> AA.VARval (v, ASDL_External ext)
      | CHARval (n,_)   => AA.CHARval n
      | INTval (n,_)    => AA.INTval n
      | FLOATval (x,_)  => AA.FLOATval x
      | STRINGval (s,_) => AA.STRINGval s
      | BOOLval (b,_)   => if b
			   then AA.BOOLval AA.TRUEval
			   else AA.BOOLval AA.FALSEval
      | UNITval _       => AA.UNITval
      | NULLval (s,_)   => AA.NULLval (*s*)

fun ASDL_UnaryOperator oper =
    case oper of
	NOTop => AA.NOTop
      | ISNULLop => AA.ISNULLop

fun ASDL_BinaryOperator oper =
    case oper of
        PLUSop    => AA.PLUSop
      | MINUSop   => AA.MINUSop
      | TIMESop   => AA.TIMESop
      | DIVop     => AA.DIVop
      | MODop     => AA.MODop
      | LANDop    => AA.LANDop
      | LORop     => AA.LORop
      | LXORop    => AA.LXORop
      | LSLop     => AA.LSLop
      | LSRop     => AA.LSRop
      | ASRop     => AA.ASRop
      | FPLUSop   => AA.FPLUSop
      | FMINUSop  => AA.FMINUSop
      | FTIMESop  => AA.FTIMESop
      | FDIVop    => AA.FDIVop
      | LESSop    => AA.LESSop
      | LEQop     => AA.LEQop
      | EQUALSop  => AA.EQUALSop
      | CONCATop  => AA.CONCATop

fun ASDL_MatchDiam m =
    case m of
        NOWHERE => AA.NOWHERE
      | SOMEWHERE n => AA.SOMEWHERE (nowhere, ASDL_Name n)
      | DISPOSE => AA.DISPOSE

fun ASDL_OOMatchPat p =
    case p of
        CLASSpat (n, n') => AA.CLASSpat (ASDL_Name n, ASDL_Name n')
      | ANYCLASSpat => AA.ANYCLASSpat

fun ASDL_Test t =
    case t of
	TEST (bop, v, v', _) => AA.TEST (ASDL_BinaryOperator bop, ASDL_Value v, ASDL_Value v')

fun ASDL_Instance i =
    case i of
        INSTANCE => AA.INSTANCE
      | STATIC => AA.STATIC

fun ASDL_Exp e =
    case e of
	VALexp (v,loc) => (toStdLoc loc, AA.VALexp(ASDL_Value v))
      | UNARYexp (oper, v, loc) => (toStdLoc loc, AA.UNARYexp (ASDL_UnaryOperator oper, ASDL_Value v))
      | BINexp (oper, v, v', loc) => (toStdLoc loc,
				      AA.BINexp (ASDL_BinaryOperator oper, ASDL_Value v, ASDL_Value v'))
      | APPexp (v, l, ext, loc) => (toStdLoc loc,
				    AA.APPexp (ASDL_Name v, map ASDL_Value l, ASDL_External ext) )
      | CONexp (v, vs, addr, loc)
	=> (toStdLoc loc, AA.CONexp (ASDL_Name v, map ASDL_Value vs, Option.map ASDL_NameL addr) )
      | INVOKEexp (obj,mname,vs,loc) => (toStdLoc loc,
					 AA.INVOKEexp (ASDL_Name obj, ASDL_Name mname, map ASDL_Value vs))
      | NEWexp (class, vs,loc) => (toStdLoc loc, AA.NEWexp (ASDL_Name class, map ASDL_Value vs))
      | GETexp (obj, var, loc) => (toStdLoc loc, AA.GETexp (ASDL_Name obj, ASDL_Name var) )
      | SGETexp (var, loc) => (toStdLoc loc, AA.SGETexp (ASDL_Name var))
      | UPDATEexp (a, v, loc) => (toStdLoc loc, AA.UPDATEexp (ASDL_Name a, ASDL_Value v) )
      | SUPERMAKERexp (vs,loc) => (toStdLoc loc, AA.SUPERMAKERexp(map ASDL_Value vs))

      | LETexp (v, e, e',loc) => (toStdLoc loc, AA.LETexp (ASDL_Name v, ASDL_Exp e, ASDL_Exp e') )
      | IFexp (tst, e1, e2, loc) => (toStdLoc loc, AA.IFexp (ASDL_Test tst, ASDL_Exp e1, ASDL_Exp e2))
      | MATCHexp (v,l,loc) => (toStdLoc loc, AA.MATCHexp (ASDL_Name v, map ASDL_MatchRule l) )
      | TYPEDexp (e,t,loc) => (toStdLoc loc, AA.TYPEDexp (ASDL_Exp e, ASDL_Ty t) )
      | COERCEexp (e,t,loc) => (toStdLoc loc, AA.COERCEexp (ASDL_Exp e, ASDL_Ty t) )
      | ASSERTexp (e, as1, as2, loc) => (toStdLoc loc, AA.ASSERTexp (ASDL_Exp e, as1, as2))

and ASDL_MatchRule r =
    case r of
        MATCHrule (con, args, d, e, loc)
	=> (toStdLoc loc, AA.MATCHrule (ASDL_NameL con, map ASDL_NameL args, ASDL_MatchDiam d,
		      ASDL_Exp e))
      | OOMATCHrule (p, e, loc) => (toStdLoc loc, AA.OOMATCHrule (ASDL_OOMatchPat p, ASDL_Exp e))


fun ASDL_ValDec vd =
    case vd of
        VALdec (name, ty, inst) => (nowhere, AA.VALdec (ASDL_Name name, ASDL_Ty ty, ASDL_Instance inst))
      | CLASSdec (cname, super, ns, vds) =>
	 (nowhere, AA.CLASSdec (ASDL_Name cname, Option.map ASDL_Name super,
		  map ASDL_Name ns, map (#2 o ASDL_ValDec) vds))

fun ASDL_VarTy UNITvar = AA.UNITvar
  | ASDL_VarTy (VAR(s,t)) = AA.VAR(s, Option.map ASDL_Ty t)

fun ASDL_FunDef (FUNdef (fname, args, inst, e, loc)) =
    (toStdLoc loc, AA.FUNdef (ASDL_Name fname, map ASDL_VarTy args, ASDL_Instance inst, ASDL_Exp e))

fun ASDL_ClassDef (CLASSdef (class, super, imps, vals, methods)) =
    (nowhere, AA.CLASSdef (ASDL_Name class,
	      Option.map ASDL_Name super,
	      map ASDL_Name imps,
	      map ASDL_ValDec vals,
	      map ASDL_FunDef methods))

fun ASDL_Program (PROG(typedecs,valdecs,classdefs,funblocks)) =
    AA.PROG (map ASDL_TypeDec typedecs,
	  map ASDL_ValDec valdecs,
	  map ASDL_ClassDef classdefs,
	  map (fn FUNblock b => AA.FUNblock (map ASDL_FunDef b)) funblocks)


end


local
open Absyn_ASDL Util
val nowhere = Absyn.nowhere
val nowhere' = Location.nilLocation
in

fun Camelot_Name x = (x, nowhere)
fun Camelot_NameL (l, x) = (x, nowhere)  (* FIX *)

fun Camelot_Ty t =
    case t of
         INTty => Normsyn.INTty
       | CHARty => Normsyn.CHARty
       | BOOLty => Normsyn.BOOLty
       | FLOATty => Normsyn.FLOATty
       | STRINGty => Normsyn.STRINGty
       | UNITty => Normsyn.UNITty
       | TVARty s => Normsyn.TVARty {name=s, eq=ref false, ord=ref false}
				    (* FIX: eqtype info not preserved *)
       | ARRAYty t => Normsyn.ARRAYty (Camelot_Ty t)
       | PRODUCTty ts => Normsyn.PRODUCTty (map Camelot_Ty ts)
       | ARROWty (t, t') => Normsyn.ARROWty (Camelot_Ty t, Camelot_Ty t')
       | CONty (ts,s) => Normsyn.CONty (map Camelot_Ty ts, s)
       | DIAMONDty s => Normsyn.DIAMONDty s
       | OBJECTty s => Normsyn.OBJECTty s

fun Camelot_HeapUsage h =
    case h of
        HEAP => Normsyn.HEAP
      | NOHEAP => Normsyn.NOHEAP

fun Camelot_TypeCon (u', TYPEcon (cname, tys, h)) =
    let
	val u = fromStdLoc u'
    in
	Normsyn.TYPEcon (Camelot_Name cname, (map Camelot_Ty tys, u), Camelot_HeapUsage h, u)
    end

fun Camelot_TypeDec (u', TYPEdec (tvars, name, cons)) =
    let
	val u = fromStdLoc u'
    in
	Normsyn.TYPEdec (map (fn (loc, v) => (v, nowhere)) tvars, Camelot_Name name, map Camelot_TypeCon cons, u)
    end

fun Camelot_FunAssns (u, FUNASSNS (name, ass1, ass2))
  = Normsyn.FUNASSNS (Camelot_Name name, ass1, ass2)
(*
fun Camelot_Hints (HINTS (decs, funassns))
  = Normsyn.HINTS (map Camelot_TypeDec decs, map Camelot_FunAssns funassns)
*)


fun Camelot_External e =
    case e of
	GLOBAL  => Normsyn.GLOBAL
      | LOCAL   => Normsyn.LOCAL
      | BUILTIN => Normsyn.BUILTIN
      | EXTERN  => Normsyn.EXTERN


fun Camelot_Value u v =
    let
    in case v of
           VARval (v, ext) => Normsyn.VARval (v, Camelot_External ext, u)
	 | CHARval n   => Normsyn.CHARval (n,u)
	 | INTval n    => Normsyn.INTval (n,u)
	 | FLOATval x  => Normsyn.FLOATval (x,u)
	 | STRINGval s => Normsyn.STRINGval (s,u)
	 | BOOLval TRUEval  =>  Normsyn.BOOLval (true, u)
	 | BOOLval FALSEval =>  Normsyn.BOOLval (false, u)
	 | UNITval     => Normsyn.UNITval u
	 | NULLval     => Normsyn.NULLval ("eh?", u)
    end

fun Camelot_UnaryOperator oper =
    case oper of
	NOTop => Normsyn.NOTop
      | ISNULLop => Normsyn.ISNULLop

fun Camelot_BinaryOperator oper =
    case oper of
        PLUSop    => Normsyn.PLUSop
      | MINUSop   => Normsyn.MINUSop
      | TIMESop   => Normsyn.TIMESop
      | DIVop     => Normsyn.DIVop
      | MODop     => Normsyn.MODop
      | LANDop    => Normsyn.LANDop
      | LORop     => Normsyn.LORop
      | LXORop    => Normsyn.LXORop
      | LSLop     => Normsyn.LSLop
      | LSRop     => Normsyn.LSRop
      | ASRop     => Normsyn.ASRop
      | FPLUSop   => Normsyn.FPLUSop
      | FMINUSop  => Normsyn.FMINUSop
      | FTIMESop  => Normsyn.FTIMESop
      | FDIVop    => Normsyn.FDIVop
      | LESSop    => Normsyn.LESSop
      | LEQop    => Normsyn.LEQop
      | EQUALSop  => Normsyn.EQUALSop
      | CONCATop  => Normsyn.CONCATop

fun Camelot_MatchDiam m =
    case m of
        NOWHERE => Normsyn.NOWHERE
      | SOMEWHERE n => Normsyn.SOMEWHERE (Camelot_NameL n)
      | DISPOSE => Normsyn.DISPOSE

fun Camelot_OOMatchPat p =
    case p of
        CLASSpat (n, n') => Normsyn.CLASSpat (Camelot_Name n, Camelot_Name n')
      | ANYCLASSpat => Normsyn.ANYCLASSpat

fun Camelot_Instance i =
    case i of
        INSTANCE => Normsyn.INSTANCE
      | STATIC => Normsyn.STATIC

fun Camelot_Test t =
    case t of
	TEST (bop, v, v') => Normsyn.TEST (Camelot_BinaryOperator bop,
					   Camelot_Value nowhere v, Camelot_Value nowhere v', nowhere)

fun Camelot_Exp (u', e) =
    let val u = fromStdLoc u'
    in case e of
	   VALexp v => Normsyn.VALexp(Camelot_Value u v, u)
	 | UNARYexp (oper, v) => Normsyn.UNARYexp (Camelot_UnaryOperator oper, Camelot_Value u v, u)
	 | BINexp (oper, v, v') => Normsyn.BINexp (Camelot_BinaryOperator oper,
						   Camelot_Value u v, Camelot_Value u v', u)
	 | IFexp (tst, e1, e2)
	   => Normsyn.IFexp (Camelot_Test tst, Camelot_Exp e1, Camelot_Exp e2, u)
	 | MATCHexp (x,l) => Normsyn.MATCHexp (Camelot_Name x, map Camelot_MatchRule l, u)
	 | LETexp (v, e, e') => Normsyn.LETexp (Camelot_Name v, Camelot_Exp e, Camelot_Exp e', u)
	 | APPexp (x, l, ext) => Normsyn.APPexp (Camelot_Name x,
						 map (Camelot_Value u) l, Camelot_External ext, u)
	 | CONexp (x, vs, addr)
	   => Normsyn.CONexp (Camelot_Name x, map (Camelot_Value u) vs,
			      Option.map Camelot_NameL addr, u)
	 | TYPEDexp (e,t) => Normsyn.TYPEDexp (Camelot_Exp e, Camelot_Ty t, u)
	 | COERCEexp (e,t) => Normsyn.COERCEexp (Camelot_Exp e, Camelot_Ty t, u)
	 | NEWexp (class, vs) => Normsyn.NEWexp (Camelot_Name class, map (Camelot_Value u) vs, u)
	 | SUPERMAKERexp vs => Normsyn.SUPERMAKERexp(map (Camelot_Value u) vs, u)
	 | INVOKEexp (obj,mname,vs) =>
	   Normsyn.INVOKEexp (Camelot_Name obj, Camelot_Name mname, map (Camelot_Value u) vs, u)
	 | UPDATEexp (a, v) => Normsyn.UPDATEexp (Camelot_Name a, Camelot_Value u v, u)
	 | GETexp (obj, var) => Normsyn.GETexp (Camelot_Name obj, Camelot_Name var, u)
	 | SGETexp var => Normsyn.SGETexp (Camelot_Name var, u)
	 | ASSERTexp (e, as1, as2) => Normsyn.ASSERTexp (Camelot_Exp e, as1, as2, u)
    end

and Camelot_MatchRule (u,r) =    case r of
        MATCHrule (con, args, d, e)
	=> Normsyn.MATCHrule (Camelot_NameL con, map Camelot_NameL args, Camelot_MatchDiam d,
		      Camelot_Exp e, fromStdLoc u)
      | OOMATCHrule (p, e) => Normsyn.OOMATCHrule (Camelot_OOMatchPat p, Camelot_Exp e, fromStdLoc u)

and Camelot_ValDec vd =
    case vd of
        VALdec (name, ty, inst) => Normsyn.VALdec (Camelot_Name name, Camelot_Ty ty, Camelot_Instance inst)
      | CLASSdec (cname, super, ns, vds) =>
		 Normsyn.CLASSdec (Camelot_Name cname,
				   Option.map Camelot_Name super,
				   map Camelot_Name ns, map Camelot_ValDec vds)

and Camelot_VarTy v =
    case v of
	UNITvar => Normsyn.UNITvar
      | VAR(s,t) => Normsyn.VAR(s, Option.map Camelot_Ty t)

and Camelot_FunDef (u, FUNdef (fname, args, inst, e)) =
    Normsyn.FUNdef (Camelot_Name fname, map Camelot_VarTy args,
		    Camelot_Instance inst, Camelot_Exp e, fromStdLoc u)

and Camelot_FunBlock (FUNblock b) =
    Normsyn.FUNblock (map Camelot_FunDef b)

and Camelot_ClassDef (u, CLASSdef (class, super, imps, vals, methods)) =
    Normsyn.CLASSdef (Camelot_Name class,
	      Option.map Camelot_Name super,
	      map Camelot_Name imps,
	      map (Camelot_ValDec o #2) vals,
	      map Camelot_FunDef methods)

and Camelot_Program (PROG(typedecs,valdecs,classdefs,funblocks)) =
    Normsyn.PROG (map Camelot_TypeDec typedecs,
	  map (Camelot_ValDec o #2) valdecs,
	  map Camelot_ClassDef classdefs,
	  map Camelot_FunBlock funblocks)
end

