;;; camelot-mode.el --- Major Emacs mode for editing (and running) Camelot programs
;;
;; Heavily based on ocaml-mode version 3.05, by Xavier Leroy et al.
;; 
;; To install put the following into your .emacs file:
;;  (setq auto-mode-alist
;;        (cons '("\\.cml?t$" . camelot-mode) auto-mode-alist))
;;  (autoload 'camelot-mode "camelot-mode" "Major mode for editing Camelot code." t)
;;  
;; Version: $Id: camelot-mode.el,v 1.1 2004/01/22 22:44:50 a1hloidl Exp $
;; State:
;;  - fontification is ok
;;  - imenu is ok
;;  - indentation doesn't work yet
;;  - form insertion doesn't work yet
;;  - menubar needs restructuring
;; ---------------------------------------------------------------------------

;@menu
;* Version of the mode::	
;* User customizable variables::  
;* Main part::			
;* Other internal variables::	
;* The major mode::		
;* Subshell support::		
;* Imenu support::		
;* Indentation stuff::		
;* Error processing::		
;* Phrases::			
;* Various constants and regexps::  
;* Fontification::		
;@end menu

;@node Version of the mode, User customizable variables
;@section Version of the mode

(defconst camelot-mode-version "$Id: camelot-mode.el,v 1.1 2004/01/22 22:44:50 a1hloidl Exp $"
  "Camelot-mode version number.")
(defun camelot-mode-version ()
  "Echo the current version of Camelot-mode in the minibuffer."
  (interactive)
  (message "Using Camelot-mode version %s" camelot-mode-version))

;@node User customizable variables, Main part, Version of the mode
;@section User customizable variables

(defvar camelot-quote-char "'"
  "*Quote for character constants.")

(defvar camelot-imenu-enable nil
  "*Enable Imenu support.")

(defvar camelot-mode-indentation 2
  "*Used for \\[camelot-unindent-command].")

(defvar camelot-lookback-limit 5000
  "*How far to look back for syntax things in Camelot mode.")

(defvar camelot-max-indent-priority 8
  "*Bounds priority of operators permitted to affect Camelot indentation.

Priorities are assigned to `interesting' caml operators as follows:

        all keywords 0 to 7     8
        type, val, ... + 0      7
        :: ^                    6
        @                       5
        := <-                   4
        if                      3
        fun, let, match ...     2
        module                  1
        opening keywords        0.")

(defvar camelot-apply-extra-indent 2
  "*How many spaces to add to indentation for an application in Camelot mode.")
(make-variable-buffer-local 'camelot-apply-extra-indent)

(defvar camelot-begin-indent 2
  "*How many spaces to indent from a begin keyword in caml mode.")
(make-variable-buffer-local 'camelot-begin-indent)

(defvar camelot-class-indent 2
  "*How many spaces to indent from a class keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-class-indent)

(defvar camelot-exception-indent 2
  "*How many spaces to indent from a exception keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-exception-indent)

(defvar camelot-for-indent 2
  "*How many spaces to indent from a for keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-for-indent)

(defvar camelot-fun-indent 2
  "*How many spaces to indent from a fun keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-fun-indent)

(defvar camelot-function-indent 4
  "*How many spaces to indent from a function keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-function-indent)

(defvar camelot-if-indent  2
  "*How many spaces to indent from a if keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-if-indent)

(defvar camelot-if-else-indent 0
  "*How many spaces to indent from an if .. else line in Camelot mode.")
(make-variable-buffer-local 'camelot-if-else-indent)

(defvar camelot-inherit-indent 2
  "*How many spaces to indent from a inherit keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-inherit-indent)

(defvar camelot-initializer-indent 2
  "*How many spaces to indent from a initializer keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-initializer-indent)

(defvar camelot-include-indent 2
  "*How many spaces to indent from a include keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-include-indent)

(defvar camelot-let-indent 2
  "*How many spaces to indent from a let keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-let-indent)

(defvar camelot-let-in-indent 0
  "*How many spaces to indent from a let .. in keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-let-in-indent)

(defvar camelot-match-indent 2
  "*How many spaces to indent from a match keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-match-indent)

(defvar camelot-method-indent 2
  "*How many spaces to indent from a method keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-method-indent)

(defvar camelot-module-indent 2
  "*How many spaces to indent from a module keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-module-indent)

(defvar camelot-object-indent 2
  "*How many spaces to indent from a object keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-object-indent)

(defvar camelot-of-indent 2
  "*How many spaces to indent from a of keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-of-indent)

(defvar camelot-parser-indent 4
  "*How many spaces to indent from a parser keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-parser-indent)

(defvar camelot-sig-indent 2
  "*How many spaces to indent from a sig keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-sig-indent)

(defvar camelot-struct-indent 2
  "*How many spaces to indent from a struct keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-struct-indent)

(defvar camelot-try-indent 2
  "*How many spaces to indent from a try keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-try-indent)

(defvar camelot-type-indent 4
  "*How many spaces to indent from a type keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-type-indent)

(defvar camelot-val-indent 2
  "*How many spaces to indent from a val keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-val-indent)

(defvar camelot-while-indent 2
  "*How many spaces to indent from a while keyword in Camelot mode.")
(make-variable-buffer-local 'camelot-while-indent)

(defvar camelot-::-indent  2
  "*How many spaces to indent from a :: operator in Camelot mode.")
(make-variable-buffer-local 'camelot-::-indent)

(defvar camelot-@-indent   2
  "*How many spaces to indent from a @ operator in Camelot mode.")
(make-variable-buffer-local 'camelot-@-indent)

(defvar camelot-:=-indent  2
  "*How many spaces to indent from a := operator in Camelot mode.")
(make-variable-buffer-local 'camelot-:=-indent)

(defvar camelot-<--indent  2
  "*How many spaces to indent from a <- operator in Camelot mode.")
(make-variable-buffer-local 'camelot-<--indent)

(defvar camelot-->-indent  2
  "*How many spaces to indent from a -> operator in Camelot mode.")
(make-variable-buffer-local 'camelot-->-indent)

(defvar camelot-lb-indent 2
  "*How many spaces to indent from a \[ operator in Camelot mode.")
(make-variable-buffer-local 'camelot-lb-indent)

(defvar camelot-lc-indent 2
  "*How many spaces to indent from a \{ operator in Camelot mode.")
(make-variable-buffer-local 'camelot-lc-indent)

(defvar camelot-lp-indent  1
  "*How many spaces to indent from a \( operator in Camelot mode.")
(make-variable-buffer-local 'camelot-lp-indent)

(defvar camelot-and-extra-indent nil
  "*Extra indent for caml lines starting with the and keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-and-extra-indent)

(defvar camelot-do-extra-indent nil
  "*Extra indent for caml lines starting with the do keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-do-extra-indent)

(defvar camelot-done-extra-indent nil
  "*Extra indent for caml lines starting with the done keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-done-extra-indent)

(defvar camelot-else-extra-indent nil
  "*Extra indent for caml lines starting with the else keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-else-extra-indent)

(defvar camelot-end-extra-indent nil
  "*Extra indent for caml lines starting with the end keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-end-extra-indent)

(defvar camelot-in-extra-indent nil
  "*Extra indent for caml lines starting with the in keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-in-extra-indent)

(defvar camelot-then-extra-indent nil
  "*Extra indent for caml lines starting with the then keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-then-extra-indent)

(defvar camelot-to-extra-indent -1
  "*Extra indent for caml lines starting with the to keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-to-extra-indent)

(defvar camelot-with-extra-indent nil
  "*Extra indent for caml lines starting with the with keyword.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-with-extra-indent)

(defvar camelot-comment-indent 3
  "*Indent inside comments.")
(make-variable-buffer-local 'camelot-comment-indent)

(defvar camelot-|-extra-indent -2
  "*Extra indent for caml lines starting with the | operator.
Usually negative. nil is align on master.")
(make-variable-buffer-local 'camelot-|-extra-indent)

(defvar camelot-rb-extra-indent -2
  "*Extra indent for caml lines statring with ].
Usually negative. nil is align on master.")

(defvar camelot-rc-extra-indent -2
  "*Extra indent for caml lines starting with }.
Usually negative. nil is align on master.")

(defvar camelot-rp-extra-indent -1
  "*Extra indent for caml lines starting with ).
Usually negative. nil is align on master.")

(defvar camelot-electric-indent t
  "*Non-nil means electrically indent lines starting with |, ] or }.

Many people find eletric keys irritating, so you can disable them if
you are one.")

(defvar camelot-electric-close-vector t
  "*Non-nil means electrically insert a | before a vector-closing ].

Many people find eletric keys irritating, so you can disable them if
you are one. You should probably have this on, though, if you also
have camelot-electric-indent on, which see.")

;@node Main part, Other internal variables, User customizable variables
;@section Main part

(if (or (not (fboundp 'indent-line-to))
        (not (fboundp 'buffer-substring-no-properties)))
    (require 'camelot-compat))

(defvar camelot-shell-active nil
  "Non nil when a subshell is running.")

(defvar emacs-type
  ;; from emacs-vers.el
  (cond ((string-match "XEmacs" emacs-version)
	 ;; NOTE that this "XEmacs" test must come before the
	 ;; "Lucid" test.
	 'xemacs)
	((string-match "Lucid" emacs-version)
	 'lucid)
	((and (boundp 'epoch::version)
	      ;; I have heard of poorly-written packages that
	      ;; bind `epoch::version' to `nil', so we're extra
	      ;; paranoid here.
	      ;;
	      ;; This `symbol-value' stuff is here in order to
	      ;; avoid a warning from the byte-compiler about
	      ;; "reference to free variable epoch::version."
	      (stringp (symbol-value 'epoch::version))
	      (string-match "Epoch"
			    (symbol-value 'epoch::version)))
	 'epoch)
	(t
	 'fsf))
  "Type of emacs: 'fsf, 'xemacs, 'lucid, or 'epoch")

(defvar camelot-mode-map nil
  "Keymap used in Camelot mode.")

(if camelot-mode-map
    ()
  (setq camelot-mode-map (make-sparse-keymap))
  (define-key camelot-mode-map "|" 'camelot-electric-pipe)
  (define-key camelot-mode-map "}" 'camelot-electric-pipe)
  (define-key camelot-mode-map "]" 'camelot-electric-rb)
  (define-key camelot-mode-map "\t" 'camelot-indent-command)
  (define-key camelot-mode-map [backtab] 'camelot-unindent-command)

  (if (eq emacs-type 'xemacs)
      (define-key camelot-mode-map 'backspace 'backward-delete-char-untabify)
    (define-key camelot-mode-map "\177" 'backward-delete-char-untabify))
  (define-key camelot-mode-map "\C-cib" 'camelot-insert-begin-form)
  (define-key camelot-mode-map "\C-cif" 'camelot-insert-for-form)
  (define-key camelot-mode-map "\C-cii" 'camelot-insert-if-form)
  (define-key camelot-mode-map "\C-cil" 'camelot-insert-let-form)
  (define-key camelot-mode-map "\C-cim" 'camelot-insert-match-form)
  (define-key camelot-mode-map "\C-cit" 'camelot-insert-try-form)
  (define-key camelot-mode-map "\C-ciw" 'camelot-insert-while-form)
  (define-key camelot-mode-map "\C-c`" 'camelot-goto-phrase-error)
  ;(define-key camelot-mode-map "\C-c\C-a" 'camelot-find-alternate-file)
  (define-key camelot-mode-map "\C-c\C-c" 'compile)
  (define-key camelot-mode-map "\C-c\C-e" 'camelot-eval-phrase)
  (define-key camelot-mode-map "\C-c\C-\[" 'camelot-backward-to-less-indent)
  (define-key camelot-mode-map "\C-c\C-\]" 'camelot-forward-to-less-indent)
  (define-key camelot-mode-map "\C-c\C-q" 'camelot-indent-phrase)
  (define-key camelot-mode-map "\C-c\C-r" 'camelot-eval-region)
  (define-key camelot-mode-map "\C-c\C-s" 'camelot-show-subshell)
  (define-key camelot-mode-map "\M-\C-h" 'camelot-mark-phrase)
  (define-key camelot-mode-map "\M-\C-q" 'camelot-indent-phrase)
  (define-key camelot-mode-map "\M-\C-x" 'camelot-eval-phrase)
  (if (eq emacs-type 'xemacs) nil ; if not running xemacs
    (let ((map (make-sparse-keymap "Camelot"))
          (forms (make-sparse-keymap "Forms")))
      (define-key camelot-mode-map "\C-c\C-d" 'camelot-show-imenu)
      (define-key camelot-mode-map [menu-bar] (make-sparse-keymap))
      (define-key camelot-mode-map [menu-bar caml] (cons "Camelot" map))
      (define-key map [run-caml] '("Start subshell..." . run-caml))
      (define-key map [compile] '("Compile..." . compile))
;;       (define-key map [switch-view]
;;         '("Switch view" . camelot-find-alternate-file))
      (define-key map [separator-format] '("--"))
      (define-key map [forms] (cons "Forms" forms))
      (define-key map [show-imenu] '("Show index" . camelot-show-imenu))
      (put 'camelot-show-imenu 'menu-enable '(not camelot-imenu-shown))
      (define-key map [show-subshell] '("Show subshell" . camelot-show-subshell))
      (put 'camelot-show-subshell 'menu-enable 'camelot-shell-active)
      (define-key map [eval-phrase] '("Eval phrase" . camelot-eval-phrase))
      (put 'camelot-eval-phrase 'menu-enable 'camelot-shell-active)
      (define-key map [indent-phrase] '("Indent phrase" . camelot-indent-phrase))
      (define-key forms [while]
        '("while .. do .. done" . camelot-insert-while-form))
      (define-key forms [match] '("match .. with .." . camelot-insert-match-form))
      (define-key forms [let] '("let .. in .." . camelot-insert-let-form))
      (define-key forms [if] '("if .. then .. else .." . camelot-insert-if-form))
      (define-key forms [begin] '("begin .. end" . camelot-insert-begin-form)))))

(defvar camelot-mode-xemacs-menu
  (if (eq emacs-type 'xemacs)
      '("Camelot"
        [ "Indent phrase" camelot-indent-phrase :keys "C-M-q" ]
        [ "Eval phrase" camelot-eval-phrase
          :active camelot-shell-active :keys "C-M-x" ]
        [ "Show subshell" camelot-show-subshell camelot-shell-active ]
        ("Forms"
         [ "match .. with .." camelot-insert-match-form t ]
         [ "let .. in .." camelot-insert-let-form t ]
         [ "if .. then .. else .." camelot-insert-if-form t ]
         [ "begin .. end" camelot-insert-begin-form t ])
        "---"
        [ "Switch view" camelot-find-alternate-file t ]
;        [ "Compile..." compile t ]
        [ "Start subshell..." run-caml t ]))
  "Menu to add to the menubar when running Xemacs")

(defvar camelot-mode-syntax-table nil
  "Syntax table in use in Caml mode buffers.")
(if camelot-mode-syntax-table
    ()
  (setq camelot-mode-syntax-table (make-syntax-table))
  ; backslash is an escape sequence
  (modify-syntax-entry ?\\ "\\" camelot-mode-syntax-table)
  ; ( is first character of comment start
  (modify-syntax-entry ?\( "()1" camelot-mode-syntax-table)
  ; * is second character of comment start,
  ; and first character of comment end
  (modify-syntax-entry ?*  ". 23" camelot-mode-syntax-table)
  ; ) is last character of comment end
  (modify-syntax-entry ?\) ")(4" camelot-mode-syntax-table)
  (modify-syntax-entry ?/ "(12b" grail-mode-syntax-table)
  ; backquote was a string-like delimiter (for character literals)
  ; (modify-syntax-entry ?` "\"" camelot-mode-syntax-table)
  ; quote and underscore are part of words
  (modify-syntax-entry ?. "w" camelot-mode-syntax-table)
  (modify-syntax-entry ?' "w" camelot-mode-syntax-table)
  (modify-syntax-entry ?_ "w" camelot-mode-syntax-table))
;; no thanks
;;   ; ISO-latin accented letters and EUC kanjis are part of words
;;   (let ((i 160))
;;     (while (< i 256)
;;       (modify-syntax-entry i "w" camelot-mode-syntax-table)
;;       (setq i (1+ i)))))

(defvar camelot-mode-abbrev-table nil
  "Abbrev table used for Camelot mode buffers.")
(if camelot-mode-abbrev-table nil
  (setq camelot-mode-abbrev-table (make-abbrev-table))
  (define-abbrev camelot-mode-abbrev-table "and" "and" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "do" "do" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "done" "done" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "else" "else" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "end" "end" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "in" "in" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "then" "then" 'camelot-abbrev-hook)
  (define-abbrev camelot-mode-abbrev-table "with" "with" 'camelot-abbrev-hook))

;@node Other internal variables, The major mode, Main part
;@section Other internal variables

(defvar camelot-last-noncomment-pos nil
  "Caches last buffer position determined not inside a Camelot comment.")
(make-variable-buffer-local 'camelot-last-noncomment-pos)

;;last-noncomment-pos can be a simple position, because we nil it
;;anyway whenever buffer changes upstream. last-comment-start and -end
;;have to be markers, because we preserve them when the changes' end
;;doesn't overlap with the comment's start.

(defvar camelot-last-comment-start nil
  "A marker caching last determined caml comment start.")
(make-variable-buffer-local 'camelot-last-comment-start)

(defvar camelot-last-comment-end nil
  "A marker caching last determined caml comment end.")
(make-variable-buffer-local 'camelot-last-comment-end)

(make-variable-buffer-local 'before-change-function)

(defvar camelot-imenu-shown nil
  "True if we have computed definition list.")
(make-variable-buffer-local 'camelot-imenu-shown)

(defconst camelot-imenu-search-regexp
  (concat "\\<in\\>\\|"
          "^[ \t]*\\(let\\|class\\|type\\|m\\(odule\\|ethod\\)"
          "\\|functor\\|and\\|val\\)[ \t]+"
          "\\(\\('[a-zA-Z0-9]+\\|([^)]+)"
          "\\|mutable\\|private\\|rec\\|type\\)[ \t]+\\)?"
          "\\([a-zA-Z][a-zA-Z0-9_']*\\)"))

;@node The major mode, Subshell support, Other internal variables
;@section The major mode

(eval-when-compile
  (if (eq emacs-type 'xemacs)
    (require 'imenu)))

(defun camelot-mode ()
  "Major mode for editing Camelot code.

\\{camelot-mode-map}"

  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'camelot-mode)
  (setq mode-name "Camelot")
  (use-local-map camelot-mode-map)
  (set-syntax-table camelot-mode-syntax-table)
  (setq local-abbrev-table camelot-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'require-final-newline)
  (setq require-final-newline t)
  (make-local-variable 'comment-start)
  (setq comment-start "(*")
  (make-local-variable 'comment-end)
  (setq comment-end "*)")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "(\\*+ *")
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments nil)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'camelot-indent-command)
  ;itz Fri Sep 25 13:23:49 PDT 1998
  (make-local-variable 'add-log-current-defun-function)
  (setq add-log-current-defun-function 'camelot-current-defun)
  ;itz 03-25-96
  ;; compile support
  (make-local-variable 'compile-command)
  (setq compile-command 
    (if (or (file-exists-p "makefile")
	    (file-exists-p "Makefile"))
	    "make -k "
	(concat "camelot -v "
		        (file-name-sans-extension buffer-file-name))))  
  (setq before-change-function 'camelot-before-change-function)
  (setq camelot-last-noncomment-pos nil)
  (setq camelot-last-comment-start (make-marker))
  (setq camelot-last-comment-end (make-marker))
  ;; font lock support
  (make-local-variable 'font-lock-defaults)
  ;; install custom faces; these must have been created when loading the file
;  (if (not (facep 'camelot-oo-face))
;     nil
   (setq font-lock-warning-face 'camelot-oo-face)
   (setq font-lock-doc-face 'camelot-pragma-face)
   (setq font-lock-stop-face 'Stop)
   (setq font-lock-doccomment-face 'Doc)
;  (if (eq emacs-type 'xemacs)
;    (setq font-lock-defaults '(camelot-font-lock-keywords-for-xemacs nil nil))
  (setq font-lock-defaults '(camelot-font-lock-keywords nil nil))
  ;;
  (cond 
    ((eq emacs-type 'xemacs)
      (if (and (featurep 'menubar)
               current-menubar)
          (progn
            ;; make a local copy of the menubar, so our modes don't
            ;; change the global menubar
            (set-buffer-menubar current-menubar)
            (add-submenu nil camelot-mode-xemacs-menu))))
    ((eq emacs-type 'fsf)
     ;imenu support (not for Xemacs)
     (progn
      (make-local-variable 'imenu-create-index-function)
      (setq imenu-create-index-function 'camelot-create-index-function)
      (make-local-variable 'imenu-generic-expression)
      (setq imenu-generic-expression camelot-imenu-search-regexp)
      (if (and camelot-imenu-enable (< (buffer-size) 10000))
          (camelot-show-imenu)))))
  (run-hooks 'camelot-mode-hook))

;;; Auxiliary function. Garrigue 96-11-01.

;; (defun camelot-find-alternate-file ()
;;   (interactive)
;;   (let ((name (buffer-file-name)))
;;     (if (string-match "^\\(.*\\)\\.\\(cmlt\\)$" name)
;;         (find-file
;;          (concat
;;           (camelot-match-string 1 name)
;;           (if (string= "ml" (camelot-match-string 2 name)) ".mli" ".ml"))))))

;@node Subshell support, Imenu support, The major mode
;@section Subshell support

(defun camelot-eval-region (start end)
  "Send the current region to the inferior Caml process."
  (interactive "r")
  (require 'inf-camelot)
  (inferior-camelot-eval-region start end))

;; old version ---to be deleted later
; 
; (defun camelot-eval-phrase ()
;   "Send the current Caml phrase to the inferior Caml process."
;   (interactive)
;   (save-excursion
;     (let ((bounds (camelot-mark-phrase)))
;     (inferior-camelot-eval-region (car bounds) (cdr bounds)))))

(defun camelot-eval-phrase (arg &optional min max)
  "Send the phrase containing the point to the CAML process.
With prefix-arg send as many phrases as its numeric value, 
If an error occurs during evalutaion, stop at this phrase and
repport the error. 

Return nil if noerror and position of error if any.

If arg's numeric value is zero or negative, evaluate the current phrase
or as many as prefix arg, ignoring evaluation errors. 
This allows to jump other erroneous phrases. 

Optional arguments min max defines a region within which the phrase
should lies."
  (interactive "p")
  (require 'inf-camelot)
  (inferior-camelot-eval-phrase arg min max))

(defun camelot-eval-buffer (arg)
  "Evaluate the buffer from the beginning to the phrase under the point.
With prefix arg, evaluate past the whole buffer, no stopping at
the current point."
  (interactive "p")
  (let ((here (point)) err)
    (goto-char (point-min))
    (setq err
          (camelot-eval-phrase 500 (point-min) (if arg (point-max) here)))
    (if err (set-mark err))
    (goto-char here)))

(defun camelot-show-subshell ()
  (interactive)
  (require 'inf-camelot)
  (inferior-camelot-show-subshell))

;@node Imenu support, Indentation stuff, Subshell support
;@section Imenu support


(defun camelot-show-imenu ()
  (interactive)
  (require 'imenu)
  (switch-to-buffer (current-buffer))
  (imenu-add-to-menubar "Defs")
  (setq camelot-imenu-shown t))

(defun camelot-prev-index-position-function ()
  (let (found data)
    (while (and (setq found
                      (re-search-backward camelot-imenu-search-regexp nil 'move))
                (progn (setq data (match-data)) t)
                (or (camelot-in-literal-p)
                    (camelot-in-comment-p)
                    (if (looking-at "in") (camelot-find-in-match)))))
    (set-match-data data)
    found))

(defun camelot-create-index-function ()
  (let (value-alist
        type-alist
        class-alist
        method-alist
        module-alist
        and-alist
        all-alist
        menu-alist
        (prev-pos (point-max))
        index)
    (goto-char prev-pos)
    (imenu-progress-message prev-pos 0 t)
    ;; collect definitions
    (while (camelot-prev-index-position-function)
      (setq index (cons (camelot-match-string 5) (point)))
      (imenu-progress-message prev-pos nil t)
      (setq all-alist (cons index all-alist))
      (cond
       ((looking-at "[ \t]*and")
        (setq and-alist (cons index and-alist)))
       ((looking-at "[ \t]*let")
        (setq value-alist (cons index (append and-alist value-alist)))
        (setq and-alist nil))
       ((looking-at "[ \t]*type")
        (setq type-alist (cons index (append and-alist type-alist)))
        (setq and-alist nil))
       ((looking-at "[ \t]*class")
        (setq class-alist (cons index (append and-alist class-alist)))
        (setq and-alist nil))
       ((looking-at "[ \t]*val")
        (setq value-alist (cons index value-alist)))
;;        ((looking-at "[ \t]*\\(module\\|functor\\)")
;;         (setq module-alist (cons index module-alist)))
       ((looking-at "[ \t]*method")
        (setq method-alist (cons index method-alist)))))
    ;; build menu
    (mapcar
     '(lambda (pair)
        (if (symbol-value (cdr pair))
            (setq menu-alist
                  (cons
                   (cons (car pair)
                         (sort (symbol-value (cdr pair)) 'imenu--sort-by-name))
                   menu-alist))))
     '(("Values" . value-alist)
       ("Types" . type-alist)
       ("Modules" . module-alist)
       ("Methods" . method-alist)
       ("Classes" . class-alist)))
    (if all-alist (setq menu-alist (cons (cons "Index" all-alist) menu-alist)))
    (imenu-progress-message prev-pos 100 t)
    menu-alist))

;@node Indentation stuff, Error processing, Imenu support
;@section Indentation stuff

(defun camelot-in-indentation ()
  "Tests whether all characters between beginning of line and point
are blanks."
  (save-excursion
    (skip-chars-backward " \t")
    (bolp)))

;;; The command
;;; Sorry, I didn't like the previous behaviour... Garrigue 96/11/01

(defun camelot-indent-command (&optional p)
  "Indent the current line in Caml mode.

Compute new indentation based on caml syntax. If prefixed, indent
the line all the way to where point is."

  (interactive "*p")
  (cond
   ((and p (> p 1)) (indent-line-to (current-column)))
   ((camelot-in-indentation) (indent-line-to (camelot-compute-final-indent)))
   (t (save-excursion
        (indent-line-to
         (camelot-compute-final-indent))))))

(defun camelot-unindent-command ()

  "Decrease indentation by one level in Caml mode.

Works only if the point is at the beginning of an indented line
\(i.e. all characters between beginning of line and point are
blanks\).  Does nothing otherwise. The unindent size is given by the
variable camelot-mode-indentation."

  (interactive "*")
  (let* ((begline
          (save-excursion
            (beginning-of-line)
            (point)))
         (current-offset
          (- (point) begline)))
    (if (and (>= current-offset camelot-mode-indentation)
             (camelot-in-indentation))
        (backward-delete-char-untabify camelot-mode-indentation))))

;@node Error processing, Phrases, Indentation stuff
;@section Error processing

;; Error positions are given in bytes, not in characters
;; This function switches to monobyte mode

(if (not (fboundp 'char-bytes))
    (defalias 'forward-byte 'forward-char)
  (defun camelot-char-bytes (ch)
    (let ((l (char-bytes ch)))
      (if (> l 1) (- l 1) l)))
  (defun forward-byte (count)
    (if (> count 0)
        (while (> count 0)
          (setq count (- count (camelot-char-bytes (char-after))))
          (forward-char))
      (while (< count 0)
        (setq count (+ count (camelot-char-bytes (char-before))))
        (backward-char)))))

(require 'compile)

;; In Emacs 19, the regexps in compilation-error-regexp-alist do not
;; match the error messages when the language is not English.
;; Hence we add a regexp.

(defconst camelot-error-regexp
  "^[A-\377]+ \"\\([^\"\n]+\\)\", [A-\377]+ \\([0-9]+\\)[-,:]"
  "Regular expression matching the error messages produced by camlc.")

(if (boundp 'compilation-error-regexp-alist)
    (or (assoc camelot-error-regexp
               compilation-error-regexp-alist)
        (setq compilation-error-regexp-alist
              (cons (list camelot-error-regexp 1 2)
               compilation-error-regexp-alist))))

;; A regexp to extract the range info

(defconst camelot-error-chars-regexp
  ".*, .*, [A-\377]+ \\([0-9]+\\)-\\([0-9]+\\):"
  "Regular expression extracting the character numbers
from an error message produced by camlc.")

;; Wrapper around next-error.

(defvar camelot-error-overlay nil)

;;itz 04-21-96 somebody didn't get the documetation for next-error
;;right. When the optional argument is a number n, it should move
;;forward n errors, not reparse.

;itz 04-21-96 instead of defining a new function, use defadvice
;that way we get our effect even when we do \C-x` in compilation buffer

(defadvice next-error (after camelot-next-error activate)
 "Reads the extra positional information provided by the Caml compiler.

Puts the point and the mark exactly around the erroneous program
fragment. The erroneous fragment is also temporarily highlighted if
possible."

 (if (eq major-mode 'camelot-mode)
     (let (bol beg end)
       (save-excursion
         (set-buffer
          (if (boundp 'compilation-last-buffer)
              compilation-last-buffer   ;Emacs 19
            "*compilation*"))           ;Emacs 18
         (save-excursion
           (goto-char (window-point (get-buffer-window (current-buffer))))
           (if (looking-at camelot-error-chars-regexp)
               (setq beg
                     (string-to-int
                      (buffer-substring (match-beginning 1) (match-end 1)))
                     end
                     (string-to-int
                      (buffer-substring (match-beginning 2) (match-end 2)))))))
       (cond (beg
              (setq end (- end beg))
              (beginning-of-line)
              (forward-byte beg)
              (setq beg (point))
              (forward-byte end)
              (setq end (point))
              (goto-char beg)
              (push-mark end t)
              (cond ((fboundp 'make-overlay)
                     (if camelot-error-overlay ()
                       (setq camelot-error-overlay (make-overlay 1 1))
                       (overlay-put camelot-error-overlay 'face 'region))
                     (unwind-protect
                         (progn
                           (move-overlay camelot-error-overlay
                                         beg end (current-buffer))
                           (sit-for 60))
                       (delete-overlay camelot-error-overlay)))))))))

;; Usual match-string doesn't work properly with font-lock-mode
;; on some emacs.

(defun camelot-match-string (num &optional string)

  "Return string of text matched by last search, without properties.

NUM specifies which parenthesized expression in the last regexp.
Value is nil if NUMth pair didn't match, or there were less than NUM
pairs.  Zero means the entire text matched by the whole regexp or
whole string."

  (let* ((data (match-data))
         (begin (nth (* 2 num) data))
         (end (nth (1+ (* 2 num)) data)))
    (if string (substring string begin end)
      (buffer-substring-no-properties begin end))))

;; itz Thu Sep 24 19:02:42 PDT 1998 this is to have some level of
;; comfort when sending phrases to the toplevel and getting errors.
(defun camelot-goto-phrase-error ()
  "Find the error location in current Caml phrase."
  (interactive)
  (require 'inf-camelot)
  (let ((bounds (save-excursion (camelot-mark-phrase))))
    (inferior-camelot-goto-error (car bounds) (cdr bounds))))

;@node Phrases, Various constants and regexps, Error processing
;@section Phrases


;itz the heuristics used to see if we're `between two phrases'
;didn't seem right to me.

(defconst camelot-phrase-start-keywords
  (concat "\\<\\(class\\|let\\|type\\|val\\)\\>")
  "Keywords starting phrases in files")

;; a phrase starts when a toplevel keyword is at the beginning of a line
(defun camelot-at-phrase-start-p ()
  (and (bolp)
       (or (looking-at "#")
           (looking-at camelot-phrase-start-keywords))))

(defun camelot-skip-comments-forward ()
  (skip-chars-forward " \n\t")
  (while (or (looking-at comment-start-skip) (camelot-in-comment-p))
    (if (= (following-char) ?\)) (forward-char)
      (search-forward comment-end))
    (skip-chars-forward " \n\t")))

(defun camelot-skip-comments-backward ()
  (skip-chars-backward " \n\t")
  (while (and (eq (preceding-char) ?\)) (eq (char-after (- (point) 2)) ?*))
    (backward-char)
    (while (camelot-in-comment-p) (search-backward comment-start))
    (skip-chars-backward " \n\t")))

(defconst camelot-phrase-sep-keywords (concat ";;\\|" camelot-phrase-start-keywords))

(defun camelot-find-phrase (&optional min-pos max-pos)
  "Find the Camelot phrase containing the point.
Return the position of the beginning of the phrase, and move point
to the end.
"
  (interactive)
  (if (not min-pos) (setq min-pos (point-min)))
  (if (not max-pos) (setq max-pos (point-max)))
  (let (beg end use-semi kwop)
    ;(camelot-skip-comments-backward)
    (cond
     ; shall we have special processing for semicolons?
     ;((and (eq (char-before (- (point) 1)) ?\;) (eq (char-before) ?\;))
     ; (forward-char)
     ; (camelot-skip-comments-forward)
     ; (setq beg (point))
     ; (while (and (search-forward ";;" max-pos 'move)
     ;    (or (camelot-in-comment-p) (camelot-in-literal-p)))))
     (t
      (camelot-skip-comments-forward)
      (if (camelot-at-phrase-start-p) (forward-char))
      (while (and (cond
                   ((re-search-forward camelot-phrase-sep-keywords max-pos 'move)
                    (goto-char (match-beginning 0)) t))
                  (or (not (or (bolp) (looking-at ";;")))
                      (camelot-in-comment-p)
                      (camelot-in-literal-p)))
        (forward-char))
      (setq end (+ (point) (if (looking-at ";;") 2 0)))
      (while (and
              (setq kwop (camelot-find-kwop camelot-phrase-sep-keywords min-pos))
              (not (string= kwop ";;"))
              (not (bolp))))
      (if (string= kwop ";;") (forward-char 2))
      (if (not kwop) (goto-char min-pos))
      (camelot-skip-comments-forward)
      (setq beg (point))
      (if (>= beg end) (error "no phrase before point"))
      (goto-char end)))
    (camelot-skip-comments-forward)
    beg))

(defun camelot-mark-phrase (&optional min-pos max-pos)
  "Put mark at end of this Caml phrase, point at beginning.
"
  (interactive)
  (let* ((beg (camelot-find-phrase min-pos max-pos)) (end (point)))
    (push-mark)
    (goto-char beg)
    (cons beg end)))
    
;;itz Fri Sep 25 12:58:13 PDT 1998 support for adding change-log entries
(defun camelot-current-defun ()
  (save-excursion
    (camelot-mark-phrase)
    (if (not (looking-at camelot-phrase-start-keywords)) nil
      (re-search-forward camelot-phrase-start-keywords)
      (let ((done nil))
        (while (not done)
          (cond
           ((looking-at "\\s ")
            (skip-syntax-forward " "))
           ((char-equal (following-char) ?\( )
            (forward-sexp 1))
           ((char-equal (following-char) ?')
            (skip-syntax-forward "w_"))
           (t (setq done t)))))
      (re-search-forward "\\(\\sw\\|\\s_\\)+")
      (match-string 0))))

(defun camelot-overlap (b1 e1 b2 e2)
  (<= (max b1 b2) (min e1 e2)))

;this clears the last comment cache if necessary
(defun camelot-before-change-function (begin end)
  (if (and camelot-last-noncomment-pos
           (> camelot-last-noncomment-pos begin))
      (setq camelot-last-noncomment-pos nil))
  (if (and (marker-position camelot-last-comment-start)
           (marker-position camelot-last-comment-end)
           (camelot-overlap begin end
                         camelot-last-comment-start
                         camelot-last-comment-end))
      (prog2
          (set-marker camelot-last-comment-start nil)
          (set-marker camelot-last-comment-end nil)))
  (let ((orig-function (default-value 'before-change-function)))
    (if orig-function (funcall orig-function begin end))))

(defun camelot-in-literal-p ()
  "Returns non-nil if point is inside a Camelot literal."
  (let* ((start-literal (concat "[\"" camelot-quote-char "]"))
         (char-literal
          (concat "\\([^\\]\\|\\\\\\.\\|\\\\[0-9][0-9][0-9]\\)"
                  camelot-quote-char))
         (pos (point))
         (eol (progn (end-of-line 1) (point)))
         state in-str)
    (beginning-of-line 1)
    (while (and (not state)
                (re-search-forward start-literal eol t)
                (<= (point) pos))
      (cond
       ((string= (camelot-match-string 0) "\"")
        (setq in-str t)
        (while (and in-str (not state)
                    (re-search-forward "\"\\|\\\\\"" eol t))
          (if (> (point) pos) (setq state t))
          (if (string= (camelot-match-string 0) "\"") (setq in-str nil)))
        (if in-str (setq state t)))
       ((looking-at char-literal)
        (if (and (>= pos (match-beginning 0)) (< pos (match-end 0)))
            (setq state t)
          (goto-char (match-end 0))))))
    (goto-char pos)
    state))

(defun camelot-forward-comment ()
  "Skip one (eventually nested) comment."
  (let ((count 1) match)
    (while (> count 0)
      (if (not (re-search-forward "(\\*\\|\\*)" nil 'move))
          (setq count -1)
        (setq match (camelot-match-string 0))
        (cond
         ((camelot-in-literal-p)
          nil)
         ((string= match comment-start)
          (setq count (1+ count)))
         (t
          (setq count (1- count))))))
    (= count 0)))

(defun camelot-backward-comment ()
  "Skip one (eventually nested) comment."
  (let ((count 1) match)
    (while (> count 0)
      (if (not (re-search-backward "(\\*\\|\\*)" nil 'move))
          (setq count -1)
        (setq match (camelot-match-string 0))
        (cond
         ((camelot-in-literal-p)
          nil)
         ((string= match comment-start)
          (setq count (1- count)))
         (t
          (setq count (1+ count))))))
    (= count 0)))

(defun camelot-in-comment-p ()
  "Returns non-nil if point is inside a caml comment.
Returns nil for the parenthesis openning a comment."
  ;;we look for comments differently than literals. there are two
  ;;reasons for this. first, caml has nested comments and it is not so
  ;;clear that parse-partial-sexp supports them; second, if proper
  ;;style is used, literals are never split across lines, so we don't
  ;;have to worry about bogus phrase breaks inside literals, while we
  ;;have to account for that possibility in comments.
  (save-excursion
    (let* ((cached-pos camelot-last-noncomment-pos)
           (cached-begin (marker-position camelot-last-comment-start))
           (cached-end (marker-position camelot-last-comment-end)))
      (cond
       ((and cached-begin cached-end
             (< cached-begin (point)) (< (point) cached-end)) t)
       ((and cached-pos (= cached-pos (point))) nil)
       ((and cached-pos (> cached-pos (point))
             (< (abs (- cached-pos (point))) camelot-lookback-limit))
        (let (end found (here (point)))
          ; go back to somewhere sure
          (goto-char cached-pos)
          (while (> (point) here)
            ; look for the end of a comment
            (while (and (if (search-backward comment-end (1- here) 'move)
                            (setq end (match-end 0))
                          (setq end nil))
                        (camelot-in-literal-p)))
            (if end (setq found (camelot-backward-comment))))
          (if (and found (= (point) here)) (setq end nil))
          (if (not end)
              (setq camelot-last-noncomment-pos here)
            (set-marker camelot-last-comment-start (point))
            (set-marker camelot-last-comment-end end))
          end))
       (t
        (let (begin found (here (point)))
          ; go back to somewhere sure (or far enough)
          (goto-char
           (if cached-pos cached-pos (- (point) camelot-lookback-limit)))
          (while (< (point) here)
            ; look for the beginning of a comment
            (while (and (if (search-forward comment-start (1+ here) 'move)
                            (setq begin (match-beginning 0))
                          (setq begin nil))
                        (camelot-in-literal-p)))
            (if begin (setq found (camelot-forward-comment))))
          (if (and found (= (point) here)) (setq begin nil))
          (if (not begin)
              (setq camelot-last-noncomment-pos here)
            (set-marker camelot-last-comment-start begin)
            (set-marker camelot-last-comment-end (point)))
          begin))))))

;@node Various constants and regexps, Fontification, Phrases
;@section Various constants and regexps

;; needs fix for Camelot!! -- HWL
(defconst camelot-before-expr-prefix
  (concat "\\<\\(asr\\|begin\\|class\\|do\\(wnto\\)?\\|else"
          "\\|i\\(f\\|n\\(herit\\|itializer\\)?\\)"
          "\\|f\\(or\\|un\\(ct\\(ion\\|or\\)\\)?\\)"
          "\\|l\\(and\\|or\\|s[lr]\\|xor\\)\\|m\\(atch\\|od\\)"
          "\\|o[fr]\\|parser\\|s\\(ig\\|truct\\)\\|t\\(hen\\|o\\|ry\\)"
          "\\|w\\(h\\(en\\|ile\\)\\|ith\\)\\)\\>\\|:begin\\>"
          "\\|[=<>@^|&+-*/$%][!$%*+-./:<=>?@^|~]*\\|:[:=]\\|[[({,;]")

  "Keywords that may appear immediately before an expression.
Used to distinguish it from toplevel let construct.")

(defconst camelot-matching-kw-regexp
  (concat
   "\\<\\(and\\|do\\(ne\\)?\\|e\\(lse\\|nd\\)\\|in\\|t\\(hen\\|o\\)"
   "\\|with\\)\\>\\|[^[|]|")
  "Regexp used in caml mode for skipping back over nested blocks.")

(defconst camelot-matching-kw-alist
  '(("|" . camelot-find-pipe-match)
    (";" . camelot-find-semi-match)
    ("," . camelot-find-comma-match)
    ("end" . camelot-find-end-match)
    ("done" . camelot-find-done-match)
    ("in"  . camelot-find-in-match)
    ("with" . camelot-find-with-match)
    ("else" . camelot-find-else-match)
    ("then" . camelot-find-then-match)
    ("to" . camelot-find-done-match)
    ("do" . camelot-find-done-match)
    ("and" . camelot-find-and-match))

  "Association list used in caml mode for skipping back over nested blocks.")

(defconst camelot-kwop-regexps (make-vector 9 nil)
  "Array of regexps representing caml keywords of different priorities.")

(defun camelot-in-expr-p ()
  (let ((pos (point)) (in-expr t))
    (camelot-find-kwop
     (concat camelot-before-expr-prefix "\\|"
             camelot-matching-kw-regexp "\\|"
             (aref camelot-kwop-regexps camelot-max-indent-priority)))
    (cond
     ; special case for ;;
     ((and (> (point) 1) (= (preceding-char) ?\;) (= (following-char) ?\;))
      (setq in-expr nil))
     ((looking-at camelot-before-expr-prefix)
      (if (not (looking-at "(\\*")) (goto-char (match-end 0)))
      (skip-chars-forward " \t\n")
      (while (looking-at "(\\*")
        (forward-char)
        (camelot-forward-comment)
        (skip-chars-forward " \t\n"))
      (if (<= pos (point)) (setq in-expr nil))))
    (goto-char pos)
    in-expr))

(defun camelot-at-sexp-close-p ()
  (or (char-equal ?\) (following-char))
      (char-equal ?\] (following-char))
      (char-equal ?} (following-char))))

(defun camelot-find-kwop (kwop-regexp &optional min-pos)
  "Look back for a caml keyword or operator matching KWOP-REGEXP.
Second optional argument MIN-POS bounds the search.

Ignore occurences inside literals. If found, return a list of two
values: the actual text of the keyword or operator, and a boolean
indicating whether the keyword was one we looked for explicitly
{non-nil}, or on the other hand one of the block-terminating
keywords."

  (let ((start-literal (concat "[\"" camelot-quote-char "]"))
        found kwop)
    (while (and (> (point) 1) (not found)
                (re-search-backward kwop-regexp min-pos 'move))
      (setq kwop (camelot-match-string 0))
      (cond
       ((looking-at "(\\*")
        (if (> (point) 1) (backward-char)))
       ((camelot-in-comment-p)
        (search-backward "(" min-pos 'move))
       ((looking-at start-literal))
       ((camelot-in-literal-p)
        (re-search-backward start-literal min-pos 'move))  ;ugly hack
       ((setq found t))))
    (if found
        (if (not (string-match "\\`[^|[]|[^]|]?\\'" kwop)) ;arrrrgh!!
            kwop
          (forward-char 1) "|") nil)))

;  Association list of indentation values based on governing keywords.
;
;Each element is of the form (KEYWORD OP-TYPE PRIO INDENT). OP-TYPE is
;non-nil for operator-type nodes, which affect indentation in a
;different way from keywords: subsequent lines are indented to the
;actual occurrence of an operator, but relative to the indentation of
;the line where the governing keyword occurs.

(defconst camelot-no-indent 0)

(defconst camelot-kwop-alist
  '(("begin"            nil     6       camelot-begin-indent)
    (":begin"           nil     6       camelot-begin-indent) ; hack
    ("class"            nil     0       camelot-class-indent)
    ("constraint"       nil     0       camelot-val-indent)
    ("sig"              nil     1       camelot-sig-indent)
    ("struct"           nil     1       camelot-struct-indent)
    ("exception"        nil     0       camelot-exception-indent)
    ("for"              nil     6       camelot-for-indent)
    ("fun"              nil     3       camelot-fun-indent)
    ("function"         nil     3       camelot-function-indent)
    ("if"               nil     6       camelot-if-indent)
    ("if-else"          nil     6       camelot-if-else-indent)
    ("include"          nil     0       camelot-include-indent)
    ("inherit"          nil     0       camelot-inherit-indent)
    ("initializer"      nil     0       camelot-initializer-indent)
    ("let"              nil     6       camelot-let-indent)
    ("let-in"           nil     6       camelot-let-in-indent)
    ("match"            nil     6       camelot-match-indent)
    ("method"           nil     0       camelot-method-indent)
    ("module"           nil     0       camelot-module-indent)
    ("object"           nil     6       camelot-object-indent)
    ("of"               nil     7       camelot-of-indent)
    ("open"             nil     0       camelot-no-indent)
    ("parser"           nil     3       camelot-parser-indent)
    ("try"              nil     6       camelot-try-indent)
    ("type"             nil     0       camelot-type-indent)
    ("val"              nil     0       camelot-val-indent)
    ("when"             nil     2       camelot-if-indent)
    ("while"            nil     6       camelot-while-indent)
    ("::"               t       5       camelot-::-indent)
    ("@"                t       4       camelot-@-indent)
    ("^"                t       4       camelot-@-indent)
    (":="               nil     3       camelot-:=-indent)
    ("<-"               nil     3       camelot-<--indent)
    ("->"               nil     2       camelot-->-indent)
    ("\["               t       8       camelot-lb-indent)
    ("{"                t       8       camelot-lc-indent)
    ("\("               t       8       camelot-lp-indent)
    ("|"                nil     2       camelot-no-indent)
    (";;"               nil     0       camelot-no-indent))
; if-else and let-in are not keywords but idioms
; "|" is not in the regexps
; all these 3 values correspond to hard-coded names

"Association list of indentation values based on governing keywords.

Each element is of the form (KEYWORD OP-TYPE PRIO INDENT). OP-TYPE is
non-nil for operator-type nodes, which affect indentation in a
different way from keywords: subsequent lines are indented to the
actual occurrence of an operator, but relative to the indentation of
the line where the governing keyword occurs.")

;;Originally, we had camelot-kwop-regexp create these at runtime, from an
;;additional field in camelot-kwop-alist. That proved way too slow,
;;although I still can't understand why. itz

(aset camelot-kwop-regexps 0
      (concat
       "\\<\\(begin\\|object\\|for\\|s\\(ig\\|truct\\)\\|while\\)\\>"
       "\\|:begin\\>\\|[[({]\\|;;"))
(aset camelot-kwop-regexps 1
      (concat (aref camelot-kwop-regexps 0) "\\|\\<\\(class\\|module\\)\\>"))
(aset camelot-kwop-regexps 2
      (concat
       (aref camelot-kwop-regexps 1)
       "\\|\\<\\(fun\\(ction\\)?\\|initializer\\|let\\|m\\(atch\\|ethod\\)"
       "\\|parser\\|try\\|val\\)\\>\\|->"))
(aset camelot-kwop-regexps 3
      (concat (aref camelot-kwop-regexps 2) "\\|\\<if\\|when\\>"))
(aset camelot-kwop-regexps 4
      (concat (aref camelot-kwop-regexps 3) "\\|:=\\|<-"))
(aset camelot-kwop-regexps 5
      (concat (aref camelot-kwop-regexps 4) "\\|@"))
(aset camelot-kwop-regexps 6
      (concat (aref camelot-kwop-regexps 5) "\\|::\\|\\^"))
(aset camelot-kwop-regexps 7
      (concat
       (aref camelot-kwop-regexps 0)
       "\\|\\<\\(constraint\\|exception\\|in\\(herit\\|clude\\)"
       "\\|o\\(f\\|pen\\)\\|type\\|val\\)\\>"))
(aset camelot-kwop-regexps 8
      (concat (aref camelot-kwop-regexps 6)
       "\\|\\<\\(constraint\\|exception\\|in\\(herit\\|clude\\)"
       "\\|o\\(f\\|pen\\)\\|type\\)\\>"))

(defun camelot-find-done-match ()
  (let ((unbalanced 1) (kwop t))
    (while (and (not (= 0 unbalanced)) kwop)
      (setq kwop (camelot-find-kwop "\\<\\(done\\|for\\|while\\)\\>"))
      (cond
       ((not kwop))
       ((string= kwop "done") (setq unbalanced (1+ unbalanced)))
       (t (setq unbalanced (1- unbalanced)))))
    kwop))

(defun camelot-find-end-match ()
  (let ((unbalanced 1) (kwop t))
    (while (and (not (= 0 unbalanced)) kwop)
      (setq kwop
            (camelot-find-kwop
             "\\<\\(end\\|begin\\|class\\)\\>"))
      (cond
       ((not kwop))
       ((string= kwop "end") (setq unbalanced (1+ unbalanced)))
       ( t (setq unbalanced (1- unbalanced)))))
    ))

(defun camelot-find-in-match ()
  (let ((unbalanced 1) (kwop t))
    (while (and (not (= 0 unbalanced)) kwop)
      (setq kwop (camelot-find-kwop "\\<\\(in\\|let\\|end\\)\\>"))
      (cond
       ((not kwop))
       ((string= kwop "end") (camelot-find-end-match))
       ((string= kwop "in") (setq unbalanced (1+ unbalanced)))
       (t (setq unbalanced (1- unbalanced)))))
    kwop))

(defun camelot-find-with-match ()
  (let ((unbalanced 1) (kwop t))
    (while (and (not (= 0 unbalanced)) kwop)
      (setq kwop
            (camelot-find-kwop
             "\\<\\(with\\|match\\)\\>\\|{\\|}"))
      (cond
       ((not kwop))
       ((or (string= kwop "with") (string= kwop "}"))
        (setq unbalanced (1+ unbalanced)))
       (t (setq unbalanced (1- unbalanced)))))
    kwop))

(defun camelot-find-paren-match (close)
  (let ((unbalanced 1)
        (regexp (cond ((= close ?\)) "[()]")
                      ((= close ?\]) "[][]")
                      ((= close ?\}) "[{}]"))))
    (while (and (> unbalanced 0)
                (camelot-find-kwop regexp))
      (if (= close (following-char))
          (setq unbalanced (1+ unbalanced))
        (setq unbalanced (1- unbalanced))))))

(defun camelot-find-then-match (&optional from-else)
  (let ((bol (if from-else
                 (save-excursion
                   (progn (beginning-of-line) (point)))))
        kwop done matching-fun)
    (while (not done)
      (setq kwop
            (camelot-find-kwop
             "\\<\\(e\\(nd\\|lse\\)\\|done\\|then\\|if\\|with\\)\\>\\|[])};]"))
      (cond
       ((not kwop) (setq done t))
       ((camelot-at-sexp-close-p)
        (camelot-find-paren-match (following-char)))
       ((string= kwop "if") (setq done t))
       ((string= kwop "then")
        (if (not from-else) (setq kwop (camelot-find-then-match))))
       ((setq matching-fun (cdr-safe (assoc kwop camelot-matching-kw-alist)))
        (setq kwop (funcall matching-fun)))))
    (if (and bol (>= (point) bol))
        "if-else"
      kwop)))

(defun camelot-find-pipe-match ()
  (let ((done nil) (kwop)
        (re (concat
             "\\<\\(try\\|match\\|with\\|function\\|parser\\|type"
             "\\|e\\(nd\\|lse\\)\\|done\\|then\\|in\\)\\>"
             "\\|[^[|]|\\|[])}]")))
    (while (not done)
      (setq kwop (camelot-find-kwop re))
      (cond
       ((not kwop) (setq done t))
       ((looking-at "[^[|]\\(|\\)")
        (goto-char (match-beginning 1))
        (setq kwop "|")
        (setq done t))
       ((camelot-at-sexp-close-p)
        (camelot-find-paren-match (following-char)))
       ((string= kwop "with")
        (setq kwop (camelot-find-with-match))
        (setq done t))
       ((string= kwop "parser")
        (if (re-search-backward "\\<with\\>" (- (point) 5) t)
            (setq kwop (camelot-find-with-match)))
        (setq done t))
       ((string= kwop "done") (camelot-find-done-match))
       ((string= kwop "end") (camelot-find-end-match))
       ((string= kwop "then") (camelot-find-then-match))
       ((string= kwop "else") (camelot-find-else-match))
       ((string= kwop "in") (camelot-find-in-match))
       (t (setq done t))))
    kwop))

(defun camelot-find-and-match ()
  (let ((done nil) (kwop))
    (while (not done)
      (setq kwop (camelot-find-kwop
                  "\\<\\(object\\|exception\\|let\\|type\\|end\\|in\\)\\>"))
      (cond
       ((not kwop) (setq done t))
       ((string= kwop "end") (camelot-find-end-match))
       ((string= kwop "in") (camelot-find-in-match))
       (t (setq done t))))
    kwop))

(defun camelot-find-else-match ()
  (camelot-find-then-match t))

(defun camelot-find-semi-match ()
  (camelot-find-kwop-skipping-blocks 2))

(defun camelot-find-comma-match ()
  (camelot-find-kwop-skipping-blocks 3))

(defun camelot-find-kwop-skipping-blocks (prio)
  "Look back for a caml keyword matching camelot-kwop-regexps [PRIO].

 Skip nested blocks."

  (let ((done nil) (kwop nil) (matching-fun)
        (kwop-list (aref camelot-kwop-regexps prio)))
    (while (not done)
      (setq kwop (camelot-find-kwop
                  (concat camelot-matching-kw-regexp
                          (cond ((> prio 3) "\\|[])},;]\\|")
                                ((> prio 2) "\\|[])};]\\|")
                                (t "\\|[])}]\\|"))
                          kwop-list)))
      (cond
       ((not kwop) (setq done t))
       ((camelot-at-sexp-close-p)
        (camelot-find-paren-match (following-char)))
       ((or (string= kwop ";;")
            (and (string= kwop ";") (= (preceding-char) ?\;)))
        (forward-line 1)
        (setq kwop ";;")
        (setq done t))
       ((and (>= prio 2) (string= kwop "|")) (setq done t))
       ((string= kwop "end") (camelot-find-end-match))
       ((string= kwop "done") (camelot-find-done-match))
       ((string= kwop "in")
        (cond ((and (camelot-find-in-match) (>= prio 2))
               (setq kwop "let-in")
               (setq done t))))
       ((and (string= kwop "parser") (>= prio 2)
             (re-search-backward "\\<with\\>" (- (point) 5) t))
        (setq kwop (camelot-find-with-match))
        (setq done t))
       ((setq matching-fun (cdr-safe (assoc kwop camelot-matching-kw-alist)))
        (setq kwop (funcall matching-fun))
        (if (looking-at kwop-list) (setq done t)))
       (t (let* ((kwop-info (assoc kwop camelot-kwop-alist))
                 (is-op (and (nth 1 kwop-info)
                             ; check that we are not at beginning of line
                             (let ((pos (point)) bti)
                               (back-to-indentation)
                               (setq bti (point))
                               (goto-char pos)
                               (< bti pos)))))
            (if (and is-op (looking-at
                            (concat (regexp-quote kwop)
                                    "|?[ \t]*\\(\n\\|(\\*\\)")))
                (setq kwop-list
                      (aref camelot-kwop-regexps (nth 2 kwop-info)))
              (setq done t))))))
    kwop))

(defun camelot-compute-basic-indent (prio)
  "Compute indent of current caml line, ignoring leading keywords.

Find the `governing node' for current line. Compute desired
indentation based on the node and the indentation alists.
Assumes point is exactly at line indentation.
Does not preserve point."

  (let* (in-expr
         (kwop (cond
                ((looking-at ";;")
                 (beginning-of-line 1))
                ((looking-at "|\\([^]|]\\|\\'\\)")
                 (camelot-find-pipe-match))
                ((and (looking-at camelot-phrase-start-keywords)
                      (camelot-in-expr-p))
                 (camelot-find-end-match))
                ((and (looking-at camelot-matching-kw-regexp)
                      (assoc (camelot-match-string 0) camelot-matching-kw-alist))
                 (funcall (cdr-safe (assoc (camelot-match-string 0)
                                      camelot-matching-kw-alist))))
                ((looking-at
                  (aref camelot-kwop-regexps camelot-max-indent-priority))
                 (let* ((kwop (camelot-match-string 0))
                        (kwop-info (assoc kwop camelot-kwop-alist))
                        (prio (if kwop-info (nth 2 kwop-info)
                                camelot-max-indent-priority)))
                   (if (and (looking-at (aref camelot-kwop-regexps 0))
                            (not (looking-at "object"))
                            (camelot-in-expr-p))
                       (setq in-expr t))
                   (camelot-find-kwop-skipping-blocks prio)))
                (t
                 (if (and (= prio camelot-max-indent-priority) (camelot-in-expr-p))
                     (setq in-expr t))
                 (camelot-find-kwop-skipping-blocks prio))))
         (kwop-info (assoc kwop camelot-kwop-alist))
         (indent-diff
          (cond
           ((not kwop-info) (beginning-of-line 1) 0)
           ((looking-at "[[({][|<]?[ \t]*")
            (length (camelot-match-string 0)))
           ((nth 1 kwop-info) (symbol-value (nth 3 kwop-info)))
           (t
            (let ((pos (point)))
              (back-to-indentation)
;             (if (looking-at "\\<let\\>") (goto-char pos))
              (- (symbol-value (nth 3 kwop-info))
                 (if (looking-at "|") camelot-|-extra-indent 0))))))
         (extra (if in-expr camelot-apply-extra-indent 0)))
         (+ indent-diff extra (current-column))))

(defconst camelot-leading-kwops-regexp
  (concat
   "\\<\\(and\\|do\\(ne\\)?\\|e\\(lse\\|nd\\)\\|in"
   "\\|t\\(hen\\|o\\)\\|with\\)\\>\\|[]|})]")

  "Regexp matching caml keywords which need special indentation.")

(defconst camelot-leading-kwops-alist
  '(("and" camelot-and-extra-indent 2)
    ("do" camelot-do-extra-indent 0)
    ("done" camelot-done-extra-indent 0)
    ("else" camelot-else-extra-indent 3)
    ("end" camelot-end-extra-indent 0)
    ("in" camelot-in-extra-indent 2)
    ("then" camelot-then-extra-indent 3)
    ("to" camelot-to-extra-indent 0)
    ("with" camelot-with-extra-indent 2)
    ("|" camelot-|-extra-indent 2)
    ("]" camelot-rb-extra-indent 0)
    ("}" camelot-rc-extra-indent 0)
    (")" camelot-rp-extra-indent 0))

  "Association list of special Camelot keyword indent values.

Each member is of the form (KEYWORD EXTRA-INDENT PRIO) where
EXTRA-INDENT is the variable holding extra indentation amount for
KEYWORD (usually negative) and PRIO is upper bound on priority of
matching nodes to determine KEYWORD's final indentation.")

(defun camelot-compute-final-indent ()
  (save-excursion
    (back-to-indentation)
    (cond
     ((and (bolp) (looking-at comment-start-skip)) (current-column))
     ((camelot-in-comment-p)
      (let ((closing (looking-at "\\*)"))
            (comment-mark (looking-at "\\*")))
        (camelot-backward-comment)
        (looking-at comment-start-skip)
        (+ (current-column)
           (cond
            (closing 1)
            (comment-mark 1)
            (t camelot-comment-indent)))))
     (t (let* ((leading (looking-at camelot-leading-kwops-regexp))
               (assoc-val (if leading (assoc (camelot-match-string 0)
                                             camelot-leading-kwops-alist)))
               (extra (if leading (symbol-value (nth 1 assoc-val)) 0))
               (prio (if leading (nth 2 assoc-val)
                       camelot-max-indent-priority))
               (basic (camelot-compute-basic-indent prio)))
          (max 0 (if extra (+ extra basic) (current-column))))))))



(defun camelot-split-string ()
  "Called whenever a line is broken inside a caml string literal."
  (insert-before-markers "\"^\"")
  (backward-char 1))

(defadvice indent-new-comment-line (around
                                    camelot-indent-new-comment-line
                                    activate)

  "Handle multi-line strings in Camelot mode."

;this advice doesn't make sense in other modes. I wish there were a
;cleaner way to do this: I haven't found one.

  (let ((hooked (and (eq major-mode 'camelot-mode) (camelot-in-literal-p)))
        (split-mark))
    (if (not hooked) nil
      (setq split-mark (set-marker (make-marker) (point)))
      (camelot-split-string))
    ad-do-it
    (if (not hooked) nil
      (goto-char split-mark)
      (set-marker split-mark nil))))

(defadvice newline-and-indent (around
                               camelot-newline-and-indent
                               activate)

  "Handle multi-line strings in caml mode."

    (let ((hooked (and (eq major-mode 'camelot-mode) (camelot-in-literal-p)))
        (split-mark))
    (if (not hooked) nil
      (setq split-mark (set-marker (make-marker) (point)))
      (camelot-split-string))
    ad-do-it
    (if (not hooked) nil
      (goto-char split-mark)
      (set-marker split-mark nil))))

(defun camelot-electric-pipe ()
  "If inserting a | or } operator at beginning of line, reindent the line.

Unfortunately there is a situation where this mechanism gets
confused. It's when | is the first character of a |] sequence. This is
a misfeature of caml syntax and cannot be fixed, however, as a
workaround, the electric ] inserts | itself if the matching [ is
followed by |."

  (interactive "*")
  (let ((electric (and camelot-electric-indent
                       (camelot-in-indentation)
                       (not (camelot-in-comment-p)))))
    (self-insert-command 1)
    (if electric (save-excursion (camelot-indent-command)))))

(defun camelot-electric-rb ()
  "If inserting a ] operator at beginning of line, reindent the line.

Also, if the matching [ is followed by a | and this ] is not preceded
by |, insert one."

  (interactive "*")
  (let* ((prec (preceding-char))
         (use-pipe (and camelot-electric-close-vector
                        (not (camelot-in-comment-p))
                        (not (camelot-in-literal-p))
                        (or (not (numberp prec))
                            (not (char-equal ?| prec)))))
         (electric (and camelot-electric-indent
                        (camelot-in-indentation)
                        (not (camelot-in-comment-p)))))
    (self-insert-command 1)
    (if electric (save-excursion (camelot-indent-command)))
    (if (and use-pipe
             (save-excursion
               (condition-case nil
                   (prog2
                       (backward-list 1)
                       (looking-at "\\[|"))
                 (error ""))))
        (save-excursion
          (backward-char 1)
          (insert "|")))))

(defun camelot-abbrev-hook ()
  "If inserting a leading keyword at beginning of line, reindent the line."
  ;itz unfortunately we need a special case
  (if (and (not (camelot-in-comment-p)) (not (= last-command-char ?_)))
      (let* ((bol (save-excursion (beginning-of-line) (point)))
             (kw (save-excursion
                   (and (re-search-backward "^[ \t]*\\(\\sw+\\)\\=" bol t)
                        (camelot-match-string 1)))))
        (if kw
            (let ((indent (save-excursion
                            (goto-char (match-beginning 1))
                            (camelot-indent-command)
                            (current-column)))
                  (abbrev-correct (if (= last-command-char ?\ ) 1 0)))
              (indent-to (- indent
                            (or
                             (symbol-value
                              (nth 1
                                   (assoc kw camelot-leading-kwops-alist)))
                             0)
                            abbrev-correct)))))))

; (defun camelot-indent-phrase ()
;   (interactive "*")
;   (let ((bounds (camelot-mark-phrase)))
;     (indent-region (car bounds) (cdr bounds) nil)))

;;; Additional commands by Didier to report errors in toplevel mode

(defun camelot-skip-blank-forward ()
  (if (looking-at "[ \t\n]*\\((\\*\\([^*]\\|[^(]\\*[^)]\\)*\\*)[ \t\n]*\\)*")
      (goto-char (match-end 0))))

;; to mark phrases, so that repeated calls will take several of them
;; knows little about Ocaml appart literals and comments, so it should work
;; with other dialects as long as ;; marks the end of phrase. 

(defun camelot-indent-phrase (arg)
  "Indent current phrase
with prefix arg, indent that many phrases starting with the current phrase."
  (interactive "p")
  (save-excursion
    (let ((beg (camelot-find-phrase)))
    (while (progn (setq arg (- arg 1)) (> arg 0)) (camelot-find-phrase))
    (indent-region beg (point) nil))))

(defun camelot-indent-buffer ()
  (interactive)
  (indent-region (point-min) (point-max) nil))

(defun camelot-backward-to-less-indent (&optional n)
  "Move cursor back  N lines with less or same indentation."
  (interactive "p")
  (beginning-of-line 1)
  (if (< n 0) (camelot-forward-to-less-indent (- n))
    (while (> n 0)
      (let ((i (current-indentation)))
        (forward-line -1)
        (while (or (> (current-indentation) i)
                   (camelot-in-comment-p)
                   (looking-at
                    (concat "[ \t]*\\(\n\\|" comment-start-skip "\\)")))
          (forward-line -1)))
      (setq n (1- n))))
  (back-to-indentation))

(defun camelot-forward-to-less-indent (&optional n)
  "Move cursor back N lines with less or same indentation."
  (interactive "p")
  (beginning-of-line 1)
  (if (< n 0) (camelot-backward-to-less-indent (- n))
    (while (> n 0)
      (let ((i (current-indentation)))
        (forward-line 1)
        (while (or (> (current-indentation) i)
                   (camelot-in-comment-p)
                   (looking-at
                    (concat "[ \t]*\\(\n\\|" comment-start-skip "\\)")))
          (forward-line 1)))
      (setq n (1- n))))
  (back-to-indentation))

(defun camelot-insert-begin-form ()
  "Inserts a nicely formatted begin-end form, leaving a mark after end."
  (interactive "*")
  (let ((prec (preceding-char)))
    (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
        (insert " ")))
  (let* ((c (current-indentation)) (i (+ camelot-begin-indent c)))
    (insert "begin\n\nend")
    (push-mark)
    (indent-line-to c)
    (forward-line -1)
    (indent-line-to i)))

;; (defun camelot-insert-for-form ()
;;   "Inserts a nicely formatted for-do-done form, leaving a mark after do(ne)."
;;   (interactive "*")
;;   (let ((prec (preceding-char)))
;;     (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
;;         (insert " ")))
;;   (let* ((c (current-indentation)) (i (+ camelot-for-indent c)))
;;     (insert "for  do\n\ndone")
;;     (push-mark)
;;     (indent-line-to c)
;;     (forward-line -1)
;;     (indent-line-to i)
;;     (push-mark)
;;     (beginning-of-line 1)
;;     (backward-char 4)))

(defun camelot-insert-if-form ()
  "Insert nicely formatted if-then-else form leaving mark after then, else."
  (interactive "*")
  (let ((prec (preceding-char)))
    (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
        (insert " ")))
  (let* ((c (current-indentation)) (i (+ camelot-if-indent c)))
    (insert "if\n\nthen\n\nelse\n")
    (indent-line-to i)
    (push-mark)
    (forward-line -1)
    (indent-line-to c)
    (forward-line -1)
    (indent-line-to i)
    (push-mark)
    (forward-line -1)
    (indent-line-to c)
    (forward-line -1)
    (indent-line-to i)))

(defun camelot-insert-match-form ()
  "Insert nicely formatted match-with form leaving mark after with."
  (interactive "*")
  (let ((prec (preceding-char)))
    (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
        (insert " ")))
  (let* ((c (current-indentation)) (i (+ camelot-match-indent c)))
    (insert "match\n\nwith\n")
    (indent-line-to i)
    (push-mark)
    (forward-line -1)
    (indent-line-to c)
    (forward-line -1)
    (indent-line-to i)))

(defun camelot-insert-let-form ()
  "Insert nicely formatted let-in form leaving mark after in."
  (interactive "*")
  (let ((prec (preceding-char)))
    (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
        (insert " ")))
  (let* ((c (current-indentation)))
    (insert "let  in  end\n")
    (indent-line-to c)
    (push-mark)
    (forward-line -1)
    (forward-char (+ c 4))))

;; (defun camelot-insert-try-form ()
;;   "Insert nicely formatted try-with form leaving mark after with."
;;   (interactive "*")
;;   (let ((prec (preceding-char)))
;;     (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
;;         (insert " ")))
;;   (let* ((c (current-indentation)) (i (+ camelot-try-indent c)))
;;     (insert "try\n\nwith\n")
;;     (indent-line-to i)
;;     (push-mark)
;;     (forward-line -1)
;;     (indent-line-to c)
;;     (forward-line -1)
;;     (indent-line-to i)))

(defun camelot-insert-while-form ()
  "Insert nicely formatted while-do-done form leaving mark after do, done."
  (interactive "*")
  (let ((prec (preceding-char)))
    (if (and (numberp prec) (not (char-equal ?\  (char-syntax prec))))
        (insert " ")))
  (let* ((c (current-indentation)) (i (+ camelot-if-indent c)))
    (insert "while  do\n\ndone")
    (push-mark)
    (indent-line-to c)
    (forward-line -1)
    (indent-line-to i)
    (push-mark)
    (beginning-of-line 1)
    (backward-char 4)))

(autoload 'run-camelot "inf-camelot" "Run an inferior Caml process." t)

;@node Fontification,  , Various constants and regexps
;@section Fontification

;@menu
;* Useful colors::		
;* Fontification strings::	
;* Fontification in inferior mode::  
;@end menu

;@node Useful colors, Fontification strings, Fontification, Fontification
;@subsection Useful colors

;; maybe set font-lock faces when entering the mode, not here already
(cond
 ((x-display-color-p)
  (cond
   ((not (memq 'font-lock-type-face (face-list)))
    ; make the necessary faces
    (make-face 'Firebrick)
    (set-face-foreground 'Firebrick "Firebrick")
    (make-face 'RosyBrown)
    (set-face-foreground 'RosyBrown "RosyBrown")
    (make-face 'Purple)
    (set-face-foreground 'Purple "Purple")
    (make-face 'MidnightBlue)
    (set-face-foreground 'MidnightBlue "MidnightBlue")
    (make-face 'DarkGoldenRod)
    (set-face-foreground 'DarkGoldenRod "DarkGoldenRod")
    (make-face 'DarkOliveGreen)
    (set-face-foreground 'DarkOliveGreen "DarkOliveGreen4")
    (make-face 'CadetBlue)
    (set-face-foreground 'CadetBlue "CadetBlue")
    ; assign them as standard faces
    (setq font-lock-comment-face 'Firebrick)
    (setq font-lock-string-face 'RosyBrown)
    (setq font-lock-keyword-face 'Purple)
    (setq font-lock-function-name-face 'MidnightBlue)
    (setq font-lock-variable-name-face 'DarkGoldenRod)
    (setq font-lock-type-face 'DarkOliveGreen)
    (setq font-lock-reference-face 'CadetBlue)))
  ; extra faces for Camelot specific constructs
  (make-face 'camelot-diamond-face)
  (make-face-bold 'camelot-diamond-face)
  (set-face-foreground 'camelot-diamond-face "Red")
  (set-face-background 'camelot-diamond-face "White")
  ;(setq font-lock-builtin-face 'camelot-diamond-face)
  (make-face 'camelot-oo-face)
  ;;(make-face-bold 'camelot-oo-face)
  (set-face-foreground 'camelot-oo-face "Blue")
  (set-face-background 'camelot-oo-face "White")
  (copy-face 'font-lock-comment-face 'camelot-pragma-face)
  (set-face-background 'camelot-pragma-face "White")
  ; extra faces for documention
  (make-face 'Stop)
  (set-face-foreground 'Stop "White")
  (set-face-background 'Stop "Red")
  (make-face 'Doc)
  (set-face-foreground 'Doc "Red")
))

;@node Fontification strings, Fontification in inferior mode, Useful colors, Fontification
;@subsection Fontification strings

; The same definition is in caml.el:
; we don't know in which order they will be loaded.
(defvar camelot-quote-char "'"
  "*Quote for character constants. \"'\" for Objective Caml, \"`\" for Camelot-Light.")

(defconst camelot-font-lock-keywords-for-xemacs
   (list
;pragmas
   '("\\(^\\|[^\"]\\)\\((\\*+#+[^#]*#+\\*+)\\)"
     2 camelot-pragma-face)
;comments
   '("\\(^\\|[^\"]\\)\\((\\*[^*]*\\*+\\([^)*][^*]*\\*+\\)*)\\)"
     2 font-lock-comment-face)
;strings
   '("\\(\"[^\"]\"\\)"
     1 font-lock-string-face)
;character literals
   (cons (concat camelot-quote-char "\\(\\\\\\([ntbr" camelot-quote-char "\\]\\|"
                 "[0-9][0-9][0-9]\\)\\|.\\)" camelot-quote-char
                 "\\|\"[^\"\\]*\\(\\\\\\(.\\|\n\\)[^\"\\]*\\)*\"")
         'font-lock-string-face)

;keywords
  '("\\<\\(if\\|then\\|else\\|match\\|with\\|let\\|rec\\|in\\|val\\|of\\|and\\|or\\|mod\\|string\\)\\>"
      0 'font-lock-keyword-face)
   '("\\(|\\|->\\)"
     0 font-lock-keyword-face)
; constants
   '("\\<\\(true\\|false\\|null\\)\\>"
     0 font-lock-constant-face)
;types
 '("\\<\\(int\\|bool\\|char\\|float\\|not\\|type\\|array\\|unit\\|is\\|isnull\\)\\>" 
     0 font-lock-type-face)
 '("\\(:\\|:>\\)"
     0 font-lock-type-face)
;blocking
 '("\\<\\(begin\\|end\\)\\>"
     0 font-lock-keyword-face)
;oo thingies
   '("\\<\\(new\\|class\\|classtype\\|object\\|method\\|field\\|virtual\\|inherit\\|implement\\|super\\|maker\\|<-\\)\\>"
     0 camelot-oo-face)
;special symbols
   '("[#!;]"
     0 font-lock-reference-face)
;Diamond thingies
   '("[@]"
     0 camelot-diamond-face)
;;    '("\\<\\(assert\\|open\\|include\\)\\>\\|[~?][ (]*[a-z][a-zA-Z0-9_']*"
;;      0 font-lock-variable-name-face)
;; ;; see Lexer.lex
;;   '("\\<[A-Z][a-zA-Z0-9_`$]*'*\\>"    ;;  ConId 
;;      0 font-lock-function-name-face) 
;; ;;   '("\\<[A-Za-z][a-zA-Z0-9_.]*\\>"    ;;  ExtId
;; ;;      . font-lock-variable-name-face) 
  '("\\<'[a-zA-Z][a-zA-Z0-9_.]*\\>"   ;;  TYVAR
     0 font-lock-type-face) 
))

(defconst camelot-font-lock-keywords
  (list
;stop special comments
;   '("\\(^\\|[^\"]\\)\\((\\*\\*/\\*\\*)\\)"
;     2 font-lock-stop-face)
;doccomments
;   '("\\(^\\|[^\"]\\)\\((\\*\\*[^*]*\\([^)*][^*]*\\*+\\)*)\\)"
;     2 font-lock-doccomment-face)
;pragmas
   '("\\(^\\|[^\"]\\)\\((\\*+#+[^#]*#+\\*+)\\)"
     2 font-lock-doc-face)
;comments
   '("\\(^\\|[^\"]\\)\\((\\*[^*]*\\*+\\([^)*][^*]*\\*+\\)*)\\)"
     2 font-lock-comment-face)
;strings
   '("\\(\"[^\"]\"\\)"
     1 font-lock-string-face)
;character literals
   (cons (concat camelot-quote-char "\\(\\\\\\([ntbr" camelot-quote-char "\\]\\|"
                 "[0-9][0-9][0-9]\\)\\|.\\)" camelot-quote-char
                 "\\|\"[^\"\\]*\\(\\\\\\(.\\|\n\\)[^\"\\]*\\)*\"")
         'font-lock-string-face)
;modules and constructors
;keywords
   '("\\<\\(if\\|then\\|else\\|match\\|with\\|let\\|rec\\|in\\|val\\|of\\|and\\|or\\|mod\\|string\\)\\>"
     1 'font-lock-keyword-face)
   '("\\(|\\|->\\)"
     1 font-lock-keyword-face)
; constants
   '("\\<\\(true\\|false\\|null\\)\\>"
     1 font-lock-constant-face)
;types
 '("\\<\\(int\\|bool\\|char\\|float\\|not\\|type\\|array\\|unit\\|is\\|isnull\\)\\>" 
     1 font-lock-type-face)
 '("\\(:\\|:>\\)"
     1 font-lock-type-face)
;blocking
 '("\\<\\(begin\\|end\\)\\>"
     1 font-lock-keyword-face)
;oo thingies
   '("\\<\\(new\\|class\\|classtype\\|object\\|method\\|field\\|virtual\\|inherit\\|implement\\|super\\|maker\\)\\>"
     1 font-lock-warning-face)
  '("<-"
     0 font-lock-warning-face)
;;infix symbols
  `(,(regexp-opt '("+." "+" "-." "-" "*." "*" "=." "<." "<" ">." ">" "<=." "<=" ">=." ">=" "||" "|" "::" ":>" ":" "&&" "&" "^" "->" "\\" "\/") t)
    0 font-lock-constant-face)
;special symbols
   '("[#!;]"
     0 font-lock-reference-face)
;Diamond thingies
   '("[@]"
     0 camelot-diamond-face) ; font-lock-builtin-face) ; font-lock-keyword-face) ; 
   '("\\<\\(assert\\|open\\|include\\)\\>\\|[~?][ (]*[a-z][a-zA-Z0-9_']*"
     1 font-lock-variable-name-face)
;; see Lexer.lex
  '("\\<\\([A-Z][a-zA-Z0-9_`$]*'*\\)\\>"    ;;  ConId 
     1 font-lock-builtin-face) ; font-lock-function-name-face) 
;;   '("\\<\\([A-Za-z][a-zA-Z0-9_.]*\\)\\>"    ;;  ExtId ; too much; use std face
;;      1 font-lock-variable-name-face) 
  '("\\<\\('[a-zA-Z][a-zA-Z0-9_.]*\\)\\>"   ;;  TYVAR
     1 font-lock-type-face) 
;;;  '("`?\\<[A-Z][A-Za-z0-9_']*\\>" . font-lock-function-name-face)
))


(defconst inferior-camelot-font-lock-keywords
  (append
   (list
;inferior
    '("^[#-]" . font-lock-comment-face))
   camelot-font-lock-keywords))

;; ;; font-lock commands are similar for camelot-mode and inferior-camelot-mode
;; (setq camelot-mode-hook
;;       '(lambda ()
;;          (cond
;;           ((fboundp 'global-font-lock-mode)
;;            (make-local-variable 'font-lock-defaults)
;;            (setq font-lock-defaults
;;                  '(camelot-font-lock-keywords nil nil ((?' . "w") (?_ . "w")))))
;;           (t
;;            (setq font-lock-keywords camelot-font-lock-keywords)))
;;          (make-local-variable 'font-lock-keywords-only)
;;          (setq font-lock-keywords-only t)
;;          (font-lock-mode 1)))

;@node Fontification in inferior mode,  , Fontification strings, Fontification
;@subsection Fontification in inferior mode

(defun inferior-camelot-mode-font-hook ()
  (cond
   ((fboundp 'global-font-lock-mode)
    (make-local-variable 'font-lock-defaults)
    (setq font-lock-defaults
          '(inferior-camelot-font-lock-keywords
            nil nil ((?' . "w") (?_ . "w")))))
   (t
    (setq font-lock-keywords inferior-camelot-font-lock-keywords)))
  (make-local-variable 'font-lock-keywords-only)
  (setq font-lock-keywords-only t)
  (font-lock-mode 1))

(add-hook 'inferior-camelot-mode-hooks 'inferior-camelot-mode-font-hook)


;;; Camelot-mode.el ends here

(provide 'Camelot-mode)
