(* 				 
   File:	$RCSfile: Semantics.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Semantics.thy,v 1.2 2004/02/13 14:16:44 amomigl1 Exp $

   Functional semantics for a (restricted version of) Grail.
*)

header {* Operational semantics *}
(*<*)
theory Semantics = Prelude:
(*>*)
text {*
  We define a big-step evaluation semantics.  For the purpose of later
  proofs, the relation is also indexed by the height of the derivation.
  *}

subsection{* Resource environments and abbreviations *}

text {* 
We define resource tuples to model resource consumption during the execution.
Currently, a resource tuple is a record with 4 components, counting execution time, 
function and method calls, and measuring maximal depth of method invocations.
*}

(*Changed from nats to ints*)
record rescomp = clock :: int        -- {* execution time *}
                 callc :: int        -- {* call counter *}
                 invkc :: int        -- {* invoke counter *}
                 invkdpth :: nat     -- {* max invoke depth *}

text {*
To add up resource consumption, we use two combinators over resource tuples.
*}

translations 
 "rescomp" <= (type) "\<lparr> clock :: int, callc :: int, invkc :: int, invkdpth :: nat \<rparr>"

constdefs rescomp_plus:: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp"  (infixr "\<oplus>" 0)
"rescomp_plus p1 p2 == \<lparr>clock = clock p1 + clock p2,
                     callc = callc p1 + callc p2,
                     invkc = invkc p1 + invkc p2,
                     invkdpth = invkdpth p1 + invkdpth p2\<rparr>"

constdefs rescomp_cup :: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp" (infixr "\<smile>" 0)
"rescomp_cup p1 p2 == \<lparr>clock = clock p1 + clock p2,
                    callc = callc p1 + callc p2,
                    invkc = invkc p1 + invkc p2,
                    invkdpth = max (invkdpth p1) (invkdpth p2)\<rparr>" 
(*<*)
declare rescomp_plus_def [simp add]
declare rescomp_cup_def [simp add]
(*>*)

text {* Some syntax for creating and manipulating resource tuples. *}

syntax mkRescomp:: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp"
translations
"mkRescomp C CC IC ID" == "\<lparr>clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax mkRescomp_HWL :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp" ("\<langle>_ _ _ _\<rangle>")
translations
"\<langle>C CC IC ID\<rangle>" == "\<lparr>rescomp.clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax tkn ::"int \<Rightarrow> rescomp \<Rightarrow> rescomp"
translations
"tkn n p" == "((mkRescomp n 0 0 0) \<smile> p)"

syntax tk::"rescomp \<Rightarrow> rescomp"
translations
"tk p" == "tkn 1 p"

syntax tkcall::"rescomp \<Rightarrow> rescomp"
translations
"tkcall p" == "((mkRescomp 1 1 0 0) \<smile> p)"

syntax tickRo::"rescomp"
translations
"tickRo" == "mkRescomp 1 0 0 0"

subsection {* Operational semantics *}

text {*
We define the operational semantics as an inductively defined set, 
connecting, for a given expression, all possible pre-states with its post-states.
Note that we define the semantics with a counter of derivation height to use this
counter later on in the soundness proof to do induction over. The counter is \emph{not}
used to capture resource consumption (we have a separate field in the resource tuple 
for that). 
*}

consts semn :: "(env \<times> heap \<times> expr \<times> nat \<times> heap \<times> val \<times> rescomp) set" -- {* operational semantics *}

syntax semn_  :: "[env, heap, expr, nat, heap, val, rescomp] \<Rightarrow> bool"   ("_ \<turnstile> _ , _ \<Down>_ '(_ , _ , _')")

translations "E \<turnstile> h,e \<Down>n (hh,v,p)" == "(E,h,e,n,hh,v,p) : semn"

inductive semn intros
 semNull:    "E \<turnstile> h,(expr.Null) \<Down>1 (h,RVal Nullref, tickRo)"

 semInt:     "E \<turnstile> h,(expr.Int i) \<Down>1 (h,IVal i, tickRo)"

 semIVar:    "E \<turnstile> h,IVar v \<Down>1 (h,IVal (E<v>), tickRo)"

 semRVar:    "E \<turnstile> h,RVar v \<Down>1 (h,RVal (E\<lfloor>v\<rfloor>), tickRo)"

 semPrimop:  "E \<turnstile> h, (Primop f x y) \<Down>1 (h,IVal (f (E<x>) (E<y>)), \<langle>3 0 0 0\<rangle>)"

 semRPrimop: "E \<turnstile> h, (RPrimop f x y) \<Down>1 (h,IVal (f (E\<lfloor>x\<rfloor>) (E\<lfloor>y\<rfloor>)), \<langle>3 0 0 0\<rangle>)"

 semGetfi: "E\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> E \<turnstile> h,(GetFi x f) \<Down>1 (h, IVal (h<a\<bullet>f>), \<langle>2 0 0 0\<rangle>)"

 semGetfr: "E\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> E \<turnstile> h,(GetFr x f) \<Down>1 (h, RVal (h\<lfloor>a\<diamondsuit>f\<rfloor>), \<langle>2 0 0 0\<rangle>)"

 semPutfi: "E\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> E \<turnstile> h, (PutFi x f y) \<Down>1 (h<a\<bullet>f:=E<y>>, arbitrary, \<langle>3 0 0 0\<rangle>)"

 semPutfr: "E\<lfloor>x\<rfloor> = Ref a \<Longrightarrow> E \<turnstile> h, (PutFr x f y) \<Down>1 (h\<lfloor>a\<diamondsuit>f:=E\<lfloor>y\<rfloor>\<rfloor>, arbitrary,\<langle>3 0 0 0\<rangle>)"

 semGetstat: "E \<turnstile> h,(GetStat c f) \<Down>1 (h, RVal (h\<lbrace>c\<struct>f\<rbrace>), \<langle>2 0 0 0\<rangle>)"

 semPutstat: "E \<turnstile> h, (PutStat c f y) \<Down>1 (h\<lbrace>c\<struct>f:=E\<lfloor>y\<rfloor>\<rbrace>, arbitrary,\<langle>3 0 0 0\<rangle>)"

 semNew: "\<lbrakk> l = freshloc (Dom h)\<rbrakk> \<Longrightarrow>
          E \<turnstile> h,(New c ifldvals rfldvals) \<Down>1 (newObj h l E c ifldvals rfldvals, RVal (Ref l), tickRo)"

 semIf_True:  "\<lbrakk>E<x> = grailbool True; E \<turnstile> h, e1 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
               E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (tkn 2 p))"

 semIf_False:  "\<lbrakk>E<x> = grailbool False; E \<turnstile> h, e2 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
                E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (tkn 2 p))"

 semLeti: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,IVal i,p1);  E<x:=i> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Leti x e1 e2) \<Down>((max n m) + 1) (h2,v, tk (p1 \<smile> p2))"

 semLetr: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,RVal r,p1);  E\<lfloor>x:=r\<rfloor> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letr x e1 e2) \<Down>((max n m) + 1) (h2,v, tk (p1 \<smile> p2))"

 semLetv: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,w,p1);  E \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letv e1 e2) \<Down>(max n m) (h2,v, p1 \<smile> p2)"

 semCall: "\<lbrakk>E \<turnstile> h, funtable f \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h,(Call f) \<Down>(n+1) (h1, v, tkcall p)"

 semInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;  
              h@@a = Some C; 
              (newframe_env (Ref a) (fst (methtable C mn)) args E) \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
             E \<turnstile> h, x\<diamondsuit>mn(args) \<Down>(n + 1) (h1,v,\<langle>5 0 1 1\<rangle> \<oplus> p)"

 semInvokeStatic: "\<lbrakk>(newframe_env Nullref (fst (methtable C mn)) args E) \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                   E \<turnstile> h, C\<bullet>mn(args) \<Down>(n + 1) (h1,v, \<langle>3 0 1 1\<rangle> \<oplus> p)"
(*Before conversion to multi-parameters:
 semInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;  
              fmap_lookup (heap.oheap h) a = Some C; 
              (newframe_env (Ref a) (E\<lfloor>y\<rfloor>)) \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
             E \<turnstile> h, (Invoke x mn y) \<Down>(n + 1) (h1,v,\<langle>5 0 1 1\<rangle> \<oplus> p)"

 semInvokeStatic: "\<lbrakk>(newframe_env Nullref (E\<lfloor>y\<rfloor>)) \<turnstile> h,methtable C mn \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                   E \<turnstile> h, (InvokeStatic C mn y) \<Down>(n + 1) (h1,v, \<langle>3 0 1 1\<rangle> \<oplus> p)"
*)

text {* Operational semantics without height index *}

constdefs
 sem :: "[env, heap, expr, heap, val, rescomp] \<Rightarrow> bool"  ("_ \<turnstile> _ , _ \<Down> _ , _ , _" 1000)
 "E \<turnstile> h,e \<Down> hh,v,p \<equiv> (\<exists> n. (E \<turnstile> h,e \<Down>n (hh,v,p)))"

text {*
  Comentary:
  \begin{enumerate}
  \item InvokeStatic: static condition: method body does not refer to ''self''.
  \item Putfield instructions (rules @{text evalPutFi} and @{text evalPutFr}) are only
     evaluated  for their side-effect and should only be used with @{text Letv}.  
     To emphasise this, they return the undetermined value @{text arbitrary}.    
  \item To explain the resource counting in Invoke: vn1 = 1, vn2 = 1,
    frame push/pop = 2, and the implicit return = 1.
    NB: we could break up the increment of ticks into before/after the invocations.
  \item Many rules omit type-checking premises.
    For example, in get field and put field rules, we should
    check that the object belongs to a class that indeed has
    the fields being accessed.  
  \item Object initialization in the New ruleallocates a fresh location,
    etends the oheap component of the heap by the class entry for the new object
    and initialises the fields.
  \item  We might tickn 2 in the If rules, but we leave it at one to
     allow for extra store/retrieve which may not be necessary in 
     some cases on the real machine.
  \item The rule for evaluating Let once contained an additional side
    condition that the expression being bound does not itself contain a
    CALL (non-tail position).  So evaluation would get stuck for non-Grail
    programs.  However, there seems to be no harm in considering the
    more general semantics for a slightly richer language, since this
    side condition is not used in the logic.  (This restriction could
    therefore be considered as part of the type-checking phase for Grail).
  \end{enumerate}
*}

subsection {* Basic properties *}
text {*We also provide elimination lemmas for the inductively defined relation*}
inductive_cases eval_cases: 
 "E \<turnstile> h,(expr.Null) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(expr.Int i) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(IVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Primop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RPrimop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFi x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFr x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFi x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFr x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetStat c f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutStat c f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(New c ifldvals rfldvals) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Ifg x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Leti x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letr x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letv e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Call f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Invoke x m y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(InvokeStatic C m y) \<Down>n  (h1,v,p)"
(*<*)


lemma no_zero_height_derivs[rule_format]: "\<forall> E h h1 v p . (E \<turnstile> h, e \<Down>0  (h1,v,p) \<longrightarrow> False)"
apply (induct_tac e)
apply (simp_all)
apply(clarsimp, elim eval_cases, simp)+

apply (subgoal_tac "n=0 & m=0")
apply clarsimp
apply (simp add: max_def)
apply (case_tac "n \<le> m", auto)
apply(erule semn.elims, simp_all)+
done
(*>*)



end
(*>*)
