(*<*)
theory FunMachine = MachineBasic:
(*>*)

subsection {* Heaps and environments *}

text {* 
  The object heap is a map from locations to objects.  
  Conceptually, objects consist of a class name together with a
  mapping of field names to values.  We use Burstall's
  idea of treating records (i.e. objects) on the heap as separate
  ``mini''-heaps themselves, so that an object heap is split into
  three components: the object heap, the integer heap,
  and the reference heap.  The latter two components are
  modelled as total functions, which simplifies
  the formalization.  Of course, the semantics only examines
  the contents of heaps at non-empty locations.
  In addition, the heap contains a (total) map
  sheap for modelling ststic (reference) fields -- we don't
  count the number of classes loaded.
*}

types 
  oheap    = "locn \<leadsto>\<^sub>f cname"           -- {* object heap; a finite map *}
  iheap    = "ifldname \<Rightarrow> locn \<Rightarrow> int"  -- {* integer heap for each field *}
  rheap    = "rfldname \<Rightarrow> locn \<Rightarrow> ref"  -- {* reference heap for each field *}
  sheap = "cname \<Rightarrow> rfldname \<Rightarrow> ref" -- {* the heap for static object fields *}

text {*
  Translations are included to fold up type definitions in printing.
  *}

translations
  "oheap"   <= (type)  "nat \<leadsto>\<^sub>f cname"
  "iheap"   <= (type)  "ifldname \<Rightarrow> nat \<Rightarrow> int"
  "rheap"   <= (type)  "rfldname \<Rightarrow> nat \<Rightarrow> ref"
  "sheap" <= (type) "cname \<Rightarrow> rfldname \<Rightarrow> ref"

text {*The three components of heaps are combined in a record.*}
record heap = 
	oheap   :: oheap		 -- {* object heap     *}
	iheap   :: iheap		 -- {* integer heap    *}
	rheap   :: rheap		 -- {* references heap *}
	sheap   :: sheap		 -- {* static heap *}

text {*
  Environments are modelled as pairs of total functions, with
  components for integer variables and object variables.
  If the source program is well-typed, it should never attempt to
  access an undefined part of the env. 
 *}

types
  ienv   = "iname \<Rightarrow> int" -- {* the integer environment *}
  renv   = "rname \<Rightarrow> ref" -- {* the reference environment *}

text {*Again, translations fold up type definitions in printing*}

translations
  "ienv"  <= (type)  "iname \<Rightarrow> int"
  "renv"  <= (type)  "rname \<Rightarrow> ref"

text {* and a record bundles the components.*}

record env =  
	ienv    :: ienv	         -- {* integer env  *}
	renv    :: renv	         -- {* ref env      *}

text {*Some more syntac translations for pretty-printing.*}
translations
  "env" <= (type) "\<lparr>ienv :: ienv, renv :: renv\<rparr>"
  "env" <= (type) "\<lparr>ienv :: ienv, renv :: renv, \<dots>::'a\<rparr>"
  "heap" <= (type) "\<lparr>oheap :: oheap, iheap :: iheap, rheap :: rheap, sheap :: sheap\<rparr>"
  "heap" <= (type) "\<lparr>oheap :: oheap, iheap :: iheap, rheap :: rheap, sheap :: sheap, \<dots>::'a\<rparr>"

syntax objhp::"heap \<Rightarrow> oheap"
translations "objhp" == "heap.oheap"
syntax inthp::"heap \<Rightarrow> iheap"
translations "inthp" == "heap.iheap"
syntax refhp::"heap \<Rightarrow> rheap"
translations "refhp" == "heap.rheap"
syntax stathp::"heap \<Rightarrow> sheap"
translations "stathp" == "heap.sheap"
syntax Dom::"heap \<Rightarrow> locn set"
translations "Dom h" == "fmap_dom(oheap h)"

subsection  {* Values *}
text {* Values may be either integer values or reference values. *}

datatype val = IVal int | RVal ref

text {*Partial project functions:*}
consts
  theival :: "val \<Rightarrow> int"
  therval :: "val \<Rightarrow> ref"
primrec "theival  (IVal i) = i"
primrec "therval  (RVal r) = r"

subsection {* Access functions *}

text {* We define functions for accessing components of the heaps and environments.
        These are treated as syntactic abbreviations, rather than as functions. *}

(* NB: only priority of 1000 is high enough to override curried application in 
  writing "f s<x>".  We might try unifying these projections using axiomatic
  type class hack -- see HOL/Bali --- but that's pretty ugly. 
  Another improvement might be to restrict to using identifiers in the
  syntax: certainly for field names and variable names. *)

syntax "_get_ivar"     :: "env \<Rightarrow> iname \<Rightarrow> int"		("_<_>" [1000,1000] 1000)
translations "_get_ivar e x" == "env.ienv e x"

syntax "_get_rvar"     :: "env \<Rightarrow> rname \<Rightarrow> ref"		("_\<lfloor>_\<rfloor>" [1000,1000] 1000)
translations "_get_rvar e x" == "env.renv e x"

syntax "_get_rvar_nonnil"     :: "env \<Rightarrow> rname \<Rightarrow> locn"		("_\<lceil>_\<rceil>" [1000,1000] 1000)
translations "_get_rvar_nonnil e x" == "theloc (env.renv e x)"

syntax "_get_stat"     :: "[heap, cname, rfldname] \<Rightarrow> ref" ("_\<lbrace>_\<struct>_\<rbrace>" [1000,1000, 1000] 1000)
translations "_get_stat h c f" == "sheap h c f"

syntax "_get_ifld" ::	  "[heap, locn, ifldname] \<Rightarrow> int"      ("_<_\<bullet>_>" [1000,1000,1000] 1000)
translations "_get_ifld h loc inm" == "iheap h inm loc"

syntax "_get_rfld" ::	  "[heap, locn, rfldname] \<Rightarrow> ref"      ("_\<lfloor>_\<diamondsuit>_\<rfloor>" [1000,1000,1000] 1000)
translations "_get_rfld h loc rnm" == "rheap h rnm loc"

syntax "_get_class" :: "[heap, locn] \<Rightarrow> cname option"         ("_ @@ _" [1000,1000] 1000)
translations "_get_class h a" == "(fmap_lookup (heap.oheap h) a)"

subsection {* Update functions *}
text {* Update functions are treated as abstract operations on heaps and environments.
        This makes propositions and intermediate assertions more readable, at the 
        expense of needing to add some simplification rules.*}

-- {* Env and heap updates *}
constdefs 
ivarupdate :: "env \<Rightarrow> iname \<Rightarrow> int \<Rightarrow> env"	("_<_:=_>" [1000,1000,0] 1000)
  "ivarupdate e v val \<equiv> e\<lparr> ienv := (env.ienv e)(v := val) \<rparr>"

rvarupdate :: "env \<Rightarrow> rname \<Rightarrow> ref \<Rightarrow> env"        ("_\<lfloor>_:=_\<rfloor>" [1000,1000,0] 1000)
  "rvarupdate e v val \<equiv> e\<lparr> renv := (env.renv e)(v := val) \<rparr>"

stat_update :: "heap \<Rightarrow> cname \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> heap"        ("_\<lbrace>_\<struct>_:=_\<rbrace>" [1000,1000,1000,0] 1000)
  "stat_update h c f val \<equiv> h\<lparr> sheap := (sheap h)(c := (sheap h c)(f := val))\<rparr>"

obj_ifieldupdate :: "heap \<Rightarrow> locn \<Rightarrow> ifldname \<Rightarrow> int \<Rightarrow> heap"   ("_<_\<bullet>_:=_>" [1000,80,1000,80] 1000)
  "obj_ifieldupdate h a f rtv \<equiv> h\<lparr> iheap := (iheap h) (f:= (iheap h f)(a:=rtv)) \<rparr>"

obj_rfieldupdate :: "heap \<Rightarrow> locn \<Rightarrow> rfldname \<Rightarrow> ref \<Rightarrow> heap"   ("_\<lfloor>_\<diamondsuit>_:=_\<rfloor>" [1000,80,1000,80] 1000)
  "obj_rfieldupdate h a g rtv \<equiv> h\<lparr> rheap := (rheap h) (g:= (rheap h g)(a:=rtv)) \<rparr>"

(*<*)
lemmas update_lemmas = ivarupdate_def rvarupdate_def stat_update_def obj_ifieldupdate_def obj_rfieldupdate_def
(*>*)

-- {* New objects *}
text {* 
  Object initialization simply sets the class of an object. We may update the 
  fields with empty maps, but assume that type-checking guarantees that
  no uninitialised fields are accessed.
 *}
consts iheapflds :: "[(ifldname \<times> iname) list, locn, iheap, ienv] \<Rightarrow> iheap"
primrec
  "iheapflds [] a ihp st = ihp"
  "iheapflds (ii#iis) a ihp st = iheapflds iis a (ihp ((fst ii):=((ihp (fst ii))(a:= (st (snd ii)))))) st"

consts rheapflds :: "[(rfldname \<times> rname) list, locn, rheap, renv] \<Rightarrow> rheap"
primrec
  "rheapflds [] a rhp st = rhp"
  "rheapflds (rr#rrs) a rhp st = rheapflds rrs a (rhp ((fst rr):=((rhp (fst rr))(a:=(st (snd rr)))))) st"

constdefs
  emptyi    :: "'a \<Rightarrow> int"
  "emptyi x \<equiv> 0"
  emptyr    :: "'a \<Rightarrow> ref"
  "emptyr x \<equiv> Nullref"

text {* Object creation initialises the fields of an object at a particular location
        (a fresh location in the case of object creation) with the given values.
        The heap of static fields remains unchanged.*}

constdefs
  newObj :: "heap \<Rightarrow> locn \<Rightarrow> env \<Rightarrow> cname \<Rightarrow> (ifldname \<times> iname) list \<Rightarrow> (rfldname \<times> rname) list \<Rightarrow> heap"
  "newObj h l e c ifldvals rfldvals \<equiv> 
       \<lparr> oheap = (heap.oheap h)(l \<mapsto>\<^sub>f c),
         iheap = iheapflds ifldvals l (iheap h) (ienv e),
         rheap = rheapflds rfldvals l (rheap h) (renv e),
         sheap = sheap h\<rparr>"

subsection {*Method parameters, arguments and frames*}
text {* Formal parameters of method declarations and actual arguments 
        of method invocations may be integer or object variables.*}
datatype PAR = INpar iname | RNpar rname
types PARAMTYPE = "PAR list"

text {*Actual arguments of method invocations are variable names or immediate
       values (in particular, integer values and null references).
       Complex (eg arithmetic) expressions which may occur as function arguments
       in Camelot programs are eliminated during compilation process (''normalisation'').*}
datatype ARG = INarg iname | RNarg rname | VALarg val
types ARGTYPE = "ARG list"

text {* Each method invocation allocates a new frame on the frame stack. 
        The function newframe\_env creates the appropriate environment, 
        given a reference to the invoking object, the formal parameters
        and the actual arguments.
        The environment contains only bindings for the self object
	and the method parameters.  (We might also give initial
        values to the local variables for the method).
	If we are invoking a static method, then the
	self variable will be set to null.*}
consts evalARG :: "env \<Rightarrow> ARG \<Rightarrow> val"
primrec
"evalARG E (INarg x) = IVal E<x>"
"evalARG E (RNarg x) = RVal E\<lfloor>x\<rfloor>"
"evalARG E (VALarg v) = v"
constdefs evalARGS::"env \<Rightarrow> ARGTYPE \<Rightarrow> val list"
"evalARGS E \<equiv> map (evalARG E)"

consts assign::"(val list \<times> PARAMTYPE) \<Rightarrow> env \<Rightarrow> env"
recdef assign "measure (\<lambda> (vals, pars) . length vals)"
"assign ([], []) = (\<lambda> E . E)"
"assign ((RVal v) # vals, (RNpar x) # pars) = (\<lambda> E . assign (vals, pars) E\<lfloor>x:=v\<rfloor>)"
"assign ((IVal v) # vals, (INpar x) # pars) = (\<lambda> E . assign (vals, pars) E<x:=v>)"

(*
constdefs  newframe_env :: "ref \<Rightarrow> ref \<Rightarrow> env"
  "newframe_env objref arg \<equiv>   \<lparr> ienv = emptyi, 
 	   	                 renv = ((emptyr(self := objref))(param := arg)) \<rparr>"
*)
text {*A new frame contains entries for all arguments and a pointer to the parent 
       object (which is \verb|null| in the case of a static method).*}
constdefs newframe_env ::"ref \<Rightarrow> PARAMTYPE \<Rightarrow> ARGTYPE \<Rightarrow> env \<Rightarrow> env"
"newframe_env r params args E \<equiv> 
  assign (evalARGS E args, params) \<lparr>ienv = emptyi, renv = emptyr(self :=r)\<rparr>"

subsection {* Heap operations: HSize, same and HpMinus*}

text {* The size of the heap is approximated as the size of the domain of @{text oheap}. *}

constdefs HSize ::"heap \<Rightarrow> int"
"HSize h == int (card (Dom h))"
declare HSize_def [simp]

text {*It is easy to prove that the heap size increments with each fresh location.*}
lemma SizeInsert[simp]: "int (card (insert (freshloc (fmap_dom H)) (fmap_dom H))) = 
                         int (card (fmap_dom H)) + 1"
(*<*)
apply (subgoal_tac "card (insert (freshloc (fmap_dom H)) (fmap_dom H)) =
Suc (card (fmap_dom H))")
apply simp
apply (rule card_insert_disjoint)
apply fastsimp
apply (rule freshloc, fastsimp)
done
(*>*)

text {* The following predicate requires two heaps to coincide 
        pointwise on all objects in a given set of locations.*}
constdefs sameOH::"locn set \<Rightarrow> heap \<Rightarrow> heap \<Rightarrow> bool"
"sameOH X h hh \<equiv> (\<forall> l. (l \<in> X \<longrightarrow> (h@@l = hh@@l \<and> 
                                 (\<forall> ifield . h<l\<bullet>ifield> = hh<l\<bullet>ifield>) \<and> 
                                 (\<forall> rfield . h\<lfloor>l\<diamondsuit>rfield\<rfloor> = hh\<lfloor>l\<diamondsuit>rfield\<rfloor>))))"

text {* The next predicate additionally stipulates that all
        static fields must coincide.*}
constdefs same::"locn set \<Rightarrow> heap \<Rightarrow> heap \<Rightarrow> bool"
"same X h hh \<equiv> (sameOH X h hh \<and> (\<forall> c rfield . h\<lbrace>c\<struct>rfield\<rbrace> = hh\<lbrace>c\<struct>rfield\<rbrace>))"

text {* Another predicate for removing one object from a heap.*}
constdefs HpMinus ::"heap \<Rightarrow> locn \<Rightarrow> heap \<Rightarrow> bool"
"HpMinus h l hh == (same (Dom hh) hh h \<and> Dom hh = Dom h - {l} \<and> l : Dom h)"

text {* Values of booleans are converted into integers *}
syntax grailbool :: "bool \<Rightarrow> int"
translations "grailbool b" == "if b then (1::int) else (0::int)"
(*<*)
end
(*>*)
