(* 				 
   File:	$RCSfile: Expr.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Expr.thy,v 1.1 2004/01/16 12:19:12 lenb Exp $

   Abstract Syntax of Grail
*)

header {* Abstract Syntax *}
(*<*)
theory Expr = FunMachine:
(*>*)
subsection{* Expressions *}

text {* Expressions are the basic unit of execution,
  and denote values.  They are built from operators, constants,
  and previously computed values (names).  An expression may have a
  side effect, but there are no directly nested expressions.
  We model untyped expressions. 

  Compound expressions are built using let expressions.  
  A function body is a let expression.  A function ends with
  tail calls to other functions or method returns.  
  The natural thing to do is to have two syntactic categories
  of terms: one for simple expressions and one for compound
  expressions.  However this makes the whole development 
  more tedious; so instead we define a single category
  and leave the distinction to the conversion to.from JVM.
*}


datatype  expr =
   Int	    int			
 | IVar     iname		
 | Primop   "int \<Rightarrow> int \<Rightarrow> int" iname iname
 | Null	          		           ("NULL")
 | RVar     rname	
 | RPrimop   "ref \<Rightarrow> ref \<Rightarrow> int" rname rname
 | New      "cname" "(ifldname \<times> iname) list" "(rfldname \<times> rname) list" ("NEW <_> '(_,_')")
 | GetFi    rname ifldname	  	   ("_\<bullet>_" [65,1000] 65)
 | GetFr    rname rfldname	  	   ("_\<diamondsuit>_" [65,1000] 65)
 | PutFi    rname ifldname iname	   ("(2_\<bullet>_ :=/ _)" [70,1000,65] 61)
 | PutFr    rname rfldname rname	   ("(2_\<diamondsuit>_ :=/ _)" [70,1000,65] 61)
 | GetStat  cname rfldname                 ("_\<struct>_" [65,1000] 65)
 | PutStat  cname rfldname rname           ("(2_\<struct>_ :=/ _)" [70,1000,65] 61)
 | Invoke   rname mname ARGTYPE              ("_\<diamondsuit>_'(_')" [70,1000,65] 61) 
 | InvokeStatic cname mname ARGTYPE          ("_\<bullet>_'(_')" [70,1000,65] 61) 
 | Leti     iname expr expr
 | Letr     rname expr expr
 | Letv     expr expr
 | Ifg      iname expr expr      	   ("(1IF _/ THEN _/ ELSE _)"  [0,0,0] 61)
 | Call     funame			   ("(CALL _)" [0] 60)

text {* Notes:
  \begin{enumerate}
  \item Primitive operations (\texttt{RPrimop}) on references might
  include a test for null, and (perhaps) a pointer equality test.

  \item In method invocations, the first argument is either a
  variable pointing to an object (instance methods), or
  a class name (static/class methods).  The second argument
  is the method name, and the third argument is the
  list of actual parameters.

  \item If we treated imperative Grail, we might instead use the traditional
  imperative language of commands together with non-side effecting
  expressions, and lets would be broken into assignment and composition.
  But that treatment de-emphasises the functional view.
  \end{enumerate}
*}
  
text {* Some nice concrete syntax: *}

(* The following black magic is Isabelle gobbledygook to
   introduce some nice concrete syntax. *)

(* let syntax *)
nonterminals
 gletbind gletbinds
syntax
  "_Gbind"      :: "[id,expr] => gletbind"                ("(1_ =/_)" 10)
  "_Grefbind"   :: "[id,expr] => gletbind"                ("(1rf _ =/_)" 10)
  "_Gvoidbind"  :: "[expr] => gletbind"                      ("(1 '_ =/_ )" 10)
  ""            :: "gletbind => gletbinds"               ("_")
  "_gbinds"     :: "[gletbind, gletbinds] => gletbinds"  ("_;/ _")
  "_GLets"      :: "[gletbinds, expr] => expr"     ("(LET (_)/ IN (_) END)" 60)

translations
  "_GLets (_gbinds b bs) e"  == "_GLets b (_GLets bs e)"
  "LET    v = e IN l END"    == "Leti v e l"
  "LET    _ = e IN l END"    == "Letv e l"
  "LET rf v = e IN l END"    == "Letr v e l"

subsection {* Functions and methods *}
text {* A Grail function is notionally parameterised on
  a subset of the current variables in scope.  Variables
  not in the formal parameters should not be accessible
  in the body.  However, a new stack frame is not 
  allocated.  We model this simply by ignoring
  the formal parameters, so a Grail function is just
  its body, which is an @{text expr}.  
  To simplify the representation for Grail, we consider
  global mapping from function names to function bodies --
  the compiler ensures the distinctness of function names across
  different methods.

  A method body is a collection of function bindings,
  together with a main body and a set of local variables used
  within.
  Methods are parameterised on the object invoking the method (slef, or null)
  and then the formal parameters. While the first dependency is implicit,
  the list of formal parameters occurs explicitly.
 *}

consts
  funtable  :: "funame \<Rightarrow> expr"                       -- {* global function names *}
  methtable :: "cname \<Rightarrow> mname \<Rightarrow> (PARAMTYPE \<times> expr)" -- {* methods by class and method names*}
(*<*)
end
(*>*)
