(*
   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: types.ml,v $
   Id:       $Id: types.ml,v 1.1 2004/12/07 17:30:41 sjost Exp $ 

	
   What this File is all about:
   ----------------------------
   The TYPES of ART

   Anything concerned with the types is contained here.
   

   Global values: 
     - the_contab / update_the_contab


   ToDos: 

   
*)

open Common
open Support


(* Identifiers *) 
type variable       = string 
type typvariable    = string 
type constructor    = string
type funcidentifier = string   
type typeidentifier = string   

(* Sets of variables *)
(* has been moved to common.ml
   module IdSet= Set.Make(struct type t = string let compare = String.compare end)
   type idset = IdSet.t
 *)

(* Built-in types *)

type plain_typ =
  | UnitTyp 
  | DiamondTyp (* Just for compatibility with Camelot -> we treat it like unit, which does not cost anything *)
  | TvarTyp of typvariable  (* More abused like an error-type rather than a real typevariable, since we're not polymorphic anyway *)
  | BoolTyp 
  | IntTyp 
  | FloatTyp
  | CharTyp 
  | StringTyp 
  | LinPairTyp of (typ * typ)                (* Lineare Paare *) 
  | ConTyp of ((typ list) * typeidentifier)  (* parameter-list, typ-identifier. Bsp: int ilist, int bool tree, etc. -- Every ConTyp should carry complete information about itself (i.e. all constructors and types), but we decided to use a globally available object "the_contab" providing this information rather than storing it mutliple times in each type information for the ART type; simplifies parsing *)
  | ArrowTyp of (typ * typ) 
	(* Listen & Produkte muss man mit den Konstruktor-Typen selber bauen! *)
and typ = plain_typ Support.withinfo

let polymorph:plain_typ = (TvarTyp("Polymorph")) (* Slightly abusing syntax - used for builtin functions *)      


(* User-defined types *)
      
type constructor_size = int (* Amount of heapcells to be allocated, hence restricted to integer rather than rational - could be rational though *)
	
type typcon = TypCon of Support.info * constructor * constructor_size * (typ list)
type typdec = TypDec of Support.info * (typvariable list) * typeidentifier * (typcon list)

module InternalTab= Map.Make(struct type t = string let compare = String.compare end)

class ['t, 'argt] par_coninfo = (* merely replaces a record, which we want to expand later *)
  fun cn tid ot arg_ts sz ord ->
    object (self: 'self)
      val v_name:  constructor      = cn
      val v_typid: typeidentifier   = tid
      val v_own_typ: 't             = ot
      val v_arg_typs: 'argt list    = arg_ts
      val v_size: constructor_size  = sz
      val v_order: int              = ord

      method name: constructor      = v_name      (* the constructor's name *)
      method typid: typeidentifier  = v_typid     (* name of the type the constructor belongs to *)
      method own_typ: 't            = v_own_typ   (* constructed type, i.e. ConTyp(_,typid) - contains info! *)
      method arg_typs: 'argt list   = v_arg_typs  (* ordered list of types of the constructor's arguments *)
      method size: constructor_size = v_size      (* number of heapcells needed for construction *)
      method order: int             = v_order     (* the number the constructor appeared in the type declaration - for neat printing purposes and commandline-argument parsing *)
    end

(*
type coninfo = (typ, typ) par_coninfo  (*  Ok, but does not allow "new" with coninfo, whereas the class definition below does. However a "new par_coninfo" of correct type is compatible with this definition. *)
*)
class coninfo =
  fun cn tid ot arg_ts sz ord ->
    object (self: 'self)
      inherit [typ, typ] par_coninfo cn tid ot arg_ts sz ord 
    end

class contab = (* This object simplifies access to all user-defined types *)
  fun tdecs -> 
    let compile_coninfo: typeidentifier -> typ -> (typcon list) -> ((coninfo) InternalTab.t)  -> ((coninfo) InternalTab.t) =     
      fun typid own_typ tcs ct -> 
	let f =
	  fun (order,ct) tc ->
	    let TypCon(info, constr, csize, args) = tc in
	    let coninfo = new coninfo constr typid own_typ args csize order in
	    if InternalTab.mem constr ct 
	    then errAt info ("Duplicate constructor '"^constr^"'. Each constructor must be unique!")
	    else ((order+1),(InternalTab.add constr coninfo ct))
	in snd (List.fold_left f (1,ct) tcs)
    in
    let compile_contab: (typdec list) -> ((coninfo) InternalTab.t) =
      fun tds ->
	let f =
	  fun ct td ->
	    let TypDec(info,param,typid,tcs) = td in
	    let own_typ: typ = 
	      let paramtyp = List.map (fun x -> addinfo info (TvarTyp(x))) param (* parameter names to types *)
	      in  addinfo info (ConTyp(paramtyp, typid))
	    in compile_coninfo typid own_typ tcs ct
	in List.fold_left f InternalTab.empty tds
    in
    (* These are VAL declarations, but I do not understand the difference to proper val declarations, except that in this way, type_table may depend upon constr_table *)
    (*val*) let constr_table = compile_contab tdecs in 
    (*val*) let type_table   =
      let add_constr: constructor -> coninfo -> (coninfo list) InternalTab.t -> (coninfo list) InternalTab.t =
	fun constr ci acc ->
	  if   InternalTab.mem ci#typid acc
	  then (* Add constructor to list *)
	    let prev = InternalTab.find ci#typid acc in
	    InternalTab.add ci#typid (ci::prev) acc
	  else (* Initalize new type to table *)
	    InternalTab.add ci#typid [ci] acc
      in InternalTab.fold add_constr constr_table InternalTab.empty
    in
    object (s)
	(* val constr_table = compile_contab tdecs - we need this table also to define type_table, and I do not understand the difference to a header-let anyway *)
	(* val type_table   = ... *)
	
      method mem:   constructor -> bool =
	fun c   -> InternalTab.mem  c constr_table

      method find:  constructor -> coninfo = 
	fun c   -> 
	  try  InternalTab.find c constr_table
	  with Not_found -> err ("Undeclared constructor '"^c^"' encountered.") 

      method fold: 'a . (constructor -> coninfo -> 'a -> 'a) -> 'a -> 'a = 
	fun f i -> InternalTab.fold f constr_table i
	    
      method get_constrs: typeidentifier -> (typcon list) =
	fun tid ->
	  try
	    let TypDec(info,param,typid,cs) = List.find (function TypDec(_,_,typid,_) -> typid = tid) tdecs  (* Efficency is ok, since it is reasonable to expect that there are only 5-10 user-defined types *)
	    in  cs
	  with Not_found -> raise (Invalid_argument ("Typeidentifier '"^tid^"' not declared."))
	  
      method get_coninfos: typeidentifier -> (coninfo list) =
	fun tid ->
	  try  InternalTab.find tid type_table
	  with Not_found -> raise (Invalid_argument ("Typeidentifier '"^tid^"' not declared."))
    end


let the_contab: contab ref = ref (new contab []) (* !IMPERATIVE! *)   

let update_the_contab: (typdec list) -> unit =   (* SHOULD BE CALLED ONCE AFTER PARSING *)
  fun tds -> let _ = the_contab := new contab tds in ()

(* Tools for User-defined Types *)

let typcon_arg_compare: typcon -> typcon -> int = (* Compares constructors by number of arguments, the more arguments means greater *)
 fun tc1 tc2 ->
   let n1 = 
     let TypCon(_,_,_,arg1) = tc1 in
     List.length arg1
   in
   let n2 = 
     let TypCon(_,_,_,arg2) = tc2 in
     List.length arg2
   in 
   (n1 - n2)

let rec select_constr: (typ list) -> (typcon list) -> typcon = (* given an (ordered) typ list, a constructor with fitting arguments is searched for --- obsolete *)
  fun typl constrl ->
    List.find
      (function 
	| TypCon(_,constr,_,arg_typs) when (Common.list_equal typl arg_typs) -> true
	| _ -> false
      )
      constrl

(* Deprecated, do we need anything like this anymore?
   let rec conTab_fold2: (constructor -> 'a -> 'b -> 'c -> 'c) -> 'a ConTab.t -> 'b ConTab.t -> 'c -> 'c =
     fun f tba tbb acc ->
       let appf = fun key reca acc -> f key reca (ConTab.find key tbb) acc in
         ConTab.fold appf tba acc
*)



(* Tools on types: *)

let rec uncurry: typ -> (typ list) = (* Unfold ArrowTyp into a list of types, omitting result type. Not really uncurrying, I know... *) 
  fun t ->
    match t.v with
    | ArrowTyp(r,d) -> r::(uncurry d)
    | _             -> []

let rec curry: (typ list) -> typ = (* Fold typlist into higher-order type for recursion, not really currying, I know *)
  function
    | []    -> raise (Invalid_argument "curry")
    | [t]   -> t
    | h::ts -> Support.fakeinfo (ArrowTyp(h,(curry ts)))

(*
let rec curry_rev: typ -> (typ list) -> typ = (* more like curry with last explicit , but 1st arg is reversed *)
  fun last l ->
    let rec cr_aux: typ -> (typ list) -> typ = 
      fun acc l ->
	match l with ->
	| []  ->
	| [t] -> Support.fakeinfo (ArrowTyp(t,acc))
	| h::ts -> cr_aux (Support.fakeinfo (ArrowTyp(h,acc))) ts
    in cr_aux last l
*)

let rec equal: typ -> typ -> bool = (* Compare does not worry about Support.info *)
  fun tya tyb ->
    match ((stripinfo tya),(stripinfo tyb)) with
    | (LinPairTyp(fa,sa),LinPairTyp(fb,sb)) 
      -> (equal fa fb) && (equal sa sb)
    | (ConTyp(pa,tida),ConTyp(pb,tidb)) when ((String.compare tida tidb)=0)
      -> 
	begin
	  try  List.fold_left2 (fun acc a b -> acc && (equal a b)) true pa pb
	  with _ -> false
	end
    | (ArrowTyp(da,ra),ArrowTyp(db,rb)) 
      -> (equal da db) && (equal ra rb)
    | (basety_a, basety_b) 
      when ((basety_a = basety_b) || (basety_a = polymorph))
      -> true
    | otherwise                                       
      -> false

(* SOME UTILITIES: *)

let rec string_of_typ: typ -> string =
  fun t ->
    match t.v with
    | UnitTyp     -> "()"
    | DiamondTyp  -> "<>"
    | TvarTyp(v)  -> ("'"^v^"'") 
    | BoolTyp     -> "bool"
    | IntTyp      -> "int"
    | FloatTyp    -> "float"
    | CharTyp     -> "char"
    | StringTyp   -> "string"
    | LinPairTyp(tya,tyb) -> ("(|"^(string_of_typ tya)^"|"^(string_of_typ tyb)^"|)")
    | ConTyp(p,id)        -> Common.string_ltrim((String.concat " " (List.map string_of_typ p))^" "^id)
    | ArrowTyp(d,r)       -> ((string_of_typ d)^" -> "^(string_of_typ r))

let to_string: typ -> string = string_of_typ (* This name might be problematic with open *)



(* Contexts --- where shall this be located? typcheck? *)
class ['t] context = 
  object (self: 'self)
    inherit ['t] lookup
    method joker       = "_"
    method key_name    = "Variable"
    method value_name  = "type"
    method lookup_name = "context"
    method error       = Support.err 
	
    method lookup_info: Support.info -> variable -> 't = 
      fun i v -> 
	try  InternalLookup.find v the_lookup
	with Not_found -> Support.errAt i (self#key_name^" '"^v^"' not found in "^self#lookup_name^".")
	    
(*  method change: variable -> 't -> 'self = DEPRECATED - use replace instead! *)
	      
  end
    
