(*
   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: syntax.ml,v $
   Id:       $Id: syntax.ml,v 1.1 2004/12/07 17:30:41 sjost Exp $ 

	
   What this File is all about:
   ----------------------------
   The TYPES & the ABSTRACT SYNTAX 

   Anything concerned with abstract syntax as well as the abstract syntax itself.


   Global values: 
    - the_contab / update_the_contab



   ToDos: 

   
*)

open Common
open Support
open Types


(* Die Operatoren haegnen nur als Blatt an einem generellen UnOP/BinOP Knoten dran, welcher auch die Operanden traegt. *)
(* Operators, unary *)
type plain_unaryoperator = (* There MUST NOT be an unary operator on constructor types! -> sharing in constraint.ml *) (* !!! The inference assumes that all built-in operators do not affect the heap, regardless of sharing! *)
  | NotOp 
  | UMinusOp
  | UFminusOp
and unaryoperator = plain_unaryoperator Support.withinfo

(* Operators, binary *)
type plain_binaryoperator = (* There MUST NOT be a binary operator on constructor types! -> sharing in constraint.ml *) (* !!! The inference assumes that all built-in operators do not affect the heap, regardless of sharing! *)
  | TimesOp 
  | DivOp 
  | PlusOp
  | MinusOp
  | FtimesOp
  | FdivOp
  | FplusOp
  | FminusOp
  | LessOp 
  | LteqOp 
  | GreaterOp
  | GteqOp
  | EqualOp
  | AppendOp
  | AndOp
  | OrOp
  | AndalsoOp       (* Sollte es gar nicht geben, da sie gleich als "if operand1 then operand2 else false" geparst werden knnten, wenn IF Ausdrcke im guard akzeptieren wrde. *)
  | OrelseOp        (* Sollte es gar nicht geben, da sie gleich als "if operand1 then true else operand2 " geparst werden knnten, wenn IF Ausdrcke im guard akzeptieren wrde. *)
  | ModOp           (* Recently added to Camelot *)
and binaryoperator = plain_binaryoperator Support.withinfo


(* Values *)
type plain_valu =
  | VarVal      of variable   (* Strictly, this should not be allowed to be a FunctionCall for 0-arity function, but... *)
  | UnitVal 
  | BoolVal     of bool       (* in Camelot wird BOOL definiert, wozu? *)
  | IntVal      of int 
  | FloatVal    of float 
  | CharVal     of char 
  | StringVal   of string 
  | UnaryOpVal  of  unaryoperator * valu
  | BinaryOpVal of binaryoperator * valu * valu
and valu = plain_valu Support.withinfo

(* A tool for the 'value-class': *)
let rec fv_valu: valu -> idset =
  fun valu -> match valu.v with
  | VarVal(x)                      -> IdSet.singleton x
  | UnaryOpVal(  _ , valu1)        -> fv_valu valu1
  | BinaryOpVal( _ , valu1, valu2) -> IdSet.union (fv_valu valu1) (fv_valu valu2)
  | _                              -> IdSet.empty
let fv_valus: valu list -> idset =  List.fold_left (fun acc valu -> IdSet.union acc (fv_valu valu)) IdSet.empty   (* == function | [] -> IdSet.empty | valu::valus -> IdSet.union (fv_valu valu) (fv_valus valus) *)  

(* Warning! Diamonds are only considered in the sandbox, they do not influence the constraint inference *)
type diamond = (* variable option - but we use a new variant type in order to avoid confusion, as we will need a (diamond option) in matchrules (only) and Some(None) seemed weird. *) 
    New                 (* this tells us that it is a @_   *)
  | Reuse of variable   (* this holds the variable of @var *)

(* Expressions *)
(* We want to versions of expressions: 
     - one containing only Support.info and a plain expression 
     - another bearing also the expressions type and its free variables

   The problem is that we want some functions like "string_of_expr" to
   work on both versions without duplicating code, but there is no
   subtyping for recordtypes (even the field names must be unique!), 
   so we must use classes and coercion instead.

   This solution still suffers one flaw: Another datatype cannot contain
   the type #expression, but must choose between expression or rich_expression only
   or must be explicitely parameterized.
*)
type +'exp plain_expression =      (* We explicitely demand that plain_expression is covariant in 'exp, but OCaml also infers this for mere variant types already on its own *) 
    ValueExp   of valu (* Include Variables, etc., but NOT Constructors! *) 
  | ConstrExp  of (constructor * (valu list) * diamond) 
  | FunExp     of (variable * (typ option) * 'exp)
  | AppExp     of (variable *  variable)
  | LetExp     of (variable * (typ option) * 'exp * 'exp) 
  | SeqExp     of ('exp * 'exp)            (* Hintereinanderausfhrung, fr die Freunde imperativer Sprachen... *)  
  | RecExp     of (variable * (typ option) * 'exp * 'exp)
  | AndExp     of (variable * (typ option) * 'exp) list * 'exp (* Mutually recursive "let rec"-expression, should make RecExp obsolete eventually (AndExp was hacked in at a later time) *)
  | IfExp      of (valu * 'exp * 'exp) (* Variable could be replaced by valu here *)
  | LinIExp    of ('exp * 'exp) 
  | LinEExp    of (bool * variable) (* Fst and Snd are not implemented as unary operators, since they are more special (more like Fun-Abstraction/Elemination) *)
  | MatchExp   of (variable * ('exp matchrule list )) 
and 'exp matchrule 
      = Matchrule of Support.info * constructor * (variable list) * (diamond option) * 'exp 
                      (* Fileinfo,  constructor, constructor_args, diamond-variable-option, action-expression *)


class expression info p_expr =  (* merely replaces the record type:  type expression = {i: Support.info; e: plain_expression}; see notes above. *)
  let freev = (* computes the free variables of an expression an construction time *)
    function
      | ValueExp(v)                         -> fv_valu v
      | ConstrExp(constr, valus, Reuse(id)) -> IdSet.add id (fv_valus valus)
      | ConstrExp(constr, valus, _)         ->              (fv_valus valus)
      | FunExp(fid, ty, fbody)              -> IdSet.remove fid (fbody#fv)
      | AppExp(fid, argid)                  -> IdSet.add fid (IdSet.singleton argid)
      | LetExp(letvar, ty, expra, exprb)    -> IdSet.union (expra#fv) (IdSet.remove letvar (exprb#fv))
      | SeqExp(expra, exprb)                -> IdSet.union (expra#fv) (exprb#fv)
      | RecExp(recvar, ty, expra, exprb)    -> IdSet.remove recvar (IdSet.union (expra#fv) (exprb#fv))
      | AndExp(rcl,expr)                    -> 
	  (* uncurry IdSet.diff *)
	  let (bound,free) =
	    let splitvar (* : (idset * idset) -> (variable * (typ option) * 'self) list -> (idset  * idset) *) =
	      fun (bound_acc,free_acc) (recvar, ty, defexp) ->
		((IdSet.add recvar bound_acc), (IdSet.union defexp#fv free_acc))
	    in List.fold_left splitvar (IdSet.empty, (expr#fv)) rcl
	  in IdSet.diff free bound
      | IfExp(ifvalu, thenexpr, elseexpr)   -> IdSet.union (fv_valu ifvalu) (IdSet.union (thenexpr#fv) (elseexpr#fv))
      | LinIExp(expra,exprb)                -> IdSet.union (expra#fv) (exprb#fv)
      | LinEExp(_, var)                     -> IdSet.singleton var
      | MatchExp(mvar, mrules)              -> 
	  let fv_mrule: 'exp matchrule -> idset = 
	    function (Matchrule(info, constr, vars, dia, mrexpr)) ->
	      let vars = 
		match dia with
		| Some(Reuse(d)) -> d::vars
		| _              -> vars
	      in List.fold_left 
		(fun acc v -> IdSet.remove v acc)
		(mrexpr#fv)
		vars
	  in List.fold_left 
	    (fun acc mr -> IdSet.union acc (fv_mrule mr))
	    (IdSet.singleton mvar) 
	    mrules
  in
  object (self: 'self)
    val v_i: Support.info       = info          (* The info where the expression is located *)
    val v_e: ('self plain_expression) = p_expr        (* The plain expression *)
    val v_fv: idset                   = freev p_expr  (* All free variables contained *)
    method i = v_i
    method e = v_e
    method fv = v_fv
	(* Now that expressions are objects, we also include some functions here by following the object-oriented faith, although ordinary functions calling them exist for compatibility with old code parts. *)
	
    method is_function = 
      match v_e with
      | FunExp(_,_,_) -> true
      | other         -> false

    method to_string: string =
      let msg = 
	match v_e with (* shall we write self#e instead of v_e ??? *)
	| ValueExp _        -> "Val"
	| ConstrExp(id,_,_) -> "Con("^id^")"
	| FunExp(id,_,_)    -> "Fun("^id^")"
	| AppExp(fid,argid) -> "App("^fid^","^argid^")"
	| LetExp(id,_,_,_)  -> "Let("^id^")"
	| SeqExp _          -> "Seq"
	| RecExp(id,_,_,_)  -> "Rec("^id^")"
	| AndExp(_,_)       -> "LetRecAnd"
	| IfExp(_,_,_)      -> "If"
	| LinIExp _         -> "Lin"
	| LinEExp(true,id)  -> "Fst("^id^")"
	| LinEExp(false,id) -> "Snd("^id^")"
	| MatchExp(id,_)    -> "Match("^id^")"
      in (string_of_info_NF v_i)^msg
  end


(* Funktionen *) 
type fundef = FunctionDef of  Support.info * funcidentifier * expression
type funblockdef = fundef list


(* Declarations *)
(* Old variant, where we had multiple functions instead of one expression:
   type valdec = 
   | ValDec    of Support.info * funcidentifier * typ 
   | AnnValDec of Support.info * funcidentifier * (float option) * rich_typ * (float option) 
*)


(* Programme *)
type program = Program of Support.info * (typdec list) * (typ) * (expression) 



(* SOME UTILITIES: *)

let string_of_expr: #expression -> string = 
  fun e -> e#to_string



