(* This code is from Benjamin Pierce, Indiana University *)

(* 
   Modified by Steffen Jost <jost@informatik.uni-muenchen.de>. 
   
   ToDo for Steffen: 
   
   - Understand how the module format is used and decide whether we should
     use it throughout or not at all.

   - I think that the different bug,err,...-functions should be rewritten 
     in a more generalised/parameterised style, allowing to easily introduce 
     new functions like cerr with ONE definition only, reducing redundancies 
     in the code! 
   
   Ok decision: All calls to err,errAt,... get changed. All they should do now is raise an exception
   so that higher programm structures might catch them, add info if they know about, alter the
   exception according to the situation.

   Therefore we define a new kind of exception, which consist of an 
     - info     (where are things happening)
     - string   (message at what stage things went wrong (parsing, etc)
     - string   (message what exactly went wrong)

*) 

open Common

(* File location info *)
type        info  = FI of string * int * int | UNKNOWN
type 'a withinfo  = {i: info; v: 'a}

(* The functions allow us to hide the record structure  ---  Not sure if we reall want to hide the record strucure *)
let unknown:info        = UNKNOWN
let create        f l c = FI(f,l,c)
let addinfo     fi valu = {i=fi; v=valu}
let fakeinfo       valu = {i=UNKNOWN; v=valu}
let fakeinfo_str s valu = {i=(FI(s,0,0)); v=valu}
let stripinfo x         = x.v 
let getinfo x           = x.i


exception ErrAt of info * string            (* Where? What?  --  e.g. (FI("file",23,5),"Parsing","Wrong token expected") *)
exception BugAt of info * string * exn      (* Something else happened, but we know where and at what stage *)

exception NotSharable of string             (* A special exception for Constraint.Rich_typ.share: shows that a type cannot be shared. We need this special exception since we intend to catch it on some occasions. *)


let exn_sep: string = " \n     "            (* A string that separates exception messages of different grain. *)


let rec string_of_info: info -> string = (* Turns info into printable string *)
  function
    | FI(f,l,c) as i ->	f^(string_of_info_NF i) 
    | UNKNOWN        -> "<Unknown file and line>: "

and string_of_info_NF: info -> string =  (* Only contains Row and Column, but no filename *)
  function
    | FI(_,l,c) -> 
	let line   = aligned_string_of_int ' ' 4 l in
	let column = aligned_string_of_int '0' 2 c in
	let out = (line^"."^column^": ") in 
	Common.string_lalign ' ' 10 out
    | UNKNOWN   ->
	("   ??.??: ")
	  
let print_info: info -> unit = (* Prints info to screen *)
  compose print_string string_of_info
    

let raise_withinfo: info -> string -> exn -> 'a =  (* adds location and stage info to an exception *)
  let add_s: string -> string -> string =
    fun s m ->
      if   (string_beginswith m s)
      then m
      else s^exn_sep^m
  in
  fun i s e->
    raise 
      begin
	match e with
	| ErrAt(UNKNOWN,m)    -> ErrAt(i,(add_s s m))
	| ErrAt(i,m)          -> ErrAt(i,(add_s s m))
	| BugAt(UNKNOWN,m,x)  -> BugAt(i,(add_s s m),x)
	| BugAt(i,m,x)        -> BugAt(i,(add_s s m),x)
	|  other              -> BugAt(i,s,other)
      end

let try_withinfo: info -> string -> ('a Lazy.t) -> 'a =   (* Convenient, but admittedly looks a bit odd. Lazy evaluation is important here. A textual compiler macro would have been what we rather wanted *)
  fun i m e -> 
    try  Lazy.force e 
    with x -> raise_withinfo i m x

let errAt: info -> string -> 'a =
  fun i s -> raise (ErrAt(i,s))

let err: string -> 'a = 
  (* errAt UNKNOWN *)
  fun s -> raise (ErrAt(UNKNOWN,s))

let bugAt: info -> string -> exn -> 'a =
  fun i s x -> raise (BugAt(i,s,x))

let bug: string -> exn -> 'a =
  (* bugAt UNKNOWN *)
  fun s x -> raise (BugAt(UNKNOWN,s,x))

let rec print_err: exn -> unit =
  let string_of_simple_exn: exn -> string =  (* We need this also for the BugAt-case, hence a separate function within the function *)
    function
      | Invalid_argument s -> ("Invalid argument '"^s^"'.")
      | Failure s          -> ("Failure '"^s^"'.")
      | Not_found          -> ("Not found.")
      | Out_of_memory      -> ("Garbage collection failed.")
      | Stack_overflow     -> ("OCaml: Stack overflow.")
      | Sys_error s        -> ("System Error: "^s^".")
      | NotSharable s      -> ("Instances of linear types cannot be shared: "^s^".")
      | other              -> (Printexc.to_string other)
  in
  fun e ->
    begin
      flush stdout;
      flush stderr;
      print_newline ();
      match e with
      | ErrAt(i,m) -> 
	  begin
	    print_newline();
	    print_info i;
	    print_string exn_sep;
	    print_string m;
	    print_newline ()
	  end
      | BugAt(i,m,x) ->
	  begin
	    print_newline();
	    print_info i;
	    print_string exn_sep;
	    print_string m;
	    print_string exn_sep;
	    print_string (string_of_simple_exn x)
	  end
      | other -> 	    
	  print_string (string_of_simple_exn other)
    end
      
      
(* WARNINGS, delayable *)
      
let warn_queue   = ref []            (* The warning message queue *)
let warn_qflag   = ref false         (* Shall we queue warning messages at all? *)
let last_warning = ref ""            (* Always remembers last warning message *)
    
let forcewarningAt: info -> string -> unit =
  fun i s -> 
    if (s <> !last_warning)  
    then
      begin
	last_warning := s;
	(if i <> UNKNOWN then print_info i); 
	print_string "\n WARNING: ";
	print_string s; 
	print_newline()
      end
	  
let warningAt: info -> string -> unit = 
  fun i s -> 
    if   !warn_qflag
    then warn_queue := ((i,s)::(!warn_queue))
    else forcewarningAt i s

let warning: string -> unit = warningAt UNKNOWN
    
let flush_warnings: unit -> unit = 
  fun () ->
    let rec flush_warnings_aux =
      function
	| [] -> ()
	| (i,s)::t ->
	    begin
	      forcewarningAt i s;
	      flush_warnings_aux t
	    end
    in
    begin
      if !warn_queue <> [] 
      then 
	begin
	  print_string "\nSOME WARNINGS HAVE BEEN DELAYED:\n";
	    flush_warnings_aux (List.rev !warn_queue);
	  warn_queue := []
	end
    end
      
let queue_warnings: unit -> unit = (* All subsequent calls to "warning" are delayed now *)
  fun () ->  (warn_qflag := true)
let print_warnings: unit -> unit = (* Queued warnings are printed, subsequent calls to "warning" are printed immediately *)
  fun () -> ((warn_qflag := false); flush_warnings ())
      


