(*

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: memory.ml,v $
   Id:       $Id: memory.ml,v 1.1 2004/12/07 17:30:41 sjost Exp $ 

	

   What this File is all about: 
   ---------------------------- 
   This models encodes the memory model (i.e. the SIZE functions),
   Operational Values and the implementation of Stack & Heap.

   ToDos: 

   - Write a .mli file in order to hide some stuff and export
     only the things that are really needed somewhere else

   Notes: One should use monads to fasciliate the passing of stack and
   heap within exec.ml, but monads are awkward to use in OCaml. OCamls
   way to deal with such things are objects, which should have
   replaced the whole module buisness, as the allow the extension of
   classes, which modules do not offer. Yet changing the whole thing
   seemed to costly, therefore we stick to the modules, except for the heap.

*)


open Common
open Support
open Types
open Syntax
open Argument


(* * * Type Definitions * * *)


type location    = int 
let  loccomp x y = x - y 
let  null: location = 0
let  string_of_loc:         location -> string = fun l -> ((aligned_string_of_int '0' 2 l)^"*")
let  print_location:        location -> unit   = fun l -> print_string (string_of_loc l)

type rvalue   = (* Operational Values, Stack-Only *) 
  | IntRVal of int 
  | FloatRVal of float 
  | CharRVal of char 
  | StringRVal of string 
  | BoolRVal of bool 
  | UnitRVal 
  | PointerRVal of location
  (* Note that these differ from the paper-work: 
       We do not support pairs and sums as stack allocated
       values. They must be defined as user-types and hence require
       heap space, although the user may specify their size manually
       as zero in order to match the paper work. 
   *)

let print_rvalue: rvalue -> unit =       (* Flat printing, i.e. non-heap dependent *)
  function 
    | IntRVal(x)    -> print_int x
    | FloatRVal(x)  -> print_float x
    | CharRVal(x)   -> print_char x
    | StringRVal(x) -> 
	let out =
	  if !the_options.debug 
	  then ("\""^x^"\"") (* Explicit strings *)
	  else string_captured x
	in print_string out
    | BoolRVal(true)   -> print_string "True"
    | BoolRVal(false)  -> print_string "False"
    | UnitRVal         -> print_string "()"
    | PointerRVal(loc) -> print_location loc


(* * * Stack * * *)    
class stack =
  object (self: 'self)
    inherit [rvalue] lookup
    method joker       = "_"
    method key_name    = "Variable"
    method value_name  = "Value"
    method lookup_name = "stack"
    method error       = Support.err 
    method lookup_info = 
      fun i v -> 
	if   self#mem v 
	then self#lookup v
	else errAt i (self#key_name^" '"^v^"' not found in "^self#lookup_name^". ")

  end

type heapcell =  (* Operational Values, Heap-Only *)
    Constructor of constructor * (rvalue list)
  | Closure     of expression  * stack 
	

(* * * Memory Model - theSIZE * * *)

module Size =  (* If option -uni is to be revived, this is the place to go... *)
  struct
    let typ: typ -> int =          (* This is the SIZE function as in paper for ART-types *)
      (
       fun t -> 
	 match t.v with 
	 | UnitTyp             -> 1
	 | DiamondTyp          -> Support.warning "Size of Diamond Type should not be required, but is. Size currently set to zero."; 
	     0 (* ? *) 
	 | TvarTyp(tvar)       -> Support.err ("Resource inference for type-variables ('"^tvar^"') not implemented. See documentation.\n")
	 | BoolTyp             -> 1
	 | IntTyp              -> 1
	 | FloatTyp            -> 1
	 | CharTyp             -> 1
	 | StringTyp           -> Support.warning "Size of String Type should not be needed, since all strings are assumed to occupy no heap-space (0). See documentation.\n";
             0 (* ? *)
	 | LinPairTyp _ (*(tya,tyb)*)
	 | ConTyp     _ (*(pars,id)*)     
	 | ArrowTyp   _ (*(dot,rat)*) ->  
             1  (* The three above are merely pointers and hence occupy themself only one heap cell *)  
      )      

    let rvalue:  rvalue -> int  = (* Operational size, as in paper *)
      let deduce_type: rvalue -> typ  = (* Deduce (or guess) a type of correct size from an rvalue *)
	function
	  | IntRVal(x)     -> fakeinfo IntTyp
	  | FloatRVal(x)   -> fakeinfo FloatTyp
	  | CharRVal(x)    -> fakeinfo CharTyp
	  | StringRVal(x)  -> fakeinfo StringTyp
	  | BoolRVal(x)    -> fakeinfo BoolTyp
	  | UnitRVal       -> fakeinfo UnitTyp
	  | PointerRVal(x) -> fakeinfo (ConTyp([],"Unknown")) (* This is just a pointer. *)
      in compose typ deduce_type 
	
    let rec heapcell: heapcell -> int =
      fun hc ->
	if   !the_options.uniform 
	then 1 
	else match hc with
	| Constructor(constr, cargs) ->
	    coninfo (!the_contab#find constr)
	| Closure(_, s) ->
	    1 + (s#fold (fun var valu c -> c + (rvalue valu)) 0)
	      
    and coninfo: <size:int; .. > -> int = (* Returns user-specified size for a constructor *)
      fun ci -> 	
	if   !the_options.uniform 
	then 1 
	else ci#size
  end



(* * * Heap * * *)

module InternalHeap  = Map.Make(struct type t = location let compare = loccomp end)  

class heap = 
  object (self: 'self)
      (* The heap itself --- a functional object! 
	 The functional object is a bit awkward to implement and use in the absence of monads, but it is much closer to the paperwork.
	 Note that even methods like read have an effect on statistics and thus return a heap object.
       *)
    val  the_heap: heapcell InternalHeap.t = InternalHeap.empty 
    val  curr_loc: int = 0              (* Counter holding the highest heapcell address in use. This counter never decreases, i.e. a removed cell goes to the freelist or is abandoned. *) 
    val  freelist: (location list) = [] (* Freelist, containing previousliy withdrawn cells *)
      (* statistics *)
    val  num_heap_store:   int = 0  (* Number of store operations *)
    val  num_heap_read:    int = 0  (* Number of read operations *)
    val  num_heap_remove:  int = 0  (* Number of remove operations *)
    val  usrsized_cur: int =     0  (* Counting according to user (declare size via '(*#*)') in typedeclaration. Set to one if unspecified. *)
    val  usrsized_max: int =     0  (* Maximum according to user sizes *)
    val  usrsized_ini: int =     0  (* Inital amount of allocated heap in user size *)
    val  unisized_ini: int =     0  (* Inital amount of allocated heap objects *)

      (* Access to heap statistics *)
    method size_num_store: int  = num_heap_store 
    method size_num_read: int   = num_heap_read 
    method size_num_remove: int = num_heap_remove 
    method size_uni_cur: int = InternalHeap.fold (fun l c n -> n+1) the_heap 0 (* Maybe replace by another counter for more efficiency *)
    method size_uni_ini: int = unisized_ini
    method size_usr_cur: int = usrsized_cur
    method size_usr_max: int = usrsized_max
    method size_usr_ini: int = usrsized_ini
	
    method reset_statistics: 'self =
      let cur = InternalHeap.fold (fun l c a -> a + (Size.heapcell c)) the_heap 0 in
      {< 
         num_heap_store  = 0;
         num_heap_read   = 0;
         num_heap_remove = 0;
         usrsized_cur    = cur;
         usrsized_max    = cur;
         usrsized_ini    = cur;
	 unisized_ini    = self#size_uni_cur
      >}
	  
	  
      (* Location/Freelist management. Removed locations must be manually added to the freelist. New_loc uses them if appropriate. *)

    method new_loc: 'self * location =  
      match (!the_options.inplace, freelist) with (* Freelist is ignored if option -nipl is set *)
      | (true, (l::ls)) -> ({< freelist = ls >}, l)  
      | _ -> 
	  try 
	    let l = save_increment curr_loc 
	    in ({< curr_loc = l >}, l)
          with _ ->  bug ("While incrementing new heap pointer:") (Failure "Overflow detected.")
	      
    method reuse_loc: location -> 'self =  (* Adds location to freelist. *)
      fun l ->
	if  (InternalHeap.mem l the_heap) 
	then err ("Location "^(string_of_loc l)^" to be added to freelist is still allocated.")
	else 
	  if (List.mem l freelist)
	  then err ("Location "^(string_of_loc l)^" to be added to freelist is already contained in freelist.")
	  else
	    if (not !the_options.inplace) 
	    then self (* No warning: the option shall alter the behaviour within this class only and nowhere else. Hence outside calls do not know this and shall call this method without producing a warning message. *)
	    else {< freelist = (l::freelist) >}
		
    (* Heap Access --- Only these functions are publicly available to manipulate a heap. *)

    method store: heapcell -> 'self * location  = (* Storing at arbitrary address - returning type result seemed to special, so use in conjunction with function "to_result" if necessary *)
      fun c ->
	let ( h,               l) = self#new_loc 
	in  ((h#store_at c l), l)
	  
    method store_at: heapcell -> location -> 'self = (* Storing in heap at a given address *)
      fun c l ->
	if  (InternalHeap.mem l the_heap) 
	then err "Reallocation of undisposed heap-cell detected. Check for duplicated '@' annotations!"
	else 
	  begin
	    let hcsz = Size.heapcell c in
	    let _ =
	      if !the_options.debug then 
		begin
		  (print_string ("\n [Allocating   address "^(string_of_loc l)
 				 ^"  ---  UsrSize:"^(signed_aligned_string_of_int ' ' 4 hcsz) ^ "  ---  "));
		  (self#print_heapcell c);
		  (print_string ("] "))
		end
	    in ({<
		  the_heap       = InternalHeap.add l c the_heap;	
		  num_heap_store = num_heap_store + 1;
		  usrsized_cur   = usrsized_cur + hcsz;
		  usrsized_max   = max usrsized_max (usrsized_cur + hcsz)
	       >})
	  end

    method read:              location -> 'self * heapcell = self#read_info unknown
    method read_info: info -> location -> 'self * heapcell = (* READ-ONLY access *)
      fun i l -> 
	try  ({< num_heap_read = (num_heap_read + 1) >} , (InternalHeap.find l the_heap))
	with Not_found -> errAt i ("Dangling pointer encountered: "^(string_of_loc l))
	    
    method remove: location -> 'self =  (* Destructs heapcell, NOT added to freelist, no reading *)
      fun l ->
	let hcsz =
	  let hc = 
	    try  (InternalHeap.find l the_heap) 
	    with Not_found -> err ("Unable to free location "^(string_of_loc l)^" - already deallocated.") 
	  in Size.heapcell hc 
	in
	let _ = 
	  if   !the_options.debug 
	  then ( print_string ("\n [Deallocating address "^(string_of_loc l)
			       ^"  ---  UsrSize:"^(signed_aligned_string_of_int ' ' 4 (-hcsz))^"] ")) 
	in {< 
	      the_heap = InternalHeap.remove l the_heap;
	      num_heap_remove = num_heap_remove + 1;
	      usrsized_cur    = usrsized_cur - hcsz;
 	   >}
	  
    method withdraw:              location -> 'self * heapcell = self#withdraw_info unknown
    method withdraw_info: info -> location -> 'self * heapcell = (* DESTRUCTIVE READ access, location ADDED to freelist, counts as read *)
      fun i l -> 
	let h1,c = self#read_info i l in (* Hence counts as a read operation *)
	let h2   =   h1#remove l      in (* Remove from heap *)
        let h3   =   h2#reuse_loc l   in (* Add l to freelist *)
	(h3,c)

   (* Printing of heap allocated rvalues and heapcells  -  all without effect on the heap object *)

    method print_rvalue: rvalue -> unit =
      function 
	| PointerRVal(loc) -> self#print_location loc
	| rv               -> print_rvalue rv     (* calls the simple, flat print_rvalue *)
	      

   (* ALL THESE PRINTING ROUTINES NEVER COUNT AS HEAP-READ ACCESSES  -  They are meant for debugging purposes *)
    method print_rvalues: rvalue list -> unit =  
      function
	| []    -> ()
	| x::xs -> 
	    self#print_rvalue x; 
	    List.iter (fun x -> (print_string ","; self#print_rvalue x)) xs

	      (*  (*Alternate version:*)
		 method print_rvalues: rvalue list -> unit =
		   let f = fun a rv -> 
		     let _ = if a then print_string "," in
		     let _ = self#print_rvalue rv in true
		   in List.fold_left f false
	       *)

    method print_location: location -> unit = 
      fun loc ->
	let _  = if !the_options.debug then (print_location loc) in     (* calls the simple print_location *) 
	let hc =  (* This read should not count! *)
	  try  InternalHeap.find loc the_heap 
	  with Not_found -> err ("Dangling pointer encountered : "^(string_of_loc loc))
	in self#print_heapcell hc
	  
    method print_heapcell: heapcell -> unit = 
      function 
	| Constructor(cname,rvals) -> 
	    begin
	      print_string cname;
	      print_char '(';
	      self#print_rvalues rvals;
	      print_string ")"
	    end
	| Closure(expr,stack)  -> (* Variant: Full stack printed *)
	    let _ = print_string ("<"^(string_trim(string_of_expr expr))) in
	    let _ = 
	      let f = 
		fun var bind acc -> 
		  let _ = print_string ", "        in 
		  let _ = print_string (var^":")   in
		  let _ = print_rvalue bind in ()     (* calls the simple, flat print_rvalue *)
	      in stack#fold f ()
	    in
	    let _ = print_string ">" in ()
	      
  end


type result = heap * rvalue (* A value, possible pointing into the heap *)
      
let to_result: heap * location -> result = (* To use in conjunction with heap#store *) 
  fun (h,l) -> (h, PointerRVal(l))


(* Probably deprecated code *)

let default_rvalue: heap -> typ -> result =
  fun h t ->
    match t.v with
    | UnitTyp   -> (h, (UnitRVal))
    | BoolTyp   -> (h, (BoolRVal(false)))
    | IntTyp    -> (h, (IntRVal(0) ))
    | FloatTyp  -> (h, (FloatRVal(0.0))) 
    | CharTyp   -> (h, (CharRVal(' ')))
    | StringTyp -> (h, (StringRVal("")))
    | ConTyp([],tid) -> 
	let constrs = !the_contab#get_constrs tid in (* get all constructor for this type *)
	let TypCon(info, constr, csize, arg_typs) =  (* select a Nil-Constructor *)
	  try  select_constr [] constrs 
	  with Not_found -> err ("Unable to generate default value for type '"^tid^"': Nil-like constructor is missing.")
	in to_result(h#store (Constructor(constr, [])))
    | _ -> err ("Unable to generate default value for type "^(string_of_typ t))
	  
	  
