(**************************************************************************************

   Author:   Steffen Jost  <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: inference.ml,v $
   Id:       $Id: inference.ml,v 1.1 2004/12/07 17:30:41 sjost Exp $ 

   This module constructs the LP, 
   feeds it to an LP-solver,
   and prints the resulting annotated type for each function.


   ToDos: 
   ------

   - Solution from lp_solve is not verified. (Since a simple recalculation may suffer the same numeric mistakes.)

   - Mutual recursive datatypes lead to non-termination. (Due to the stupid enrichment algorithm - may be fixed if necessary.)

   - Support parametric/polymorphic variant types. 
     (camelot -a2 eliminates those, so this is of low priority)


   Notes:
   ------

   - Performance may be considerably increased by:
      - Eliminate quadratic runtime in printing routines 
        (do not remove duplicates, do not sort), 
        and call this routine only once instead of twice (once for the file, once for piping to the lp_solver),
        i.e. feed the file to the lp_solver instead of piping directly.
       
      - Removing all queries of the variable global.options.debug 


**************************************************************************************)


open Common
open Support
open Types
open Syntax
open Builtin
open Argument
open Constraints
open Typcheck


let poly_acc: (variable * rich_typ) list ref = ref [] (* In order to get a more informative screen print about the lp, we remember all polytypes in this list. Yes, this is a bad hack, but OCaml has no monads and I'm to weary to put that collector in everywhere *)


let lp_solve: ((constr list) * idset) -> objective -> solution =  (* Forks to lp_solve, solves lp and returns solution - hopefully *)
  let args = Array.make 2 "-S2" in                                            (* Array.make is only sufficient if we need to pass a single argument alone. *)
  fun lp_cvs obj ->
    let (lpout,lpin) = Common.pipe2prog Argument.lp_solver args in            (* Generate unix pipes to newly forked lp_solve subprocess *)
    let  _  = Constrs.to_file  lp_cvs obj "" lpout in                     (* Write constraints to lp_solver via pipe now! *)
    let  _  = try close_out lpout with _ -> err "Closing lpout failed!" in
    let sol = file_to_solution lpin in                                           (* Read  solution  from lp_solver via pipe now! *)
    let  _  = try close_in  lpin  with _ -> err "Closing lpin failed!" in
    sol

let rec iprogram: program -> (out_channel * string) -> unit =
  let scrw = !the_options.screen_width - 18 in
  fun (Program(info, typdecs, maintyp, mainexpr)) (outch,oname) ->
    let runtime    = new timer   in                                            (* A timer for runtime measurement *)
    let typed_main_expr =                                                      (* Add type information *)
      let  _  = (print_raligned_string '.' scrw " Reconstructing type derivation.."); flush stdout in
      let  _  = runtime#reset in
      let tme = Typcheck.expression (new context) mainexpr in
      let  _  = print_string ("..finished. ") in
      let  _  = runtime#print_pretty in
      let  _  = (* Check against valdec *)
	if not (!the_options.performance || (equal maintyp tme#t))
	then
	  let _ = warning ("Ignoring val declaration for inference: \n Type '"^(Types.to_string tme#t)^"' mismatches declaration '"^(Types.to_string maintyp)^"'.\n") in
	  if   !the_options.debug && !the_options.diamond
	  then begin print_newline(); tme#pretty_print end
      in
      let  _  = print_newline() in tme
    in

    let din                 = Cvar.generate Cvar.Lhs in
    let dout                = Cvar.generate Cvar.Rhs in
    let rt: rich_typ        = Rich_typ.from_typ(typed_main_expr#t) in               (* Generate enriched VAL of main expression *)
    let main_obj: objective =
      let (lhs,rhs) = Rich_typ.cvar_signature rt in
      (IdSet.singleton(din),lhs,rhs,IdSet.singleton(dout))
    in

    let lp: (constr list) =                                                         (* Construct constraints *)
      let  _  = (print_raligned_string '.' scrw " Constructing ARTHUR type constraints.."); flush stdout in
      let  _  = runtime#reset in
      let raw = iexpression (new rich_context) din typed_main_expr rt dout in       (* <- *) 
      let  _  = print_string ("..finished. ") in
      let  _  = runtime#print_pretty in
      let  _  = print_newline() in raw
    in

    let ((lp,cvs), ((num_lp, num_triv, num_cvs) as lp_stat)) = Constrs.stats lp main_obj in  (* simplify, sort and count*)

    let  _  =                                                                       (* Protocol constraints to file... *)
      if  (not !the_options.performance) && (oname <> "stdout")
      then 
	let  _  = print_string (" Generated "^num_lp^" constraints ") in
	let  _  = if num_triv <> "0" then print_string ("(of which "^num_triv^" are trivial) ") in
	let  _  = print_string ("over "^num_cvs^" variables.\n") in
	let  _  = print_raligned_string '.' scrw (" Writing constraints to file '"^oname^"'.."); flush stdout in
	let  _  = Constrs.to_file (lp,cvs) main_obj (Constrs.file_header oname (Rich_typ.to_string rt) lp_stat) outch in  (* Write constraints to file oname. *)
	let  _  = print_string ("..finished.\n") in ()
    in
    let _ =                                                                         (* Call lp_solver via pipe and print solution...*)
      if !the_options.solvelp && (lp <> [])
      then
       let  _  = try flush stdout with _ -> () in                              (* Before we pipe around, lets see what happened so far... *)
       let  _  = print_raligned_string '.' scrw (" Calling '"^Argument.lp_solver^"' for main expression via pipe.."); flush stdout in
       let  _  = runtime#reset in
       let sol = lp_solve (lp,cvs) main_obj in
       let  _  = print_string ("..finished. ") in
       let  _  = runtime#print_pretty in
       let  _  = print_newline() in
       let _ =  (* Verify that all constraints are indeed satisfied... *)
	 () (* Not sure what to do here, as me might suffer from the same numerical instabilities *)
       in
       
       let _ = (* things to be done in case of a non-empty solution *)
	 if sol#is_empty 
	 then 	  
	   print_string "\n     ---   LP is infeasible   ---    \n\n"
	 else
	 let _ = (* Write solved constraints to file...*)
	   if  (!the_options.performance) || (oname = "stdout") || (oname = "") || (oname = "_") or (oname = "@") then () else
	   let oname_sol = oname^Argument.ext_solutions in
	   try 
	     let  _        = print_raligned_string '.' scrw (" Writing solved lp to file '"^oname_sol^"'.."); flush stdout in	 
	     let solch     = (open_out oname_sol) in 
	     let  _        = solution_to_file sol (lp,cvs) main_obj (Constrs.file_header oname (Rich_typ.to_string_sol sol rt) lp_stat) solch in  (* Write constraints to file oname. *)
	     let  _        = close_out solch in
	     let  _        = print_string ("..finished.\n") in ()
	   with Sys_error _ as x -> bug ("Can't open file '"^oname_sol^"' for writing solved lp.") x 
	 in

	 let _ =  (* Search for memory leaks... *)
	   if !the_options.performance then () else
	   let strict_cnstrs = Constrs.filter_strict lp sol in
	   if strict_cnstrs <> [] 
	   then
	     let _ = print_string " Memory leaks detected in the following branches of computation: \n" in
	     List.iter (fun c -> print_string ("  "^(Constrs.to_string c))) strict_cnstrs
	   else
	     print_string " No memory leaks detected with respect to this particular solution of the lp! \n" 
	 in ()
       in
       
       let _ = (* print all poly types *)
	 if !the_options.performance then () else
	 let _ = print_string (" Arbitrary instances of polymorphic types of let/letrec variables\n  (calling lp solver again individually): \n") in
	 let print_poly: (variable * rich_typ) -> unit =
	   fun (letvar, rt) -> 
	     match rt with
	     | RPolyTyp(alpha, phi, poly_rt) -> 
		 let _ = print_string ((string_rl_align ' ' 16 12 letvar)^": ") in
		 let _ = flush stdout in 
		 let poly_obj =
		   let (lhs,rhs) = Rich_typ.cvar_signature rt in (* Rich_typ.cvar_signature ignores all variables in alpha! *)
		   (alpha, lhs, rhs, IdSet.empty)
		 in
		 let ((phi,poly_cvs), poly_lp_stat) = Constrs.stats phi poly_obj in
		 let (phi_bounded, sol_consts) = 
		   let set_to_sol: cvar -> ((constr list) * idset) -> ((constr list) * idset) = (* Restricts a variable to a solution if known *)
		     fun cv (constr_acc, consts_acc) ->
		       if   sol#mem cv
		       then (* Restrict to the value of that particular solution *)
			 let const = Constant.generate(Some(sol#lookup cv)) in (* The solution is a float, but constraint coefficients must be integers, hence we must generate a new konstant *)
			 (
			  ((Constrs.eq "Poly" (1,cv) [(1,const)]) 
			   :: constr_acc),
			  (IdSet.add const consts_acc)
			 )
		       else (* We dont know much about this particular cvar, its not in alpha neither in the solution... *)
			 (constr_acc, consts_acc)
		   in IdSet.fold set_to_sol (IdSet.diff poly_cvs alpha) (phi,IdSet.empty) (* All non-polymorphic cvars are bound here to the value of the overall solution. *)
		 in
                 let poly_sol = lp_solve (phi_bounded,(IdSet.union sol_consts poly_cvs)) poly_obj in
		 (*	 let _ = Constrs.to_file (phi_bounded,poly_cvs) poly_obj ("DEBUG PRINT\n"^(Rich_typ.to_string poly_rt)^"\n"^(idset_to_string alpha)) (open_out "debug.debug") in (* This line was used for DEBUGGING only! *) *)
		 let _ = Rich_typ.print_sol 19 poly_sol poly_rt in
		 let _ = print_string (";\n") in 
		 ()
	     | other -> ()   (* Should not occur, but does not hurt anyway *)
	 in  List.iter print_poly (!poly_acc)
       in

       let _ =  (* Print enriched instantiated signature to screen... *)
	 let _ = print_string ("\n ARTHUR Type of main expression: \n    ") in
	 let _ = print_string (sol#print din) in
	 let _ = print_string (", ") in
	 let _ = Rich_typ.print_sol 10 sol rt in
	 let _ = print_string (", ") in
	 let _ = print_string (sol#print dout) in
	 let _ = print_string ("\n") in ()
       in ()
    in  ()
      

and iexpression: #rich_context -> cvar -> typed_expression -> rich_typ -> cvar -> (constr list) =
  fun ctxt din expr rt dout ->
    let _ = 
      if !the_options.debug
      then 
	let _ = print_newline () in
	let _ = print_string (string_ralign ' ' 25 ((expr#to_string))) in
        let _ = print_string (": "^din^", ") in
	let _ = Rich_typ.print 32 rt in
        let _ = print_string (", "^dout) in ()
    in
    let linetag: string -> string = Constrs.linetag expr#i in
    try_withinfo expr#i "Building constraints failed:" 
      begin lazy 
	  begin match (expr#e , rt) with
	    (* Important: The context may contain polymoprh types, so whenever we look up something in the context, we must Rich_typ.restrict it as well.
  	                  Can we otherwise assume that rt is never a polymorphic type? At least polymorphics cannot be contained within rt!
	     *)
	  | (ValueExp(vl), _ ) 
	    -> 
	      (Constrs.dominate (linetag "Val") din dout) ::  
	      (List.map (Constrs.complete_linetag expr#i) (ivalu ctxt vl rt))
		
	  | (ConstrExp(constr, args, dia), RConTyp(params, tid, rtct))
	    -> 
	      (* We know that the program typchecks, hence we ignore dia here, as these annotations do not affect the inference anyway *)
	      if (params = []) && (rtct#mem constr)
	      then
		let rtci = rtct#lookup constr in
		let args_constrs = List.map (Constrs.complete_linetag expr#i)
		    begin
		      try  iargs ctxt args (Rich_typ.unfold_args (tid,rt) rtci#arg_typs)
		      with 
		      | (Invalid_argument "iargs: not enough arguments") -> err ("Not enough arguments for constructor '"^constr^"'.")
		      | x -> bugAt expr#i ("Problem with constructor '"^constr^"': ") x 
		    end
		in
   		let base_constr = Constrs.gt_eq (linetag "Con") (1, din) [(1, rtci#cvar); (rtci#size, Constant.one); (1, dout)] in
		base_constr :: args_constrs
	      else err ("Constructor '"^constr^"' does not fit type '"^tid^"'.")

	  | (AppExp(funv, farg), _ ) 
	    ->
	      (* Naming conventions from paper to implementation:
		   arg_rt = B  -> from farg
		     f_rt = D  -> from funv
		     fdom = A     *freshly generated*
		     fdin = m     *freshly generated*
		    fdout = m'    *freshly generated*
  	             frng = C   = rt
	       *)
	      let fv_rt   =
		if   ctxt#mem funv 
		then ctxt#lookup funv (* User-defined function *)
		else 
		  if   Builtin.is_function funv
		  then Rich_typ.of_builtin funv (* Built-in function *)
		  else (* Error *)
		    err ("Applying function '"^funv^"' to argument '"^farg^"' not possible: function unknown.")
	      in
	      let arg_rt = ctxt#lookup farg in                     
	      let fdom   = Rich_typ.from_typ (Rich_typ.to_typ arg_rt) in  (* A most general rich type of arg_rt... *)
	      let fdin   = Cvar.generate Cvar.Lhs in
	      let fdout  = Cvar.generate Cvar.Rhs in
	      let frng   = rt in
	      let arg_constrs = Rich_typ.restrict(arg_rt, fdom) in                             (* \restrict{B}{A}                    -  see paper for meaning *)
	      let fun_constrs = Rich_typ.restrict(fv_rt, RArrowTyp(fdom,fdin,fdout,frng)) in    (* \restrict{D}{A --fdin-fdout--> C}  -  see paper for meaning *)
	      let app0_constr = Constrs.dominate (linetag "Ap0")    din      fdin  in
	      let app1_constr = Constrs.gt_eq    (linetag "App") (1,din) [(1,fdin); (-1,fdout); (1,dout)] in
	      app0_constr :: app1_constr ::(list_fast_append arg_constrs fun_constrs)

	  | (FunExp(absvar, absv_tyo, funexpr), RArrowTyp(fdom,fdin,fdout,frng)) 
	    -> 
	      errAt expr#i ("Syntax error: Function definition 'fun("^absvar^") -> (..)' is not preceded by let/letrec")
		(* Old code, before merging let/letrec with fun:
		   let fun_ctxt = 
		   try  ctxt#restrict_to_nonlinear expr#fv  (* Since the size of the context is important, we restrict to the free variables as well *)
		   with (Invalid_argument s) -> errAt expr#i ("Function '"^absv^"-> (..)': "^s) (* Probably a linear variable encountered, which is not allowed for function contexts. *)
		   in
		   let fun_constrs =
		   let _ = assert (not ((Rich_typ.is_polymorphic fdom) || (Rich_typ.is_polymorphic frng))) in
                   (* errAt expr#i ("Function '"^absv^"-> (..)': Domain or range is of polymorphic type. Must be specialized first, probably a bug!") *)
		   (* This should never happen, but it doesnt hurt to check. See if we can simplify the paper for this rule then!*)
		   iexpression (fun_ctxt#bind absv fdom) fdin funexpr frng fdout
		   in (Constrs.gt_eq (linetag "Fun") (1,din) [(fun_ctxt#size,Constant.one); ((Rich_typ.size rt),Constant.one); (1,dout)]) :: fun_constrs
		 *)																

	  | (LetExp(letv, _ , letexpr, inexpr), _ ) 
	    -> 
	      begin match letexpr#e with 
	      | FunExp(absv, absv_tyo, bodyexpr) ->                   (* LetFun-Rule *)
		  let dlet    = Cvar.generate Cvar.Let in                (* This is n' in the paper *)
		  let fdin    = Cvar.generate Cvar.Lhs in                (* This is m  in the paper *)
		  let fdout   = Cvar.generate Cvar.Rhs in                (* This is m' in the paper *)
		  let fdom_rt =                                          (* This is A  in the paper *)
		    match absv_tyo with
		    | None     -> errAt expr#i ("Function definition without type declaration for '"^absv^"'.")
		    | Some(ty) -> Rich_typ.from_typ ty
		  in
		  let frng_rt = Rich_typ.from_typ bodyexpr#t in          (* This is B in the paper *)
		  let fun_rt  = RArrowTyp(fdom_rt,fdin,fdout,frng_rt) in (* This is D in the paper *)
		  let delta   = 
		    try  ctxt#restrict_to_nonlinear letexpr#fv  (* Since the size of the context is important, we restrict to the free variables as well *)
		    with (Invalid_argument s) -> errAt expr#i ("Function '"^absv^"-> (..)': "^s) (* Probably a linear variable encountered, which is not allowed for function contexts. *)
		  in 
		  let delta_xA= (delta#bind absv fdom_rt)             in (* We dont have to pay for xA anyway - hence we dont check whether it is contained in bodyexpr *)
		  let psi     = iexpression delta_xA    fdin bodyexpr frng_rt fdout in
		  let _       = debug_string ("\n   "^(String.concat "   " (List.map Constrs.to_string psi))) in
		  let alpha   = IdSet.diff (Cvar.filter_constants (IdSet.union (Rich_typ.free_cvars fun_rt)(Constrs.free_cvars psi))) (delta#fv_rts) in
		  let polytyp = RPolyTyp(alpha,psi,fun_rt) in
		  let   _     = poly_acc := (letv, polytyp) :: !poly_acc in      (* Remember this polytyp for later printing *)
		  ((Constrs.gt_eq (linetag "Fun") (1,din) [(delta#size,Constant.one); ((Rich_typ.size polytyp),Constant.one); (1,dlet)]) (* We dont have to pay for xA, hence delta sufficient *)
		   :: (iexpression (ctxt#bind letv polytyp) dlet inexpr rt dout)
		  )
	      | _ ->                                                  (* Ordinary Let-Rule *)
		  let dlet = Cvar.generate Cvar.Let in
		  let (ctxt_let, ctxt_in, ctxt_share_constrs) = ctxt#split (letexpr#fv, inexpr#fv) in
		  let let_rt = Rich_typ.from_typ letexpr#t in
		  let let_constrs = iexpression  ctxt_let                  din  letexpr let_rt dlet in
		  let in_constrs  = iexpression (ctxt_in#bind letv let_rt) dlet inexpr      rt dout in
		  list_fast_append ctxt_share_constrs (list_fast_append let_constrs in_constrs)
	      end
		
          | (SeqExp(letexpr, inexpr), _ ) 
            ->    (* This is almost identical to the ordinary let-clause *)
	      let dlet = Cvar.generate Cvar.Let in
	      let (ctxt_let, ctxt_in, ctxt_share_constrs) = ctxt#split (letexpr#fv, inexpr#fv) in
	      let let_rt = Rich_typ.from_typ letexpr#t in
	      let let_constrs = iexpression  ctxt_let                  din  letexpr let_rt dlet in
	      let in_constrs  = iexpression  ctxt_in                   dlet inexpr      rt dout in
	      list_fast_append ctxt_share_constrs (list_fast_append let_constrs in_constrs)

	  | (RecExp(recv, recv_tyo, recexpr, inexpr), _ ) 
            ->
	      begin match recexpr#e with 
	      | FunExp(absv, absv_tyo, bodyexpr) ->                   (* LetFun-Rule *)
		  let drec    = Cvar.generate Cvar.Rec in                (* This is n' in the paper *)
		  let fdin    = Cvar.generate Cvar.Lhs in                (* This is m  in the paper *)
		  let fdout   = Cvar.generate Cvar.Rhs in                (* This is m' in the paper *)
		  let fdom_rt =                                          (* This is A  in the paper *)
		    match absv_tyo with
		    | None     -> errAt expr#i ("Function definition without type declaration for '"^absv^"'.")
		    | Some(ty) -> Rich_typ.from_typ ty
		  in
		  let frng_rt = Rich_typ.from_typ bodyexpr#t in          (* This is B in the paper *)
		  let fun_rt  = RArrowTyp(fdom_rt,fdin,fdout,frng_rt) in (* This is D in the paper *)
		  let delta   = 
		    try  ctxt#restrict_to_nonlinear recexpr#fv  (* Since the size of the context is important, we restrict to the free variables as well *)
		    with (Invalid_argument s) -> errAt expr#i ("Function '"^absv^"-> (..)': "^s) (* Probably a linear variable encountered, which is not allowed for function contexts. *)
		  in 
		  let psi     = iexpression ((delta#bind absv fdom_rt)#bind recv fun_rt) fdin bodyexpr frng_rt fdout in (* ctxt = delta, x:A, y:D *)
		  let _       = debug_string ("\n   "^(String.concat "   " (List.map Constrs.to_string psi))) in
		  let alpha   = IdSet.diff (Cvar.filter_constants(IdSet.union (Rich_typ.free_cvars fun_rt)(Constrs.free_cvars psi))) (delta#fv_rts) in
		  let polytyp = RPolyTyp(alpha,psi,fun_rt) in 
		  let   _     = poly_acc := (recv, polytyp) :: !poly_acc in      (* Remember this polytyp for later printing *)
		  (Constrs.gt_eq (linetag "Rec") (1,din) [(delta#size,Constant.one); ((Rich_typ.size fun_rt),Constant.one); ((Rich_typ.size polytyp),Constant.one); (1,drec)]) (* We dont have to pay for x:A, but we must pay for y:D, even if it does not occur recursively. We check this while parsing and turn a letrec into a let if possible! *)
		  :: (iexpression (ctxt#bind recv polytyp) drec inexpr rt dout) 
	      | _ ->                        
		  errAt expr#i ("Syntax error: Recursive definition 'let rec "^recv^"= (..)' is not followed by a function definition.")
	      end

	  | (AndExp(vardefs, inexpr), _)
	    ->
	      let drec   = Cvar.generate Cvar.Rec  in              (* This is n' in the paper *)

	      let delta  = ctxt#restrict_nonlinear in              (* Basic-Delta, beware: \Delta does not contain the D_i, which we store in ctxt_D_i here (unnamed in the paper! *)
	      
	      let proc_e_i (* : (#rich_context * ((???) list)  * ((???) list))
			      -> (variable * (typ option) * typed_expression) 
			      -> (#rich_context * ((???) list)  * ((???) list)) 
                            *)			    
		  (* Receiving a 3-tuple of accumulators,
		     it calculates the non-polymoprhic Type D_i for the i-th recursive definition,
		     which is given as the second argument. (a 3-tuple again)
 		     The D_i are added to the first accumulator (a context),
		     the second and third accumulator holds continuations which depend on the gathered previous results
		   *)
		  =
		fun (pre_ctxt_D, get_psi_list, get_poly_list) (recv, recv_tyo, recexpr) ->
		  begin match recexpr#e with
		  | FunExp(absv, absv_tyo, bodyexpr) ->
		      let fdin    = Cvar.generate Cvar.Lhs in                (* This is m  in the paper *)
		      let fdout   = Cvar.generate Cvar.Rhs in                (* This is m' in the paper *)
		      let fdom_rt =                                          (* This is A  in the paper *)
			match absv_tyo with
			| None     -> errAt expr#i ("Function definition without type declaration for '"^absv^"'.")
			| Some(ty) -> Rich_typ.from_typ ty
		      in
		      let frng_rt = Rich_typ.from_typ bodyexpr#t in          (* This is B in the paper *)
		      let fun_rt  = RArrowTyp(fdom_rt,fdin,fdout,frng_rt) in (* This is D in the paper *)
		      
		      let get_psi_i: (#rich_context) -> (int * (constr list)) = 
                        (* Given the biggest possible context for the function definition, i.e. all non-polymorphic functions and all non-linear parts of ctxt,
                           returns the size of the used/alloactaed context  as well as the constraint set for that particular function *)
			fun rec_delta -> (* rec_delta = delta \cup { D_i | all i} *) 
			  let rec_delta = (rec_delta#restrict_to bodyexpr#fv) in (* Minimze closure size to what is needed; the association x:A needs not to be added for counting the size, for the abstracted variable is obvioulsy not stored in the closure. *)
			  let psi_i = iexpression (rec_delta#bind absv fdom_rt) fdin bodyexpr frng_rt fdout in    
			  (* let _       = debug_string ("\n   "^(String.concat "   " (List.map Constrs.to_string psi))) in *)
			  (rec_delta#size, psi_i)
		      in
		      
		      let poly_of_D_i: (#rich_context) -> idset -> (constr list) -> (#rich_context) =
			(* Adds the polymorphic type of D_i to a given context (accumulator is intended),
			   requires the full alpha and the union of all Psi_i-constraints *)
			fun ctxt_acc alpha psi -> 
			  let polyD_i =
			    RPolyTyp(
			    alpha,
			    psi,
			    fun_rt)
			  in  (* Poly(D_i) *)
			  let _ = poly_acc := (recv, polyD_i) :: !poly_acc in  (* Remember this polytyp for later onscreen printing *)
			  ctxt_acc#bind recv polyD_i
		      in 
		      
		      ((pre_ctxt_D#bind recv fun_rt), (get_psi_i::get_psi_list), (poly_of_D_i::get_poly_list))
			
		  | _ -> errAt recexpr#i ("Syntax error: Recursive definition 'and let rec "^recv^"= (..)' is not followed by a function definition.")
		  end
		    
	      in 
	      
	      let (ctxt_D,get_psi_list,get_poly_list) =
		List.fold_left proc_e_i ((new rich_context),[],[]) vardefs (* Gathering all y_i : D_i *)
	      in

	      let (dsize_sum, psi) = 			
		let delta_cup_ctxt_D = 
		  delta#merge ctxt_D 
		in
	      
		let gather_psi: (int * (constr list)) ->  ((#rich_context) -> (int * (constr list))) -> (int * (constr list)) =
		  fun (dsize_acc, psi_acc) get_psi_i ->
		    let (dsize_i, psi_i) = get_psi_i delta_cup_ctxt_D in
		    ((dsize_i+dsize_acc), (list_fast_append psi_i psi_acc))
		in List.fold_left gather_psi (0,[]) get_psi_list (* Gathering all psi_i to psi *)
	      in
	      
	      let alpha   = 
		Cvar.filter_constants
		  (IdSet.diff 
		     (IdSet.union (ctxt_D#fv_rts)(Constrs.free_cvars psi))
		     (delta#fv_rts)
		  )
	      in
	      
	      let ctxt_poly_D =
		let gather_poly_D: (#rich_context) -> ((#rich_context) -> idset -> (constr list) -> (#rich_context)) -> (#rich_context) =
		  fun ctxt_poly_D_acc poly_of_D_i -> poly_of_D_i ctxt_poly_D_acc alpha psi
		in 
		List.fold_left 
		  gather_poly_D  
		  (new rich_context)
		  get_poly_list 
	      in
	      
(* We dont have to pay for x:A, but we must pay for y:D, even if it does not occur recursively. We check this while parsing and turn a letrec into a let if possible! *)		      
	      (Constrs.gt_eq (linetag "Rec") (1,din) 
		 [(dsize_sum, Constant.one); 
		  (ctxt_poly_D#size , Constant.one);
		  (1,drec)]
	      )
	      :: (iexpression (ctxt#merge ctxt_poly_D) drec inexpr rt dout) 


	  | (IfExp(ifvalu, thenexpr, elseexpr), _ ) 
            ->
	      list_fast_append
		(iexpression ctxt din thenexpr rt dout)
		(iexpression ctxt din elseexpr rt dout)

	  | (LinIExp(fstexpr,sndexpr), RLinPairTyp((fstdin,fstrt,fstdout),(snddin,sndrt,snddout))) 
            ->
	      let dfst    = Cvar.generate Cvar.Aux in   (* corresponds to n_a in the paper *)
	      let dsnd    = Cvar.generate Cvar.Aux in   (* corresponds to n_b in the paper *)
	      let delta   = ctxt#restrict_to expr#fv in (* = (IdSet.union fstexpr#fv sndexpr#fv) *)
	      let d_delta = delta#size + (Rich_typ.size rt) in
	      (Constrs.gt_eq (linetag "Lin") (1,   din) [( d_delta,Constant.one); (1,dout)]) ::
	      (Constrs.gt_eq (linetag "LiF") (1,fstdin) [(-d_delta,Constant.one); (1,dfst)]) ::
	      (Constrs.gt_eq (linetag "LiS") (1,snddin) [(-d_delta,Constant.one); (1,dsnd)]) ::
	      (list_fast_append 
		 (iexpression ctxt dfst fstexpr fstrt fstdout) 
		 (iexpression ctxt dsnd sndexpr sndrt snddout))

	  | (LinEExp(fstsnd, linpairvar), _ ) 
            ->
	      begin match (ctxt#lookup linpairvar) with
	      | RLinPairTyp((fstdin,fstrt,fstdout),(snddin,sndrt,snddout)) -> 
		  if   fstsnd
		  then (* This is a 'FST(...)'-expression *)
		    (Constrs.dominate (linetag "Fst")    din  fstdin ) ::
		    (Constrs.identify (linetag "Fst") fstdout    dout) ::
		    (List.map (Constrs.complete_linetag expr#i) (Rich_typ.restrict (fstrt, rt)))
		  else (* This is a 'SND(...)'-expression *)
		    (Constrs.dominate (linetag "Snd")    din  snddin ) ::
		    (Constrs.identify (linetag "Snd") snddout    dout) ::
		    (List.map (Constrs.complete_linetag expr#i) (Rich_typ.restrict (sndrt, rt)))
	      | other -> err ("Variable '"^linpairvar^"' is not bound to linear pair as required.")
	      end

	  | (MatchExp(matchvar, matchrules), _ ) 
            ->     
	      let mvrt:    rich_typ     = ctxt#lookup matchvar in
	      let mr_ctxt: #rich_context = ctxt#remove matchvar in
	      (List.fold_left (fun acc rul -> list_fast_append (imrule (matchvar, mvrt) mr_ctxt din rul rt dout) acc)  []  matchrules)

	  | _ -> err ("Expression '"^(expr#to_string)^"' does not fit typ '"^(Rich_typ.to_string rt)^"'.")
	  end
      end


and imrule: (variable * rich_typ) -> #rich_context -> cvar -> typed_expression matchrule -> rich_typ -> cvar -> (constr list) = (* We assume that mvar has been removed from mr_ctxt already! *)
  fun (mvar, mvrt) mr_ctxt din (Matchrule(info, constru, argvars, dia_opt, mexpr)) rt dout -> 
    let imrule_aux: int -> rich_typ -> #rich_context -> (constr list) -> (constr list) =
      fun destr mvrt mr_ctxt acc_constrs ->  	(* destr = 0 -> read-only, destr = 1 -> destructive *)
	let rtci =
	  match mvrt with
	  | RConTyp(param_rts, tid, rtct) when (param_rts = []) -> rtct#lookup constru
	  | other -> err ("Matched variable '"^mvar^"' is not bound to a constructor type.")
	in
	let mr_ctxt = mr_ctxt#compile argvars (Rich_typ.unfold_args (rtci#typid,mvrt) rtci#arg_typs) in
	let ltag = 
	  let ord = unsigned_string_of_int rtci#order in
	  let tag = if destr = 0 then "M'" else "Ma" in 
	  Constrs.linetag info (tag^ord)
	in
	let daux = Cvar.generate Cvar.Aux in
	(Constrs.lt_eq ltag (1,daux) [(1,din); (1,rtci#cvar); ((destr*rtci#size),Constant.one)])    (* destr = 0 -> read-only, destr = 1 -> destructive *) 
	  (* CHECK HERE WHY gt_eq instead of eq leads to something wrong!!! *)
	:: (list_fast_append acc_constrs (iexpression mr_ctxt daux mexpr rt dout))
    in
    match dia_opt with
    | None ->    (* Read-only match *)
	if   IdSet.mem mvar mexpr#fv 
	then (* mvar is shared here in this read-only match *)
	  let (mvrt_a, mvrt_b, constr_shr) = Rich_typ.share mvrt in
	  imrule_aux 0 mvrt_a (mr_ctxt#bind mvar mvrt_b) constr_shr
	else (* mvar is not shared in read-only match *)
	  imrule_aux 0 mvrt    mr_ctxt                   []
    | Some(New)      -> (* Destructive anonymous match *) 
	imrule_aux 1 mvrt  mr_ctxt                     []  (* Whether we have a named or anonymous diamond variable does not affect the inference at all *)
    | Some(Reuse(d)) -> (* Destructive named match *)
	imrule_aux 1 mvrt (mr_ctxt#bind d RDiamondTyp) []

and iargs: #rich_context -> valu list -> rich_typ list -> (constr list) =
  fun ctxt args rts ->
    let fv_args = (List.map Syntax.fv_valu args) in (* Obtain a list of free variable sets per value - maybe this should be done once only and saved with each valu *)
    (* The following is way to complicated. Rewriting it using recursion instead of iterators might be make it comprehensible! *)
    (* What we do here:
       We go over the valu list. If the valu contains some variables which also occur in the remaining valu list,
       then these variables are shared. One context is used to call ivalu, the other one to handle the remaining valu list.
       Obviously for the last valu, this_ctxt and other_ctxt are identical, hence the last other_ctxt is discarded.
       We already take into account, that a valu might contain more than one variable.
     *)
    let iarg =
      fun (remaining_fv_args, ctxt, acc) arg rt ->
	match remaining_fv_args with
	| []      -> raise (Invalid_argument "iargs: not enough arguments")
	| fv::fvs ->
	    let (this_ctxt, other_ctxt, split_constrs) = 
	      let split_context =
		fun var (this_ctxt, other_ctxt, constr_acc) -> 
		  if List.exists (IdSet.mem var) fvs
		  then (* share var *)
		    let (this_rt, other_rt, share_constrs) = Rich_typ.share (this_ctxt#lookup var) in
		    (( this_ctxt#replace var this_rt), 
	             (other_ctxt#replace var other_rt), 
		     (list_fast_append share_constrs constr_acc))
		  else (* var occurs only here, no sharing *)
		    (this_ctxt, other_ctxt, constr_acc)
	      in IdSet.fold split_context fv (ctxt,ctxt,[])
	    in 
	    let valu_constrs = ivalu this_ctxt arg rt in (* call ivalue with a context that accounts for sharing *)
	    (fvs, other_ctxt, (list_fast_append valu_constrs (list_fast_append split_constrs acc)))
    in tripel_trd (List.fold_left2 iarg (fv_args, ctxt, []) args rts)
		  

and ivalu: #rich_context -> valu -> rich_typ -> (constr list) =
  fun ctxt vl rt ->
    match vl.v with
    | VarVal(v) -> (Rich_typ.restrict ((ctxt#lookup v), rt))
    | _ -> [] (* All other values, including all operators, do not
    affect heap space at all, and we know that the program typchecks
    on unannotated types. If this is to be changed, then ensure that
    contexts are properly split and multiple variables are properly
    shared, especially in the constructor rule of iexpression!!! *)

