(*

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: exec.ml,v $
   Id:       $Id: exec.ml,v 1.1 2004/12/07 17:30:40 sjost Exp $ 

	

   What this File is all about:
   ----------------------------
   Doing a 'sandboxed' execution of lfd-programs, that is,
   interpret the program and count all the heap-cell operations.

 
   ToDos: 

   - Parse the command-line options into the appropriate datatype! (Char List remain to be done only)
 
   - rename variable "h" to "heap" for more verbosity
*)

open Support
open Common
open Types
open Syntax
open Argument
open Memory


(* Command-Line-Argument parsing --- this is messy old code that might deserve its own module, however this is of low priority, as command line parsing is not relevant and not used to anywhere else *) 
exception Parsing_failed of string (* Signals that a call to parse_rval_aux failed *)

let parse_rval: (string list) -> heap -> (typ list) -> heap * (rvalue list) =
  fun input_ ->
    if input_ = [] 
    then (fun h_ tl_ -> (h_,[]))
    else
      let _ = print_string ("Parsing command line execution arguments...") in
      let max_rec_depth = (List.length input_) + 5 in
      let rec parse_aux: int -> (string list) -> heap -> (typ list) -> (heap * (rvalue list)) =
	fun lc input h tyl -> (* lc is a loop-counter to ensure termination *)
	  let padding = (String.make (2*lc) ' ') in
	  if (lc > max_rec_depth) 
	  then 
	    let _ = debug_ext_string (padding^"Max loop depth exceeded\n") in
	    raise (Parsing_failed "Max loop depth exceeded")						
	  else
	    let _ = (* DEBUG *)
	      if !the_options.debug && !the_options.diamond
	      then
		begin
		  print_string padding;
		  print_string "loopcount:";
		  print_int lc;
		  print_string " - "; 
		  print_string "[";
		  print_string (String.concat "; " (List.map string_of_typ tyl));
		  print_string "]";
		  print_string " - "; 
		  print_string "['";
		  print_string (String.concat "'; '" input);
		  print_string "']";
		  print_string "\n"
		end
	    in
	    match tyl with
	    | []    -> 
		if   input = []   
		then (h,[])
  		else raise (Parsing_failed "Leftover input")
	    | t::ts -> 
		begin
		  let simple_parse: string -> (string -> rvalue) -> heap * (rvalue list) =
		    fun descr action ->
		      try
			let  rv        = (action (List.hd input)) in
			let   _        = debug_ext_string (padding^descr^" argument parsed") in
			let (h1, rvs1) = parse_aux (lc) (List.tl input) h ts in
			(h1, rv::rvs1)
		      with _ -> raise (Parsing_failed (descr^" argument expected"))
		  in
		  match t.v with
		  | DiamondTyp      (* treated as UnitTyp *)
		  | UnitTyp      -> simple_parse "Unit"   (function "()" | "*" -> UnitRVal | _ -> raise (Parsing_failed "Unit argument expected")) 
		  | BoolTyp      -> simple_parse "Bool"   (fun s -> BoolRVal(bool_of_string_ext(s)))
		  | IntTyp       -> simple_parse "Int"    (fun s -> IntRVal(int_of_string(s)))
		  | FloatTyp     -> simple_parse "Float"  (fun s -> FloatRVal(float_of_string(s)))
		  | CharTyp      -> simple_parse "Char"   (fun s -> CharRVal(char_of_string s)) 
		  | StringTyp    -> simple_parse "String" (fun s -> StringRVal(s))
		  | ConTyp([],tid)  -> 
		      begin 
			let sorted_cs =   (* We distinguish the constructors in order to try some certain constructors first... *)
			  let cs = !the_contab#get_constrs tid in   (* first get all constructors for this type *)
			  match (!the_options.clap_sort, !the_options.clap_ord) with (* sorting constrs: (sort?, reverse?) -- we sort in every try anew, which is pretty inefficient. It seems ok for the mere command-line parsing for the sandbox though *) 
			  | (true,  true) -> List.rev (List.stable_sort typcon_arg_compare cs)  (* Sorting is in increasing order *)
			  | (true,  false)->          (List.stable_sort typcon_arg_compare cs) 
			  | (false, true) ->          cs
			  | (false, false)-> List.rev cs
			in
			let rec loop_constrs: (typcon list) -> (heap * (rvalue list)) =  (* try constructors subsequently *)
			  fun tcons ->
			    match tcons with
			    | []        -> raise (Parsing_failed "No suitable constructor found")
			    | TypCon(info,constr,csize,arg_types) :: rest ->
				begin 
				  try					    
				    (* let _ = debug_ext_string (padding^"Trying constructor '"^constr^"'\n") in *)
				    let input = (* Remove explicitly given constructor *)
				      if  (List.length input > 0) && (constr = (List.hd input)) 
				      then List.tl input 
				      else         input
				    in
				    let (h1, rvs1) = 
				      let input1 =
					if (* Special Case: Parsing a string as a char lists *)
					  ((List.map stripinfo arg_types) = [CharTyp; ConTyp([],tid)])
					    &&
					  (try (String.length (List.hd input) > 1) with _ -> false)
					then (* Chop the next input element into pieces *)
					  match input with 
					  | []   -> bug "While parsing commandline:" (Failure "This line should never be reached") (* Cannot fail here due to prior if *)
					  | h::t -> ((string_chop h) @ t)
					else (input)
				      in parse_aux (lc+1) input1 h (arg_types @ ts) 
				    in
				    let (cs_args,rem_rvs) = list_splitat rvs1 (List.length arg_types) in
				    let (h1,l) = h1#store (Constructor(constr,cs_args)) in
				    let _ = debug_ext_string (padding^"Constructor "^constr^" parsed.") in
				    (h1, ((PointerRVal l)::rem_rvs))           
				  with _ -> 
				    let _ = debug_ext_string (padding^"Backtracking\n") in
				    loop_constrs rest
				end
			in loop_constrs sorted_cs 
		      end
		  | other_typ -> bug "While parsing commandline:" (Invalid_argument ("Argument parsing of type "^(string_of_typ t)^" not supported."))
		end
      in parse_aux 0 input_ 


(* EVALUATING FUNCTIONS: *)
(* eterm shall mean "evaluate term" *)

(* MAIN *)
let rec eprogram: program -> unit = 
  function Program(info, typdecs, maintyp, mainexpression) -> 
    let ((initial_heap, main_args): (heap * (rvalue list))) = (* Create heap with identified arguments for main expression on it! *)
      let ((heap, main_args): (heap * (rvalue list))) =
	try
	  parse_rval !the_options.argmain (new heap) (uncurry maintyp) 
	with (Parsing_failed s) -> err (" Commandline argument mismatches type declaration for main expression: "^(string_of_typ maintyp)^"\n Error:("^s^").") 
      in 
      let _ =
	if List.length (main_args) > 0 
	then
	  begin
	    print_string ("\n Evaluating main expression with argument: \n   ");  (* Execute expression first, then apply arguments if appropriate! *)
	    heap#print_rvalues main_args;
	    print_string " \n " 
	  end
      in ((heap#reset_statistics), main_args)
    in
    let ecmdargs: result -> (rvalue list) -> result = (* Applying the command line arguments to the result value *)
      fun r arg ->
	let ecmdsingarg: (result * int) -> rvalue -> (result * int) = (* Applying a single command line argument to a result value *)
	  fun ((h,rv),n) farg ->
	    let exp_i_name = (Argument.main_expr_name^"#"^(string_of_int n)^"_") in (* internal name binder for main expression *)
	    let s = (new stack)#bind exp_i_name rv in
	    ((efunction (mainexpression#i) h s exp_i_name farg),(n+1))
	in fst(List.fold_left ecmdsingarg (r,1) arg)
    in
    let initial_stack = new stack in
    let sep_bar = ("\n"^(String.make !the_options.screen_width '_')^"\n") in
    let _ = 
      begin
	(if not !the_options.debug then queue_warnings ());   (* delay warnings *)
	print_string "Program output follows: ";
	print_string sep_bar
      end
    in
    let runtime = new timer in (* runtime measurement *)
    let (((final_heap:heap), (final_rv:rvalue)) as final_result:result) =
      let pre_result = eexpression initial_heap initial_stack mainexpression in  (* evaluate main expression *) 
      ecmdargs pre_result main_args  (* apply command line arguments to result of main expression *)
    in
    let _ = runtime#stop in
    let _ = 
      begin
	print_string sep_bar;
	print_warnings ();     (* undelay warnings *)
	print_string "\n Result of program evaluation: \n   ";
        final_heap#print_rvalue final_rv; 
	print_string "   \n   ";
	begin
	  if (runtime#time >= 0.03 ) then
	    begin
	      print_string "\n Evaluation runtime: ";
              print_float runtime#time
            end
        end;
	print_string "\n Heap statistics:";
	print_string "\n  * Uniform heap objects (one cell for all constructors/closures):";
	print_string "\n     - Number of initial objects           : ";
	print_aligned_int ' ' 5 final_heap#size_uni_ini;
	print_string "\n     - Number of heap read operations      : ";
	print_aligned_int ' ' 5 final_heap#size_num_read;
	print_string "\n     - Number of heap store operations     : ";
	print_aligned_int ' ' 5 final_heap#size_num_store;
	print_string "\n     - Number of heap remove operations    : ";
	print_aligned_int ' ' 5 final_heap#size_num_remove;
	print_string "\n     - Number of current objects           : ";
	print_aligned_int ' ' 5 final_heap#size_uni_cur;
	print_string "\n\n  * Heap cells according to user defined constructor sizes:";
	print_string "\n     - Initial heap size before evaluation : ";
	print_aligned_int ' ' 5 final_heap#size_usr_ini;
	print_string "\n     - Maximum heap size during evaluation : ";
	print_aligned_int ' ' 5 final_heap#size_usr_max;
	print_string "\n     - Current heap size after evaluation  : ";
	print_aligned_int ' ' 5 final_heap#size_usr_cur;
	print_string "\n     - Additional heap cell consumption    : ";
	print_aligned_int ' ' 5 (final_heap#size_usr_max - final_heap#size_usr_ini);
	print_string "\n\n"
      end
    in  ()

and efunction: info -> heap -> stack -> funcidentifier -> rvalue -> result = (* Apply an rvalue to a functionid *)
  fun i h s fid farg ->
    begin
      if not (s#mem fid) 
      then Builtin.call_info i fid h farg
      else  
	match (s#lookup_info i fid) with
	| PointerRVal(l) -> 
	    begin
	      match (h#read_info i l) with
	      | (h, (Closure(fe,fs))) -> 
		  begin
		    match fe#e with
		    | FunExp(fvar,typ_opt,fbody) -> 
			eexpression h (fs#bind fvar farg) fbody
		    | _ -> errAt i ("Identifier '"^fid^"' is currently not bound to function closure within closure as expected.")
		  end
	      | _ -> errAt i ("Identifier '"^fid^"' is currently not bound to function closure within heap as expected.")
	    end
	| _ -> errAt i ("Identifier '"^fid^"' is currently not bound to function closure within stack as expected.")
    end


and eexpression: heap -> stack -> expression -> (heap * rvalue) =
  fun h s expr -> 
    let _ = if !the_options.diamond then print_string ("\n"^expr#to_string) in
    match expr#e with
    | ValueExp(valu) -> (h, evalue s valu) (* Basic values never access/modifiy the heap *)
	  
    | ConstrExp(constr,vals,dia)  -> 
	let (rvals:(rvalue list)) =  List.map (fun v -> evalue s v) vals in
	let (chc:  heapcell)      = (Constructor(constr, rvals)) in
	begin match dia with
	| New         -> to_result(h#store chc)
	| Reuse(dvar) -> 
	    begin match (s#lookup_info expr#i dvar) with
	    | PointerRVal(l) -> to_result((h#store_at chc l),l)
	    |  _             -> bugAt expr#i "While executing:" (Invalid_argument "PointerRVal expected as value of diamond type.")
	    end
	end

    | AppExp(fid, argid) ->
	let farg = s#lookup_info expr#i argid in 
	efunction expr#i h s fid farg 

    | FunExp(absvar, typ_opt, bodyexp) ->   (* Functions Expressions shall only occur within a let or letrec statement !!! *)
	errAt expr#i ("Syntax error: Function definition 'fun("^absvar^") -> ...' is not preceded by let/letrec")
	  (*  This was the original way to handle a function expression, only left here to fasciliate understandig:
	     to_result(h#store (Closure(expr,(s#restrict_to expr#fv))))
	   *)  

    | LetExp(letvar, v_typ_opt, letexpr, inexpr) when letexpr#is_function ->   (* LetFun *)
	let    fclosure  = Closure(letexpr,(s#restrict_to letexpr#fv)) in
	let h, l         = h#store fclosure in
	let    fpointer  = PointerRVal(l) in 
	let s            = s#bind letvar fpointer 
	in  eexpression h s inexpr

    | RecExp(recvar, v_typ_opt, recexpr, inexpr) when recexpr#is_function ->   (* RecFun *)
	let h, l         = h#new_loc in
	let    fpointer  = PointerRVal(l) in
	let    fclosure  = Closure(recexpr, ((s#restrict_to recexpr#fv)#bind recvar fpointer)) in  (* We must keep a minimal Kontext, but we do not remove recvar! *)
	let h            = h#store_at fclosure l in
	let s            = s#bind recvar fpointer 
	in  eexpression h s inexpr

    | RecExp(recvar, typ_opt, expr1, expr2) ->   (* The non-function letrec... not needed anymore *) 
	errAt expr#i ("Syntax error: Recursive definition 'let rec "^recvar^"=...' is not followed by a function definition.")
(*  This was the original way to handle a letrec expression, only left here to fasciliate understandig:
        let h,l = h#new_loc in
	let s   = s#bind var (PointerRVal l) in
	let h   = h#store_at (Closure(expr1, (s#restrict_to expr1#fv))) l in
	eexpression h s expr2
 *)  

    | AndExp(deflist,bodyexpr) ->
	let (h, s) =
	  let add_pointer: (heap * stack) -> (variable * (typ option) * expression) -> (heap * stack) =
	    fun (h, s) (recvar, typ_opt, recexpr) ->
	      let h, l = h#new_loc in
	      let s    = s#bind recvar (PointerRVal(l)) in
	      (h,s)
	  in List.fold_left add_pointer (h,s) deflist
	in
	let h =
	  let add_closure: heap -> (variable * (typ option) * expression) -> heap =
	    fun h (recvar, typ_opt, recexpr) ->
	      let fclosure = Closure(recexpr, (s#restrict_to recexpr#fv)) in  (* We keep a minimal context *)
	      let l        = 
		match (s#lookup_info expr#i recvar) with
		| PointerRVal(loc) -> loc
		| _                -> errAt expr#i ("Variable '"^recvar^"' not bound to function closure pointer as expected.")  
	      in
	      let h        = h#store_at fclosure l in
	      h
	  in List.fold_left add_closure h deflist
	in eexpression h s bodyexpr

    | LetExp(letvar, typ_opt, expr1, expr2) ->                                     (* Let: the ordinary let-statment *)
	let (h, rval1) = eexpression h s expr1 in
	let s = s#bind letvar rval1 
	in eexpression h s expr2

    | SeqExp(expr1, expr2) ->
	let (h, rval1) = eexpression h s expr1 
	in eexpression h s expr2

    | IfExp(ifvalu , expr1, expr2) -> 
	let istrue = evalue s ifvalu in
	begin match istrue with
	| BoolRVal(true)  -> eexpression h s expr1
	| BoolRVal(false) -> eexpression h s expr2 
	| _ -> errAt expr#i "If-construct expects boolean value."
	end
	  
    | LinIExp(expr1,expr2) ->
	to_result(h#store (Closure(expr,s#restrict_to expr#fv)))
	  
    | LinEExp(fst,var) ->
	begin match (s#lookup_info expr#i var) with
	| PointerRVal(l) -> 
	    begin match (h#withdraw_info expr#i l) with
	    | (h, Closure(lpe,lps)) -> 
		begin match lpe#e with
		| LinIExp(expr1,expr2) -> 
		    if fst 
		    then eexpression h lps expr1
		    else eexpression h lps expr2
		| _ -> errAt expr#i ("Identifier '"^var^"' is currently not bound to linear pair within closure as expected.")
		end
	    | _ -> errAt expr#i ("Identifier '"^var^"' is currently not bound to linear pair within heap as expected.")
	    end
	| _ -> errAt expr#i ("Identifier '"^var^"' is currently not bound to linear pair within stack as expected.")
	end
	  
    | MatchExp(var,  mrules) -> 
	let rv = s#lookup_info expr#i var in 
	let debug_msg = fun () ->
	  begin
	    print_string ("\n DEBUG: '"^var^" : ");
	    h#print_rvalue rv;  (* DEBUG *)
	    print_string "'. ";
	    errAt expr#i "Matching non-constructor types is not allowed."
	      (* By assumption we would know that the program is well typed *)
	  end
	in
	begin match rv with
	| PointerRVal(loc) -> 
	    let (h, cell) = h#read_info expr#i loc in 
	    let mv = 
	      match cell with 
	      | Constructor(c,rvs) -> (c,rvs)
	      |  _                 -> debug_msg ()
	    in ematchrule h s expr#i loc mv mrules
	| _ -> debug_msg () 
	end

and ematchrule: heap -> stack -> info -> location -> (constructor * (rvalue list)) -> 't matchrule list -> result =
  fun h s info loc (constr, rvals) mrules ->
    match mrules with
    | Matchrule(info', constr', vars, dia_o, expr)::_ when (constr = constr') ->
	let (h,s) = 
	  begin match dia_o with
	  | None               ->   (* read-only match *) 
	      (h,s) 
	  | Some(New)          ->   (* destruct to freelist *)
	      (((h#remove loc)#reuse_loc loc), s)
	  | Some(Reuse(dvar))  ->   (* destruct to variable *)
	      ( (h#remove loc), (s#bind dvar (PointerRVal loc)))
	  end
	in
	let s = 
	  try  s#compile vars rvals 
	  with Invalid_argument s ->
	    (h#print_rvalues rvals);  (* DEBUG *)
	    errAt info ("Numbers of variables ("^(string_of_int(List.length vars))^") and values ("^(string_of_int(List.length rvals))^") mismatch in matchingrule.")
	in
	eexpression h s expr
    | _::rs -> ematchrule h s info loc (constr, rvals) rs
    | []    -> errAt info ("Non-exhaustive match: constructor '"^constr^"' not covered.")


and evalue: stack -> valu -> rvalue =
  fun s v -> match v.v with
  | VarVal(vari) -> s#lookup_info v.i vari  
  | IntVal(x) -> IntRVal(x)
  | FloatVal(x) -> FloatRVal(x)
  | CharVal(x) -> CharRVal(x)
  | StringVal(x) -> StringRVal(x)
  | BoolVal(x) -> BoolRVal(x)
  | UnitVal -> UnitRVal
  | UnaryOpVal(unop, val1) -> 
      let rval1 = evalue s val1 
      in eunop unop rval1
  | BinaryOpVal(binop, val1, val2) -> 
      let rval1 = evalue s val1 in
      (* Da OCaml leider nicht LAZY ist, knnen wir hier nicht "let rval2 = evalue s val2" schreiben, sondern muessen es dreimal wiederholen...  *)
      (
        match (binop.v, rval1) with    (* Catch all partial-binary operators here... admittingly not very elegant! AND NOT EVEN NECESSARY -> No Side-Effects anymore! *)
	                               (* Note that andalso/orelse should be parsed as if-expressions, but this is not possible here. See comment in syntax.ml *)     
        | (AndalsoOp, BoolRVal(true))  -> (evalue s val2)  (* By assumption we know that the program type-checks... *)
        | (AndalsoOp, BoolRVal(false)) -> BoolRVal(false)
        | (OrelseOp,  BoolRVal(true))  -> BoolRVal(true)
        | (OrelseOp,  BoolRVal(false)) -> (evalue s val2)  (* By assumption we know that the program type-checks... *)
        | _ -> let rval2 = evalue s val2 
	       in ebinop binop rval1 rval2 
      )

and eunop: unaryoperator -> rvalue -> rvalue =
  fun op rval -> match (op.v,rval) with
    (NotOp,     BoolRVal(b)) -> BoolRVal(not b)
  | (UMinusOp,  IntRVal(w))  -> IntRVal(-w)
  | (UFminusOp, FloatRVal(w)) -> FloatRVal(-.w)
  | _                        -> errAt op.i "Type mismatch between unary operator and operand."
   
and ebinop: binaryoperator -> rvalue -> rvalue -> rvalue =
  fun op rv1 rv2 -> match (op.v,rv1,rv2) with
  | (TimesOp, IntRVal(i1), IntRVal(i2)) -> IntRVal(i1 * i2)
  | (DivOp, IntRVal(i1), IntRVal(i2))   -> IntRVal(i1 / i2)
  | (PlusOp, IntRVal(i1), IntRVal(i2))  -> IntRVal(i1 + i2)
  | (MinusOp, IntRVal(i1), IntRVal(i2)) -> IntRVal(i1 - i2)
  | (FtimesOp, FloatRVal(v1), FloatRVal(v2)) -> FloatRVal(v1 *. v2)
  | (FdivOp, FloatRVal(v1), FloatRVal(v2))   -> FloatRVal(v1 /. v2)
  | (FplusOp, FloatRVal(v1), FloatRVal(v2))  -> FloatRVal(v1 +. v2)
  | (FminusOp, FloatRVal(v1), FloatRVal(v2)) -> FloatRVal(v1 -. v2)
  | (LessOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 < v2)
  | (LessOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 < v2)
  | (LessOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 < v2)
  | (LessOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 < v2)
  | (LteqOp, IntRVal(v1),IntRVal(v2))        -> BoolRVal(v1 <= v2)
  | (LteqOp, FloatRVal(v1),FloatRVal(v2))    -> BoolRVal(v1 <= v2)
  | (LteqOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 <= v2)
  | (LteqOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 <= v2)
  | (GreaterOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 > v2)
  | (GreaterOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 > v2)
  | (GreaterOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 > v2)
  | (GreaterOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 > v2)
  | (GteqOp, IntRVal(v1), IntRVal(v2))       -> BoolRVal(v1 >= v2)
  | (GteqOp, FloatRVal(v1), FloatRVal(v2))   -> BoolRVal(v1 >= v2)
  | (GteqOp, CharRVal(v1), CharRVal(v2))     -> BoolRVal(v1 >= v2)
  | (GteqOp, StringRVal(v1), StringRVal(v2)) -> BoolRVal(v1 >= v2)
  | (EqualOp, IntRVal(v1), IntRVal(v2))         -> BoolRVal(v1 = v2)
  | (EqualOp, FloatRVal(v1), FloatRVal(v2))     -> BoolRVal(v1 = v2)
  | (EqualOp, CharRVal(v1), CharRVal(v2))       -> BoolRVal(v1 = v2)
  | (EqualOp, StringRVal(v1), StringRVal(v2))   -> BoolRVal(v1 = v2)
  | (EqualOp, BoolRVal(v1), BoolRVal(v2))       -> BoolRVal(v1 = v2)
  | (EqualOp, UnitRVal, UnitRVal)               -> BoolRVal(true)
  | (EqualOp, PointerRVal(v1), PointerRVal(v2)) -> BoolRVal(v1 = v2)
  | (AppendOp, StringRVal(v1), StringRVal(v2)) -> StringRVal(v1^v2)
  | (AppendOp, StringRVal(v1), CharRVal(v2))   -> StringRVal(v1^(Char.escaped v2))
  | (AppendOp, CharRVal(v1), StringRVal(v2))   -> StringRVal((Char.escaped v1)^v2)
  | (AppendOp, CharRVal(v1), CharRVal(v2))     -> StringRVal((Char.escaped v1)^(Char.escaped v2))
  | (AndOp, BoolRVal(b1), BoolRVal(b2)) -> BoolRVal(b1 && b2)
  | (OrOp, BoolRVal(b1), BoolRVal(b2))  -> BoolRVal(b1 || b2)
  | (AndalsoOp, v1, v2) ->  errAt op.i "Not implemented: ANDALSO-operator." (* Now catched in eexpression... *)
  | (OrelseOp,  v1, v2)  ->  errAt op.i "Not implemented: ORELSE-operator." (* Now catched in eexpression... *)
  | (ModOp, IntRVal(i1), IntRVal(i2))   -> IntRVal(i1 mod i2)
  | _ -> errAt op.i "Type mismatch between binary operator and operands."   (* Should not occur by our assumption on input programs *)




