(*
   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: constraints.ml,v $
   Id:       $Id: constraints.ml,v 1.1 2004/12/07 17:30:40 sjost Exp $ 

	
   What this File is all about:
   ----------------------------
   The TYPES of ARTHUR and constraints over them.
   
   - Anything concerned with the rich types is contained here.
   - Basic tools for manipulating constraints are present here as well.

   Global values: 


   ToDos: 
   - RichType.from_typ may not terminate in all cases, e.g. for mutually recurively defined types.
   
*)


open Common
open Types
open Argument

exception IncompatibleFoldDepth of bool * typeidentifier (* Exception produced by restricting a[b[a[b[#{a}]]]] to a[b[#{a}]] - "false" signals a vice versa pairing *)

let diamond_str: string -> string = (* A question of fashion, dont know where to place *)
  fun s -> if !the_options.diamond then "<"^s^">" else s      

  (* * * Constraint Variables * * *)

type cvar = string

module Cvar = (* yields unique variable names *)
  struct
    type use = Lhs | Rhs | Let | Rec | Lin | Dat | Aux | Misc | Konst (* Indicates the usage of a constraint variable and shall influence its name... *)

(* Unique counters for naming conventions *)
    let num_lhs    = ref 0
    let num_rhs    = ref 0
    let num_let    = ref 0
    let num_rec    = ref 0
    let num_lin    = ref 0
    let num_dat    = ref 0
    let num_aux    = ref 0
    let num_misc   = ref 0

    let convention: use -> string =
      function 
	| Lhs    -> "x" (* Lefthandside  of an arrow *)
	| Rhs    -> "y" (* Righthandside of an arrow *)
	| Let    -> "l" (* An intermediate variable introduced by a let statement *)
	| Rec    -> "r" (* An intermediate variable introduced by a rec statement *)
	| Lin    -> "p" (* An annotation of contained in an additive pair *)
	| Dat    -> "d" (* Any Datatype annotation, i.e. assorted to some constructor *)
	| Aux    -> "a" (* Any other intermediate variable *)
	| Misc   -> "m" (* ??? A konstant which isnt one, ie AnnValDec containing a '*' instead of a value *)
	| Konst  -> "K" (* Prefix for constants. Necessary as lp_solve does not allow arithmetic expressions as factors. Must be a string of length 1 *)
	      
    let generate: use -> cvar =
      fun vu ->
	let x = 
	  match vu with
	  | Lhs    -> num_lhs
	  | Rhs    -> num_rhs
	  | Let    -> num_let
	  | Rec    -> num_rec
	  | Lin    -> num_lin
	  | Dat    -> num_dat
	  | Aux    -> num_aux
	  | Misc   -> num_misc
	  | Konst  -> Support.bug "Constraint generation:" (Invalid_argument "Constant constraint variables cannot be generated arbitrarily.")
	in
	let nums = string_of_int (save_ref_increment x) in
	let fill = if String.length(nums) = 1 then "0" else "" in
	((convention vu)^fill^nums)
	  
    let get_use: cvar -> use =
      fun cv ->
	match String.sub cv 0 1 with
	| x when (convention Lhs)    = x -> Lhs
	| x when (convention Rhs)    = x -> Rhs
	| x when (convention Let)    = x -> Let
	| x when (convention Rec)    = x -> Rec
	| x when (convention Lin)    = x -> Lin
	| x when (convention Dat)    = x -> Dat
	| x when (convention Aux)    = x -> Aux
	| x when (convention Misc)   = x -> Misc
	| x when (convention Konst)  = x -> Konst
	| _                              -> raise (Invalid_argument "Constraint variable of unknown sort encountered.")

    let is_constant: cvar -> bool =
      let k = String.get (convention Konst) 0 in
      fun v -> ((String.get v 0) = k) 
	  
    let filter_constants: idset -> idset = (* Removes constant variables from a set *)
      let is_not_constant = compose not is_constant 
      in  IdSet.filter is_not_constant

    let rename: cvar -> cvar =          (* Give a fresh name of the same sort as input *)
      fun cv -> 
	if is_constant cv
	then cv 
	else generate (get_use cv)
	    
	    (* Printing *)
    
    let to_string: cvar -> string = diamond_str

  end

     
module Constant = (* Encodes constants as constraint variables *)
  struct
    let decode: cvar -> string = (* Decode a variable name into a string-float *)
      fun v ->
	try
	  if (Cvar.is_constant v) 
	  then (* A coded constant *)
	    let v = String.sub v 1 (String.length v - 1) in       (* Remove leading 'K'   *)
	    let v = string_drop_while '0' v              in       (* Remove leading '0''s *)
	    let v = string_replace v 'P' '+'             in       (* lp_solve does not permit names containing '+' *)
	    let v = string_replace v 'M' '-'             in       (* lp_solve does not permit names containing '-' *)
	    let v = if v = "" then "0" else v            in       (* Check if string was all alone zero *)
	    let _ = float_of_string v                    in v     (* Test if truly convertible *)
	  else raise (Invalid_argument "Constraints.Constant.decode")
	with _ -> Support.err ("Floatconstant-variable '"^v^"' cannot be reverted to float.")

    let to_float: cvar -> float =
      compose float_of_string decode

    let generate: float option -> cvar = (* We encode a float into a variable name *)
      function
	| None   -> Cvar.generate Cvar.Misc (* Generate from joker symbol, i.e. unknown constant. We cannot know where variable is used, hence Misc *)
	| Some c ->
	    let o1 = aligned_string_of_float '0' 2 c in
	    let o2 = string_replace o1 '-' 'M' in (* there may be operands in the exponent *)
	    let o3 = string_replace o2 '+' 'P' in
	    if (string_contains_operator o3)
	    then Support.bug "While generating constraints:" (Invalid_argument ("Cannot convert '"^o1^"' into a syntactic correct constant-variable.\n Fix: Manually edit constraint-file!"))
	    else 
	      let nc = ((Cvar.convention Cvar.Konst)^o3) in
	      let _  = assert (float_of_string(decode nc) = c) in  nc

    let zero: cvar = (generate (Some 0.0))  (* Constant 0 *)
    let one:  cvar = (generate (Some 1.0))  (* Constant 1 *)
	
  end


class solution =                (* lookup assorting floats to constraint variables *)
  object (self: 'self)
    inherit [float] lookup
    method joker       = "_"
    method key_name    = "constraint variable"
    method value_name  = "float value"
    method lookup_name = "solution"
    method error       = Support.err 

    method bind: cvar -> float -> 'self =
      fun v f -> 
	if   v = self#joker 
	then self 
	else 
	  if (self#mem v)
	  then 
	    if (self#lookup v) = f
	    then self
	    else self#error (self#lookup_name^" "^v^"="^(self#print_human v)^" already known,  contradicting "^v^"="^(pretty_string_of_float f)^".")
	  else self#add v f

    method print: cvar -> string = (* Does NOT decodes constants into human-readable format *)
      fun v ->
	if   (self#mem v)
	then pretty_string_of_float(self#lookup v)
	else v 

    method print_human: cvar -> string = (* Decodes constants into human-readable format *)
      fun v ->
	if   (Cvar.is_constant v) && not (self#mem v)
	then Constant.decode v
	else self#print v

  end


type objective = idset * idset * idset * idset (* Stores the variables for the objective function: fixedin, lhs, rhs fixedout *)

let objective_free_cvars: objective -> idset = 
  fun (dins,lhss,rhss,douts) ->
    IdSet.union
      (IdSet.union dins douts)
      (IdSet.union lhss rhss)
      

(* * * Constraints * * *)
module Constrs =
  struct
    
    module InternalConstrs= Map.Make(struct type t = cvar let compare = String.compare end) 
    type row = int InternalConstrs.t (* A map structure helps simplifying a constraint, i.e. "1*a + 2*b >= -1*a" turns into "2*a + 2*b >= 0" *)

    type constr = 
      | Eq0   of string * row   (* corresponds to 0  = (find var1)*var1 + (find var2)*var2 + ... *) (* the string argument may contain a helpful label *)
      | Geq0  of string * row   (* corresponds to 0 >= (find var1)*var1 + (find var2)*var2 + ... *)
      | Trivial of string       (* The empty constraint, merely for technical convenience (had been added later) *)
      | Comment of string          (* Comments that help a reader understanding some constraints (had been added later) *)

        (* Exports *)

    let empty: row = InternalConstrs.empty
	
    let fold: (cvar -> int -> 'b -> 'b) -> row -> 'b -> 'b = InternalConstrs.fold

	(* Constrs generation *)

    let add: row -> (int * cvar) -> row =  (* Add coefficient of a variable to row  ---  Identical to Solution.add, but there is no overloading available in OCaml. *)
      fun c (i,v) ->
	let oldi = 
	  if   InternalConstrs.mem  v c
	  then InternalConstrs.find v c
	  else 0
	in
	let newi = oldi + i in
	if  newi = 0
	then InternalConstrs.remove v c
	else InternalConstrs.add v newi c

    let strengthen: constr -> constr =
      function
	| Geq0(l,r) -> Eq0(l,r)
	| other     -> other

    let weaken: constr -> constr =
      function
	| Eq0(l,r)  -> Geq0(l,r)
	| other     -> other

    let    eq_0: string -> (int * cvar) list -> constr =
      fun l vs ->                                         (* 0  =  a1*v1 + a2*v2 + ... *)
	let addrow = (List.fold_left add empty vs) in
	if  addrow = empty
	then Trivial(l)
	else Eq0(l, addrow)
	    
    let gt_eq_0: string -> (int * cvar) list -> constr = compose2 weaken eq_0             
                                                          (* 0 >= a1*v1 + a2*v2 + ... *)

    let    eq: string -> (int * cvar) -> (int * cvar) list -> constr = (* For compatibility with geq_l *)
      fun l (a,v) vs ->   (* a*v  = a1*v1 + a2*v2 +...  ===  0  = -a*v +a1*v1 +a2*v2 +...*)
	eq_0 l ((-a,v)::vs)

    let gt_eq: string -> (int * cvar) -> (int * cvar) list -> constr = compose3 weaken eq
(*    fun l (a,v) vs ->   (* a*v >= a1*v1 + a2*v2 +...  ===  0 >= -a*v +a1*v1 +a2*v2 +...*) *)

    let lt_eq: string -> (int * cvar) -> (int * cvar) list -> constr = 
      fun l (a,v) vs ->   (* a*v <= a1*v1 + a2*v2 +...  ===  0 >= +a*v -a1*v1 -a2*v2 -...*)
        weaken (eq_0 l ((a,v)::(List.map (fun (i,v) -> (-i,v)) vs)))
	
    let identify: string -> cvar -> cvar -> constr =
      fun l va vb -> eq_0 l [(-1,va);(1,vb)]             (*  0  =  -1*va + 1*vb *)

    let dominate: string -> cvar -> cvar -> constr =
      compose3 weaken identify          (*  va >= vb   ===   0 >=  -1*va + 1*vb *)


(* Linetags *)
    let undefined_linetag: string     = "L_____"  
    let undefined_linetag_length: int = String.length undefined_linetag

    let substitutions_linetag: string    = undefined_linetag ^ "  Substitutions:\n  "  (* The undefined linetag will be replace by complete linetag, so that we now where substitutions happenend. *)
    let substitutions_linetag_negth: int = String.length substitutions_linetag

    let is_substitutions_tag: string -> bool =
      let subs_tag = string_drop undefined_linetag_length substitutions_linetag in
      fun tag ->
	try 
	  string_beginswith (string_drop undefined_linetag_length tag) subs_tag
	with _ -> false (* string_drop raises exception if input string is to small. Since this function is not security relevant, we catch all other exceptions as well, since then it wasnt a proper substitution tag... *)

    let linetag: Support.info -> string -> string =
      fun i ->
	if !the_options.performance 
	then (fun _ -> "")
	else 
	  match i with
	  | Support.UNKNOWN   -> (fun s -> (                                undefined_linetag                 ^ s))
	  | Support.FI(_,x,_) -> (fun s -> ("L"^(aligned_string_of_int '0' (undefined_linetag_length-2) x)^"_"^ s))
  
    let complete_linetag: Support.info -> constr -> constr =
      fun i ->
	let add_i_to: string -> string = 
	  fun s ->
	    if (String.length s < undefined_linetag_length)
	    then linetag i s
	    else 
	      if   (string_beginswith s undefined_linetag)
	      then linetag i (string_drop undefined_linetag_length  s)
	      else s
	in
	function
	  | Trivial(l) -> Trivial(add_i_to l)
	  | Comment(l) -> Comment(add_i_to l)
	  | Eq0(l,r)   ->     Eq0(add_i_to l, r)
	  | Geq0(l,r)  ->    Geq0(add_i_to l, r)

    let get_linetag: constr -> string =
      begin
	function
	  | Comment(l)
	  | Trivial(l) 
	  | Eq0(l,_)   
	  | Geq0(l,_)  -> l 
      end
	
	(* Misc Tools *)

    let sort: (constr list) -> (constr list) = (* Sort constraints by label *)
      List.sort (fun a b -> String.compare (get_linetag a) (get_linetag b))
      
    let substitute: cvar -> cvar -> constr -> constr = (* replace first variable by second *)
      fun cvout cvin constr ->
	if   cvout = cvin
	then 
	  if   Cvar.is_constant cvout
	  then constr
	  else 
	    let _ = Support.warning ("Ignored constraint variable substitution: Replace '"^cvout^"' with '"^cvin^"'.") in
	    constr 
	else
	  let substitute_row: row -> row =
	    fun r ->
	      if   InternalConstrs.mem cvout r
	      then 
		let fac = (InternalConstrs.find cvout r) +
		    (
		     if   InternalConstrs.mem cvin r 
		     then (* Already contained *)
		       let _ = Support.warning ("Substituting '"^cvout^"' with '"^cvin^"': '"^cvin^"' already contained in constraint. Adding both factors for this constraint row.") 
		       in (InternalConstrs.find cvin r)
		     else 0 (* cvin is new *)
		    )
		in InternalConstrs.add cvin fac (InternalConstrs.remove cvout r)
	      else r
	  in match constr with
	  | Geq0(l,r)                                ->    Geq0(l,(substitute_row r))
	  | Eq0(l,r)                                 ->     Eq0(l,(substitute_row r))
	  | Comment(l) when (is_substitutions_tag l) -> Comment(l^" "^cvout^"<-"^cvin^",")
	  | other                                    ->  other
       	
    let map_substitute: cvar -> cvar -> constr list -> constr list =
      fun cvout cvin -> List.map (substitute cvout cvin)

    let free_cvars: (constr list) -> idset =  (* Returns set of contained constraint variables, including constant variables! --- Maybe include set of variables within constr type right from the start? *)
      let aux_add_cvar: cvar -> int -> idset -> idset = 
	fun v _ acc -> IdSet.add v acc  
      in
      let aux_add_constr: idset -> constr -> idset =
	fun acc c ->
	  match c with 
	  |  Geq0(l,r)  
	  |   Eq0(l,r) -> InternalConstrs.fold aux_add_cvar r acc
	  | Comment(_)
	  | Trivial(_) -> acc
      in List.fold_left aux_add_constr IdSet.empty 

    let stats_aux: (constr list) -> (int * int * int * int) = (* counts the different constraint kinds and eliminates trivials*)
      let ss_aux =
	fun (acc_e, acc_g, acc_t, acc_c) constr ->
	  match constr with
	  | Eq0(_,_)   -> (acc_e+1, acc_g  , acc_t  , acc_c  )
	  | Geq0(_,_)  -> (acc_e ,  acc_g+1, acc_t  , acc_c  )
	  | Trivial(_) -> (acc_e  , acc_g  , acc_t+1, acc_c  )
	  | Comment(_) -> (acc_e  , acc_g  , acc_t  , acc_c+1)
      in List.fold_left ss_aux (0,0,0,0)
	
    let simplify: (constr list) -> (constr list) = (* No ideas except removing trivials, but there are only so few that it takes too much time to remove them compared to the gain *) 
      fun lp -> lp (* Change 'stats' as well if this function actually does something *)

    let stats: (constr list) -> objective -> (((constr list) * idset) * (string * string * string)) = (* Returns simplified lp and its free constraint variables, the number of constraints, trivial constraints and variables. The three numbers are turned into strings as well. *)
      fun lp obj ->
	if    !the_options.performance 
	then  	
	  let cvs = IdSet.union (free_cvars lp) (objective_free_cvars obj) in
	  ((lp,cvs),("???", "???", "???"))
	else
	  let slp = sort(simplify lp) in (* Should we count before or after simplifying? *)
	  let (eqs,geqs,trivs,coms) = stats_aux slp in
	  let cvs = IdSet.union (free_cvars slp) (objective_free_cvars obj) in
	  let nlp = string_of_int (eqs+geqs) in
	  let ntv = string_of_int  trivs     in
	  let nvs = string_of_int(idset_size cvs) in
	  ((slp,cvs),(nlp,ntv,nvs))


(* printing constraints *)	  

    let to_string_sol: solution -> constr -> string =
      let print_label: string -> string =
	let cut_sym = "$" in
	(fun s ->
	  if !the_options.performance then "" else
	  begin match s with
	  | ""     -> "     "
	  | _ as l -> 
	      let lc = 
		let len = String.length l in
		if len < 27 (* lp_solve does not accept huge labels, and simply ignores inequalities with huge labels! *)
		then l
		else
		  try
		    let cut = (String.rindex_from l (len - 5) '_') in
		    let (st,en) = string_split l cut in
		    let st = string_rdrop (len - 24) st in
		    (st^cut_sym^en)
		  with _ -> ((String.sub l 0 20)^cut_sym^(String.sub l (len-4) 4))   (* Just in case... *)
	      in (lc^": ")
	  end
	)
      in
      fun sol -> 
	let print_row: row -> string = 
	  (fun r -> (InternalConstrs.fold (fun cv fak acc -> (" "^(signed_aligned_string_of_int ' ' 3 fak)^"*"^(sol#print cv)^acc)) r ""))
	in
	(
	 function 
	   | Eq0(l,r)   -> ((print_label l)^(print_row r)^"  = 0 ; \n")
	   | Geq0(l,r)  -> ((print_label l)^(print_row r)^" <= 0 ; \n")
	   | Comment(l) ->
	       if (!the_options.performance)
	       then ""
	       else ("/* "^l^" */ \n")           
	   | Trivial(l) -> 
	       if (!the_options.performance || not (!the_options.debug))
	       then ""
	       else ("/* "^(print_label l)^ " : trivial ; */ \n")
	)
	
    let to_string: constr -> string =
      to_string_sol (new solution)
  
    let calc_row: row -> solution -> float =
      fun r sol ->
	let calc: cvar -> int -> float -> float =
	  fun v a sum -> ((float_of_int a) *. (sol#lookup v)) +. sum
	in InternalConstrs.fold calc r 0.

    let filter_strict: (constr list) -> solution -> (constr list) = (* All strict constraints with respect to given solution, except constraints ending with "Ap0" *)
      fun constrs sol ->
	let strict: constr -> bool =
	  function 
	    | Geq0(l,tb) -> 
		let rowval = calc_row tb sol in
		if (rowval = 0.) || (string_endswith l "Ap0") (* Ap0-Constrsaints are almost never strict, and that's ok. *)
		then false (* non-strict *)
		else 
		  if  (rowval < 0.) 
		  then true  (* strict *)
		  else 
		    let _ = Support.warning ("Precision error: computed solution '"^(string_of_float rowval)^"' greater than 0, not valid for '"^l^"'!\n") in
		    true
	    | Eq0(l,tb)  -> 
		let rowval = calc_row tb sol in
		if   rowval = 0. 
		then false 
		else 
		  let _ = Support.warning ("Precision error: computed solution '"^(string_of_float rowval)^"' unequal to 0, not valid for '"^l^"'!\n") in
		  true
	    | Trivial(_)
	    | Comment(_) -> false
	in List.filter strict constrs


(* Writing constraints to a file... *)

    let file_header: string -> string -> (string * string * string) -> string = (* Returns number of constraints, simplified constraints, trivial constraints, variables and number of variables. All numbers are turned into strings as well. *)
      fun fname rt_str (num_lp, num_triv, num_cvs) ->
	begin
	  ("   This file is an automatically generated lp for '"^lp_solver^"'. \n")^
          ("       (Filename should be: "^fname^") \n")^
	  ("\n")^
	  ("   Contains "^(num_lp)^ " inequalties in "^num_cvs^" variables. \n")^
          (if "0" = num_triv then "" else
	  ("       ("^num_triv^" trivial constraints suppressed.) \n"))^
	  ("\n")^
	  ("   Annotated Type of main expression: \n")^
	  ("       "^rt_str^"\n")
	end


    let to_file: ((constr list) * idset) -> objective -> string -> out_channel -> unit = 
      fun (lp, cvs) (dins, lcvs, rcvs, douts) fheader outch ->
	(* This function shall print the constraints in lp_solver-format (and is written in an iterative style...) *)
	(* CURRENTLY WRITTEN FOR: "lp_solve" version 4.0.1.0 (ftp://ftp.es.ele.tue.nl/pub/lp_solve ,  LesserGNU public license) *)
	let write:string -> unit = output_string outch in
	let _ = (* Print some optional header comment *)
	  if not (!the_options.performance || fheader = "") then (* print some informative header *)
	    begin
	      write "/* \n";
	      write fheader;
	      write "\n*/ \n\n"
	    end
	in
	let _ = (* Print objective-function *)
	  let obj_aux: string -> idset -> string = 
	    fun fac set -> IdSet.fold (fun v acc -> (acc^" "^fac^"*"^v)) set ""
	  in
	  begin
	    (write "MIN: ");
	    (write (obj_aux (!the_options.obj_din)  dins));
	    (write (obj_aux (!the_options.obj_lhs)  lcvs));
	    (write (obj_aux (!the_options.obj_rhs)  rcvs));
	    (write (obj_aux (!the_options.obj_dout) douts));
	    (write " ;\n\n")
	  end 
	in
	let _ =  (* Print all constraints *)
	  List.iter (fun c -> write (to_string c)) lp
	in
	let _ =  (* Print non-negativity constraints - All constants must be positive as well! *)
	  IdSet.iter (fun v -> write (v^" >= 0 ;\n")) cvs
	in
	let _ =  (* Print constants OR upper-bounds constraints - prohibits unbounded LPs *)
	  IdSet.iter 
	    (fun v ->
	      if (Cvar.is_constant v) 
	      then (* A coded constant *)
		let c = Constant.decode v     (* Extract constant value from name *)
		in  write (v^" =  "^c^" ;\n")
	      else (* An ordinary variable, not a constant *)
		write (v^" <= "^(!the_options.infinity)^" ;\n")
	    ) cvs 
	in
	let _ =  (* Print integer restrictions if desired *)
	  if   !the_options.integer_lp
          then IdSet.iter (fun v -> write ("int "^v^" ;\n")) cvs 
	in
	write (" \n")

  end

type constr = Constrs.constr (* Exporting the vital type, but still hiding the previous definitions *)

      
(* Tools for constraints and solutions *)

let rec file_to_solution: in_channel -> solution =
  fun inch ->
    let read: unit -> string = function () -> (input_line inch) in  (* Read a line from input channel *)
    let ifty: float = ((float_of_string (!the_options.infinity)) -. (float_of_string (!the_options.delta))) in
    try 
      match read() with
      | "This problem is infeasible" ->
          new solution
      | "This problem is unbounded" ->
	  let _ = Support.err "\n   ---   LP is unbounded   --- \n\n  Should never occur, since all variables are bounded. Hence LP must contain an error." in
          new solution
      | _ ->
    	  let _ = 
	    while (read()) <> "Actual values of the variables:"  (* Header that usually precedes solution of lp_solver! *)
            do () done 
	  in
	  let rec parselines: solution -> solution =
	    fun sol ->
	      try 
		let line   = read () in
		let br     = String.index line ' ' in
		let var    = String.sub line  0 br in
		let flvalu = float_of_string (String.sub line (br+1) ((String.length line)-(br+1))) in
		if   flvalu <= ifty
		then parselines (sol#bind var flvalu)
		else parselines  sol
	      with 
	      | End_of_file -> sol (* Shall we close here ? *)
	      |  _          -> Support.err ("Problem parsing output of '"^Argument.lp_solver^"'. See written file for input to '"^Argument.lp_solver^"'.")
	  in parselines (new solution)
    with  End_of_file -> Support.err ("Problem reading output of '"^Argument.lp_solver^"'. Probably no solution found.")
	

let solution_to_file: solution -> ((constr list) * idset) -> objective -> string -> out_channel -> unit =  (* This method is pretty similar to Constrs.to_file, but serves only a protocolling function *)
  fun sol (lp, cvs) (dins, lcvs, rcvs, douts) fheader outch ->
    let write:string -> unit = output_string outch in
    let _ = (* Print some optional header comment *)
      if not (!the_options.performance || fheader = "") then (* print some informative header *)
	begin
	  write "/* \n";
	  write fheader;
	  write "*/ \n\n"
	end
    in
    let _ = (* Print objective-function *)
      let obj_aux: string -> idset -> string = 
	fun fac set -> IdSet.fold (fun v acc -> (acc^" "^fac^"*"^(sol#print v))) set ""
      in
      begin
	(write "MIN: ");
	(write (obj_aux (!the_options.obj_din)  dins));
	(write (obj_aux (!the_options.obj_lhs)  lcvs));
	(write (obj_aux (!the_options.obj_rhs)  rcvs));
	(write (obj_aux (!the_options.obj_dout) douts));
	(write " ;\n\n")
      end 
    in
    let _ =  (* Print all constraints *)
      List.iter (fun c -> write (Constrs.to_string_sol sol c)) lp
    in
    let _ =  (* Print all variables with their values *) 
      IdSet.iter (fun v -> write (v^" = "^(sol#print v)^" ;\n")) cvs
    in
    write (" \n")


(* Enriched Arthur Types, i.e. types with resource informations *)

let diamant = "<>" (* simply the diamond-symbol *)

(* This class contains all info for a constructor of an enriched type *)
class ['t, 'arg_rt] par_rt_coninfo = 
  fun cn tid ot arg_ts sz dv ord ->
    object (self:'self)
      inherit ['t, 'arg_rt] par_coninfo cn tid ot arg_ts sz ord
      val  v_cvar: cvar = dv
      method cvar: cvar = v_cvar  (* Each constructor of a rich ARTHUR type carries its own constraint variable, which are NOT unique, but vary which each use! Hence a global contab as for ART-types does not do the trick anymore, so we also include the static constructor infos for simplicity as well now. *)
    end

(* This class replaces the ConTab module, i.e. it is a lookup-table for each constructor to its coninfo *)
(* module ConTab= Map.Make(struct type t = constructor let compare = String.compare end) *)
class ['t] par_rt_contab = (* Instead of map *)
  object (self: 'self)
    inherit ['t] lookup
    method key_name    = "Constructor"
    method value_name  = "rich constructor info"
    method lookup_name = "rich constructor table"
    method joker       = ""
    method error       = Support.err
    method map:                 ('t -> 't) -> 'self = 
      fun f -> {< the_lookup = InternalLookup.map  f the_lookup >}
    method mapi: (constructor -> 't -> 't) -> 'self = 
      fun f -> {< the_lookup = InternalLookup.mapi f the_lookup >} 
  end

type rich_typ = (* Maybe is should be a class as well and inherit from typ, but then again, they are differen variant types, and not even polymorphic variants seem good enough. I guess rich types are really something different. *)
  | RUnitTyp 
  | RDiamondTyp (* For compatibilty with camelot *) 
  | RTvarTyp of typvariable 
  | RBoolTyp 
  | RIntTyp 
  | RFloatTyp
  | RCharTyp 
  | RStringTyp 
  | RLinPairTyp of ((cvar * rich_typ * cvar) * (cvar * rich_typ * cvar))
  | RConTyp of ((rich_typ list) * typeidentifier * rt_contab)  (* parameter-list, typ-identifier and resource variable, which we expanded to the full constructor info class. *)
  | RArrowTyp of rich_typ * cvar * cvar * rich_typ   (* argument type, din, dout, result type *)
  | RPolyTyp  of  idset * (constr list) * rich_typ   
and rec_rich_typ =
  | RSelfTyp  of typeidentifier  (* Necessary for constructors, which have their own type as an argument (like Cons)  *)
  | RFlatTyp of rich_typ
and rt_coninfo = (typ, rec_rich_typ) par_rt_coninfo (* First parameter must still be typ instead of rich_type: A rich_coninfo must contain itself within own_type, which is not possible functionally, i.e. without pointers. Does not allow "new" command *)
and rt_contab  =  rt_coninfo par_rt_contab          (* Does not allow "new" command *)

let new_rt_coninfo: constructor -> typeidentifier -> typ -> rec_rich_typ list -> constructor_size -> cvar -> int -> rt_coninfo 
    = new par_rt_coninfo  (* since class and type definition cannot be mutually recursive, this is a bit clumsy *)
let new_rt_contab:  rt_contab  = new par_rt_contab   (* since class and type definition cannot be mutually recursive, this is a bit clumsy *)


(* class type rt_coninfo = [rich_typ, rec_rich_typ] par_rt_coninfo --- does not help to allow "new" either *)

let rSelfTypUnknown = RSelfTyp("Unknown Self")  (* Sometimes we must infer self later - use this abbreviation in these cases *)
let selfSymbol      = "#"                       (* Just in case that we need to alter this symbol - alter in parser.mly as well. Current token: SHARP *)


module Rich_typ = (* Shall we include some of the type definitions here and export only some types? *)
  struct
    
    let rec strip: rich_typ -> typ = (* Removes annotations *)
      fun rt ->
	Support.fakeinfo 
	  begin match rt with
	  | RUnitTyp         -> UnitTyp
	  | RDiamondTyp      -> UnitTyp
	  | RTvarTyp(tvar)   -> TvarTyp(tvar)
	  | RBoolTyp         -> BoolTyp
	  | RIntTyp          -> IntTyp
	  | RFloatTyp        -> FloatTyp
	  | RCharTyp         -> CharTyp
	  | RStringTyp       -> StringTyp
	  | RLinPairTyp((_,f,_),(_,s,_))  -> LinPairTyp(strip f, strip s)
	  | RConTyp(pars,tid,rtct)        -> ConTyp((List.map strip pars),tid)
	  | RArrowTyp(rta,din,dout,rtb)   -> ArrowTyp((strip rta), (strip rtb))
	  | RPolyTyp(vs,cnstrs,rt)        -> Support.stripinfo (strip rt)
	  end

    let to_typ: rich_typ -> typ = strip  (* simply renaming it to something more logical *)



    let from_typ: typ -> rich_typ = 
      let rec from_typ_aux: typeidentifier list -> typ -> rich_typ = 
	fun encountered_ts t ->
	  let from_typ = from_typ_aux encountered_ts in (* sort of a hack, no?*)
	  begin match (Support.stripinfo t) with
	  | UnitTyp             ->  RUnitTyp
	  | DiamondTyp          ->  RDiamondTyp
	  | TvarTyp(v)          ->  RTvarTyp(v)
	  | BoolTyp             ->  RBoolTyp
	  | IntTyp              ->  RIntTyp
	  | FloatTyp            ->  RFloatTyp
	  | CharTyp             ->  RCharTyp
	  | StringTyp           ->  RStringTyp
	  | LinPairTyp(fst,snd) ->
	      let fdin  = Cvar.generate Cvar.Lhs in
	      let fdout = Cvar.generate Cvar.Rhs in
	      let sdin  = Cvar.generate Cvar.Lhs in
	      let sdout = Cvar.generate Cvar.Rhs in
	      RLinPairTyp((fdin,(from_typ fst),fdout),(sdin,(from_typ snd),sdout))
	  | ConTyp(params, tid) -> 
	      let rich_params = List.map from_typ params in
	      let rtct = 
		let enrich_coninfo: rt_contab -> coninfo -> rt_contab =
		  fun rtct ci ->
		    let rcinf: rt_coninfo =
		      let arg_typs' =  
			let fold_tid: typ -> rec_rich_typ =
			  fun ty ->
			    let encountered_ts' = tid :: encountered_ts in
			    match Support.stripinfo ty with
			    (* | ConTyp(params',tid') when (tid' = tid) && (params' = params)
			       -> RSelfTyp(tid)
			       This clause does not support mutually recursive types, but
			       a recursive constructor is prohibited to have parameters without this clause! 
			     *)
			    | ConTyp(params',tid') when (List.mem tid' encountered_ts') 
			      -> 
				if (params' = [])  
				then RSelfTyp(tid')
				else Support.err ("User defined datatypes with parameters are not supported yet! ('"^tid^"').")
			    | _ -> RFlatTyp(from_typ_aux encountered_ts' ty)
			in List.map fold_tid ci#arg_typs
		      in new_rt_coninfo ci#name tid ci#own_typ arg_typs' ci#size (Cvar.generate Cvar.Dat) ci#order (* Note that own_typ remains a 'typ' rather than 'rich_typ', since a rich own_typ would have to know all rt_coninfo already *)
		    in rtct#bind ci#name rcinf
		in List.fold_left enrich_coninfo new_rt_contab (!the_contab#get_coninfos tid)
	      in RConTyp(rich_params, tid, rtct)
	  | ArrowTyp(dom,rng)   ->
	      RArrowTyp(
	      (from_typ dom), 
	      (Cvar.generate Cvar.Lhs), 
	      (Cvar.generate Cvar.Rhs),
	      (from_typ rng)
	     )
	  end
      in from_typ_aux []
	
    let enrich: typ -> rich_typ = from_typ

    let size: rich_typ -> int = compose Memory.Size.typ strip

    let rec to_string_sol: solution -> rich_typ -> string = (* For tracing rich_types *)
      fun sol t -> 
	let to_str_s = to_string_sol sol in
	match t with
	| RUnitTyp              -> "unit"
	| RDiamondTyp           -> if !the_options.diamond then "diamond" else "<>" (* diamond=true means we do the nice & fancy printout, thus there are already a lot of '<' and '>' symbols which might lead to confusion! *)
	| RTvarTyp(typvar)      -> "tvar("^typvar^")"
	| RBoolTyp              -> "bool"
	| RIntTyp               -> "int"
	| RFloatTyp             -> "float"
	| RCharTyp              -> "char"
	| RStringTyp            -> "string"
	| RLinPairTyp((fdin,frt,fdout),(sdin,srt,sdout))      
	  -> 
	    let fi = sol#print_human fdin  in
	    let fo = sol#print_human fdout in
	    let si = sol#print_human sdin  in
	    let so = sol#print_human sdout in
	    ("(|"^fi^","^(to_str_s frt)^","^fo^"|"^si^","^(to_str_s srt)^","^so^"|)")
	| RConTyp(rts,id,rtci)  
	  -> 
	    let params:  string = List.fold_left (fun a x -> a ^" "^(to_str_s x)) "" rts in
	    let cn_args: string = (* constructors with arguments, ordered *)
	      let raw_cn_args : string list = 
		let cn_to_str: constructor -> rt_coninfo -> string list -> string list =
		  fun cn ci acc ->
		    let ord    = string_of_int(ci#order)^"" in
		    let cn_opt = if !the_options.diamond then (cn^":") else "" in
		    let dv     = sol#print ci#cvar in
		    let args   =
 		      let rec_rt_to_str_s: rec_rich_typ -> string = 
			function
			  | RSelfTyp(sid) when sid = id -> selfSymbol
			  | RSelfTyp(sid)               -> (selfSymbol^"{"^sid^"}")   (* Should this line raise an error immediately? - No! -> Mutually recursive Types. *)
			  | RFlatTyp(rt)                -> to_str_s rt
		      in List.map rec_rt_to_str_s ci#arg_typs  
		    in (ord^cn_opt^""^(String.concat "," (list_snoc args dv))^"")::acc (* Add "(" and ")" if printout is too messy *)
		in List.sort String.compare (rtci#fold cn_to_str [])
	      in
	      let ord_cn_args : string list = (* remove order tags *)
		if   !the_options.debug && !the_options.diamond
		then raw_cn_args 
		else let remove_ord: string -> string = compose (string_drop 1) (string_drop_until '') in
		     List.map remove_ord raw_cn_args
	      in String.concat "|" ord_cn_args
	    in 
	    if (params = "")
	    then id ^ "[" ^ cn_args ^ "]" 
	    else "(" ^ params ^ id ^ "[" ^ cn_args ^ "]" ^ ")"
	| RArrowTyp(rtd,fdin,fdout,rtr)    -> 
	    let dom = 
	      let raw_dom = (to_str_s rtd) in
	      match rtd with 
	      | RArrowTyp _
	      | RPolyTyp  _ -> "("^raw_dom^")"
	      | _           ->     raw_dom
	    in 
	    let res = to_str_s rtr in
	    let vdi = sol#print fdin  in
	    let vdo = sol#print fdout in
	    (dom^"-"^vdi^"/"^vdo^"->"^res)
	| RPolyTyp(alpha, phi, rt) ->
	    let vars = String.concat "," (IdSet.elements alpha) in
	    let cnts = "C["^(string_of_int (List.length phi))^"]" in
	    let solv = 
	      let cap = IdSet.inter alpha sol#domain in
	      if   IdSet.is_empty cap 
	      then ""
	      else 
		let vss =
		  let aux: variable -> string list -> string list = 
		    fun v acc -> 
		      (v^"->"^(sol#print_human v))::acc 
		  in String.concat "," (IdSet.fold aux cap []) 
		in (" with ["^vss^"] ")
	    in ("V "^vars^" in "^cnts^".("^(to_str_s rt)^")"^solv)
	      
    let to_string: rich_typ -> string = to_string_sol (new solution)

    let print_sol: int -> solution -> rich_typ -> unit =
      fun indent s rt -> 
	print_string (string_break (to_string_sol s rt) !the_options.screen_width '>' ("\n"^(String.make indent ' ')))
	  
    let print: int -> rich_typ -> unit =
      fun indent rt -> print_sol indent (new solution) rt      


    let is_polymorphic: rich_typ -> bool = (* Checks if this type is polymorphic *)
      function
	| RPolyTyp(_,_,_) -> true
	| other           -> false

    let rec is_function: rich_typ -> bool = (* Checks if this type is a function *)
      function
	| RArrowTyp(_,_,_,_) -> true
	| RPolyTyp(_,_,rt)   -> (is_function rt)
	| other              -> false

    let of_builtin: funcidentifier -> rich_typ = (* Builts rich_type of a given built-in function *)
      fun fid ->
	let (fdom, fin, fout, frng) = Builtin.signature fid 
	in RArrowTyp
	  (
	   (from_typ fdom),
	   (Constant.generate (Some(float_of_int fin ))), 
	   (Constant.generate (Some(float_of_int fout))), 
	   (from_typ frng)
	  )

    let is_builtin_polymorph: rich_typ -> bool =
      let rich_polymorph = from_typ (Builtin.polymorph) in
      fun rt -> (rt = rich_polymorph)

    let rec unfold_with: (typeidentifier * rich_typ) -> rich_typ -> rich_typ = 
     (* First argument is a constructor type tid and the type itself, 
        which encloses the second argument RSelfTypes(tid), 
        which are replaced by the given type.
	It shall be ok if the second of the first argument equals the second argument, i.e. "unfold_with (id,rt) rt" = "unfold_deep id rt"
      *)
      fun ((replacing_tid, replacing_rt) as rep_pair) ->
	let rec unfold_with_tid: rich_typ -> rich_typ =
	  function
	    | RLinPairTyp((fi,frt,fo),(si,srt,so)) ->
		let frt = unfold_with_tid frt in
		let srt = unfold_with_tid srt in
		RLinPairTyp((fi,frt,fo),(si,srt,so))
	    | (RConTyp(pars,id,rtct)) as rt ->
		let pars = List.map unfold_with_tid pars in
		let rtct =
		  if   id = replacing_tid 
		  then (* interior enclosing type found -> switch change replacing_rt type! *) 
		    rtct#map (unfold_rtci_with (id, rt)) 
		  else
		    rtct#map (unfold_rtci_with rep_pair) 
		in RConTyp(pars,id,rtct)
	    | RArrowTyp(fdom,fdin,fdout,frng) ->
		let fdom = unfold_with_tid fdom in
		let frng = unfold_with_tid frng in
		RArrowTyp(fdom,fdin,fdout,frng)
	    | RPolyTyp(vs,cnstrs,rt) ->
		let rt = unfold_with_tid rt in (*Is this really correct???*)
		RPolyTyp(vs,cnstrs,rt)  
	    | other -> other
	in unfold_with_tid


    and unfold_rtci_with:(typeidentifier * rich_typ) -> rt_coninfo -> rt_coninfo =
      fun  ((replacing_tid, replacing_rt) as rep_pair) ->
	let unfold_recrt: rec_rich_typ -> rec_rich_typ =
	  function
	    | RFlatTyp(rt)        -> RFlatTyp(unfold_with rep_pair rt)
	    | RSelfTyp(id) as rrt -> 
		if   id = replacing_tid 
		then RFlatTyp(replacing_rt)
		else rrt
	in
	fun rcti ->
	  let arg_typs = List.map unfold_recrt rcti#arg_typs in
	  new_rt_coninfo rcti#name rcti#typid rcti#own_typ arg_typs rcti#size rcti#cvar rcti#order (* Rebuild the coninfo with the changed arg_types *)      
	    
		
    let unfold_deep: typeidentifier -> rich_typ -> rich_typ = 
      (* Replace one level of  all occurrences of RSelfTyp(u_tid) by its proper enclosing type within a given rich_type *)
      (* We merely search for the proper enclosing type first. Unfolding it is then done by function "unfold". *)
      fun u_tid ->
	let rec unfold_deep_tid: rich_typ -> rich_typ = 
	  let unfold_deep_tid_rtci: rt_coninfo -> rt_coninfo =
	    let unfold_deep_tid_rec: rec_rich_typ -> rec_rich_typ =
	      function
		| RFlatTyp(rt)        -> RFlatTyp(unfold_deep_tid rt)
		| RSelfTyp(id) as rrt -> 
		    if   u_tid = id 
		    then Support.err ("While unfolding type '"^u_tid^"': Self parameter encountered which is not enclosed by its own type.")  (* We shall replace occurences of u_tid, but in order to do this, we must first identify the enclosing type! I.e. we can unfold a[#{a}] or a[b[#{a}]] but not a[#{b}] *)
		    else rrt
	    in
	    fun rcti ->
	      let arg_typs = List.map unfold_deep_tid_rec rcti#arg_typs 
	      in  new_rt_coninfo rcti#name rcti#typid rcti#own_typ arg_typs rcti#size rcti#cvar rcti#order (* Rebuild the coninfo with the changed arg_types *)
	  in
	  fun rt ->
	    match rt with
	    | RLinPairTyp((fi,frt,fo),(si,srt,so)) ->
		let frt = unfold_deep_tid frt in
		let srt = unfold_deep_tid srt in
		RLinPairTyp((fi,frt,fo),(si,srt,so))
	    | RConTyp(pars,id,rtct) ->
		if id = u_tid 
		then unfold_with (u_tid, rt) rt
		else
		  let pars = List.map unfold_deep_tid pars in
		  let rtct = rtct#map unfold_deep_tid_rtci in
		  RConTyp(pars,id,rtct)
	    | RArrowTyp(fdom,fdin,fdout,frng) ->
		let fdom = unfold_deep_tid fdom in
		let frng = unfold_deep_tid frng in
		RArrowTyp(fdom,fdin,fdout,frng)
	    | RPolyTyp(vs,cnstrs,rt) ->
		let rt = unfold_deep_tid rt in (*Is this really correct???*)
		RPolyTyp(vs,cnstrs,rt)  
	    | other -> other
	in unfold_deep_tid

    let unfold_args: (typeidentifier * rich_typ) -> (rec_rich_typ list) -> (rich_typ list) =
      fun (replacing_tid, replacing_rt) ->
	let unfold_args_aux: rec_rich_typ -> rich_typ =
	  function
	    | RFlatTyp(rt) -> unfold_with (replacing_tid, replacing_rt) rt
	    | RSelfTyp(id) -> 
		if   id = replacing_tid 
		then replacing_rt
		else Support.err ("Unenclosed top-level self #{"^id^"} encountered while unfolding type '"^replacing_tid^"'.") (* A self without a proper enclosing type encountered! *)
	in List.map unfold_args_aux


    let unfold_a2b: (rich_typ * rich_typ) -> rich_typ = 
      (* Expand first argument by (partially) unfolding/folding to match second, i.e.
	 (a[b[#{a}]], a[b[a[#{b}]]) returns first argument partially unfolded to  a[b[a[#{b}]]
	 No constraints are produced, as net operation will be partial unfolding if first is shorter.
	 So the first argument must be shorter to ensure soundness!
       *)
      let rec unfold_a2b_aux: (rich_typ context) -> (rich_typ * rich_typ) -> rich_typ = 
	fun ctxt (rta, rtb) ->
	  let unfold_a2b = unfold_a2b_aux ctxt in
	  match (rta, rtb) with
	  | (_,_) when rta = rtb -> rta

	  | (RLinPairTyp((a_fin,a_frt,a_fout),(a_sin,a_srt,a_sout)),
	     RLinPairTyp((b_fin,b_frt,b_fout),(b_sin,b_srt,b_sout))) 
	    -> 
	      let a_frt = unfold_a2b (a_frt,b_frt) in   
	      let a_srt = unfold_a2b (a_srt,b_srt) in   
	      RLinPairTyp((a_fin,a_frt,a_fout),(a_sin,a_srt,a_sout))

	  | (RArrowTyp(a_dom,a_din,a_dout,a_rng),
	     RArrowTyp(b_dom,b_din,b_dout,b_rng)) 
	    ->
	      let a_dom = unfold_a2b (a_dom,b_dom) in 
	      let a_rng = unfold_a2b (a_rng,b_rng) in 
	      (* RECONSIDER: Rather contravariant instead of covariant? *)
	      RArrowTyp(a_dom,a_din,a_dout,a_rng)

	  | (RConTyp(a_params,a_tid,a_rtct),
	     RConTyp(b_params,b_tid,b_rtct)) when a_tid = b_tid 
	    ->
	      let a_params =List.map2 (Common.curry unfold_a2b) a_params b_params in
	      let unfold_a2b = unfold_a2b_aux (ctxt#add a_tid rta) in
	      let a_rtct =
		let a2b_rtci: constructor -> rt_coninfo -> rt_coninfo =
		  fun con a_rtci ->
		    let b_rtci = b_rtct#lookup con in
		    let a_arg_typs =
		      let a2b_arg: rec_rich_typ -> rec_rich_typ -> rec_rich_typ =
			fun a_rrt b_rrt ->
			  match (a_rrt, b_rrt) with
			  | (RFlatTyp(RConTyp(_,a_arg_id,_)),
			     RSelfTyp(b_arg_id)) 
			    when a_arg_id = b_arg_id 
			    -> b_rrt 
			  | (RSelfTyp(a_arg_id),
			     RFlatTyp((RConTyp(_,b_arg_id,_)) as b_arg_rt ))
			    when a_arg_id = b_arg_id ->
			      RFlatTyp(unfold_a2b((ctxt#lookup a_arg_id), b_arg_rt)) (* Unfold, first argument, but recurse to ensure partial unfold! - Otherwise nonterminating *)
				
			  | (RSelfTyp(a_arg_id),RSelfTyp(b_arg_id)) 
			    when a_arg_id = b_arg_id 
			    -> a_rrt
				
			  | (RFlatTyp(a_arg_rt),RFlatTyp(b_arg_rt)) -> 
			      RFlatTyp(unfold_a2b(a_arg_rt, b_arg_rt))
				
			  | _ -> raise (Invalid_argument "unfold_a2b: pair of structurally mismatching types")
				
		      in  List.map2 a2b_arg a_rtci#arg_typs b_rtci#arg_typs
		    in new_rt_coninfo a_rtci#name a_rtci#typid a_rtci#own_typ a_arg_typs a_rtci#size a_rtci#cvar a_rtci#order (* Rebuild the coninfo with the changed arg_types *)      
		in a_rtct#mapi a2b_rtci
	      in RConTyp(a_params,a_tid,a_rtct)

	  | (_,_) -> raise (Invalid_argument "unfold_a2b: pair of structurally mismatching types")

      in  unfold_a2b_aux (new context)

	  
    let rec free_cvars: rich_typ -> idset =   (* returns all _free_ constraint variables contained in a rich_typ; polymorph type bind resource variables. *)
      function
	| RUnitTyp         
	| RDiamondTyp  
	| RTvarTyp(_)
	| RBoolTyp    
	| RIntTyp   
	| RFloatTyp 
	| RCharTyp
	| RStringTyp       
	  ->
	    IdSet.empty
	| RLinPairTyp((fi,frt,fo),(si,srt,so)) 
	  -> 
	    IdSet.add fi (
	    IdSet.add fo (
	    IdSet.add si (
	    IdSet.add so (
	    IdSet.union (free_cvars frt) (free_cvars srt) ))))
	| RConTyp(pars,tid,rtct)      
	  -> 
	    let pars_fcvs = List.fold_left (fun acc par_rt -> IdSet.union (free_cvars par_rt) acc) IdSet.empty pars in
	    let constructor_fcvs: constructor -> rt_coninfo -> idset -> idset =
	      fun constr rtci acc ->
		List.fold_left 
		  (fun acc2 rt -> 
		    match rt with
		    | RSelfTyp(tid') ->
			acc2  (* - we need not check selftype here, and we MUST not if we are to accept mutual recursive types *)
                              (* if   tid = tid' 
				 then acc2
				 else Support.err ("Constructor-argument with different self encountered ("^tid^"/"^tid'^").")
			       *)
		    | RFlatTyp(rtfl) ->
			IdSet.union (free_cvars rtfl) acc2
		  ) 
		  (IdSet.add rtci#cvar acc) 
		  rtci#arg_typs
	    in rtct#fold constructor_fcvs pars_fcvs  (* Instead of a final union, we feed pars_fcvs as the initial accumulator *)
	| RArrowTyp(fdom,fdin,fdout,frng) 
	  ->
	    IdSet.add fdin (IdSet.add fdout (IdSet.union (free_cvars fdom) (free_cvars frng)))
	| RPolyTyp(vs,cnstrs,rt)      
	  ->
	    IdSet.diff (IdSet.union (free_cvars rt) (Constrs.free_cvars cnstrs)) vs
	      
    let cvar_signature: rich_typ -> (idset * idset) = (* identifies in-dvars and out-dvars in order to construct objective function for lp *)
      (* Example: ((A -> B) -> (C -> D)) -> (E -> (F -> G)), assume a = all cvars in A, b = all cvars in B, ...
	 then we get (acef,bdg) 
       *)
      let rec c_sig_aux: bool -> rich_typ -> (idset * idset) = (* identifies in-dvars and out-dvars in order to construct objective function for lp *)
	fun lhs rt ->
	  begin match rt with
	  | RLinPairTyp((fin,frt,fout),(sin,srt,sout)) ->
	      let (left,right) = 
		let (finvs, foutvs) = c_sig_aux lhs frt in
		let (sinvs, soutvs) = c_sig_aux lhs srt in
		((IdSet.union finvs  sinvs ),
		 (IdSet.union foutvs soutvs))
	      in
	      ((IdSet.add fin  (IdSet.add sin  left)),
	       (IdSet.add fout (IdSet.add sout right)))
	  | RArrowTyp(fdom,fdin,fdout,frng) ->
	      let (domin,domout) = c_sig_aux true  fdom in
	      let (rngin,rngout) = c_sig_aux false frng in
	      ((IdSet.add fdin  (IdSet.union domin  rngin )),
	       (IdSet.add fdout (IdSet.union domout rngout)))
	  | RPolyTyp(alpha,phi,rt) ->
	      let (invs, outvs) = c_sig_aux lhs rt in
	      (* We do not know any preference on the cvars only occurring free in phi, hence we ignore them here *)
	      ((IdSet.diff  invs alpha),
	       (IdSet.diff outvs alpha))
	  | other -> 
	      if   lhs
	      then ((free_cvars other),       IdSet.empty )
	      else ( IdSet.empty,       (free_cvars other))
	  end
      in c_sig_aux false

    let rec substitute: cvar -> cvar -> rich_typ -> rich_typ = (* replaces all occurrences of the first variable by the second within the given rich_typ *)
      fun cvout cvin rt ->
	let replace: cvar -> cvar = 
	  fun old -> 
	    if   old = cvout 
	    then cvin 
	    else 
	      let _ = if old = cvin then Support.warning ("Substituting '"^cvout^"' with '"^cvin^"', but '"^old^"' already contained in given rich type.")
	      in      old
	in
	let rec_subs: rich_typ -> rich_typ = substitute cvout cvin in (* Just a convenient shortcut for readability *)
	match rt with
	| RLinPairTyp((fin,frt,fout),(sin,srt,sout))        
	  -> RLinPairTyp(
	    ((replace fin),(rec_subs frt),(replace fout)),
	    ((replace sin),(rec_subs srt),(replace sout))
	   )

	| RConTyp(params,tid,rtct)    ->
	    let substitute_coninfo: rt_coninfo -> rt_coninfo =
	      fun ci ->
		let substed_args = 
		  let subs_rec_rich_typ: rec_rich_typ -> rec_rich_typ =
		    begin
		      function
			| RSelfTyp(tid)     -> RSelfTyp(tid)
			| RFlatTyp(frt)     -> RFlatTyp(rec_subs frt)
		    end
		  in List.map subs_rec_rich_typ ci#arg_typs
		in new_rt_coninfo ci#name tid ci#own_typ substed_args ci#size (replace ci#cvar) ci#order
	    in  RConTyp((List.map rec_subs params), tid, rtct#map substitute_coninfo)

	| RArrowTyp(dom,din,dout,rng) -> RArrowTyp(rec_subs dom, replace din, replace dout, rec_subs rng)

	| RPolyTyp(alpha, phi, rt)    -> (* Nested polymoprh types should not occur, but I doesnt hurt to have this routine ready yet *)
	    if  (IdSet.mem cvout alpha) 
	    then (* Stop substituting *)
	      let _ = Support.warning  ("Substituting '"^cvout^"' with '"^cvin^"', but '"^cvout^"' is bound in polymorph type, hence further substitution omitted.")
	      in RPolyTyp(alpha, phi, rt)
	    else (* Substitute anyway *)
	      let _ = 
		if (IdSet.mem cvin  alpha) 
		then Support.warning  ("Substituting '"^cvout^"' with '"^cvin^"', but '"^cvin^"' is bound in polymorph type, hence there might be something wrong here. Still Substituting without renaming.")
	      in RPolyTyp(alpha, (Constrs.map_substitute cvout cvin phi), (rec_subs rt))

	| other -> other (* All other rich types do not carry cvars *)

    let rec restrict_aux: (rich_typ * rich_typ) -> constr list = (* Restricts first type to second, exactly as in the theoretical paper *)
      function
	| (RUnitTyp, RUnitTyp)     
	| (RDiamondTyp, RDiamondTyp)           -> []
	
	| (RTvarTyp(a),RTvarTyp(b)) when a = b -> []
	
	| (RBoolTyp, RBoolTyp)     
	| (RIntTyp, RIntTyp)       
	| (RFloatTyp, RFloatTyp)   
	| (RCharTyp, RCharTyp)     
	| (RStringTyp, RStringTyp)             -> [] 
	
	| (RLinPairTyp((a_fin,a_frt,a_fout),(a_sin,a_srt,a_sout)),RLinPairTyp((b_fin,b_frt,b_fout),(b_sin,b_srt,b_sout)))
	   -> 
	     (Constrs.dominate "Res" b_fin  a_fin ) ::
	     (Constrs.dominate "Res" a_fout b_fout) ::
	     (Constrs.dominate "Res" b_sin  a_sin ) ::
	     (Constrs.dominate "Res" a_sout b_sout) ::
	     (list_fast_append (restrict_aux (a_frt,b_frt)) (restrict_aux (a_srt,b_srt)))
	
	| (RConTyp(a_params,a_tid,a_rtct),RConTyp(b_params,b_tid,b_rtct)) when a_tid = b_tid 
	  ->
	    let restrict_aux_fold2: (constr list) -> (rich_typ list) -> (rich_typ list) -> (constr list) =
	      List.fold_left2 (fun acc a b -> list_fast_append (restrict_aux (a,b)) acc) (* maybe catch different length here:      with (Invalid_argument "List.fold_left2") -> raise (Invalid_argument "Restricting type A to B failed: different number of parameters/arguments with constructor types.") *)
	    in
	    let param_constrs = restrict_aux_fold2 [] a_params b_params in
	    let constructors_constraints =
	      let restrict_aux_coninfo: constructor -> rt_coninfo -> (constr list) -> (constr list) =
		fun constr_name a_rtci acc ->
		  let b_rtci = b_rtct#lookup constr_name 
		  in (* restrict_aux_fold2 - doesnt work, as we have to deal with the rec_rich_typ first *) 			
		  let this_constr_constraints =
		    List.fold_left2 
		    (fun acc a b -> 
		      match (a,b) with
		      | (RSelfTyp(ida), RSelfTyp(idb)) -> 
			  if ida = idb (* This check is necessary, especially in the presence of mutual types! *)
			  then acc 
			  else Support.err ("Cannot restrict different self types: '"^ida^"' and '"^idb^"'.") 
		      | (RFlatTyp(RConTyp(_,ida,_)), RSelfTyp(idb)) ->
			  if ida = idb
			  then raise (IncompatibleFoldDepth(true, idb))     (* Exception produced by restricting a[b[a[b[#{a}]]]] to a[b[#{a}]] *)
			  else Support.err ("Cannot restrict different self types (with different fold depth): '"^ida^"' and '"^idb^"'.") 
		      | (RSelfTyp(ida), RFlatTyp(RConTyp(_,idb,_))) ->
			  if ida = idb
			  then raise (IncompatibleFoldDepth(false, ida))     (* Exception produced by restricting a[b[a[b[#{a}]]]] to a[b[#{a}]] *)
			  else Support.err ("Cannot restrict different self types  (with different fold depth): '"^ida^"' and '"^idb^"'.") 
		      | (RFlatTyp(ta),  RFlatTyp(tb)) -> 
			  list_fast_append (restrict_aux (ta,tb)) acc
		      | _ -> Support.err ("Restricting constructor types failed: Argument typs of constructor '"^constr_name^"' differ.") 
		    )
		    [(Constrs.dominate "Res" a_rtci#cvar b_rtci#cvar)]  (* The one constraint originating directly from the constructor here! *)
		    a_rtci#arg_typs 
		    b_rtci#arg_typs
		  in list_fast_append this_constr_constraints acc
	      in  a_rtct#fold restrict_aux_coninfo []
	    in list_fast_append param_constrs constructors_constraints
	
	| (RArrowTyp(a_dom,a_din,a_dout,a_rng),RArrowTyp(b_dom,b_din,b_dout,b_rng)) ->
	    (Constrs.dominate "Res" b_din  a_din)  :: 
	    (Constrs.dominate "Res" a_dout b_dout) ::
	    (
	     if is_builtin_polymorph a_dom 
	     then
	       if is_builtin_polymorph a_rng
	       then (* Builtin-function is truly polymorphic in both arguments, hence returns its argument, hence result restricts input *)
		 restrict_aux (b_rng, b_dom)
	       else (* Builtin-function is truly polymorphic only within argument: Only restrict result *)
		 restrict_aux (a_rng, b_rng)
	     else (* No true polymorphism due to a possibly builtin function *)
	       (list_fast_append
		  (restrict_aux (b_dom, a_dom))
		  (restrict_aux (a_rng, b_rng))
	       )
	    )
	    	
	| (RPolyTyp(alpha, phi, rt), other) ->
	    let phi = (Constrs.Comment Constrs.substitutions_linetag) :: phi in (* To record all following subsitutions *)
	    let sigma_phi, sigma_rt =
	      let sub_single: cvar -> ((constr list) * rich_typ) -> ((constr list) * rich_typ) =
		fun old_cv (cnstr_acc,rt_acc) -> 
		  let new_cv = Cvar.rename old_cv in (* Get the new name and replace it in the constraints and the type at once, therefore we do not need to record the substitution sigma *)
		  ((Constrs.map_substitute old_cv new_cv cnstr_acc),
		   (substitute old_cv new_cv rt_acc))
	      in IdSet.fold sub_single alpha (phi,rt)
	    in list_fast_append (restrict_aux (sigma_rt, other)) sigma_phi
	      
	| (ty_a,ty_b) -> Support.err ("Type restriction failed: "^(Support.exn_sep)^
				      "  "^(to_string ty_a)^(Support.exn_sep)^
				      "cannot be restricted to "^(Support.exn_sep)^
				      "  "^(to_string ty_b)^".")

    let rec restrict: (rich_typ * rich_typ) -> constr list = (* Restricts first type to second, exactly as in the theoretical paper *)
      fun (rta,rtb) ->
	let restriction_info = (Constrs.undefined_linetag^": \n  "^(to_string rta)^" restrict to \n  "^(to_string rtb)^" \n") in 
	(* let _ = print_string restriction_info (* !!! DEBUG !!! *) in *)
	begin
	  try 
	    Constrs.Comment(restriction_info)::restrict_aux (rta,rtb)
	  with IncompatibleFoldDepth(firstlonger,tid) ->
	    if   firstlonger
	    then (* unfold second *)
	      restrict (rta, (unfold_deep tid rtb)) 
	    else (* we cannot unfold first, for we must suffer non-termination: 
		    a[b[a[#{b}]] restrict to
		    a[b[#{a}]] -unfold-> a[b[a[b[#a]]]]
		  *)
	      let rta = 
		try  unfold_a2b (rta,rtb)
		with x -> Support.bug ("Restricting failed: "^restriction_info) x 
	      in restrict (rta,rtb)
	end

    let rec share: rich_typ -> (rich_typ * rich_typ * (constr list)) =   (* Sharing of rich types mostly done as in the theoretical paper - throws special exception 'NotShareable "..."' in order to safely test linearity of a type. *)
      begin
      function
	| RDiamondTyp               -> raise (Support.NotSharable "Diamond type is not shareable")
	| RUnitTyp        
	| RTvarTyp(_)     
	| RBoolTyp        
	| RIntTyp         
	| RFloatTyp       
	| RCharTyp        
	| RStringTyp      
	| RArrowTyp(_,_,_,_)  
	| RPolyTyp(_,_,_)  as t     -> (t, t, []) (* These types may freely be shared *)
	| RLinPairTyp(f,s)          -> raise (Support.NotSharable "All linear pairs are not shareable")
	| RConTyp(params,tid,rtci)  -> 
	    let param_a, param_b, param_constr = list_map_map_fold share params [] in (* List order must be preserved *)
	    let rtct_a, rtct_b, rtct_constrs =
	      let rec share_rtci: rt_coninfo -> (rt_coninfo * rt_coninfo * (constr list)) =
		fun rtci ->
		  let cvar_a = Cvar.rename rtci#cvar in 
		  let cvar_b = Cvar.rename rtci#cvar in (* Yes, name generation uses side-effects to generate unique names *)
		  let args_a, args_b, args_constrss =
		    let share_recrt: rec_rich_typ -> (rec_rich_typ * rec_rich_typ * (constr list)) =
		      begin
			function
			  | RSelfTyp(i) -> (RSelfTyp(tid), RSelfTyp(tid), []) 
			  | RFlatTyp(t) -> 
			      let rta,rtb,c = share t 
			      in (RFlatTyp(rta), RFlatTyp(rtb), c)
		      end
		    in list_map_map_fold share_recrt rtci#arg_typs [] (* List order must be preserved *)
		  in
		  let rtci_a = new_rt_coninfo rtci#name tid rtci#own_typ args_a rtci#size cvar_a rtci#order in
		  let rtci_b = new_rt_coninfo rtci#name tid rtci#own_typ args_b rtci#size cvar_b rtci#order in
		  (rtci_a, rtci_b, ((Constrs.gt_eq "Shr" (1,rtci#cvar) [(1,cvar_a);(1,cvar_b)])::args_constrss) )
	      in rtci#fold (* its a map map fold really *)
		(fun constr rtci (acc_rtct_a, acc_rtct_b, acc_constr) -> 
		  let rtci_a, rtci_b, rtci_constr = share_rtci rtci
		  in  ((acc_rtct_a#bind constr rtci_a), (acc_rtct_b#bind constr rtci_b), (list_fast_append rtci_constr acc_constr))
		)
		(new_rt_contab, new_rt_contab, [])
	    in (RConTyp(param_a,tid,rtct_a), RConTyp(param_b,tid,rtct_b), (list_fast_append param_constr rtct_constrs))
      end   

    let is_linear: rich_typ -> bool = (* Checks if this type is linear *)
      fun rt ->
	try 
	  begin
	    match share rt with
	    | (_, _,  [] ) -> false
	    | (_, _, _::_) -> true
	  end
	with Support.NotSharable _ -> true 
	  
  end


(* Enriched Contexts *)
class rich_context = 
  object (self: 'self)
    inherit [rich_typ] context
    method value_name  = "rich type"

    method linear: idset =
      self#fold 
	(fun v rt acc -> 
	  if   Rich_typ.is_linear rt 
	  then IdSet.add v acc 
	  else acc) 
	IdSet.empty
	
    method restrict_nonlinear: 'self =
      self#restrict (fun v rt -> not (Rich_typ.is_linear rt))
	
    method restrict_to_nonlinear: idset -> 'self = (* More efficient than calling #restrict_to and #restrict_nonlinear *)
      fun vars -> 
	self#restrict 
	  (fun v rt -> 
	    if   (IdSet.mem v vars)
	    then 
	      begin 
		if   (Rich_typ.is_linear rt)
		then raise (Invalid_argument ("Context contains linear variable '"^v^":"^(Rich_typ.to_string rt)^"'."))
		else true
	      end 
	    else false
	  )

    method size: int =
      let count: variable -> rich_typ -> int -> int =
	fun v rt acc -> acc + (Rich_typ.size rt)
      in self#fold count 0

    method strip: typ context =
      self#fold (fun v rt acc -> acc#bind v (Rich_typ.to_typ rt)) (new context)

    method split: (idset * idset) -> ('self * 'self * (constr list)) = (* Splits a context, sharing when needed *)
      fun (a_vs,b_vs) ->
	let split_aux: variable -> rich_typ -> ('self * 'self * constr list) -> ('self * 'self * constr list) =
	  fun v rt (ctxt_a, ctxt_b, acc_share) ->
	    match ((IdSet.mem v a_vs), (IdSet.mem v b_vs)) with
	    | (true,  true ) -> let (rta,rtb,share) = Rich_typ.share rt in
 	                        ((ctxt_a#bind v rta), (ctxt_b#bind v rtb), (list_fast_append share acc_share))
	    | (true,  false) -> ((ctxt_a#bind v rt ),  ctxt_b            ,                         acc_share )
	    | (false, true ) -> ( ctxt_a            , (ctxt_b#bind v rt ),                         acc_share )
	    | (false, false) -> ( ctxt_a            ,  ctxt_b            ,                         acc_share )
	in  self#fold split_aux (self#empty, self#empty, [])  
	  
    method fv_rts: idset = (* Returns all resource variables contained in the rich types of the contained variables *)
      self#fold (fun v rt acc -> IdSet.union (Rich_typ.free_cvars rt) acc) IdSet.empty
end













    

