(* 

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: common.ml,v $
   Id:       $Id: common.ml,v 1.1 2004/12/07 17:23:39 sjost Exp $ 

	

   What this File is all about:
   ----------------------------
   This file captures commonly used functions and definitions or general functions,
   it _shall_ not relate to relate to anything specific for lfd_infer.
   
   Sometimes we add/replace functions of the OCaml library, e.g. 
   like "List.rev_append" which is 'replaced' by "list_fast_append".
   This is useful for changing them easily later and distinguishing between the cases were reversal is essential and when not.

*)

(* ToDos / ToReconsiders: 

    - rewrite functions tagged with 'rewrite efficiently', eg. 'list_remove_last'
    - rewrite some functions using imperative features for performance reason if necessary.
*)


(* UNIX macht Spass *)
let pipe2prog: string -> (string array) -> (out_channel * in_channel) =
  fun prog args ->
    let (  toext_read,   toext_write) = Unix.pipe () in (* Pipe from Parent zum Child *)
    let (fromext_read, fromext_write) = Unix.pipe () in (* Pipe from Child zum Parent *)
    if (Unix.fork ()) <> 0                           (* Jetzt bin ich zweimal da! *)
    then (* Parent Process *)
      (
       Unix.close    toext_read;       (* Der Vater hoert sich nicht selbst zu *)
       Unix.close  fromext_write;      (* Der Vater spricht sich nicht selbst an *)
       let  inch = Unix.in_channel_of_descr  fromext_read  in
       let outch = Unix.out_channel_of_descr   toext_write in
       set_binary_mode_in   inch false;
       set_binary_mode_out outch false;
       (outch, inch)
      )
    else (* Child Process *)
      (
       Unix.dup2   toext_read Unix.stdin;   (* Das Kind soll dem Vater ueber stdin zuhoeren *)
       Unix.close  toext_read;
       Unix.close  toext_write;               (* Das Child soll nicht zu sich reden *)
       Unix.dup2   fromext_write Unix.stdout; (* Das Kind soll dem Vater ueber stdout ansprechen *)
       Unix.close  fromext_write;
       Unix.close  fromext_read;            (* Das Child soll sich nicht selbst zuhoeren *)
       try 
	 Unix.execvp prog args;
	 (* This line is never reached *)
	 assert false
       with _ -> raise (Sys_error ("Calling the external program '"^prog^"' failed."))
      )


(* A simple timing fascility *)
class timer =          (* This class produces side-effects! *)
  object (self)
    val mutable t_start   = 0.
    val mutable t_time    = 0.
    val mutable t_running = false
    val mutable t_epsilon = 0.03     (* All measured times smaller than epsilon are suppressed by print_pretty *)

    method reset  = 
      t_time    <- 0.; 
      self#resume

    method resume =
      t_running <- true;
      t_start   <- (Sys.time ())

    method stop   = 
      t_time    <- self#time;
      t_running <- false

    method time   = 
      let   t_stop = Sys.time () in
      if    t_running 
      then (t_time +. (t_stop -. t_start))
      else  t_time

    method print =                  (* prints time in parenthesis *)
      begin
	print_string "(";
	print_float self#time;
	print_string "s)"
      end
	
    method set_epsilon e =
      t_epsilon <- e

    method print_pretty =           (* prints time in parenthesis, but only if greater than epsilon *)
      let t_current = self#time in
      if    t_current >= t_epsilon
      then  
	begin
	  print_string "(";
	  print_float t_current;
	  print_string "s)"
	end

    initializer self#reset
  end
    
(* General functions *)

let pr_snd: 'a -> 'b -> 'b = (* Curried second projection *)
  fun a b -> b

let tripel_fst: ('a * 'b * 'c) -> 'a =
  fun (a,_,_) -> a
let tripel_snd: ('a * 'b * 'c) -> 'b =
  fun (_,b,_) -> b
let tripel_trd: ('a * 'b * 'c) -> 'c =
  fun (_,_,c) -> c


let compose: ('b -> 'c) -> ('a -> 'b) -> ('a -> 'c) =
  fun f g x ->          f (g x)

let compose2: ('c -> 'd) -> ('a -> 'b -> 'c) -> 'a -> 'b -> 'd =
  fun f g x -> compose  f (g x)

let compose3: ('d -> 'e) -> ('a -> 'b -> 'c -> 'd) -> 'a -> 'b -> 'c -> 'e =
  fun f g x -> compose2 f (g x)

let compose4 =
  fun f g x -> compose3 f (g x)
      
(* We cant type the general definition:
  function
   | 0              -> (fun f g x -> f (g x))
   | i when (i > 0) -> (fun f -> compose_n (i-1) (compose f))
   | _              -> raise (Invalid_argument "compose_n")
*)

let curry: (('a * 'b) -> 'c) -> ('a -> 'b -> 'c) =
  fun f a b -> f (a,b)
      
let uncurry: ('a -> 'b -> 'c) -> ('a * 'b) -> 'c =
  fun f (a,b) -> f a b

let flip: ('a -> 'b -> 'c) -> 'b -> 'a -> 'c =
  fun f y x -> f x y 



let save_increment: int -> int = (* Increments and checks for overflow *)
  fun x -> 
    if x >= x+1 
    then
      failwith "Integer overflow detected!"  
    else x+1

let save_ref_increment: int ref -> int = (* Increments references and checks for overflow *)
  fun x ->
    let old_val = !x in
    let _ = x := !x + 1 in
    if old_val >= !x 
    then 
      failwith "Integer overflow detected!"  
    else !x

  


(* General functions on lists *)
let mapfold: (('a * 'b) -> ('a * 'c)) -> ('a * ('b list)) -> ('a * ('c list)) =
  fun f (ini,l) ->
    let (res,l') =
      List.fold_left (fun (a,cs) b -> let (a',c) = f (a,b) in (a',c::cs)) (ini,[]) l
    in (res, List.rev l')
      
let mapfold': (('a * 'b) -> ('a * 'c)) -> ('a * ('b list)) -> ('a * ('c list)) =
  fun f (ini,l) ->
    List.fold_right (fun b (a,cs) -> let (a',c) = f (a,b) in (a',c::cs)) l (ini,[])  
      
      (* Example usage:
	 # mapfold  (fun (a,b) -> (a+1,(a,b))) (1,[1;2;3]);;
	 - : int * (int * int) list = (4, [(1, 1); (2, 2); (3, 3)])
	 # mapfold' (fun (a,b) -> (a+1,(a,b))) (1,[1;2;3]);;
	 - : int * (int * int) list = (4, [(3, 1); (2, 2); (1, 3)])
       *)

let rec list_snoc: 'a list -> 'a -> 'a list = (* Append an element at the end of a list *)
  fun l e ->
    match l with
    | []    -> [e]
    | x::xs -> x::(list_snoc xs e)

let rec list_fast_append: 'a list -> 'a list -> 'a list = (* List.rev_append *) (* Beware: Lists get reversed, hence verifiy that it is only applied to constr lists, where the order does not matter! *)
  fun xs1 xs2 ->
    match xs1 with
    | []       -> xs2
    | x::xs1'  -> list_fast_append xs1' (x::xs2) 



let rec list_concat: (('a list) list) -> 'a list = (* List.concat *)
   fun xss -> list_concat_aux [] xss 
       
and list_concat_aux: 'a list -> (('a list) list) -> 'a list =
  fun acc xss ->
    (
     match xss with
     | [] -> acc
     | x::xs -> 
	 let acc' = list_fast_append x acc in
	 list_concat_aux acc' xs
    )

let list_concat_map: ('a -> 'b list) -> 'a list -> 'b list =
  fun f -> List.fold_left (fun acc x -> List.append (f x) acc) [] 
 (* Is this more efficient? It seems that inlining the code below  is more efficient *)
(* fun f l -> List.concat (List.map f l) *)

let list_fold_left3: ('a -> 'b -> 'c -> 'd -> 'a) -> 'a -> 'b list -> 'c list -> 'd list -> 'a = (* tail-recursive *)
  fun f ->
    let rec lfl3_aux = 
      fun acc bs cs ds ->
	match (bs,cs,ds) with
	| ((hb::tb), (hc::tc), (hd::td))  ->  lfl3_aux (f acc hb hc hd) tb tc td
	| (     [] ,      [] ,      [])   ->              acc
	|  _                              ->  raise (Invalid_argument "list_fold_left3")
    in lfl3_aux

(* Is this a better variant than the one below ?
let list_fold_right3: ('b -> 'c -> 'd -> 'a -> 'a) -> 'b list -> 'c list -> 'd list -> 'a -> 'a = (* NOT tail-recursive *)
   fun f ->
   let rec lfl3_aux = 
   fun bs cs ds ->
	match (bs,cs,ds) with
	| ((hb::tb), (hc::tc), (hd::td))  ->  f' hb hc hd (lfl3_aux tb tc td)
	| (     [] ,      [] ,      [])   ->  fun x -> x
	|  _                              ->  fun _ -> raise (Invalid_argument "list_fold_right3")
    in lfl3_aux
  *)

let list_fold_right3: ('b -> 'c -> 'd -> 'a -> 'a) -> 'b list -> 'c list -> 'd list -> 'a -> 'a = (* NOT tail-recursive *)
  fun f ->
    let rec lfl3_aux = 
      fun bs cs ds x ->
	match (bs,cs,ds) with
	| ((hb::tb), (hc::tc), (hd::td))  ->  f hb hc hd (lfl3_aux tb tc td x)
	| (     [] ,      [] ,      [])   ->  x
	|  _                              ->  raise (Invalid_argument "list_fold_right3")
    in lfl3_aux


    
let list_map3: ('d -> 'a * 'b * 'c) -> 'd list -> (('a list) * ('b list) * ('c list)) = (* Preserves order *)
  fun f ds -> 
    let aux_f = 
      fun d (acc_a,acc_b,acc_c) -> 
	let a,b,c = f d in (a::acc_a, b::acc_b, c::acc_c)
    in List.fold_right aux_f ds ([],[],[])

let list_map_map_fold: ('d -> 'a * 'b * 'c list) -> 'd list -> 'c list -> (('a list) * ('b list) * ('c list)) = (* Preserves order *)
  fun f ds cs -> 
    let aux_f = 
      fun d (acc_a,acc_b,acc_c) -> 
	let a,b,c = f d in (a::acc_a, b::acc_b, c@acc_c)
    in List.fold_right aux_f ds ([],[],cs)
      
let list_map_over_two:  ('a * 'b -> 'c * 'd) -> ('a list * 'b list) -> ('c list * 'd list) =
  fun f (xs,ys) ->
  List.split (List.map f (List.combine xs ys)) (* INEFFICIENT!!!  rewrite efficiently *)

let list_remove: 'a -> 'a list -> 'a list = (* Remove _all_ occurences from list *)
  fun e -> List.filter ((<>) e) 

let rec list_remove_once: 'a -> 'a list -> 'a list = (* Removes given element exactly once *)
  fun a l ->
    match l with
    | []     -> raise Not_found
    | x::xs  -> 
	if a = x 
	then xs
	else x::(list_remove_once a xs)

let rec list_nub: 'a list -> 'a list = (* Remove all duplicates from a list - Very Expensive ! *)
  fun xs -> list_nub_aux [] xs 
and list_nub_aux: 'a list -> 'a list -> 'a list = 
  fun acc l ->
    match l with
    | []     -> acc
    | x::xs  -> 
	if   (List.mem x acc)
	then list_nub_aux acc xs
	else list_nub_aux (x::acc) xs

(* The one BELOW IS quite INEFFICIENT, but preserves order..
let rec list_nub: 'a list -> 'a list = (* Remove all duplicates from a list *)
  function 
    | []     -> []
    | x::xs  -> x :: (list_nub (list_remove x xs))
*)

		      
(* Same problem as in GET_LAST here: eta-expansion is required! *)
(*
let list_diff: 'a list -> 'a list -> 'a list = 
  List.fold_left (fun xs y -> list_remove y xs) 
*)

let rec list_diff: 'a list -> 'a list -> 'a list = (* List-Difference, always removes all occurences *)
  function 
    | []    -> fun id -> []
    | x::xs -> fun ys -> 
	if   List.mem x ys
	then     list_diff xs ys
	else x::(list_diff xs ys)

let list_intersection: 'a list -> 'a list -> 'a list =
  fun xs -> List.filter (fun y -> List.mem y xs)

let list_splitat: 'a list -> int -> ('a list * 'a list) = (* Rewrite more efficently or use fold *)
  fun l i ->
    let rec lsplit_aux: 'a list -> 'a list -> int -> ('a list * 'a list) =
      fun acc l1 i1 ->
	if i1 = 0 
	then (acc, l1)
	else 
	  match l1 with
	  | h::t -> lsplit_aux (h::acc) t (i1-1) 
	  | _    -> raise (Invalid_argument "list_splitat")
    in 
    let (f,s) = lsplit_aux [] l i in
    ((List.rev f),s)

let rec list_equal: 'a list -> 'a list -> bool =
  fun l1 l2 ->
    try
      (*  let eq = (fun ac a b -> ac && (a = b)) in  morally right, but inefficient *)
      let eq = (fun _ a b ->  if (a = b) then true else raise (Failure "false") ) in 
      List.fold_left2 eq true l1 l2
    with _ -> false

let rec list_unordered_equal: 'a list -> 'a list -> bool = (* Compares list like _unordered_ sets -- propably inefficient using list_remove_once! *)
  fun la lb ->
    match (la,lb) with
    | ([],[])        -> true
    | (x::xs, y::ys) ->
	if x=y 
	then (list_unordered_equal xs ys)
	else 
	  (
 	   try  list_unordered_equal xs (list_remove_once x (y::ys))
	   with Not_found -> false
	  )
    | _              -> false     
	  
let sum: int list -> int =
  List.fold_left (+) 0

let list_remove_last: 'a list -> 'a list = (* Rewrite efficently! *)
  fun a -> List.rev(List.tl(List.rev a))

let get_last: 'a -> 'a list -> 'a = (* Get last element if it exists, otherwise use default... *)
    fun a b -> (List.fold_left pr_snd a b) (* Why is eta-reduction prohibited here ?*) 
(*  fun a   ->  List.fold_left pr_snd a  (* This works... *) *)
(*             (List.fold_left pr_snd)   (* This doesnt work: *)
					    The type of this expression, '_a -> '_a list -> '_a,
					    contains type variables that cannot be generalized
                                          *) 
let list_orderedfind: ('a -> bool) -> 'a list -> ('a * int) =  (* Yields searched element and its place within the list. First element has address 1! *)
  fun p l ->
    let rec lof_aux: ( int * 'a list ) -> ('a * int) = 
      function 
	| (_,[])     -> raise Not_found
	| (i, x::xs) -> 
	    if   p x 
	    then (x,i)
	    else lof_aux ((i+1), xs)
    in lof_aux (1,l) 

let rec list_insert: ('a -> 'a -> bool) -> 'a -> 'a list -> 'a list =
  fun p e l ->
    match l with 
    | []   -> [e]
    | h::t -> 
	if   (p e h) 
	then e::l
	else h::(list_insert p e t)

(* Strings and Things (aka Files) *)

(* I hate it that ocaml doesnt treat a string as a 'char list'. This simply produces duplicated code parts! *)

let string_of_char: char -> string = String.make 1

let char_of_string: string -> char = 
  fun s -> 
    if  (String.length s) = 1 
    then String.get    s    0
    else raise (Invalid_argument "char_of_string: string too long")

let bool_of_string_ext: string -> bool = (* More generous than built_in bool_of string *)
  fun s ->
    match String.lowercase s with
    | "t" | "tt" -> true
    | "f" | "ff" -> false
    | other      -> bool_of_string other
	
let string_fold_left: ('a -> char -> 'a) -> 'a -> string -> 'a =  (* Its a shame that OCaml doesnt provide this *)
  fun f base_acc str ->
    let acc_ref = ref base_acc in
    let aux:char -> unit =
      fun c -> acc_ref := f !acc_ref c
    in
    let _ = String.iter aux str in
    !acc_ref

let string_beginswith: string -> string -> bool =   (* Reprogram using the imperative feature for the sake of performance ??? rewrite *)
  fun mas sub ->
    let n_sub = String.length sub in
    try (String.sub mas 0 n_sub) = sub 
    with Invalid_argument("String.sub") -> false      (* z.b. Knnte n_mas < n_sub sein *)
    
  
let string_endswith: string -> string -> bool =   (* Reprogram using the imperative feature for the sake of performance ??? rewrite *)
  fun mas sub ->
    let n_mas = String.length mas in
    let n_sub = String.length sub in
    try (String.sub mas (n_mas - n_sub) n_sub) = sub 
    with Invalid_argument("String.sub") -> false      (* z.b. Knnte n_mas < n_sub sein *)

let string_drop: int -> string -> string =  (* Drops i characters from left *)
  fun i s ->
    let n = String.length s in
    try String.sub s i (n-i)
    with Invalid_argument("String.sub") -> raise (Invalid_argument ("string_drop: "^(string_of_int i)^",'"^s^"'."))

let string_rdrop: int -> string -> string = (* Drops i characters from right *)
  fun i s ->
    let n = String.length s in
    try String.sub s 0 (n-i)
    with Invalid_argument("String.sub") -> raise (Invalid_argument ("string_rdrop: "^(string_of_int i)^",'"^s^"'."))

let string_drop_until: char -> string -> string =
  fun c s ->
    try
      let p = String.index s c in
      string_drop p s
    with
      Not_found -> s (* Char does not occur, hence string is not altered *)

let rec string_drop_while: char -> string -> string =
  fun c s ->
    try  
      let n = (String.length s) in
      if (n <= 0) 
      then ""
      else
	let cut = 
	  let i = ref 0 in
	  let _ = while (!i<n) && ((s.[!i]) = c) do i:=!i+1 done in !i
	in String.sub s cut (n-cut)
   with _ -> raise (Invalid_argument ("string_drop_while: '"^(string_of_char c)^"','"^s^"'."))

let rec string_rdrop_while: char -> string -> string = 
  fun c s ->
    try
      let n = String.length s in
      if (n <= 0) 
      then ""
      else
	let cut = 
	  let i = ref (n-1) in
	  let _ = while (!i>=0) && ((s.[!i]) = c) do i:=!i-1 done in (!i+1)
	in String.sub s 0 cut
    with _ -> raise (Invalid_argument "Common.string_rdrop_while")

let string_ltrim: string -> string = (* Removes leading blanks *)
  string_drop_while ' '

let string_rtrim: string -> string = (* Removes trailing blanks *)
  string_rdrop_while ' '

let string_trim: string -> string = (* Removes leading and trailing blanks *)
  compose (string_ltrim) (string_rtrim)
    
let string_lalign: char -> int -> string -> string =  (* Fills up string from left if string is to short *)
  fun c l s -> 
    let diff = l - (String.length s) in
    if   diff > 0
    then (String.make diff c) ^ s
    else s

let string_ralign: char -> int -> string -> string =  (* Fills up string from right if string is to short *)
  fun c l s -> 
    let diff = l - (String.length s) in
    if   diff > 0
    then s ^ (String.make diff c) 
    else s

let string_rl_align: char -> int -> int -> string -> string =
  fun c l r s -> string_lalign c l (string_ralign c r s)

let string_lr_align: char -> int -> int -> string -> string =
  fun c l r s -> string_ralign c l (string_lalign c r s)
		
let print_laligned_string: char -> int -> string -> unit =
  compose3 print_string string_lalign

let print_raligned_string: char -> int -> string -> unit =
  compose3 print_string string_ralign

let string_split: string -> int -> (string * string) =
  fun s i ->
    let n   = String.length s in
    let fst = String.sub s 0 i in
    let snd = String.sub s i (n-i) in
    (fst,snd)

let string_chop: string -> string list = (* Returns a list of strings of length 1  --- should produce a char list, but string list is more useful due to a lack of string/char conversions *)
  fun s ->
    let slen = (String.length s) - 1 in
    let rec chop_aux: string list -> int -> string list =
      fun acc d ->
	if (d > slen)
	then List.rev acc
	else chop_aux ((string_of_char (String.get s d))::acc) (d+1)
    in chop_aux [] 0

let string_replace: string -> char -> char -> string = (* Returns a copy of the input string with _all_ occurences of a character replaces by another character. *)
  fun s o n ->
    let s' = String.copy s in (* No side-effects! *)
    let _  =
      for i = 0 to (String.length s' - 1) do
	if o = (String.get s' i)
	then String.set s' i n
      done
    in s'

let string_captured: string -> string = (* Inverse to String.escaped, except for numeral characters (written like \123) *)
  fun s ->
    let out = ref "" in
    let last_c = ref false in
    let add_char: char -> unit =
      fun c -> out := (!out^(string_of_char c))
    in
    let cap_aux: char -> unit =
      fun act_c ->
	if !last_c 
	then 
	  begin
	    last_c := false;
	    match act_c with
	    | '\\' 
	    | '\"' -> add_char act_c 
	 (* | '\'' does not occur as escaped character in strings *)
	 (* | '0'    These number probably dont work correctly yet...
	    | '1'
	    | '2'
	    | '3'
	    | '4'
	    | '5'
	    | '6'
	    | '7'
	    | '8'
	    | '9'  -> add_char act_c *)
	    | 'n'  -> add_char '\n'
	    | 'r'  -> add_char '\r'
	    | 't'  -> add_char '\t'
	    | 'b'  -> add_char '\b'
	    | _    -> begin add_char '\\'; add_char act_c end
	  end
	else
	  match act_c with
	  | '\\'  -> last_c := true
	  | _     -> add_char act_c
    in
    let _ = String.iter cap_aux s in
    !out
      
let string_intersperse: string -> int -> string -> string = (* Every i chars, insert the third argument *)
  fun s i sep ->
    let rec inters_aux: string list -> string -> string list =
      fun acc rs ->
	if ((String.length rs) > i)
	then 
	  let (h,t) = string_split rs i in
	  inters_aux (h::acc) t
	else 
	  List.rev (rs::acc)
    in String.concat sep (inters_aux [] s)

let string_break: string -> int -> char -> string -> string = (* Trys to break 'the_s' into chunks with at most 'width' parts, breaking only AFTER each 'break_c' character, inserting 'sep_s' as the break *)
  fun the_s width break_c sep_s ->
    let rec br_aux: string list -> string -> string list =
      fun acc rs ->
	if ((String.length rs) > width)
	then
	  try
	    let indx = (String.rindex_from rs width break_c)+1 in
	    let (fst, snd) = string_split rs indx in
	    if indx <> 0 
	    then br_aux (fst::acc) snd
	    else raise Not_found
	  with Not_found -> (* No suitable breakpoint found *)
	    begin 
	      try
		let indx = (String.index_from rs width break_c)+1 in
		let (fst, snd) = string_split rs indx in
		if indx <> 0 
		then br_aux (fst::acc) snd
		else raise Not_found
	      with Not_found -> (* No breakpoint at all *)
		List.rev (rs::acc)
	    end
	else List.rev (rs::acc)
    in String.concat sep_s (br_aux [] the_s)

let filename_has_suffix: string -> bool =
  fun s -> 
    try  String.rcontains_from s ((String.length s) - 2) '.'
    with _ -> false

(*
let print_some: string option -> string = (* Prints names if any... *) 
  function
    | None   -> ""
    | Some s -> s
*)



(* Some printing functions for printing integers *)

let signed_string_of_int: int -> string =
  fun i -> 
    if   i >= 0 
    then "+"^(string_of_int i)
    else     (string_of_int i)

let unsigned_string_of_int: int -> string =
  fun i -> 
    let o = (string_of_int i) in
    match (String.get o 0) with
    | '*'
    | '+'  
    | '-' -> string_drop 1 o
    |  _  -> o

let aligned_string_of_int: char -> int -> int -> string =
  fun c l x -> (* x is the number to print, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (string_of_int x)
						  
let signed_aligned_string_of_int: char -> int -> int -> string =
  fun c l x -> (* x is the number to print, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (signed_string_of_int x)

let unsigned_aligned_string_of_int: char -> int -> int -> string =
  fun c l x -> (* x is the number to print, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (unsigned_string_of_int x)
						  
let print_singed_int: int -> unit =
  compose print_string signed_string_of_int

let print_unsinged_int: int -> unit =
  compose print_string unsigned_string_of_int

let print_aligned_int: char -> int -> int -> unit =
  compose3 print_string aligned_string_of_int

let print_signed_aligned_int: char -> int -> int -> unit =
  compose3 print_string signed_aligned_string_of_int

let print_unsigned_aligned_int: char -> int -> int -> unit =
  compose3 print_string unsigned_aligned_string_of_int

let rec print_pretty_int_list: int list -> string = (* Printing integer lists... *)
  function 
    | [] -> "[]"
    | h::t -> ("["^(string_of_int h)^(print_pretty_int_list_aux t))
and print_pretty_int_list_aux: int list -> string =
  function
    | []   -> "]"
    | h::t -> (","^(string_of_int h)^(print_pretty_int_list_aux t))

let print_normal_float: float -> string =
  fun f ->
    let raw = 
      if (f = 0.0) (* to prevent "-0" print-outs *)
      then "0"
      else string_of_float f 
    in
    let n   = (String.length raw) - 1 in
    if   (String.get raw n) = '.' 
    then String.sub raw 0 n
    else raw


(* Some printing functions for printing floats *)
let pretty_string_of_float : float -> string = (* removes a trailing '.' *)
  fun f ->
    let o = string_of_float f in
    if   string_endswith o "." 
    then string_rdrop  1 o
    else                 o

let signed_string_of_float: float -> string =
  fun i -> 
    if   i >= 0. 
    then "+"^(pretty_string_of_float i)
    else     (pretty_string_of_float i)

let unsigned_string_of_float: float -> string =
  fun i -> 
    let o = (pretty_string_of_float i) in
    match (String.get o 0) with
    | '*'
    | '+'  
    | '-' -> string_drop 1 o
    |  _  -> o
	  
let aligned_string_of_float: char -> int -> float -> string =
  fun c l x -> (* x is the number to prfloat, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (pretty_string_of_float x)
						  
let signed_aligned_string_of_float: char -> int -> float -> string =
  fun c l x -> (* x is the number to print, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (signed_string_of_float x)

let unsigned_aligned_string_of_float: char -> int -> float -> string =
  fun c l x -> (* x is the number to print, l the entire length of the string, c the character fo filling up space *)
    string_lalign c l (unsigned_string_of_float x)
						  
let print_singed_float: float -> unit =
  compose print_string signed_string_of_float

let print_unsinged_float: float -> unit =
  compose print_string unsigned_string_of_float

let print_aligned_float: char -> int -> float -> unit =
  compose3 print_string aligned_string_of_float

let print_signed_aligned_float: char -> int -> float -> unit =
  compose3 print_string signed_aligned_string_of_float

let print_unsigned_aligned_float: char -> int -> float -> unit =
  compose3 print_string unsigned_aligned_string_of_float



let string_contains_operator : string -> bool =
  fun s -> (* We would need a fold-operator on strings for an efficient implementation! *)
    String.contains s '+' ||
    String.contains s '-' ||
    String.contains s '*' ||
    String.contains s '/' ||
    String.contains s '=' ||
    String.contains s '>' ||
    String.contains s '<' ||
    String.contains s '^' 

(* Lookup tables --- Map.make as a class *)

(* Doesnt work this way, as we could not use Map.Make anymore
class type ['k] compareable =
  object
    method elt: 'k
    method compare: 'k -> 'k -> bool
  end
*)

module IdSet= Set.Make(struct type t = string let compare = String.compare end)
type idset = IdSet.t 
     
let idset_size: idset -> int =  (* Count the number of elements *)
  flip (IdSet.fold (fun _ acc -> acc+1)) 0

let idset_to_string: idset -> string =
  fun set ->
    if IdSet.is_empty set 
    then "{}"
    else ((string_rdrop 1 (IdSet.fold (fun v acc -> acc^v^",") set "{"))^"}")


(* How to hide this in class lookup?*)
module InternalLookup = Map.Make(struct type t = string let compare = String.compare end)
      
class virtual ['t] lookup = (* Essentially Map-structure, but with explicit replace/bind instead of add. Includes neat error messages *)
  object (self: 'self)
    val the_lookup: 't InternalLookup.t = InternalLookup.empty
    method virtual    key_name: string  (* parameter for error messages *)
    method virtual  value_name: string  (* parameter for error messages *)
    method virtual lookup_name: string  (* parameter for error messages *)
    method virtual joker: string (* a key to be ignored *)
    method virtual error: 'e. string -> 'e (* OCaml does not provide an error-handler for objects *) 

    method private empty: 'self = (* Important to heirs, so that they can create new empty selves *)
      {< the_lookup = InternalLookup.empty >}

    method add: string -> 't -> 'self =
      fun k v ->
	{< the_lookup = InternalLookup.add k v the_lookup >}

    method private del: string -> 'self =
      fun k ->
	{< the_lookup = InternalLookup.remove k the_lookup >}

    method bind: string -> 't -> 'self =
      fun k v -> 
	if   k = self#joker 
	then self 
	else 
	  if self#mem k 
	  then self#error (self#key_name^" '"^k^"' already bound in "^self#lookup_name^".")
	  else self#add k v

    method remove: string -> 'self =
      fun k ->
	if   k = self#joker 
	then self 
	else 
	  if not (self#mem k)
	  then self#error (self#key_name^" '"^k^"' not bound in "^self#lookup_name^", removing failed.")
	  else self#del k 
    
    method replace: string -> 't -> 'self =
      fun k v -> 
	if   k = self#joker 
	then self 
	else 
	  if not (self#mem k)
	  then self#error (self#key_name^" '"^k^"' not bound in "^self#lookup_name^", replacing failed.")
	  else self#add k v     

    method lookup: string -> 't =
      fun k -> 
	try  InternalLookup.find k the_lookup
	with Not_found -> self#error (self#key_name^" '"^k^"' not found in "^self#lookup_name^".")

    method mem: string -> bool =
      fun k -> InternalLookup.mem k the_lookup
	
    method is_empty: bool =
      (* (InternalLookup.is_empty  the_lookup)   --- is_empty seems to be unknown to OCaml! *)
      the_lookup = InternalLookup.empty
	  
    method fold: 'b. (string -> 't -> 'b -> 'b) -> 'b -> 'b =
      fun f acc -> InternalLookup.fold f the_lookup acc

 (* We rather want type "'b.('t -> 'b) -> ['b] self" and thus must leave the definition to the unparameterized heirs
    method map: ('t -> 't) -> 'self =
      fun f -> {< the_lookup = InternalLookup.map f the_lookup >}
  *)

    method restrict: (string -> 't -> bool) -> 'self = (* Only those who fulfill the predicate may remain *)
      fun predicate ->	
	self#fold 
	  (fun key binding acc_lu ->
	    if   predicate key binding 
	    then acc_lu#bind key binding
	    else acc_lu
	  )
	  (self#empty) 

    method restrict_to: idset -> 'self = (* Restrict to a set *)
      fun keyset -> self#restrict (fun k b -> IdSet.mem k keyset)
	  
    method domain: idset = (* Returns set of bound keys *)
      self#fold (fun k _ acc -> IdSet.add k acc) IdSet.empty

    method compile: string list -> 't list -> 'self =
      try  List.fold_left2 (fun acc_lu k v -> acc_lu#bind k v) self
      with (Invalid_argument "list_fold_left2") -> 
	raise (Invalid_argument("Mismatching number of "^self#key_name^"/"^self#value_name^" for building "^self#lookup_name^"."))

    method merge: 'self -> 'self =
      fun other ->
	other#fold (fun k b acc -> acc#bind k b) self

  end

 
(*
      (* Practising a very bad style of recursion: *)
exception MTry
let rec mtry: int -> int -> unit =
  fun x y ->
    let _ = (print_int x; print_string "-"; print_int y; print_string "  #  ") in
    if x > 9 then raise MTry
    else if y > 9 then raise MTry
    else 
      (* First version: *)
      try mtry x (y+1)
      with _ -> 
	begin
	  try mtry (x+1) 0
	  with _ -> ()
	end
      (*
      (* Alternate and equivalent version: *)
	try
	  begin
	    try
	      mtry x (y+1)
	    with _ -> mtry (x+1) 0
	  end
	with _ -> ()
       *)
*)	


