(*

   Author:   Steffen Jost <jost@informatik.uni-muenchen.de>
   Name:     $Name:  $
   File:     $RCSfile: builtin.ml,v $
   Id:       $Id: builtin.ml,v 1.1 2004/12/07 17:23:39 sjost Exp $ 

	

   What this File is all about: 
   ---------------------------- 

   This is supposed to be the only place where information about
   built-in functions is stored in the source code!  Needed in exec.ml
   and constraint.ml.

   ToDos: 

*)

open Support
open Common
open Types
open Memory
  
(* Built-in Functions *)

type func = 
        typ                         (* Input type  *)
      * int                         (* Fixed din   *)
      * int                         (* Fixed dout  *)
      * typ                         (* Output type *) 
      * ((heap * rvalue) -> result) (* Implementation: Maps heap and an rvalue to new heap & rvalue *)

module BuiltIn = Map.Make(struct type t = funcidentifier let compare = String.compare end)
type builtin = func BuiltIn.t
      
let bi_info = fakeinfo_str "built-in function"
let polymorph:typ = bi_info(Types.polymorph) (* Slightly abusing syntax *)

let the_funcs:builtin = 
    let builtin_list = [
      ("int_of_float", (bi_info FloatTyp, 0, 0, bi_info IntTyp, (function 
	| (h, FloatRVal(f)) -> (h, IntRVal(int_of_float f)) 
	| _                 -> raise (Invalid_argument "Float expected")))
      );
      ("float_of_int", (bi_info IntTyp, 0, 0, bi_info FloatTyp, (function 
	| (h, IntRVal(i)) -> (h, FloatRVal(float_of_int i)) 
	| _               -> raise (Invalid_argument "Integer expected.")))
      ); 
      ("char_of_int", (bi_info IntTyp, 0, 0, bi_info CharTyp, (function 
	| (h, IntRVal(i)) -> (h, CharRVal(char_of_int(i))) 
	| _               -> raise (Invalid_argument "Integer expected.")))
      ); 
      ("int_of_char", (bi_info CharTyp, 0, 0, bi_info IntTyp, (function 
	| (h, CharRVal(x)) -> (h, IntRVal(int_of_char x)) 
	| _                -> raise (Invalid_argument "Character expected.")))
      ); 
      ("float_of_string", (bi_info StringTyp, 0, 0, bi_info FloatTyp, (function 
	| (h, StringRVal(x)) -> (h, FloatRVal(float_of_string x)) 
	| _                  -> raise (Invalid_argument "String expected.")))
      ); 
      ("string_of_float", (bi_info FloatTyp, 0, 0, bi_info StringTyp, (function 
	| (h, FloatRVal(x)) -> (h, StringRVal(string_of_float x)) 
	| _                 -> raise (Invalid_argument "Float expected.")))
      ); 
      ("int_of_string", (bi_info StringTyp, 0, 0, bi_info IntTyp, (function 
	| (h, StringRVal(x)) -> (h, IntRVal(int_of_string x)) 
	| _                  -> raise (Invalid_argument "String expected.")))
      ); 
      ("string_of_int", (bi_info IntTyp, 0, 0, bi_info StringTyp, (function 
	| (h, IntRVal(x)) -> (h, StringRVal(string_of_int x)) 
	| _               -> raise (Invalid_argument "Integer expected.")))
      ); 
      ("print_int", (bi_info IntTyp, 0, 0, bi_info UnitTyp, (function 
	| (h, IntRVal(x)) -> let _ = print_rvalue (IntRVal x) in (h, UnitRVal) 
	| _               -> raise (Invalid_argument "Integer expected.")))
      ); 
      ("print_float", (bi_info FloatTyp, 0, 0, bi_info UnitTyp, (function 
	| (h, FloatRVal(x)) -> let _ = print_rvalue (FloatRVal x) in (h, UnitRVal) 
	| _                 -> raise (Invalid_argument "Float expected.")))
      ); 
      ("print_char", (bi_info CharTyp, 0, 0, bi_info UnitTyp, (function 
	| (h, CharRVal(x)) -> let _ = print_rvalue (CharRVal x) in (h, UnitRVal) 
	| _                -> raise (Invalid_argument "Character expected.")))
      ); 
      ("print_string", (bi_info StringTyp, 0, 0, bi_info UnitTyp, (function 
	| (h, StringRVal(x)) -> let _ = print_rvalue (StringRVal x) in (h, UnitRVal) 
	| _                  -> raise (Invalid_argument "String expected.")))
      ); 
      ("print", (polymorph, 0, 0, bi_info UnitTyp, (function     (* THIS PRINTING ROUTINES NEVER COUNTS AS HEAP ACCESS  *)
 	| (h, x) -> let _ = h#print_rvalue x in (h, UnitRVal)))
      ); 
      ("print'", (polymorph, 0, 0, polymorph, (function          (* THIS PRINTING ROUTINES NEVER COUNTS AS HEAP ACCESS  *)
	| (h, x) -> let _ = h#print_rvalue x in (h, x)))
      ); 
      ("print_newline", (bi_info UnitTyp, 0, 0, bi_info UnitTyp, (function 
 	| (h, UnitRVal) -> let _ = print_newline (); flush stdout in (h, UnitRVal) 
	| _             -> raise (Invalid_argument "Unit expected.")))
      ); 
      ("print_heap", (bi_info UnitTyp, 0, 0, bi_info UnitTyp, (function 
 	| (h, UnitRVal) -> let _ = 
	                     print_string " Current heapsize (uniform/user): ";
	                     print_aligned_int ' ' 3 (h#size_uni_cur);
	                     print_string " / ";
	                     print_aligned_int ' ' 3 (h#size_usr_cur);
	                     print_string " \n"
	                   in (h, UnitRVal) 
	| _             -> raise (Invalid_argument "Unit expected.")))
      ); 
      ("free", (bi_info DiamondTyp, 0, 0, bi_info UnitTyp, (function   (* For compatibility with Camelot *)
 	| (h, PointerRVal(l)) -> (h#reuse_loc l, UnitRVal)
	| _                   -> raise (Invalid_argument "Pointer expected.")))
      )]
    in (* For the ease of notation, we use a list first and rely on the compiler's optimisation. Otherwise we would suffocate beneath parenthesis... *)
    List.fold_left (fun acc (id,f) -> BuiltIn.add id f acc)  BuiltIn.empty builtin_list


let lookup: funcidentifier -> func =
  fun fid -> 
    try  (BuiltIn.find fid the_funcs)
    with Not_found -> err ("Functionidentifier '"^fid^"' is unknown.")
	

let call_info: info -> funcidentifier -> heap -> rvalue -> result =
  fun i fid h rv ->
    try
      let (ity, din, dout, oty, f) = BuiltIn.find fid the_funcs in 
      f (h, rv)
    with
      Not_found          -> errAt i ("Function '"^fid^"' unknown.")
    | Invalid_argument m -> errAt i ("Call to built-in function "^fid^" failed: "^m)

let call:              funcidentifier -> heap -> rvalue -> result = call_info unknown

let typ: funcidentifier -> typ =
  fun fid ->
    let (ity, din, dout, oty, f) = lookup fid in 
    bi_info (ArrowTyp(ity,oty))

let signature: funcidentifier -> (typ * int * int * typ) = (* Useful to compute the rich_typ in module Rich_typ *)
  fun fid ->
    let (ity, din, dout, oty, f) = lookup fid in 
    (ity, din, dout, oty)

let is_function: funcidentifier -> bool =
  fun fid -> BuiltIn.mem fid the_funcs
      
