(* 				 
   File:	$RCSfile: Semantics.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Semantics.thy,v 1.1 2004/07/07 11:46:48 lenb Exp $

   Functional semantics for a (restricted version of) Grail.
*)

header {* Operational semantics *}
(*<*)
theory Semantics = Prelude:
(*>*)
text {*
  We define a big-step evaluation semantics.  For the purpose of later
  proofs, the relation is also indexed by the height of the derivation.
  *}

subsection{* Resource environments and abbreviations *}

text {* 
We define resource tuples to model resource consumption during the execution.
Currently, a resource tuple is a record with 4 components, counting execution time, 
function and method calls, and measuring maximal depth of method invocations.
*}

(*Changed from nats to ints*)
record rescomp = clock :: int        -- {* execution time *}
                 callc :: int        -- {* call counter *}
                 invkc :: int        -- {* invoke counter *}
                 invkdpth :: nat     -- {* max invoke depth *}

text {*
To add up resource consumption, we use two combinators over resource tuples.
*}

translations 
 "rescomp" <= (type) "\<lparr> clock :: int, callc :: int, invkc :: int, invkdpth :: nat \<rparr>"

constdefs rescomp_plus:: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp"  (infixr "\<oplus>" 0)
"rescomp_plus p1 p2 == \<lparr>clock = clock p1 + clock p2,
                     callc = callc p1 + callc p2,
                     invkc = invkc p1 + invkc p2,
                     invkdpth = invkdpth p1 + invkdpth p2\<rparr>"

constdefs rescomp_cup :: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp" (infixr "\<smile>" 0)
"rescomp_cup p1 p2 == \<lparr>clock = clock p1 + clock p2,
                    callc = callc p1 + callc p2,
                    invkc = invkc p1 + invkc p2,
                    invkdpth = max (invkdpth p1) (invkdpth p2)\<rparr>" 
(*<*)
declare rescomp_plus_def [simp add]
declare rescomp_cup_def [simp add]
(*>*)

text {* Some syntax for creating and manipulating resource tuples. *}

syntax mkRescomp:: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp"
translations
"mkRescomp C CC IC ID" == "\<lparr>clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax mkRescomp_HWL :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp" ("\<langle>_ _ _ _\<rangle>")
translations
"\<langle>C CC IC ID\<rangle>" == "\<lparr>rescomp.clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax tkn ::"int \<Rightarrow> rescomp \<Rightarrow> rescomp"
translations
"tkn n p" == "((mkRescomp n 0 0 0) \<smile> p)"

syntax tk::"rescomp \<Rightarrow> rescomp"
translations
"tk p" == "tkn 1 p"

syntax tkcall::"rescomp \<Rightarrow> rescomp"
translations
"tkcall p" == "((mkRescomp 1 1 0 0) \<smile> p)"

syntax tickRo::"rescomp"
translations
"tickRo" == "mkRescomp 1 0 0 0"

subsection {* Operational Effects 'a*}
(*type variable 'a gives the domain of effects int, nat, bool,
  while type variable 'b gives the domain on which the
  instruction-effects may depend.
  For example, for rescomp, we have 
    'a = (int \<times> int \<times> int \<times> nat) and
    'b = unit
  while the sematic below uses
    'a = arbitrary and
    'b = epxr \<times> env \<times> heap
*)
record ('a, 'b) Effect = 
  Int_effect:: "'b \<Rightarrow> 'a"
  IVar_effect:: "'b \<Rightarrow> 'a"
  Primop_effect:: "'b \<Rightarrow> 'a"
  Null_effect:: "'b \<Rightarrow> 'a"
  RVar_effect:: "'b \<Rightarrow> 'a"
  RPrimop_effect:: "'b \<Rightarrow> 'a"
  New_effect:: "'b \<Rightarrow> 'a"
  GetFi_effect:: "'b \<Rightarrow> 'a"
  GetFr_effect:: "'b \<Rightarrow> 'a"
  PutFi_effect:: "'b \<Rightarrow> 'a"
  PutFr_effect:: "'b \<Rightarrow> 'a"
  GetStat_effect:: "'b \<Rightarrow> 'a"
  PutStat_effect:: "'b \<Rightarrow> 'a"
  InvV_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a"
  InvS_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a"
  Leti_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a \<Rightarrow> 'a"
  Letr_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a \<Rightarrow> 'a"
  Letv_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a \<Rightarrow> 'a"
  If_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a"
  Call_effect:: "'b \<Rightarrow> 'a \<Rightarrow> 'a"

subsection {* Operational semantics *}
text {*
We define the operational semantics as an inductively defined set, 
connecting, for a given expression, all possible pre-states with its post-states.
Note that we define the semantics with a counter of derivation height to use thiscounter later on in the soundness proof to do induction over. The counter is \emph{not}
used to capture resource consumption (we have a separate field in the resource tuple 
for that). 
*}

types 'a EFF = "('a, expr \<times> env \<times> heap) Effect"

consts semn :: "('a EFF \<times> env \<times> heap \<times> expr \<times> nat \<times> heap \<times> val \<times> 'a ) set" -- {* operational semantics *}

syntax semn_  :: "['a EFF, env, heap, expr, nat, heap, val, 'a] \<Rightarrow> bool"   ("_ | _ \<turnstile> _ , _ \<Down>_ '(_ , _ , _')")

translations "P | E \<turnstile> h,e \<Down>n (hh,v,p)" == "(P,E,h,e,n,hh,v,p) : semn"

inductive semn intros
 semNull:    "P | E \<turnstile> h,NULL \<Down>1 (h,RVal Nullref, (Null_effect P) (NULL, E, h))"

 semInt:     "P | E \<turnstile> h,(expr.Int i) \<Down>1 (h,IVal i, (Int_effect P) (expr.Int i, E, h))"

 semIVar:    "P | E \<turnstile> h,IVar x \<Down>1 (h,IVal (E<x>), (IVar_effect P) (IVar x, E, h))"

 semRVar:    "P | E \<turnstile> h,RVar x \<Down>1 (h,RVal (E\<lfloor>x\<rfloor>), (RVar_effect P) (RVar x, E, h))"

 semPrimop:  "P | E \<turnstile> h, (Primop f x y) \<Down>1 (h,IVal (f (E<x>) (E<y>)), (Primop_effect P) (Primop f x y, E, h))"

 semRPrimop: "P | E \<turnstile> h, (RPrimop f x y) \<Down>1 (h,IVal (f (E\<lfloor>x\<rfloor>) (E\<lfloor>y\<rfloor>)), (RPrimop_effect P) (RPrimop f x y, E, h))"

 semGetfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> P | E \<turnstile> h,(GetFi x f) \<Down>1 (h, IVal (h<a\<bullet>f>), (GetFi_effect P) (GetFi x f, E, h))"

 semGetfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> P | E \<turnstile> h,(GetFr x f) \<Down>1 (h, RVal (h\<lfloor>a\<diamondsuit>f\<rfloor>), (GetFr_effect P) (GetFr x f, E, h))"

 semPutfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> P | E \<turnstile> h, (PutFi x f y) \<Down>1 (h<a\<bullet>f:=E<y>>, arbitrary, (PutFi_effect P) (PutFi x f y, E, h))"

 semPutfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> P | E \<turnstile> h, (PutFr x f y) \<Down>1 (h\<lfloor>a\<diamondsuit>f:=E\<lfloor>y\<rfloor>\<rfloor>, arbitrary, (PutFr_effect P) (PutFr x f y, E, h))"

 semGetstat: "P | E \<turnstile> h,(GetStat c f) \<Down>1 (h, RVal (h\<lbrace>c\<struct>f\<rbrace>), (GetStat_effect P) (GetStat c f, E, h))"

 semPutstat: "P | E \<turnstile> h, (PutStat c f y) \<Down>1 (h\<lbrace>c\<struct>f:=E\<lfloor>y\<rfloor>\<rbrace>, arbitrary, (PutStat_effect P) (PutStat c f y, E, h))"

 semNew: "\<lbrakk> l = freshloc (Dom h)\<rbrakk> \<Longrightarrow>
          P | E \<turnstile> h,(New c ifldvals rfldvals) \<Down>1 (newObj h l E c ifldvals rfldvals, RVal (Ref l), (New_effect P) (New c ifldvals rfldvals, E, h))"

 semIf_True:  "\<lbrakk>E<x> = grailbool True; P | E \<turnstile> h, e1 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
               P | E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (If_effect P) (Ifg x e1 e2, E, h) p)"

 semIf_False:  "\<lbrakk>E<x> = grailbool False; P | E \<turnstile> h, e2 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
                P | E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, (If_effect P) (Ifg x e1 e2, E, h) p)"

 semLeti: "\<lbrakk>P | E \<turnstile> h,e1 \<Down>n (h1,IVal i,p1);  P | E<x:=i> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           P | E \<turnstile> h, (Leti x e1 e2) \<Down>((max n m) + 1) (h2,v, (Leti_effect P) (Leti x e1 e2, E, h) p1 p2)"

 semLetr: "\<lbrakk>P | E \<turnstile> h,e1 \<Down>n (h1,RVal r,p1);  P | E\<lfloor>x:=r\<rfloor> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           P | E \<turnstile> h, (Letr x e1 e2) \<Down>((max n m) + 1) (h2,v, (Letr_effect P) (Letr x e1 e2, E, h) p1 p2)"

 semLetv: "\<lbrakk>P | E \<turnstile> h,e1 \<Down>n (h1,w,p1);  P | E \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           P | E \<turnstile> h, (Letv e1 e2) \<Down>(max n m) (h2,v, (Letv_effect P) (Letv e1 e2, E, h) p1 p2)"

 semCall: "\<lbrakk>P | E \<turnstile> h, snd(funtable f) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
           P | E \<turnstile> h,(Call f) \<Down>(n+1) (h1, v,  (Call_effect P) (Call f, E, h) p)"

 semInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;  
              h@@a = Some C; 
              EE = newframe_env (Ref a) (fst (methtable C mn)) args E;
              P | EE \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
             P | E \<turnstile> h, x\<diamondsuit>mn(args) \<Down>(n + 1) (h1,v, (InvV_effect P) (x\<diamondsuit>mn(args), E, h) p)"

 semInvokeStatic: "\<lbrakk>EE = (newframe_env Nullref (fst (methtable C mn)) args E);
                    P | EE \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                   P | E \<turnstile> h, C\<bullet>mn(args) \<Down>(n + 1) (h1,v, (InvS_effect P) (C\<bullet>mn(args), E, h) p)"

text {* Operational semantics without height index *}

constdefs
 sem :: "['a EFF, env, heap, expr, heap, val, 'a] \<Rightarrow> bool"  ("_ | _ \<turnstile> _ , _ \<Down> _ , _ , _" 1000)
 "P | E \<turnstile> h,e \<Down> hh,v,p \<equiv> (\<exists> n. (P | E \<turnstile> h,e \<Down>n (hh,v,p)))"

text {*
  Comentary:
  \begin{enumerate}
  \item InvokeStatic: static condition: method body does not refer to ''self''.
  \item Putfield instructions (rules @{text evalPutFi} and @{text evalPutFr}) are only
     evaluated  for their side-effect and should only be used with @{text Letv}.  
     To emphasise this, they return the undetermined value @{text arbitrary}.    
  \item To explain the resource counting in Invoke: vn1 = 1, vn2 = 1,
    frame push/pop = 2, and the implicit return = 1.
    NB: we could break up the increment of ticks into before/after the invocations.
  \item Many rules omit type-checking premises.
    For example, in get field and put field rules, we should
    check that the object belongs to a class that indeed has
    the fields being accessed.  
  \item Object initialization in the New ruleallocates a fresh location,
    etends the oheap component of the heap by the class entry for the new object
    and initialises the fields.
  \item  We might tickn 2 in the If rules, but we leave it at one to
     allow for extra store/retrieve which may not be necessary in 
     some cases on the real machine.
  \item The rule for evaluating Let once contained an additional side
    condition that the expression being bound does not itself contain a
    CALL (non-tail position).  So evaluation would get stuck for non-Grail
    programs.  However, there seems to be no harm in considering the
    more general semantics for a slightly richer language, since this
    side condition is not used in the logic.  (This restriction could
    therefore be considered as part of the type-checking phase for Grail).
  \end{enumerate}
*}

subsection {* Basic properties *}
text {*We also provide elimination lemmas for the inductively defined relation*}
inductive_cases eval_cases: 
 "P | E \<turnstile> h,(expr.Null) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(expr.Int i) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(IVar x) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(RVar x) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Primop f x y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(RPrimop f x y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(GetFi x f) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(GetFr x f) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(PutFi x f y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(PutFr x f y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(GetStat c f) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(PutStat c f y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(New c ifldvals rfldvals) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Ifg x e1 e2) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Leti x e1 e2) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Letr x e1 e2) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Letv e1 e2) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Call f) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(Invoke x m y) \<Down>n  (h1,v,p)"
 "P | E \<turnstile> h,(InvokeStatic C m y) \<Down>n  (h1,v,p)"
(*<*)
lemma no_zero_height_derivsAux: "\<forall> E h h1 v p . (P | E \<turnstile> h, e \<Down>0  (h1,v,p) \<longrightarrow> False)"
apply (induct_tac e)
apply (simp_all)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply (subgoal_tac "n=0 & m=0")
apply clarsimp
apply (simp add: max_def)
apply (case_tac "n \<le> m", auto)
apply(erule semn.elims, simp_all)
apply(erule semn.elims, simp_all)
done
(*>*)

text {*and observe that no derivations of height 0 exist.*}
lemma no_zero_height_derivs: "P | E \<turnstile> h, e \<Down>0  (h1,v,p) \<Longrightarrow> False"
(*<*)by (insert no_zero_height_derivsAux, fastsimp)(*>*)

text {*The following lemma shows that the functional components of the 
       evaluation do not depend on the effects. In particular, a computational
       effect does not affect termination.*}
lemma "(P::'a EFF) | E \<turnstile> h,e \<Down> hh,v,p \<Longrightarrow> (\<forall> Q::'b EFF . \<exists> q .Q | E \<turnstile> h,e \<Down> hh,v,q)"
apply (simp add: sem_def, clarsimp)
apply (erule semn.induct)
apply (rule, rule, rule semNull)
apply (rule, rule, rule semInt)
apply (rule, rule, rule semIVar)
apply (rule, rule, rule semRVar)
apply (rule, rule, rule semPrimop)
apply (rule, rule, rule semRPrimop)
apply (rule, rule, erule semGetfi, assumption)
apply (rule, rule, erule semGetfr, assumption)
apply (rule, rule, erule semPutfi, assumption)
apply (rule, rule, erule semPutfr, assumption)
apply (rule, rule, rule semGetstat)
apply (rule, rule, rule semPutstat)
apply (rule, rule, erule semNew)
apply clarsimp apply (rule, rule, rule semIf_True)  apply simp apply simp
apply clarsimp apply (rule, rule, rule semIf_False)  apply simp apply simp
apply clarsimp apply (rule, rule, erule semLeti) apply simp 
apply clarsimp apply (rule, rule, erule semLetr) apply simp 
apply clarsimp apply (rule, rule, erule semLetv) apply simp 
apply clarsimp apply (rule, rule, erule semCall) 
apply clarsimp apply (rule, rule, erule semInvoke)  apply simp apply simp apply simp
apply clarsimp apply (rule, rule, rule semInvokeStatic)  apply simp apply simp 
done


text {*We can form the product of two effects with equal domain.*}
constdefs prodEffect ::"('a,'c) Effect \<Rightarrow> ('b,'c) Effect \<Rightarrow> ('a \<times> 'b, 'c) Effect"
"prodEffect aE bE == \<lparr>Int_effect = \<lambda> c . (Int_effect aE c, Int_effect bE c),
                      IVar_effect = \<lambda> c . (IVar_effect aE c, IVar_effect bE c),
                      Primop_effect = \<lambda> c . (Primop_effect aE c, Primop_effect bE c),
                      Null_effect = \<lambda> c . (Null_effect aE c, Null_effect bE c),
                      RVar_effect = \<lambda> c . (RVar_effect aE c, RVar_effect bE c),
                      RPrimop_effect = \<lambda> c . (RPrimop_effect aE c, RPrimop_effect bE c),
                      New_effect = \<lambda> c . (New_effect aE c, New_effect bE c),
                      GetFi_effect = \<lambda> c . (GetFi_effect aE c, GetFi_effect bE c),
                      GetFr_effect = \<lambda> c . (GetFr_effect aE c, GetFr_effect bE c),
                      PutFi_effect = \<lambda> c . (PutFi_effect aE c, PutFi_effect bE c),
                      PutFr_effect = \<lambda> c . (PutFr_effect aE c, PutFr_effect bE c),
                      GetStat_effect = \<lambda> c . (GetStat_effect aE c, GetStat_effect bE c),
                      PutStat_effect = \<lambda> c . (PutStat_effect aE c, PutStat_effect bE c),
                      InvV_effect = \<lambda> c a . (InvV_effect aE c (fst a), InvV_effect bE c (snd a)),
                      InvS_effect = \<lambda> c a . (InvS_effect aE c (fst a), InvS_effect bE c (snd a)),
                      Leti_effect = \<lambda> c a1 a2 . (Leti_effect aE c (fst a1) (fst a2), Leti_effect bE c (snd a1) (snd a2)),
                      Letr_effect = \<lambda> c a1 a2 . (Letr_effect aE c (fst a1) (fst a2), Letr_effect bE c (snd a1) (snd a2)),
                      Letv_effect = \<lambda> c a1 a2 . (Letv_effect aE c (fst a1) (fst a2), Letv_effect bE c (snd a1) (snd a2)),
                      If_effect = \<lambda> c a . (If_effect aE c (fst a), If_effect bE c (snd a)),
                      Call_effect = \<lambda> c a . (Call_effect aE c (fst a), Call_effect bE c (snd a))\<rparr>"

(* Also, the trivial effects exist for all domains. The evaulation 
   relation instantiated with this trivial effects amounts to
   the evaluation without any effects, i.e. the
   purely computational evaluation.
   Also I expect the (unit,'b) Effect to be initial in the category
   of (_,'b) Effeects (with suitable morphisms..)*)
constdefs unitEffect ::"(unit, 'b) Effect"
"unitEffect == \<lparr>Int_effect = \<lambda> c . (),
                IVar_effect = \<lambda> c . (),
                Primop_effect = \<lambda> c . (),
                Null_effect = \<lambda> c . (),
                RVar_effect = \<lambda> c . (),
                RPrimop_effect = \<lambda> c . (),
                New_effect = \<lambda> c . (),
                GetFi_effect = \<lambda> c . (),
                GetFr_effect = \<lambda> c . (),
                PutFi_effect = \<lambda> c . (),
                PutFr_effect = \<lambda> c . (),
                GetStat_effect = \<lambda> c . (),
                PutStat_effect = \<lambda> c . (),
                InvV_effect = \<lambda> c a . (),
                InvS_effect = \<lambda> c a . (),
                Leti_effect = \<lambda> c a1 a2. (),
                Letr_effect = \<lambda> c a1 a2. (),
                Letv_effect = \<lambda> c a1 a2. (),
                If_effect = \<lambda> c a . (),
                Call_effect = \<lambda> c a . ()\<rparr>"

constdefs RC::"rescomp EFF"
"RC == \<lparr>Int_effect = \<lambda> c . tickRo,
                IVar_effect = \<lambda> c . tickRo,
                Primop_effect = \<lambda> c . \<langle>3 0 0 0\<rangle>,
                Null_effect = \<lambda> c . tickRo,
                RVar_effect = \<lambda> c . tickRo,
                RPrimop_effect = \<lambda> c . \<langle>3 0 0 0\<rangle>,
                New_effect = \<lambda> c . tickRo, (*number of args?*)
                GetFi_effect = \<lambda> c . \<langle>2 0 0 0\<rangle>,
                GetFr_effect = \<lambda> c . \<langle>2 0 0 0\<rangle>,
                PutFi_effect = \<lambda> c . \<langle>3 0 0 0\<rangle>,
                PutFr_effect = \<lambda> c . \<langle>3 0 0 0\<rangle>,
                GetStat_effect = \<lambda> c . \<langle>2 0 0 0\<rangle>,
                PutStat_effect = \<lambda> c . \<langle>2 0 0 0\<rangle>,
                InvV_effect = \<lambda> c a . \<langle>5 0 1 1\<rangle>,
                InvS_effect = \<lambda> c a . \<langle>3 0 1 1\<rangle>,
                Leti_effect = \<lambda> c a1 a2. tk (a1 \<smile> a2),
                Letr_effect = \<lambda> c a1 a2. tk (a1 \<smile> a2),
                Letv_effect = \<lambda> c a1 a2. (a1 \<smile> a2),
                If_effect = \<lambda> c a . \<langle>2 0 0 0\<rangle>,
                Call_effect = \<lambda> c a . \<langle>1 1 0 0\<rangle>\<rparr>"

end
(*>*)
