(* 				 
   File:	$RCSfile: Semantics.thy,v $
   Authors:	David Aspinall, Lennart Beringer, Hans-Wolfgang Loidl
   Id:		$Id: Semantics.thy,v 1.1 2004/07/07 11:46:48 lenb Exp $

   Functional semantics for a (restricted version of) Grail.
*)

header {* Operational semantics *}
(*<*)
theory Semantics = Prelude:
(*>*)
text {*
  We define a big-step evaluation semantics.  For the purpose of later
  proofs, the relation is also indexed by the height of the derivation.
  *}

subsection{* Resource environments and abbreviations *}

text {* 
We define resource tuples to model resource consumption during the execution.
Currently, a resource tuple is a record with 4 components, counting execution time, 
function and method calls, and measuring maximal depth of method invocations.
*}

(*Changed from nats to ints*)
record rescomp = clock :: int        -- {* execution time *}
                 callc :: int        -- {* call counter *}
                 invkc :: int        -- {* invoke counter *}
                 invkdpth :: nat     -- {* max invoke depth *}

text {*
To add up resource consumption, we use two combinators over resource tuples.
*}

translations 
 "rescomp" <= (type) "\<lparr> clock :: int, callc :: int, invkc :: int, invkdpth :: nat \<rparr>"

constdefs rescomp_plus:: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp"  (infixr "\<oplus>" 0)
"rescomp_plus p1 p2 == \<lparr>clock = clock p1 + clock p2,
                     callc = callc p1 + callc p2,
                     invkc = invkc p1 + invkc p2,
                     invkdpth = invkdpth p1 + invkdpth p2\<rparr>"

constdefs rescomp_cup :: "rescomp \<Rightarrow> rescomp \<Rightarrow> rescomp" (infixr "\<smile>" 0)
"rescomp_cup p1 p2 == \<lparr>clock = clock p1 + clock p2,
                    callc = callc p1 + callc p2,
                    invkc = invkc p1 + invkc p2,
                    invkdpth = max (invkdpth p1) (invkdpth p2)\<rparr>" 
(*<*)
declare rescomp_plus_def [simp add]
declare rescomp_cup_def [simp add]
(*>*)

text {* Some syntax for creating and manipulating resource tuples. *}

syntax mkRescomp:: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp"
translations
"mkRescomp C CC IC ID" == "\<lparr>clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax mkRescomp_HWL :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> rescomp" ("\<langle>_ _ _ _\<rangle>")
translations
"\<langle>C CC IC ID\<rangle>" == "\<lparr>rescomp.clock = C, callc = CC, invkc = IC, invkdpth = ID\<rparr>"

syntax tkn ::"int \<Rightarrow> rescomp \<Rightarrow> rescomp"
translations
"tkn n p" == "((mkRescomp n 0 0 0) \<smile> p)"

syntax tk::"rescomp \<Rightarrow> rescomp"
translations
"tk p" == "tkn 1 p"

syntax tkcall::"rescomp \<Rightarrow> rescomp"
translations
"tkcall p" == "((mkRescomp 1 1 0 0) \<smile> p)"

syntax tickRo::"rescomp"
translations
"tickRo" == "mkRescomp 1 0 0 0"

subsection {* Operational semantics *}

(*expressions have at most two subexpressions, so their effect has two arguments*)
axclass Effect_C < type

(*arities
  nat :: Effect_C
  bool :: Effect_C
  * :: (Effect_C,Effect_C) Effect_C
*)

consts Int_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  IVar_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  Primop_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  Null_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  RVar_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  RPrimop_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  New_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  GetFi_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  GetFr_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  PutFi_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  PutFr_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  GetStat_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  PutStat_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C)"
  Invoke_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  InvokeStatic_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  Leti_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  Letr_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  Letv_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  Ifg_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"
  Call_effect:: "env \<Rightarrow> heap \<Rightarrow> ('a::Effect_C) \<Rightarrow> ('a::Effect_C)"

instance * :: (Effect_C,Effect_C) Effect_C by (intro_classes)

(*now a class where one of the operations satisifies an axiom*)
axclass Effect < Effect_C
  Call_effectInj: "(Call_effect E h p = Call_effect E h q) = (p=q)"

instance bool::Effect_C by (intro_classes)
instance nat::Effect_C by (intro_classes)
instance int::Effect_C by (intro_classes)

defs (overloaded)
Call_effect_def_bool: "Call_effect E h (b::bool) == b"
Int_effect_def_bool: "(Int_effect::(env \<Rightarrow> heap \<Rightarrow> bool)) E h == True"

instance bool::Effect by (intro_classes, simp add: Call_effect_def_bool)

defs (overloaded)
Call_effect_def_nat: "Call_effect E h n == (Suc n)"
Int_effect_def_nat: "(Int_effect::(env \<Rightarrow> heap \<Rightarrow> nat)) E h == 1"

instance nat::Effect by (intro_classes, simp add: Call_effect_def_nat)

axclass C1 < Effect_C

instance int::C1 by (intro_classes)

defs (overloaded)
  Call_effect_def_C1:"Call_effect E h (n::C1) == n"

defs (overloaded)
Call_effect_def_natbool: "Call_effect E h (nb::nat \<times> bool) == (Call_effect E h (fst nb), Call_effect E h (snd nb))"
Int_effect_def_natbool: "(Int_effect::(env \<Rightarrow> heap \<Rightarrow> (nat \<times> bool))) E h == ((Int_effect::(env \<Rightarrow> heap \<Rightarrow> nat)) E h, 
                                                                             (Int_effect::(env \<Rightarrow> heap \<Rightarrow> bool)) E h)"


(*defs (overloaded)
Call_effect_def_natbool: "Call_effect E h (nb::nat \<times> bool) == (Suc (fst nb), \<not> (snd nb))"
Int_effect_def_natbool: "(Int_effect::(env \<Rightarrow> heap \<Rightarrow> (nat \<times> bool))) E h == (Suc 0, True)"
*)

instance "*" :: (Effect,Effect) Effect
apply (intro_classes, simp add: Call_effect_def_prod)
apply (simp add: Call_effectInj)

instance "'a \<times> 'b" ::Effect

types NB = "nat \<times> bool"
(*instance "nat \<times> bool"::Effect*)

consts Int_effect_fst::"env \<Rightarrow> heap \<Rightarrow> 'a::Effect"
consts Int_effect_snd::"env \<Rightarrow> heap \<Rightarrow> 'a::Effect"
consts Call_effect_fst::"env \<Rightarrow> heap \<Rightarrow> 'a::Effect \<Rightarrow> 'a"
consts Call_effect_snd::"env \<Rightarrow> heap \<Rightarrow> 'a::Effect \<Rightarrow> 'a"

defs (overloaded)
Call_effect_def_prod: "Call_effect E h p == (Call_effect E h (fst p), Call_effect E h (snd p))"
Int_effect_def_prod: "Int_effect E h == (Int_effect_fst E h, Int_effect_snd E h)"

instance "*" :: (Effect,Effect) Effect
apply (intro_classes, simp add: Call_effect_def_prod)
apply (simp add: Call_effectInj)

text {*
We define the operational semantics as an inductively defined set, 
connecting, for a given expression, all possible pre-states with its post-states.
Note that we define the semantics with a counter of derivation height to use thiscounter later on in the soundness proof to do induction over. The counter is \emph{not}
used to capture resource consumption (we have a separate field in the resource tuple 
for that). 
*}

consts semn :: "(env \<times> heap \<times> expr \<times> nat \<times> heap \<times> val \<times> ('a::Effect)) set" -- {* operational semantics *}

syntax semn_  :: "[env, heap, expr, nat, heap, val, 'a::Effect] \<Rightarrow> bool"   ("_ \<turnstile> _ , _ \<Down>_ '(_ , _ , _')")

translations "E \<turnstile> h,e \<Down>n (hh,v,p)" == "(E,h,e,n,hh,v,p) : semn"

inductive semn intros
 semNull:    "E \<turnstile> h,(expr.Null) \<Down>1 (h,RVal Nullref, Null_effect E h)"

 semInt:     "E \<turnstile> h,(expr.Int i) \<Down>1 (h,IVal i, Int_effect E h)"

 semIVar:    "E \<turnstile> h,IVar v \<Down>1 (h,IVal (E<v>), IVar_effect E h)"

 semRVar:    "E \<turnstile> h,RVar v \<Down>1 (h,RVal (E\<lfloor>v\<rfloor>), RVar_effect E h)"

 semPrimop:  "E \<turnstile> h, (Primop f x y) \<Down>1 (h,IVal (f (E<x>) (E<y>)), Primop_effect E h)"

 semRPrimop: "E \<turnstile> h, (RPrimop f x y) \<Down>1 (h,IVal (f (E\<lfloor>x\<rfloor>) (E\<lfloor>y\<rfloor>)), RPrimop_effect E h)"

 semGetfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> E \<turnstile> h,(GetFi x f) \<Down>1 (h, IVal (h<a\<bullet>f>), GetFi_effect E h)"

 semGetfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> E \<turnstile> h,(GetFr x f) \<Down>1 (h, RVal (h\<lfloor>a\<diamondsuit>f\<rfloor>), GetFr_effect E h)"

 semPutfi: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> E \<turnstile> h, (PutFi x f y) \<Down>1 (h<a\<bullet>f:=E<y>>, arbitrary, PutFi_effect E h)"

 semPutfr: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a; a:Dom h\<rbrakk> \<Longrightarrow> E \<turnstile> h, (PutFr x f y) \<Down>1 (h\<lfloor>a\<diamondsuit>f:=E\<lfloor>y\<rfloor>\<rfloor>, arbitrary, PutFr_effect E h)"

 semGetstat: "E \<turnstile> h,(GetStat c f) \<Down>1 (h, RVal (h\<lbrace>c\<struct>f\<rbrace>), GetStat_effect E h)"

 semPutstat: "E \<turnstile> h, (PutStat c f y) \<Down>1 (h\<lbrace>c\<struct>f:=E\<lfloor>y\<rfloor>\<rbrace>, arbitrary, PutStat_effect E h)"

 semNew: "\<lbrakk> l = freshloc (Dom h)\<rbrakk> \<Longrightarrow>
          E \<turnstile> h,(New c ifldvals rfldvals) \<Down>1 (newObj h l E c ifldvals rfldvals, RVal (Ref l), New_effect E h)"

 semIf_True:  "\<lbrakk>E<x> = grailbool True; E \<turnstile> h, e1 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
               E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, Ifg_effect E h p)"

 semIf_False:  "\<lbrakk>E<x> = grailbool False; E \<turnstile> h, e2 \<Down>m (h1,v,p)\<rbrakk> \<Longrightarrow>
                E \<turnstile> h, (Ifg x e1 e2) \<Down>(m + 1) (h1,v, Ifg_effect E h p)"

 semLeti: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,IVal i,p1);  E<x:=i> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Leti x e1 e2) \<Down>((max n m) + 1) (h2,v, Leti_effect E h p1 p2)"

 semLetr: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,RVal r,p1);  E\<lfloor>x:=r\<rfloor> \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letr x e1 e2) \<Down>((max n m) + 1) (h2,v, Letr_effect E h p1 p2)"

 semLetv: "\<lbrakk>E \<turnstile> h,e1 \<Down>n (h1,w,p1);  E \<turnstile> h1, e2 \<Down>m (h2,v,p2)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h, (Letv e1 e2) \<Down>(max n m) (h2,v, Letv_effect E h p1 p2)"

 semCall: "\<lbrakk>E \<turnstile> h, snd(funtable f) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
           E \<turnstile> h,(Call f) \<Down>(n+1) (h1, v,  Call_effect E h p)"

 semInvoke: "\<lbrakk>E\<lfloor>x\<rfloor> = Ref a;  
              h@@a = Some C; 
              EE = newframe_env (Ref a) (fst (methtable C mn)) args E;
              EE \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
             E \<turnstile> h, x\<diamondsuit>mn(args) \<Down>(n + 1) (h1,v, Invoke_effect E h p)"

 semInvokeStatic: "\<lbrakk>EE = (newframe_env Nullref (fst (methtable C mn)) args E);
                    EE \<turnstile> h,(snd (methtable C mn)) \<Down>n (h1,v,p)\<rbrakk> \<Longrightarrow>
                   E \<turnstile> h, C\<bullet>mn(args) \<Down>(n + 1) (h1,v, InvokeStatic_effect E h p)"

text {* Operational semantics without height index *}

constdefs
 sem :: "[env, heap, expr, heap, val, 'a::Effect] \<Rightarrow> bool"  ("_ \<turnstile> _ , _ \<Down> _ , _ , _" 1000)
 "E \<turnstile> h,e \<Down> hh,v,p \<equiv> (\<exists> n. (E \<turnstile> h,e \<Down>n (hh,v,p)))"

lemma "E \<turnstile> h,(expr.Int i) \<Down> hh,v,p \<Longrightarrow> (p = Suc 0)"
apply (simp add: sem_def, clarsimp)
apply (erule semn.elims, simp_all, clarsimp)
apply (simp add: Int_effect_def_nat)
done
lemma "E \<turnstile> h,(expr.Int i) \<Down> hh,v,p \<Longrightarrow> (p = True)"
apply (simp add: sem_def, clarsimp)
apply (erule semn.elims, simp_all, clarsimp)
apply (simp add: Int_effect_def_bool)
done

defs (overloaded)
F_def: "F E h == Suc 0"

consts H::"env \<Rightarrow> heap \<Rightarrow> 'a::Effect"

defs (overloaded)
G_def: "G E h == (0::nat, H E h)"

defs (overloaded)
H_def: "H E h == tickRo"



lemma "E \<turnstile> h,(expr.Int i) \<Down> hh,v,p \<Longrightarrow> ((fst p) = Suc 0 \<and> (snd p) = (0::nat, True))"
apply (simp add: sem_def, clarsimp)
apply (erule semn.elims, simp_all, clarsimp)
apply (simp add: Int_effect_def_prod)
apply (simp add: F_def G_def H_def)
(*apply (simp add: Int_effect_def_nat Int_effect_def_bool)*)
done
lemma "E \<turnstile> h,(expr.Int i) \<Down> hh,v,p \<Longrightarrow> ((snd p) = Suc 0 \<and> (fst p) = True)"
apply (simp add: sem_def, clarsimp)
apply (erule semn.elims, simp_all, clarsimp)
apply (simp add: Int_effect_def_prod)
apply (simp add: Int_effect_def_nat Int_effect_def_bool)
done

defs 

text {*
  Comentary:
  \begin{enumerate}
  \item InvokeStatic: static condition: method body does not refer to ''self''.
  \item Putfield instructions (rules @{text evalPutFi} and @{text evalPutFr}) are only
     evaluated  for their side-effect and should only be used with @{text Letv}.  
     To emphasise this, they return the undetermined value @{text arbitrary}.    
  \item To explain the resource counting in Invoke: vn1 = 1, vn2 = 1,
    frame push/pop = 2, and the implicit return = 1.
    NB: we could break up the increment of ticks into before/after the invocations.
  \item Many rules omit type-checking premises.
    For example, in get field and put field rules, we should
    check that the object belongs to a class that indeed has
    the fields being accessed.  
  \item Object initialization in the New ruleallocates a fresh location,
    etends the oheap component of the heap by the class entry for the new object
    and initialises the fields.
  \item  We might tickn 2 in the If rules, but we leave it at one to
     allow for extra store/retrieve which may not be necessary in 
     some cases on the real machine.
  \item The rule for evaluating Let once contained an additional side
    condition that the expression being bound does not itself contain a
    CALL (non-tail position).  So evaluation would get stuck for non-Grail
    programs.  However, there seems to be no harm in considering the
    more general semantics for a slightly richer language, since this
    side condition is not used in the logic.  (This restriction could
    therefore be considered as part of the type-checking phase for Grail).
  \end{enumerate}
*}

subsection {* Basic properties *}
text {*We also provide elimination lemmas for the inductively defined relation*}
inductive_cases eval_cases: 
 "E \<turnstile> h,(expr.Null) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(expr.Int i) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(IVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RVar x) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Primop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(RPrimop f x y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFi x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetFr x f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFi x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutFr x f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(GetStat c f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(PutStat c f y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(New c ifldvals rfldvals) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Ifg x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Leti x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letr x e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Letv e1 e2) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Call f) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(Invoke x m y) \<Down>n  (h1,v,p)"
 "E \<turnstile> h,(InvokeStatic C m y) \<Down>n  (h1,v,p)"
(*<*)
lemma no_zero_height_derivsAux: "\<forall> E h h1 v p . (E \<turnstile> h, e \<Down>0  (h1,v,p) \<longrightarrow> False)"
apply (induct_tac e)
apply (simp_all)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply(clarsimp, elim eval_cases, simp)
apply (subgoal_tac "n=0 & m=0")
apply clarsimp
apply (simp add: max_def)
apply (case_tac "n \<le> m", auto)
apply(erule semn.elims, simp_all)
apply(erule semn.elims, simp_all)
done
(*>*)

text {*and observe that no derivations of height 0 exist.*}
lemma no_zero_height_derivs: "E \<turnstile> h, e \<Down>0  (h1,v,p) \<Longrightarrow> False"
(*<*)by (insert no_zero_height_derivsAux, fastsimp)(*>*)


lemma "E \<turnstile> h,e \<Down> hh,v,p \<Longrightarrow> 
done
(*<*)
end
(*>*)
